use std::fs::{copy, create_dir_all, remove_dir_all, File};
use std::io::Write;
use std::path::PathBuf;
use std::process::Command;

use rust_hdl_core::prelude::*;
use rust_hdl_ok_core::prelude::find_ok_bus_collisions;
use rust_hdl_toolchain_vivado::xdc_gen::generate_xdc;

#[derive(Clone, Debug)]
pub struct VivadoOptions {
    pub vivado_path: String,
    pub add_mig: bool,
    pub assets: Vec<String>,
}

const VIVADO_PATH: &str = "/opt/Xilinx/Vivado/2018.1/bin/";
const FP_PATH: &str = "/opt/FrontPanel-Ubuntu16.04LTS-x64-5.2.0/FrontPanelHDL/XEM7010-A50";

impl Default for VivadoOptions {
    fn default() -> Self {
        Self {
            vivado_path: VIVADO_PATH.into(),
            add_mig: true,
            assets: [
                "okLibrary.v",
                "okCoreHarness.v",
                "okWireIn.v",
                "okWireOut.v",
                "okTriggerIn.v",
                "okTriggerOut.v",
                "okPipeIn.v",
                "okPipeOut.v",
                "okBTPipeIn.v",
                "okBTPipeOut.v",
            ]
            .iter()
            .map(|x| format!("{}/{}", FP_PATH, x))
            .collect(),
        }
    }
}

pub fn add_mig_core_xem_7010(prefix: &str, _options: VivadoOptions) -> String {
    let mig_path = PathBuf::from(prefix).join("mig_a.prj");
    std::fs::write(&mig_path,
r##"<?xml version='1.0' encoding='UTF-8'?>
<!-- IMPORTANT: This is an internal file that has been generated by the MIG software. Any direct editing or changes made to this file may result in unpredictable behavior or data corruption. It is strongly advised that users do not edit the contents of this file. Re-run the MIG GUI with the required settings if any of the options provided below need to be altered. -->
<Project NoOfControllers="1" >
    <ModuleName>mig7</ModuleName>
    <dci_inouts_inputs>1</dci_inouts_inputs>
    <dci_inputs>1</dci_inputs>
    <Debug_En>OFF</Debug_En>
    <DataDepth_En>1024</DataDepth_En>
    <LowPower_En>ON</LowPower_En>
    <XADC_En>Enabled</XADC_En>
    <TargetFPGA>xc7a50t-fgg484/-1</TargetFPGA>
    <Version>4.1</Version>
    <SystemClock>Differential</SystemClock>
    <ReferenceClock>Use System Clock</ReferenceClock>
    <SysResetPolarity>ACTIVE HIGH</SysResetPolarity>
    <BankSelectionFlag>FALSE</BankSelectionFlag>
    <InternalVref>1</InternalVref>
    <dci_hr_inouts_inputs>60 Ohms</dci_hr_inouts_inputs>
    <dci_cascade>0</dci_cascade>
    <Controller number="0" >
        <MemoryDevice>DDR3_SDRAM/Components/MT41K256M16XX-125</MemoryDevice>
        <TimePeriod>2500</TimePeriod>
        <VccAuxIO>1.8V</VccAuxIO>
        <PHYRatio>4:1</PHYRatio>
        <InputClkFreq>200</InputClkFreq>
        <UIExtraClocks>0</UIExtraClocks>
        <MMCM_VCO>800</MMCM_VCO>
        <MMCMClkOut0> 1.000</MMCMClkOut0>
        <MMCMClkOut1>1</MMCMClkOut1>
        <MMCMClkOut2>1</MMCMClkOut2>
        <MMCMClkOut3>1</MMCMClkOut3>
        <MMCMClkOut4>1</MMCMClkOut4>
        <DataWidth>16</DataWidth>
        <DeepMemory>1</DeepMemory>
        <DataMask>1</DataMask>
        <ECC>Disabled</ECC>
        <Ordering>Normal</Ordering>
        <BankMachineCnt>4</BankMachineCnt>
        <CustomPart>FALSE</CustomPart>
        <NewPartName></NewPartName>
        <RowAddress>15</RowAddress>
        <ColAddress>10</ColAddress>
        <BankAddress>3</BankAddress>
        <MemoryVoltage>1.5V</MemoryVoltage>
        <UserMemoryAddressMap>BANK_ROW_COLUMN</UserMemoryAddressMap>
        <PinSelection>
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="W6" SLEW="" name="ddr3_addr[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="T4" SLEW="" name="ddr3_addr[10]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="V7" SLEW="" name="ddr3_addr[11]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="T6" SLEW="" name="ddr3_addr[12]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="Y9" SLEW="" name="ddr3_addr[13]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="W9" SLEW="" name="ddr3_addr[14]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="U7" SLEW="" name="ddr3_addr[1]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="W7" SLEW="" name="ddr3_addr[2]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="Y6" SLEW="" name="ddr3_addr[3]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="U6" SLEW="" name="ddr3_addr[4]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AB7" SLEW="" name="ddr3_addr[5]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="Y8" SLEW="" name="ddr3_addr[6]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AB8" SLEW="" name="ddr3_addr[7]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="Y7" SLEW="" name="ddr3_addr[8]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AA8" SLEW="" name="ddr3_addr[9]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AB6" SLEW="" name="ddr3_ba[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="R6" SLEW="" name="ddr3_ba[1]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AA6" SLEW="" name="ddr3_ba[2]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="U5" SLEW="" name="ddr3_cas_n" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="DIFF_SSTL15" PADName="V8" SLEW="" name="ddr3_ck_n[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="DIFF_SSTL15" PADName="V9" SLEW="" name="ddr3_ck_p[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="R4" SLEW="" name="ddr3_cke[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AA1" SLEW="" name="ddr3_dm[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="V2" SLEW="" name="ddr3_dm[1]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AB1" SLEW="" name="ddr3_dq[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="U2" SLEW="" name="ddr3_dq[10]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="Y2" SLEW="" name="ddr3_dq[11]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="U1" SLEW="" name="ddr3_dq[12]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="Y1" SLEW="" name="ddr3_dq[13]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="T1" SLEW="" name="ddr3_dq[14]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="W1" SLEW="" name="ddr3_dq[15]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="Y4" SLEW="" name="ddr3_dq[1]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AB2" SLEW="" name="ddr3_dq[2]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="V4" SLEW="" name="ddr3_dq[3]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AB5" SLEW="" name="ddr3_dq[4]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AA5" SLEW="" name="ddr3_dq[5]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AB3" SLEW="" name="ddr3_dq[6]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="AA4" SLEW="" name="ddr3_dq[7]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="U3" SLEW="" name="ddr3_dq[8]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="W2" SLEW="" name="ddr3_dq[9]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="DIFF_SSTL15" PADName="AA3" SLEW="" name="ddr3_dqs_n[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="DIFF_SSTL15" PADName="R2" SLEW="" name="ddr3_dqs_n[1]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="DIFF_SSTL15" PADName="Y3" SLEW="" name="ddr3_dqs_p[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="DIFF_SSTL15" PADName="R3" SLEW="" name="ddr3_dqs_p[1]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="W5" SLEW="" name="ddr3_odt[0]" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="V5" SLEW="" name="ddr3_ras_n" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="LVCMOS15" PADName="T3" SLEW="" name="ddr3_reset_n" IN_TERM="" />
            <Pin VCCAUX_IO="" IOSTANDARD="SSTL15" PADName="T5" SLEW="" name="ddr3_we_n" IN_TERM="" />
        </PinSelection>
        <System_Clock>
            <Pin PADName="K4/J4(CC_P/N)" Bank="35" name="sys_clk_p/n" />
        </System_Clock>
        <System_Control>
            <Pin PADName="No connect" Bank="Select Bank" name="sys_rst" />
            <Pin PADName="No connect" Bank="Select Bank" name="init_calib_complete" />
            <Pin PADName="No connect" Bank="Select Bank" name="tg_compare_error" />
        </System_Control>
        <TimingParameters>
            <Parameters twtr="7.5" trrd="7.5" trefi="7.8" tfaw="40" trtp="7.5" tcke="5" trfc="260" trp="13.75" tras="35" trcd="13.75" />
        </TimingParameters>
        <mrBurstLength name="Burst Length" >8 - Fixed</mrBurstLength>
        <mrBurstType name="Read Burst Type and Length" >Sequential</mrBurstType>
        <mrCasLatency name="CAS Latency" >6</mrCasLatency>
        <mrMode name="Mode" >Normal</mrMode>
        <mrDllReset name="DLL Reset" >No</mrDllReset>
        <mrPdMode name="DLL control for precharge PD" >Slow Exit</mrPdMode>
        <emrDllEnable name="DLL Enable" >Enable</emrDllEnable>
        <emrOutputDriveStrength name="Output Driver Impedance Control" >RZQ/7</emrOutputDriveStrength>
        <emrMirrorSelection name="Address Mirroring" >Disable</emrMirrorSelection>
        <emrCSSelection name="Controller Chip Select Pin" >Disable</emrCSSelection>
        <emrRTT name="RTT (nominal) - On Die Termination (ODT)" >RZQ/6</emrRTT>
        <emrPosted name="Additive Latency (AL)" >0</emrPosted>
        <emrOCD name="Write Leveling Enable" >Disabled</emrOCD>
        <emrDQS name="TDQS enable" >Enabled</emrDQS>
        <emrRDQS name="Qoff" >Output Buffer Enabled</emrRDQS>
        <mr2PartialArraySelfRefresh name="Partial-Array Self Refresh" >Full Array</mr2PartialArraySelfRefresh>
        <mr2CasWriteLatency name="CAS write latency" >5</mr2CasWriteLatency>
        <mr2AutoSelfRefresh name="Auto Self Refresh" >Enabled</mr2AutoSelfRefresh>
        <mr2SelfRefreshTempRange name="High Temparature Self Refresh Rate" >Normal</mr2SelfRefreshTempRange>
        <mr2RTTWR name="RTT_WR - Dynamic On Die Termination (ODT)" >Dynamic ODT off</mr2RTTWR>
        <PortInterface>NATIVE</PortInterface>
    </Controller>
</Project>
"##).unwrap();
    format!("create_ip  -vlnv xilinx.com:ip:mig_7series:4.1 -module_name mig7
set_property -dict [list CONFIG.XML_INPUT_FILE {{ {mig_path} }} CONFIG.RESET_BOARD_INTERFACE {{Custom}} CONFIG.MIG_DONT_TOUCH_PARAM {{Custom}} CONFIG.BOARD_MIG_PARAM {{Custom}}] [get_ips mig7]
generate_target {{instantiation_template}} [get_files mig7.xci]", mig_path=mig_path.canonicalize().unwrap().to_string_lossy())
}

pub fn generate_bitstream_xem_7010<U: Block>(mut uut: U, prefix: &str, options: VivadoOptions) {
    uut.connect_all();
    check_connected(&uut);
    let verilog_text = filter_blackbox_directives(&generate_verilog(&uut));
    let xdc_text = generate_xdc(&uut);
    let dir = PathBuf::from(prefix);
    let out_file = dir.join("top.out");
    if out_file.exists() {
        if String::from_utf8_lossy(&std::fs::read(out_file).unwrap())
            .contains("Vivado Run Complete")
        {
            println!("Skipped synthesis!  Bitfile should exist");
            return;
        }
    }
    let _ = remove_dir_all(&dir);
    let _ = create_dir_all(&dir);
    let assets: Vec<String> = options.assets.clone();
    std::fs::write(dir.clone().join("top.v"), verilog_text).unwrap();
    std::fs::write(dir.clone().join("top.xdc"), xdc_text).unwrap();
    for asset in &assets {
        let src = PathBuf::from(asset);
        let dest = dir.clone().join(src.file_name().unwrap());
        println!("Copy from {:?} -> {:?}", asset, dest);
        copy(asset, dest).unwrap();
    }
    let mig = if options.add_mig {
        add_mig_core_xem_7010(prefix, options.clone())
    } else {
        "".to_string()
    };
    let mut tcl_file = File::create(dir.clone().join("top.tcl")).unwrap();
    write!(
        tcl_file,
        r#"
create_project top . -part xc7a50tfgg484-1 -force

add_files {{top.v top.xdc {assets} }}

{mig}

update_compile_order

launch_runs synth_1 -jobs 8
wait_on_run synth_1

set status [ get_property STATUS [ get_runs synth_1 ] ]
if {{ $status != "synth_design Complete!" }} {{
 puts "Synthesis Failed"
 exit
}}

launch_runs impl_1 -to_step write_bitstream -jobs 8
wait_on_run impl_1

set status [ get_property STATUS [ get_runs impl_1 ] ]
if {{ $status != "write_bitstream Complete!" }} {{
 puts "Implementation Failed"
 exit
}}

puts "Vivado Run Complete"
exit
"#,
        assets = assets
            .iter()
            .map(|x| PathBuf::from(x)
                .file_name()
                .unwrap()
                .to_string_lossy()
                .to_string())
            .collect::<Vec<_>>()
            .join(" "),
        mig = mig
    )
    .unwrap();
    let output = Command::new(format!("{}/vivado", options.vivado_path))
        .current_dir(dir.clone())
        .arg("-mode")
        .arg("tcl")
        .arg("-source")
        .arg("top.tcl")
        .output()
        .unwrap();
    let stdout = String::from_utf8(output.stdout).unwrap();
    let stderr = String::from_utf8(output.stderr).unwrap();
    std::fs::write(dir.clone().join("top.out"), &stdout).unwrap();
    std::fs::write(dir.clone().join("top.err"), &stderr).unwrap();
    assert!(stdout.contains("Vivado Run Complete"));
    copy(
        dir.clone().join("top.runs/impl_1/top.bit"),
        dir.clone().join("top.bit"),
    )
    .unwrap();
}

pub fn synth_obj<U: Block>(uut: U, dir: &str) {
    check_connected(&uut);
    let vlog = generate_verilog(&uut);
    find_ok_bus_collisions(&vlog);
    let _xcd = rust_hdl_toolchain_vivado::xdc_gen::generate_xdc(&uut);
    rust_hdl_yosys_synth::yosys_validate(dir, &vlog).unwrap();
    generate_bitstream_xem_7010(uut, dir, Default::default());
}
