use super::super::context::Context;
use super::Shader;
use web_sys::WebGl2RenderingContext;

pub type VertexShader = Shader<{ WebGl2RenderingContext::VERTEX_SHADER }>;
pub type FragmentShader = Shader<{ WebGl2RenderingContext::FRAGMENT_SHADER }>;

///Builds a webgl program given a context, vertex- and fragment-shader.
pub struct ProgramBuilder<'context, 'vertex, 'fragment, U = (), A = ()> {
    context: &'context Context,
    vertex_shader: &'vertex VertexShader,
    fragment_shader: &'fragment FragmentShader,
    uniforms: Option<U>,
    attributes: Option<A>,
}

impl<'context, 'vertex, 'fragment> Context {
    ///Constructs a program builder.
    pub fn program<U, A>(
        &'context self,
        vertex_shader: &'vertex VertexShader,
        fragment_shader: &'fragment FragmentShader,
    ) -> ProgramBuilder<'context, 'vertex, 'fragment, U, A> {
        ProgramBuilder {
            context: self,
            vertex_shader,
            fragment_shader,
            uniforms: None,
            attributes: None,
        }
    }
}

#[derive(Debug, PartialEq, Eq)]
pub enum Error<U, A> {
    Compile(String),
    CompileStatus,
    CreateProgram,
    Attributes,
    ToAttribute(A),
    Uniforms,
    ToUniform(U),
    Missing {
        uniforms: Vec<String>,
        attributes: Vec<String>,
    },
}

use std::error;
use std::fmt;
impl<U, A> fmt::Display for Error<U, A>
where
    U: error::Error,
    A: error::Error,
{
    fn fmt(&self, fmt: &mut fmt::Formatter<'_>) -> Result<(), fmt::Error> {
        match self {
            Error::Compile(x) => write! {fmt, "{}", x},
            Error::CompileStatus => write! {fmt, "Could not get compile status"},
            Error::CreateProgram => write! {fmt, "Could not create program"},
            Error::Attributes => write! {fmt, "Attributes not set."},
            Error::ToAttribute(x) => write! {fmt, "{}", x},
            Error::Uniforms => write! {fmt, "Uniforms not set."},
            Error::ToUniform(x) => write! {fmt, "{}", x},
            Error::Missing {
                uniforms,
                attributes,
            } => {
                write! {fmt, "Missing uniforms : {:?}, missing attributes : {:?}", uniforms, attributes}
            }
        }
    }
}

impl<U, A> error::Error for Error<U, A>
where
    U: error::Error,
    A: error::Error,
{
}
use super::super::attribute::ToAttribute;
use super::super::uniform::ToUniform;
use super::Program;

impl<'context, 'vertex, 'fragment, U, A> ProgramBuilder<'context, 'vertex, 'fragment, U, A> {
    pub fn attributes(mut self, attributes: A) -> Self {
        self.attributes = Some(attributes);
        self
    }
    pub fn uniforms(mut self, uniforms: U) -> Self {
        self.uniforms = Some(uniforms);
        self
    }

    ///Builds a webgl program from a program builder.
    pub fn build<'program, Uo, Ue, Ao, Ae>(self) -> Result<Program<Uo, Ao>, Error<Ue, Ae>>
    where
        A: ToAttribute<Output = Ao, Error = Ae>,
        U: ToUniform<Output = Uo, Error = Ue>,
    {
        let ProgramBuilder {
            context,
            vertex_shader,
            fragment_shader,
            uniforms,
            attributes,
        } = self;
        let program = context.create_program().ok_or(Error::CreateProgram)?;

        context.attach_shader(&program, &*vertex_shader);
        context.attach_shader(&program, &*fragment_shader);
        context.link_program(&program);
        context.validate_program(&program);

        let log = context
            .get_program_info_log(&program)
            .ok_or(Error::CompileStatus)?;

        if !log.is_empty() {
            Err(Error::Compile(log))?
        }

        context.use_program(Some(&program));

        let num_uniforms = context
            .get_program_parameter(&program, WebGl2RenderingContext::ACTIVE_UNIFORMS)
            .as_f64()
            .unwrap() as u32;
        let mut map_uniforms = (0..num_uniforms)
            .flat_map(|index| {
                context
                    .get_active_uniform(&program, index)
                    .map(|x| (x.name().clone(), x))
            })
            .collect();

        let uniforms = uniforms
            .ok_or(Error::Uniforms)?
            .to_uniform(context.gl.clone(), &program, None, &mut map_uniforms)
            .map_err(Error::ToUniform)?;

        let num_attributes = context
            .get_program_parameter(&program, WebGl2RenderingContext::ACTIVE_ATTRIBUTES)
            .as_f64()
            .unwrap() as u32;
        let mut map_attributes = (0..num_attributes)
            .flat_map(|index| {
                context
                    .get_active_attrib(&program, index)
                    .map(|x| (x.name().clone(), x))
            })
            .collect();

        let attributes = attributes
            .ok_or(Error::Attributes)?
            .to_attribute(context.gl.clone(), &program, "", &mut map_attributes)
            .map_err(Error::ToAttribute)?;

        if !(map_uniforms.is_empty() && map_attributes.is_empty()) {
            Err(Error::Missing {
                uniforms: map_uniforms.into_keys().collect(),
                attributes: map_attributes.into_keys().collect(),
            })
        } else {
            let program = Program {
                program,
                uniforms,
                attributes,
            };

            Ok(program)
        }
    }
}
