use super::*;
use std::rc::Rc;
use web_sys::WebGlProgram;

#[derive(Debug)]
pub struct Base<T> {
    index: Index,
    name: String,
    gl: Rc<GL>,
    constant: T,
}

impl<T> Attribute<T> for Base<T>
where
    T: GLType,
{
    fn index(&self) -> &Index {
        &self.index
    }
    fn constant(&mut self, mut constant: T) -> T {
        use core::mem::swap;
        swap(&mut self.constant, &mut constant);
        self.constant.constant(&*self.gl, self.index);
        constant
    }
}

#[derive(Debug)]
pub enum Error {
    ///If the given attribute does not exist.
    NonExistent {
        name: String,
        index: i32,
    },
    Attribute(gltype::Error),
    ///If opengl cannot retrive the attribute information.
    ActiveAttrib {
        name: String,
        index: u32,
    },
}

use std::fmt;
impl fmt::Display for Error {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Error::NonExistent { name, index: _ } => {
                write!(formatter, "Attribute '{}' does not exists.", name)
            }
            Error::ActiveAttrib { name, index } => {
                write!(
                    formatter,
                    "Attribute '{}' at '{}' is not active.",
                    name, index
                )
            }
            Error::Attribute(x) => {
                write!(formatter, "{}", x)
            }
        }
    }
}

impl std::error::Error for Error {}

use std::collections::HashMap;

impl<T> ToAttribute for T
where
    T: GLType,
    Base<T>: Attribute<T>,
{
    type Output = Base<T>;
    type Error = Error;
    fn to_attribute(
        self,
        gl: Rc<GL>,
        program: &WebGlProgram,
        name: &str,
        map: &mut HashMap<String, WebGlActiveInfo>,
    ) -> Result<Self::Output, Self::Error> {
        let name = String::from(name);
        //Get the index of the attrrbute from the gl context.
        let index = gl.get_attrib_location(&program, &name);

        //If number is negative then the attribute does not exists.
        let index = if 0 <= index {
            Ok(index as u32)
        } else {
            Err(Error::NonExistent {
                name: name.clone(),
                index,
            })
        }?;

        let attribute = map.remove(&name).ok_or(Error::ActiveAttrib {
            name: name.clone(),
            index,
        })?;

        //Compare if the type is compatible with the attribute.
        T::compatible(&attribute).map_err(Error::Attribute)?;

        use core::mem::MaybeUninit;
        let mut base = Base {
            index,
            name,
            gl,
            constant: unsafe { MaybeUninit::uninit().assume_init() },
        };

        base.constant(self);

        Ok(base)
    }
}
