use super::*;

#[derive(Debug)]
pub struct Base<T> {
    index: Index,
    name: String,
    value: T,
}

#[derive(Debug, PartialEq, Eq)]
pub enum Error {
    GLError(super::super::context::Error),
    Type(gltype::Error),
    Uniform(String),
    Active(String),
}

use core::fmt;
impl fmt::Display for Error {
    fn fmt(&self, fmt: &mut fmt::Formatter<'_>) -> Result<(), fmt::Error> {
        match self {
            Error::GLError(x) => x.fmt(fmt),
            Error::Type(x) => x.fmt(fmt),
            Error::Uniform(x) => write! {fmt, "Uniform '{}' does not exists.", x},
            Error::Active(x) => write! {fmt, "Uniform '{}' is not active.", x},
        }
    }
}

impl<T> Uniform<T> for Base<T>
where
    T: GLType,{
    fn index(&self) -> &Index {
        &self.index
    }
    fn value(&mut self, gl: &GL, mut value: T) -> T{
        use core::mem::swap;
        value.send(gl, &self.index);
        swap(&mut self.value, &mut value);
        value
    }
}

impl<T> ToUniform for T
where
    T: GLType,{
    type Output = Base<T>;
    type Error = Error;
    fn to_uniform(self, 
        gl: &GL,
        program: &WebGlProgram,
        member : Option<Rc<Member>>,
        map : &mut HashMap<String, WebGlActiveInfo>,
    ) -> Result<Self::Output, Self::Error>{
        let name = member.map(|x| x.to_string()).unwrap_or_default();
        let index = match gl.get_uniform_location(program, &name) {
            Some(x) => Ok(x),
            _ => {
                use crate::context::GLError;
                gl.error().map_err(Error::GLError)?;
                Err(Error::Uniform(name.clone()))
            }
        }?;

        let info = map.remove(&name).ok_or(Error::Active(name.clone()))?;

        //Compare if the type is compatible with the attribute.
        T::compatible(&info).map_err(Error::Type)?;
        
        use core::mem::MaybeUninit;
        let mut base = Base {
            index,
            name,
            value: unsafe { MaybeUninit::uninit().assume_init() },
        };

        base.value(gl, self);   

        Ok(base)
    }
}