use web_sys::WebGlShader;
///Simple webgl shader typed wrapper.
pub struct Shader<const TYPE: u32> {
    shader: WebGlShader,
}

use core::ops::*;

impl<const TYPE: u32> Deref for Shader<TYPE> {
    type Target = WebGlShader;
    fn deref(&self) -> &Self::Target {
        &self.shader
    }
}

pub trait ShaderCreator<const TYPE: u32> {
    type Error: std::error::Error;
    fn shader(&self, src: &str) -> Result<Shader<TYPE>, Self::Error>;
}

use web_sys::WebGl2RenderingContext;
#[derive(Debug)]
pub enum Error {
    ShaderCreation,
    ShaderLog,
    Compiler(String),
}

use std::fmt::{self, Display, Formatter};
impl Display for Error {
    fn fmt(&self, fmt: &mut Formatter<'_>) -> Result<(), fmt::Error> {
        match self{
            Error::ShaderCreation => write!{fmt, "Could not create shader."},
            Error::ShaderLog => write!{fmt, "Could not retrive shader log."},
            Error::Compiler(x) => write!{fmt, "{}", x},
        }
    }
}

impl std::error::Error for Error {}

impl<const TYPE: u32> ShaderCreator<TYPE> for WebGl2RenderingContext {
    type Error = Error;
    fn shader(&self, src: &str) -> Result<Shader<TYPE>, Self::Error> {
        //Create the opengl shader.
        let shader = self.create_shader(TYPE).ok_or(Error::ShaderCreation)?;
        //Set the source of the shader.
        self.shader_source(&shader, src);
        //Compile the shader.
        self.compile_shader(&shader);

        //Get the shader compile log.
        let log = self.get_shader_info_log(&shader).ok_or(Error::ShaderLog)?;

        //The log contains errors as well as warnings, so if there is any error or warning, return a compile error.
        if log.is_empty() {
            Ok(Shader { shader })
        } else {
            Err(Error::Compiler(log))
        }
    }
}
