use std::rc::Rc;
use web_sys::WebGl2RenderingContext;
pub struct Target<C, const N: u32>(pub Rc<C>);

use core::ops::Deref;
impl<C, const N: u32> Deref for Target<C, N> {
    type Target = C;
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

pub struct TextureTarget<C, const N: u32>(Rc<C>);
impl<C, const N: u32> Deref for TextureTarget<C, N> {
    type Target = C;
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

pub struct Context {
    pub context: Rc<WebGl2RenderingContext>,
    pub array_buffer: Target<WebGl2RenderingContext, { WebGl2RenderingContext::ARRAY_BUFFER }>,
    pub copy_read_buffer:
        Target<WebGl2RenderingContext, { WebGl2RenderingContext::COPY_READ_BUFFER }>,
    pub copy_write_buffer:
        Target<WebGl2RenderingContext, { WebGl2RenderingContext::COPY_WRITE_BUFFER }>,
    pub element_array_buffer:
        Target<WebGl2RenderingContext, { WebGl2RenderingContext::ELEMENT_ARRAY_BUFFER }>,
    pub pixel_pack_buffer:
        Target<WebGl2RenderingContext, { WebGl2RenderingContext::PIXEL_PACK_BUFFER }>,
    pub pixel_unpack_buffer:
        Target<WebGl2RenderingContext, { WebGl2RenderingContext::PIXEL_UNPACK_BUFFER }>,
    pub transform_feedback_buffer:
        Target<WebGl2RenderingContext, { WebGl2RenderingContext::TRANSFORM_FEEDBACK_BUFFER }>,
    pub uniform_buffer: Target<WebGl2RenderingContext, { WebGl2RenderingContext::UNIFORM_BUFFER }>,
    pub texture_2d: TextureTarget<WebGl2RenderingContext, { WebGl2RenderingContext::TEXTURE_2D }>,
    pub texture_cube_map:
        TextureTarget<WebGl2RenderingContext, { WebGl2RenderingContext::TEXTURE_CUBE_MAP }>,
    pub texture_3d: TextureTarget<WebGl2RenderingContext, { WebGl2RenderingContext::TEXTURE_3D }>,
    pub texture_2d_array:
        TextureTarget<WebGl2RenderingContext, { WebGl2RenderingContext::TEXTURE_3D }>,
}
impl Deref for Context {
    type Target = WebGl2RenderingContext;
    fn deref(&self) -> &Self::Target {
        &self.context
    }
}
impl From<WebGl2RenderingContext> for Context {
    fn from(context: WebGl2RenderingContext) -> Context {
        let rc = Rc::new(context);
        Self {
            context: rc.clone(),
            array_buffer: Target(rc.clone()),
            copy_read_buffer: Target(rc.clone()),
            copy_write_buffer: Target(rc.clone()),
            element_array_buffer: Target(rc.clone()),
            pixel_pack_buffer: Target(rc.clone()),
            pixel_unpack_buffer: Target(rc.clone()),
            transform_feedback_buffer: Target(rc.clone()),
            uniform_buffer: Target(rc.clone()),
            texture_2d: TextureTarget(rc.clone()),
            texture_cube_map: TextureTarget(rc.clone()),
            texture_3d: TextureTarget(rc.clone()),
            texture_2d_array: TextureTarget(rc.clone()),
        }
    }
}

///Internal errors from opengl.
#[derive(Debug, PartialEq, Eq)]
pub enum Error {
    InvalidEnum,
    InvalidValue,
    InvalidOperation,
    InvalidFramebufferOperation,
    OutOfMemory,
    ContextLost,
}

use core::fmt;
impl fmt::Display for Error {
    fn fmt(&self, format: &mut fmt::Formatter<'_>) -> Result<(), fmt::Error> {
        write! {format, "{:?}", self}
    }
}

impl std::error::Error for Error {}

pub trait GLError {
    type Error;
    fn error(&self) -> Result<(), Self::Error>;
}

impl GLError for WebGl2RenderingContext {
    type Error = Error;
    ///Retrive the current opengl error, if there is no error Ok(()) is returned.
    fn error(&self) -> Result<(), Error> {
        match self.get_error() {
            WebGl2RenderingContext::INVALID_ENUM => Err(Error::InvalidEnum),
            WebGl2RenderingContext::INVALID_VALUE => Err(Error::InvalidValue),
            WebGl2RenderingContext::INVALID_OPERATION => Err(Error::InvalidOperation),
            WebGl2RenderingContext::INVALID_FRAMEBUFFER_OPERATION => {
                Err(Error::InvalidFramebufferOperation)
            }
            WebGl2RenderingContext::OUT_OF_MEMORY => Err(Error::OutOfMemory),
            WebGl2RenderingContext::CONTEXT_LOST_WEBGL => Err(Error::ContextLost),
            _ => Ok(()),
        }
    }
}
use super::attribute::GLType;
use core::ops::Range;
impl Context {
    //Rust like version of the opengl draw_arrays function
    pub fn draw_arrays(&mut self, mode: u32, range: Range<i32>) {
        self.context
            .draw_arrays(mode, range.start, range.end - range.start);
    }
    //Rust like version of the opengl draw_elements function
    pub fn draw_elements<T>(&mut self, mode: u32, range: Range<i32>)
    where
        T: GLType,
    {
        use std::mem::size_of;
        self.context.draw_elements_with_i32(
            mode,
            range.end - range.start,
            T::TYPE,
            range.start * size_of::<T>() as i32,
        );
    }
}
