mod traits;
use std::fmt::{Debug, Display};
use crate::traits::{Minimize, Summary, SummaryDefaulted, DevilDisplay};

struct Point<T> {
	x: T,
	y: T,
}

struct Point2<T, U> {
    x: T,
    y: U,
}

enum Option<T> {
    Some(T),
    None,
}

enum Result<T, E> {
    Ok(T),
    Err(E),
}

impl<T> Point<T> {
	fn x(&self) -> &T {
		&self.x
	}
}

impl Point<f32> {
	fn distance_from_origin(&self) -> f32 {
		(self.x.powi(2) + self.y.powi(2)).sqrt()
	}
}

impl<T, U> Point2<T, U> {
	fn mixup<V, W>(self, other: Point2<V, W>) -> Point2<T, W> {
		Point2 {x: self.x, y: other.y}
	}
}

// Trait as parameter
fn notify(item: &impl Summary) {
	// call summary on item that implements Summary Trait
	println!("Breaking news! {}", item.summarize());
}
// this is syntatic sugar for
// Trait Bound Syntax
fn notify_impl<T: Summary>(item: &T) {
	println!("Breaking news! {}", item.summarize());
}

fn notify2(item1: &impl Summary, item2: &impl Summary) {	// item 1 and 2 can be different
	println!("Breaking news! {}", item1.summarize());
	println!("Breaking news! {}", item2.summarize());
}
fn notify_impl2<T: Summary>(item1: &T, item2: &T) {	// item 1 and 2 have to be same type
	println!("Breaking news! {}", item1.summarize());
	println!("Breaking news! {}", item2.summarize());
}


// multiple Traits
fn notify_mult(item: &(impl Summary + Minimize) ) {
	println!("Breaking news! {}", item.summarize());
	println!("Breaking news! {}", item.minimize());
}

fn notify_mult_impl<T: Summary + Minimize>(item: &T) {
	println!("Breaking news! {}", item.summarize());
	println!("Breaking news! {}", item.minimize());
}



// using where clause to organize generic's trait bounds
// this function...
fn some_function<T: Display + Debug, U: Clone + Debug>(t: &T, u: &U) -> i32 {5}
// can be summarized to 
fn some_function_where_<T, U>(t: &T, u: &U) -> i32 
	where T: Display + Debug,
		  U: Clone + Debug 
		  {5}


fn return_summarizable() -> impl Summary {
	crate::traits::Tweet {
		username: String::from(""),
		content: String::from(""),
		reply: true,
		retweet: true
	}
}

fn largest<T: PartialOrd + Copy>(list: &[T]) -> T {
	let mut largest = list[0];
	for &item in list {
		if item > largest {
			largest = item;
		}
	}
	return largest
}

// using cloning instead
fn largest_clone<T: PartialOrd + Clone>(list: &[T]) -> T {
	let mut largest = list[0].clone();
	for item in list {
		if item > &largest {
			largest = item.clone();
		}
	}
	return largest
}

// using a return reference imlementation
fn largest_using_return_ref<T: PartialOrd>(list: &[T]) -> &T {
	let mut largest = &list[0];
	for item in list {
		if item > largest {
			largest = item;
		}
	}
	return largest
}

// conditionally implement methods based on genrics trait types

struct Pair<T> {
	x: T,
	y: T
}

impl<T> Pair<T> {
	fn new(x: T, y: T) -> Self {
		Pair {x, y}
	}
}

impl <T: Display + PartialOrd> Pair<T> {
	fn cmp_display(&self) { // fn is only implement when generic type of T conform to PartialOrd and Display trait
		if self.x < self.y {
			println!("smaller number is {}", self.x);
		} else {
			println!("smaller number is {}", self.y)
		}	
	}
}

// Blanket Implementations
// conditionally implement a trait for any type that implements another trait.
// implements TToString for any type that implements Display
impl<T: Display> DevilDisplay for T {
	fn devil_print(&self) -> String {
		format!("😈 : {}", self)
	}
}



fn main() {
	println!("Generics");

	let values = vec![1,2,3,3,4];
	let value = largest(&values);
	println!("Largest Vlaue: {}", value);

	// generic struct 
	let integer =Point {x: 32, y: 23};
	let float =Point {x: 32.3, y: 23.1};

	let intfloat = Point2 {x: 3.2, y: 32};

	println!("p.x  = {}", float.x());

	// using traits

	let tweet = crate::traits::Tweet {
		username: String::from("Nameless"),
		content: String::from("example cotnent message"),
		reply: true,
		retweet: true,
	};
	let article = crate::traits::NewsArticle {
		headline: String::from("Yeah!!!!!!!!"),
		location: String::from("Japan"),
		author: String::from("u3241"),
		content: String::from("some weird content"),
	};

	println!("article: {}", article.summarize());
	println!("article: {}", article.summarizeDef());
	// using default impl
	println!("1 new tweet: {}", tweet.minimize());

	// notify can take both an article and tweet
	notify(&article);
	notify(&tweet);
	
	notify_impl(&article);
	notify_impl(&tweet);

	notify2(&article, &article);	// works
	notify2(&article, &tweet);		// works
	notify_impl2(&article, &article);	// works
	// notify_impl2(&article, &tweet);		// error: different types

	// multiple traits
	notify_mult(&tweet);
	// notify_mult(&article);	// error: article doesn't satify Trait Minimize
	let new_tweet = return_summarizable();
	// println!("tweet: {}", new_tweet);	// error: this item is not guaranteed to implement the Display trait
	
	let x = String::from("i like strings");
	let y = x.devil_print();
	println!("{}", y);


}