use std::fs::{self, File};
use std::io::{self, ErrorKind, Read};

fn main() {

	// handle nested errors 
	let filename = "hello.txt";
	let f = File::open(filename);
	// NOTE: outer and innermost matches are
	// matching on enums and the middle match is 
	// actually matching on a struct like a switch statement 
	let file = match f {
		Ok(f) => f,
		Err(io_error) => match io_error.kind() {
			ErrorKind::NotFound => match File::create(filename) {
				Ok(fc) => fc,
				Err(err) => panic!("{}", err),
			}
			other_error => panic!("{:?}", other_error)
		}
	};
	
	// concise version
	let f = File::open("hello.txt").unwrap_or_else(|error| {
		if error.kind() == ErrorKind::NotFound {
			File::create("hello.txt").unwrap_or_else(|error| {
				panic!("Problem creating the file: {:?}", error);
			})
		} else {
			panic!("Problem opening the file: {:?}", error);
		}
	});

	// Shortcuts On Error
	// `unwrap()`
	let f = File::open("hello.txt").unwrap();
	// If the Result value is the Ok variant,
	// unwrap will return the value inside the Ok.
	// If the Result is the Err variant, 
	// unwrap will call the panic! macro for us

	// `expect()`
	let f =File::open("hello.txt").expect("error message couldn't blah blah...");
	// expect is similar to unwrap, lets us also choose the panic! error message

	// propagating the error

	let result_string = read_username_from_file("hello.txt");
	match result_string {
		Ok(s) => println!("returned string with data"),
		Err(err) => panic!("{}", err),
	}

	let result_string = read_username_from_file_shortcut("hello.txt");
	match result_string {
		Ok(s) => println!("{}", s),
		Err(err) => panic!("{}", err)
	}

	let result_string = read_username_from_file_concise("hello.txt");
	match result_string {
		Ok(s) => println!("{}", s),
		Err(err) => panic!("{}", err)
	}
	read_username_from_file_oneline("hello.txt");
	// using `?` shortcut operator

}

fn read_username_from_file(path: &str) -> Result<String, io::Error> {
	let f = File::open(path);
	let mut f = match f {
		Ok(file) => file,	// put file in let mut f
		Err(e) => return Err(e),	// instead of panic! we will return Err(e) form this fn
	};
	let mut s = String::new();
	match f.read_to_string(&mut s) {
		Ok(_) => Ok(s),
		Err(e) => Err(e),
	}	// this match is an expression at the end of fn
		// so whatever is matched is return from the function 
		// either an Err(s) or an Ok(s) because this fn has to 
		// return a Result<String, io::Error> type
}

fn read_username_from_file_shortcut(path: &str) -> Result<String, io::Error> {
	let mut f = File::open(path)?;
	// If the value of the Result is an Ok, 
	// the value inside the Ok will get returned 
	// from this expression, and the program will 
	// continue. If the value is an Err, the Err will be returned from the whole function as if we had used the return keyword so the error value gets propagated to the calling code.
	let mut s = String::new();
	f.read_to_string(&mut s)?;
	Ok(s)
}
fn read_username_from_file_concise(path: &str) -> Result<String, io::Error> {
	let mut s = String::new();
	File::open(path)?.read_to_string(&mut s)?;
	Ok(s)
} 

fn read_username_from_file_oneline(path: &str) -> Result<String, io::Error> {
	fs::read_to_string(path)
}