use std::{env, fs, path::Path, process::Command};

#[derive(Default, PartialEq)]
struct Slide {
    title: Option<String>,
    // Background image
    image: Option<String>,
    lines: Vec<String>,
}

impl Slide {
    fn as_tex(&self) -> String {
        let mut tex: String = String::new();
        let mut current_env: Vec<String> = Vec::new();
        let mut prev_leader: Option<char> = None;

        macro_rules! format_env {
            ($env_name: literal, $title: literal, $line_prefix: literal) => {{
                let mut iter = current_env.iter();
                if $title {
                    tex.push_str(&format!(
                        "\\begin{{{}}}{{{}}}",
                        $env_name,
                        iter.next().unwrap()
                    ));
                } else {
                    tex.push_str(&format!("\\begin{{{}}}", $env_name));
                }
                for line in iter {
                    tex.push('\n');
                    tex.push('\t');
                    tex.push_str($line_prefix);
                    tex.push_str(line);
                    tex.push('\n');
                }
                tex.push_str(&format!("\\end{{{}}}\n", $env_name));
            }};
        }

        macro_rules! push_env {
            () => {
                match prev_leader {
                    Some(prev_leader) => match prev_leader {
                        '-' => format_env!("itemize", false, "\\item "),
                        '>' => format_env!("block", true, ""),
                        '<' => format_env!("exampleblock", true, ""),
                        '!' => format_env!("alertblock", true, ""),
                        _ => panic!("Somehow the previous leader is '{}', even when it should be \"None\"", prev_leader)
                    },
                    None => {}
                }
            };
        }

        for line in &self.lines {
            if !line.is_empty() {
                let leader = line.chars().next().unwrap();
                if Some(leader) != prev_leader {
                    push_env!();
                    current_env = Vec::new();
                }
                if ['-', '>', '<', '!'].contains(&leader) {
                    current_env.push(String::from(line[1..].trim()));
                    prev_leader = Some(leader);
                } else {
                    tex.push_str(line);
                    tex.push('\n');
                    prev_leader = None;
                }
            } else {
                push_env!();
                current_env = Vec::new();
                prev_leader = None;
            }
        }

        push_env!();

        match &self.title {
            Some(title) => {
                tex.insert_str(0, &format!("\\begin{{frame}}{{{}}}\n", title));
                tex.push_str(r#"\end{frame}"#);
            }
            None => {
                tex.insert_str(0, "\\begin{frame}\n");
                tex.push_str(r#"\end{frame}"#);
            }
        }

        match &self.image {
            Some(image) => {
                let mut prepend_str = String::from(
                    r#"{\setbeamertemplate{background}{\includegraphics[width=\paperwidth,height=\paperheight]{../"#,
                );
                prepend_str.push_str(image);
                prepend_str.push_str("}}\n");

                tex.insert_str(0, &prepend_str);
                tex.push_str("\n}");
            }
            None => {}
        }

        tex
    }

    fn is_empty(&self) -> bool {
        self == &Slide::default()
    }
}

enum Content {
    Section(String),
    Slide(Slide),
}

struct Presentation {
    preamble: Vec<String>,
    contents: Vec<Content>,
}

impl Presentation {
    fn as_tex(&self) -> String {
        let mut tex: String = String::new();
        for line in &self.preamble {
            tex.push_str(line);
            tex.push('\n');
        }
        tex.push('\n');
        tex.push_str(r#"\begin{document}"#);
        tex.push('\n');
        for content in &self.contents {
            match content {
                Content::Section(title) => {
                    tex.push_str(&format!("\\section{{{}}}", title));
                }
                Content::Slide(slide) => {
                    tex.push_str(&slide.as_tex());
                }
            }
            tex.push('\n');
            tex.push('\n');
        }
        tex.push_str(r#"\end{document}"#);
        tex
    }
}

// Parses the contents of a .beam file into an instance of the Presentation struct

fn beam_to_presentation(string: &str) -> Presentation {
    let mut preamble: Vec<String> = Vec::new();
    let mut contents: Vec<Content> = Vec::new();
    let mut current_slide: Slide = Slide::default();

    macro_rules! push_slide {
        () => {
            if !current_slide.is_empty() {
                contents.push(Content::Slide(current_slide));
                current_slide = Slide::default();
            }
        };
    }

    for (num, line) in string.split('\n').enumerate() {
        let line = line.trim();
        if line.is_empty() {
            push_slide!();
        } else if !line.is_empty() {
            match line.chars().next().unwrap() {
                '#' => {
                    push_slide!();
                    contents.push(Content::Section(String::from(line[1..].trim())));
                }
                '^' => {
                    push_slide!();
                    preamble.push(String::from(line[1..].trim()));
                }
                '~' => {
                    push_slide!();
                    current_slide.title = Some(String::from(line[1..].trim()));
                }
                '@' => {
                    if current_slide.image != None {
                        panic!(
                            "Background image set twice for the same slide in line {}",
                            num
                        );
                    } else {
                        current_slide.image = Some(String::from(line[1..].trim()));
                    }
                }
                _ => {
                    current_slide.lines.push(String::from(line));
                }
            }
        }
    }
    Presentation { preamble, contents }
}

fn main() {
    let mut args: Vec<String> = env::args().collect();
    args.remove(0);

    for arg in &args {
        let build_dir = Path::new("build");
        let name = if arg.len() > 5 && &arg[arg.len() - 5..] == ".beam" {
            &arg[..arg.len() - 5]
        } else {
            arg
        };
        let tex_name = &format!("{}.tex", name);
        let pdf_name = &format!("{}.pdf", name);
        if build_dir.exists() && !build_dir.is_dir() {
            panic!("Path \"build\" exists but is not a directory");
        } else if !build_dir.exists() {
            fs::create_dir(build_dir).expect("Fialed to create directory \"build\"");
        }
        match fs::read_to_string(arg) {
            Ok(string) => {
                env::set_current_dir(build_dir)
                    .expect("Failed to set current working directory to \"build\"");
                fs::write(tex_name, beam_to_presentation(&string).as_tex())
                    .unwrap_or_else(|_| panic!("Failed to write to {}.tex", arg));
                let latexmk = Command::new("latexmk").args(["-pdf", "-f", tex_name]).output().expect("\"latexmk\" could not be executed. Check that it is installed, in PATH, and executable");
                if !latexmk.status.success() {
                    println!("Failed to compile {}", tex_name);
                }
                env::set_current_dir("..").expect("Failed to reset current working directory");
                if latexmk.status.success() {
                    fs::rename(build_dir.join(pdf_name), pdf_name).unwrap_or_else(|_| {
                        panic!("Could not rename {:?} to {}", build_dir.join(pdf_name), pdf_name)
                    });
                }
            }
            Err(e) => panic!("Path \"{}\" could not be read\n{}", arg, e),
        }
    }
}
