use crate::usp::Msg;
use crate::usp_record::Record;
use quick_protobuf::message::MessageRead;
use quick_protobuf::BytesReader;

use anyhow::{Context, Result};

/// Decodes a slice of bytes containing a protobuf encoded USP Record into a Record structure for
/// further processing
///
/// # Arguments
///
/// * `bytes` - A slice of bytes containing the protobuf encoded USP Record
///
/// # Example
///
/// ```
/// use rusp::usp_decoder::decode_record;
/// let record =
///     decode_record(&vec![
///         0x0a, 0x03, 0x31, 0x2e, 0x30, 0x1a, 0x23, 0x70, 0x72, 0x6f, 0x74, 0x6f, 0x3a, 0x3a,
///         0x61, 0x78, 0x2d, 0x75, 0x73, 0x70, 0x2d, 0x61, 0x67, 0x65, 0x6e, 0x74, 0x2d, 0x6e,
///         0x6f, 0x73, 0x73, 0x6c, 0x2d, 0x77, 0x65, 0x62, 0x73, 0x6f, 0x63, 0x6b, 0x65, 0x74,
///         0x3a, 0x4d, 0x12, 0x4b, 0x0a, 0x08, 0x0a, 0x04, 0x74, 0x65, 0x73, 0x74, 0x10, 0x03,
///         0x12, 0x3f, 0x0a, 0x3d, 0x42, 0x3b, 0x0a, 0x0f, 0x73, 0x75, 0x62, 0x73, 0x63, 0x72,
///         0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x69, 0x64, 0x42, 0x28, 0x0a, 0x03, 0x6f,
///         0x75, 0x69, 0x12, 0x0d, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x5f, 0x63, 0x6c,
///         0x61, 0x73, 0x73, 0x1a, 0x0d, 0x73, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x5f, 0x6e, 0x75,
///         0x6d, 0x62, 0x65, 0x72, 0x22, 0x03, 0x31, 0x2e, 0x30,
///     ]);
/// ```
#[deprecated(
    since = "0.10.0",
    note = "this function will panic on unexpected data, use `try_decode_record` instead"
)]
pub fn decode_record(bytes: &[u8]) -> Record {
    let mut reader = BytesReader::from_bytes(&bytes);
    Record::from_reader(&mut reader, &bytes).expect("Cannot read Record")
}

/// Decodes a slice of bytes containing a protobuf encoded USP Record into a Record structure for
/// further processing
///
/// # Arguments
///
/// * `bytes` - A slice of bytes containing the protobuf encoded USP Record
///
/// # Example
///
/// ```
/// use rusp::usp_decoder::try_decode_record;
/// let record =
///     try_decode_record(&vec![
///         0x0a, 0x03, 0x31, 0x2e, 0x30, 0x1a, 0x23, 0x70, 0x72, 0x6f, 0x74, 0x6f, 0x3a, 0x3a,
///         0x61, 0x78, 0x2d, 0x75, 0x73, 0x70, 0x2d, 0x61, 0x67, 0x65, 0x6e, 0x74, 0x2d, 0x6e,
///         0x6f, 0x73, 0x73, 0x6c, 0x2d, 0x77, 0x65, 0x62, 0x73, 0x6f, 0x63, 0x6b, 0x65, 0x74,
///         0x3a, 0x4d, 0x12, 0x4b, 0x0a, 0x08, 0x0a, 0x04, 0x74, 0x65, 0x73, 0x74, 0x10, 0x03,
///         0x12, 0x3f, 0x0a, 0x3d, 0x42, 0x3b, 0x0a, 0x0f, 0x73, 0x75, 0x62, 0x73, 0x63, 0x72,
///         0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x69, 0x64, 0x42, 0x28, 0x0a, 0x03, 0x6f,
///         0x75, 0x69, 0x12, 0x0d, 0x70, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x5f, 0x63, 0x6c,
///         0x61, 0x73, 0x73, 0x1a, 0x0d, 0x73, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x5f, 0x6e, 0x75,
///         0x6d, 0x62, 0x65, 0x72, 0x22, 0x03, 0x31, 0x2e, 0x30,
///     ]);
/// ```
pub fn try_decode_record(bytes: &[u8]) -> Result<Record> {
    let mut reader = BytesReader::from_bytes(&bytes);
    Record::from_reader(&mut reader, &bytes).context("while parsing protobuf as USP Record")
}

/// Decodes a slice of bytes containing a protobuf encoded USP Msg into a Msg structure for further
/// processing
///
/// # Arguments
///
/// * `bytes` - A slice of bytes containing the protobuf encoded USP Message
///
/// # Example
///
/// ```
/// use rusp::usp_decoder::decode_msg;
/// let msg =
///     decode_msg(&vec![
///         0x0a, 0x1a, 0x0a, 0x16, 0x41, 0x58, 0x53, 0x53, 0x2d, 0x31, 0x35, 0x34,
///         0x34, 0x31, 0x31, 0x34, 0x30, 0x34, 0x35, 0x2e, 0x34, 0x34, 0x32, 0x35,
///         0x39, 0x36, 0x10, 0x02, 0x12, 0x46, 0x12, 0x44, 0x0a, 0x42, 0x0a, 0x40,
///         0x0a, 0x22, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x2e, 0x4c, 0x6f, 0x63,
///         0x61, 0x6c, 0x41, 0x67, 0x65, 0x6e, 0x74, 0x2e, 0x4d, 0x54, 0x50, 0x2e,
///         0x31, 0x2e, 0x57, 0x65, 0x62, 0x53, 0x6f, 0x63, 0x6b, 0x65, 0x74, 0x2e,
///         0x15, 0x62, 0x1b, 0x00, 0x00, 0x1a, 0x15, 0x55, 0x6e, 0x73, 0x75, 0x70,
///         0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x20, 0x70, 0x61, 0x72, 0x61, 0x6d,
///         0x65, 0x74, 0x65, 0x72
///     ]);
/// ```
#[deprecated(
    since = "0.10.0",
    note = "this function will panic on unexpected data, use `try_decode_msg` instead"
)]
pub fn decode_msg(bytes: &[u8]) -> Msg {
    let mut reader = BytesReader::from_bytes(&bytes);
    Msg::from_reader(&mut reader, &bytes).expect("Cannot read Msg")
}

/// Decodes a slice of bytes containing a protobuf encoded USP Msg into a Msg structure for further
/// processing
///
/// # Arguments
///
/// * `bytes` - A slice of bytes containing the protobuf encoded USP Message
///
/// # Example
///
/// ```
/// use rusp::usp_decoder::try_decode_msg;
/// let msg =
///     try_decode_msg(&vec![
///         0x0a, 0x1a, 0x0a, 0x16, 0x41, 0x58, 0x53, 0x53, 0x2d, 0x31, 0x35, 0x34,
///         0x34, 0x31, 0x31, 0x34, 0x30, 0x34, 0x35, 0x2e, 0x34, 0x34, 0x32, 0x35,
///         0x39, 0x36, 0x10, 0x02, 0x12, 0x46, 0x12, 0x44, 0x0a, 0x42, 0x0a, 0x40,
///         0x0a, 0x22, 0x44, 0x65, 0x76, 0x69, 0x63, 0x65, 0x2e, 0x4c, 0x6f, 0x63,
///         0x61, 0x6c, 0x41, 0x67, 0x65, 0x6e, 0x74, 0x2e, 0x4d, 0x54, 0x50, 0x2e,
///         0x31, 0x2e, 0x57, 0x65, 0x62, 0x53, 0x6f, 0x63, 0x6b, 0x65, 0x74, 0x2e,
///         0x15, 0x62, 0x1b, 0x00, 0x00, 0x1a, 0x15, 0x55, 0x6e, 0x73, 0x75, 0x70,
///         0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x20, 0x70, 0x61, 0x72, 0x61, 0x6d,
///         0x65, 0x74, 0x65, 0x72
///     ]);
/// ```
pub fn try_decode_msg(bytes: &[u8]) -> Result<Msg> {
    let mut reader = BytesReader::from_bytes(&bytes);
    Msg::from_reader(&mut reader, &bytes).context("while parsing protobuf as USP Message")
}
