// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//
// Copyright (c) DUSK NETWORK. All rights reserved.

use crate::modules::{InstrumentationError, ModuleConfig};
use crate::VMError;
use std::sync::Arc;
use wasmer::wasmparser::Operator;
use wasmer::wasmparser::Operator::*;
use wasmer::CompilerConfig;
use wasmer_middlewares::Metering;

#[cfg(target_arch = "x86_64")]
use wasmer_compiler_singlepass::Singlepass as Compiler;

#[cfg(not(target_arch = "x86_64"))]
use wasmer_compiler_cranelift::Cranelift as Compiler;

pub struct CompilerConfigProvider;

impl CompilerConfigProvider {
    pub fn from_config(
        module_config: &ModuleConfig,
    ) -> Result<Compiler, VMError> {
        if !module_config.has_forbidden_floats {
            return Err(VMError::InstrumentationError(
                InstrumentationError::InvalidInstructionType,
            ));
        }
        fn op_type_cost(
            op_type: impl AsRef<str>,
            module_config: &ModuleConfig,
        ) -> u64 {
            *module_config
                .per_type_op_cost
                .get(op_type.as_ref())
                .unwrap_or(&module_config.regular_op_cost) as u64
        }

        let op_cost = |op_type: &str| op_type_cost(op_type, module_config);

        let regular_op_cost: u64 = module_config.regular_op_cost as u64;

        struct CostsPerOpType {
            bit: u64,
            add: u64,
            mul: u64,
            div: u64,
            float: u64,
            conversion: u64,
            float_conversion: u64,
            reinterpret: u64,
            unreachable: u64,
            nop: u64,
            flow: u64,
            local: u64,
            global: u64,
            load: u64,
            store: u64,
            current_mem: u64,
            grow_mem: u64,
            int_const: u64,
            integer_comp: u64,
            float_comp: u64,
        }

        let costs = CostsPerOpType {
            bit: op_cost("bit"),
            add: op_cost("add"),
            mul: op_cost("mul"),
            div: op_cost("div"),
            float: op_cost("float"),
            conversion: op_cost("conversion"),
            float_conversion: op_cost("float_conversion"),
            reinterpret: op_cost("reinterpret"),
            unreachable: op_cost("unreachable"),
            nop: op_cost("nop"),
            flow: op_cost("flow"),
            local: op_cost("local"),
            global: op_cost("global"),
            load: op_cost("load"),
            store: op_cost("store"),
            current_mem: op_cost("current_mem"),
            grow_mem: op_cost("grow_mem"),
            int_const: op_cost("const"),
            integer_comp: op_cost("integer_comp"),
            float_comp: op_cost("float_comp"),
        };

        let cost_function = move |operator: &Operator| -> u64 {
            match operator {
                Unreachable => costs.unreachable,
                Nop => costs.nop,
                Block { .. } => costs.flow,
                Loop { .. } => costs.flow,
                If { .. } => costs.flow,
                Else => costs.flow,
                Try { .. } => costs.flow,
                Catch { .. } => costs.flow,
                Throw { .. } => costs.flow,
                Rethrow { .. } => costs.flow,
                Unwind => costs.flow,
                End => costs.flow,
                Br { .. } => costs.flow,
                BrIf { .. } => costs.flow,
                BrTable { .. } => costs.flow,
                Return => costs.flow,
                Call { .. } => costs.flow,
                CallIndirect { .. } => costs.flow,
                ReturnCall { .. } => costs.flow,
                ReturnCallIndirect { .. } => costs.flow,
                Delegate { .. } => costs.flow,
                CatchAll => costs.flow,
                Drop => costs.flow,
                Select => costs.flow,
                TypedSelect { .. } => costs.flow,
                LocalGet { .. } => costs.local,
                LocalSet { .. } => costs.local,
                LocalTee { .. } => costs.local,
                GlobalGet { .. } => costs.global,
                GlobalSet { .. } => costs.global,
                I32Load { .. } => costs.load,
                I64Load { .. } => costs.load,
                F32Load { .. } => costs.load,
                F64Load { .. } => costs.load,
                I32Load8S { .. } => costs.load,
                I32Load8U { .. } => costs.load,
                I32Load16S { .. } => costs.load,
                I32Load16U { .. } => costs.load,
                I64Load8S { .. } => costs.load,
                I64Load8U { .. } => costs.load,
                I64Load16S { .. } => costs.load,
                I64Load16U { .. } => costs.load,
                I64Load32S { .. } => costs.load,
                I64Load32U { .. } => costs.load,
                I32Store { .. } => costs.store,
                I64Store { .. } => costs.store,
                F32Store { .. } => costs.store,
                F64Store { .. } => costs.store,
                I32Store8 { .. } => costs.store,
                I32Store16 { .. } => costs.store,
                I64Store8 { .. } => costs.store,
                I64Store16 { .. } => costs.store,
                I64Store32 { .. } => costs.store,
                MemorySize { .. } => costs.current_mem,
                MemoryGrow { .. } => costs.grow_mem,
                I32Const { .. } => costs.int_const,
                I64Const { .. } => costs.int_const,
                F32Const { .. } => regular_op_cost,
                F64Const { .. } => regular_op_cost,
                RefNull { .. } => regular_op_cost,
                RefIsNull => regular_op_cost,
                RefFunc { .. } => regular_op_cost,
                I32Eqz => costs.integer_comp,
                I32Eq => costs.integer_comp,
                I32Ne => costs.integer_comp,
                I32LtS => costs.integer_comp,
                I32LtU => costs.integer_comp,
                I32GtS => costs.integer_comp,
                I32GtU => costs.integer_comp,
                I32LeS => costs.integer_comp,
                I32LeU => costs.integer_comp,
                I32GeS => costs.integer_comp,
                I32GeU => costs.integer_comp,
                I64Eqz => costs.integer_comp,
                I64Eq => costs.integer_comp,
                I64Ne => costs.integer_comp,
                I64LtS => costs.integer_comp,
                I64LtU => costs.integer_comp,
                I64GtS => costs.integer_comp,
                I64GtU => costs.integer_comp,
                I64LeS => costs.integer_comp,
                I64LeU => costs.integer_comp,
                I64GeS => costs.integer_comp,
                I64GeU => costs.integer_comp,
                F32Eq => costs.float_comp,
                F32Ne => costs.float_comp,
                F32Lt => costs.float_comp,
                F32Gt => costs.float_comp,
                F32Le => costs.float_comp,
                F32Ge => costs.float_comp,
                F64Eq => costs.float_comp,
                F64Ne => costs.float_comp,
                F64Lt => costs.float_comp,
                F64Gt => costs.float_comp,
                F64Le => costs.float_comp,
                F64Ge => costs.float_comp,
                I32Clz => costs.bit,
                I32Ctz => costs.bit,
                I32Popcnt => costs.bit,
                I32Add => costs.add,
                I32Sub => costs.add,
                I32Mul => costs.mul,
                I32DivS => costs.div,
                I32DivU => costs.div,
                I32RemS => costs.div,
                I32RemU => costs.div,
                I32And => costs.bit,
                I32Or => costs.bit,
                I32Xor => costs.bit,
                I32Shl => costs.bit,
                I32ShrS => costs.bit,
                I32ShrU => costs.bit,
                I32Rotl => costs.bit,
                I32Rotr => costs.bit,
                I64Clz => costs.bit,
                I64Ctz => costs.bit,
                I64Popcnt => costs.bit,
                I64Add => costs.add,
                I64Sub => costs.add,
                I64Mul => costs.mul,
                I64DivS => costs.div,
                I64DivU => costs.div,
                I64RemS => costs.div,
                I64RemU => costs.div,
                I64And => costs.bit,
                I64Or => costs.bit,
                I64Xor => costs.bit,
                I64Shl => costs.bit,
                I64ShrS => costs.bit,
                I64ShrU => costs.bit,
                I64Rotl => costs.bit,
                I64Rotr => costs.bit,
                F32Abs => costs.float,
                F32Neg => costs.float,
                F32Ceil => costs.float,
                F32Floor => costs.float,
                F32Trunc => costs.float,
                F32Nearest => costs.float,
                F32Sqrt => costs.float,
                F32Add => costs.float,
                F32Sub => costs.float,
                F32Mul => costs.float,
                F32Div => costs.float,
                F32Min => costs.float,
                F32Max => costs.float,
                F32Copysign => costs.float,
                F64Abs => costs.float,
                F64Neg => costs.float,
                F64Ceil => costs.float,
                F64Floor => costs.float,
                F64Trunc => costs.float,
                F64Nearest => costs.float,
                F64Sqrt => costs.float,
                F64Add => costs.float,
                F64Sub => costs.float,
                F64Mul => costs.float,
                F64Div => costs.float,
                F64Min => costs.float,
                F64Max => costs.float,
                F64Copysign => costs.float,
                I32WrapI64 => costs.conversion,
                I64ExtendI32S => costs.conversion,
                I64ExtendI32U => costs.conversion,
                I32TruncF32S => costs.float_conversion,
                I32TruncF32U => costs.float_conversion,
                I32TruncF64S => costs.float_conversion,
                I32TruncF64U => costs.float_conversion,
                I64TruncF32S => costs.float_conversion,
                I64TruncF32U => costs.float_conversion,
                I64TruncF64S => costs.float_conversion,
                I64TruncF64U => costs.float_conversion,
                F32ConvertI32S => costs.float_conversion,
                F32ConvertI32U => costs.float_conversion,
                F32ConvertI64S => costs.float_conversion,
                F32ConvertI64U => costs.float_conversion,
                F32DemoteF64 => costs.float_conversion,
                F64ConvertI32S => costs.float_conversion,
                F64ConvertI32U => costs.float_conversion,
                F64ConvertI64S => costs.float_conversion,
                F64ConvertI64U => costs.float_conversion,
                F64PromoteF32 => costs.float_conversion,
                I32ReinterpretF32 => costs.reinterpret,
                I64ReinterpretF64 => costs.reinterpret,
                F32ReinterpretI32 => costs.reinterpret,
                F64ReinterpretI64 => costs.reinterpret,
                I32Extend8S => regular_op_cost,
                I32Extend16S => regular_op_cost,
                I64Extend8S => regular_op_cost,
                I64Extend16S => regular_op_cost,
                I64Extend32S => regular_op_cost,

                // 0xFC operators
                // Non-trapping Float-to-int Conversions
                I32TruncSatF32S => regular_op_cost,
                I32TruncSatF32U => regular_op_cost,
                I32TruncSatF64S => regular_op_cost,
                I32TruncSatF64U => regular_op_cost,
                I64TruncSatF32S => regular_op_cost,
                I64TruncSatF32U => regular_op_cost,
                I64TruncSatF64S => regular_op_cost,
                I64TruncSatF64U => regular_op_cost,

                // 0xFC operators
                // bulk memory https://github.com/WebAssembly/bulk-memory-operations/blob/master/proposals/bulk-memory-operations/Overview.md
                MemoryInit { .. } => regular_op_cost,
                DataDrop { .. } => regular_op_cost,
                MemoryCopy { .. } => regular_op_cost,
                MemoryFill { .. } => regular_op_cost,
                TableInit { .. } => regular_op_cost,
                ElemDrop { .. } => regular_op_cost,
                TableCopy { .. } => regular_op_cost,
                TableFill { .. } => regular_op_cost,
                TableGet { .. } => regular_op_cost,
                TableSet { .. } => regular_op_cost,
                TableGrow { .. } => regular_op_cost,
                TableSize { .. } => regular_op_cost,

                // 0xFE operators
                // https://github.com/WebAssembly/threads/blob/master/proposals/threads/Overview.md
                MemoryAtomicNotify { .. } => regular_op_cost,
                MemoryAtomicWait32 { .. } => regular_op_cost,
                MemoryAtomicWait64 { .. } => regular_op_cost,
                AtomicFence { .. } => regular_op_cost,
                I32AtomicLoad { .. } => regular_op_cost,
                I64AtomicLoad { .. } => regular_op_cost,
                I32AtomicLoad8U { .. } => regular_op_cost,
                I32AtomicLoad16U { .. } => regular_op_cost,
                I64AtomicLoad8U { .. } => regular_op_cost,
                I64AtomicLoad16U { .. } => regular_op_cost,
                I64AtomicLoad32U { .. } => regular_op_cost,
                I32AtomicStore { .. } => regular_op_cost,
                I64AtomicStore { .. } => regular_op_cost,
                I32AtomicStore8 { .. } => regular_op_cost,
                I32AtomicStore16 { .. } => regular_op_cost,
                I64AtomicStore8 { .. } => regular_op_cost,
                I64AtomicStore16 { .. } => regular_op_cost,
                I64AtomicStore32 { .. } => regular_op_cost,
                I32AtomicRmwAdd { .. } => regular_op_cost,
                I64AtomicRmwAdd { .. } => regular_op_cost,
                I32AtomicRmw8AddU { .. } => regular_op_cost,
                I32AtomicRmw16AddU { .. } => regular_op_cost,
                I64AtomicRmw8AddU { .. } => regular_op_cost,
                I64AtomicRmw16AddU { .. } => regular_op_cost,
                I64AtomicRmw32AddU { .. } => regular_op_cost,
                I32AtomicRmwSub { .. } => regular_op_cost,
                I64AtomicRmwSub { .. } => regular_op_cost,
                I32AtomicRmw8SubU { .. } => regular_op_cost,
                I32AtomicRmw16SubU { .. } => regular_op_cost,
                I64AtomicRmw8SubU { .. } => regular_op_cost,
                I64AtomicRmw16SubU { .. } => regular_op_cost,
                I64AtomicRmw32SubU { .. } => regular_op_cost,
                I32AtomicRmwAnd { .. } => regular_op_cost,
                I64AtomicRmwAnd { .. } => regular_op_cost,
                I32AtomicRmw8AndU { .. } => regular_op_cost,
                I32AtomicRmw16AndU { .. } => regular_op_cost,
                I64AtomicRmw8AndU { .. } => regular_op_cost,
                I64AtomicRmw16AndU { .. } => regular_op_cost,
                I64AtomicRmw32AndU { .. } => regular_op_cost,
                I32AtomicRmwOr { .. } => regular_op_cost,
                I64AtomicRmwOr { .. } => regular_op_cost,
                I32AtomicRmw8OrU { .. } => regular_op_cost,
                I32AtomicRmw16OrU { .. } => regular_op_cost,
                I64AtomicRmw8OrU { .. } => regular_op_cost,
                I64AtomicRmw16OrU { .. } => regular_op_cost,
                I64AtomicRmw32OrU { .. } => regular_op_cost,
                I32AtomicRmwXor { .. } => regular_op_cost,
                I64AtomicRmwXor { .. } => regular_op_cost,
                I32AtomicRmw8XorU { .. } => regular_op_cost,
                I32AtomicRmw16XorU { .. } => regular_op_cost,
                I64AtomicRmw8XorU { .. } => regular_op_cost,
                I64AtomicRmw16XorU { .. } => regular_op_cost,
                I64AtomicRmw32XorU { .. } => regular_op_cost,
                I32AtomicRmwXchg { .. } => regular_op_cost,
                I64AtomicRmwXchg { .. } => regular_op_cost,
                I32AtomicRmw8XchgU { .. } => regular_op_cost,
                I32AtomicRmw16XchgU { .. } => regular_op_cost,
                I64AtomicRmw8XchgU { .. } => regular_op_cost,
                I64AtomicRmw16XchgU { .. } => regular_op_cost,
                I64AtomicRmw32XchgU { .. } => regular_op_cost,
                I32AtomicRmwCmpxchg { .. } => regular_op_cost,
                I64AtomicRmwCmpxchg { .. } => regular_op_cost,
                I32AtomicRmw8CmpxchgU { .. } => regular_op_cost,
                I32AtomicRmw16CmpxchgU { .. } => regular_op_cost,
                I64AtomicRmw8CmpxchgU { .. } => regular_op_cost,
                I64AtomicRmw16CmpxchgU { .. } => regular_op_cost,
                I64AtomicRmw32CmpxchgU { .. } => regular_op_cost,

                // 0xFD operators
                // SIMD https://webassembly.github.io/simd/core/binary/instructions.html
                V128Load { .. } => regular_op_cost,
                V128Load8x8S { .. } => regular_op_cost,
                V128Load8x8U { .. } => regular_op_cost,
                V128Load16x4S { .. } => regular_op_cost,
                V128Load16x4U { .. } => regular_op_cost,
                V128Load32x2S { .. } => regular_op_cost,
                V128Load32x2U { .. } => regular_op_cost,
                V128Load8Splat { .. } => regular_op_cost,
                V128Load16Splat { .. } => regular_op_cost,
                V128Load32Splat { .. } => regular_op_cost,
                V128Load64Splat { .. } => regular_op_cost,
                V128Load32Zero { .. } => regular_op_cost,
                V128Load64Zero { .. } => regular_op_cost,
                V128Store { .. } => regular_op_cost,
                V128Load8Lane { .. } => regular_op_cost,
                V128Load16Lane { .. } => regular_op_cost,
                V128Load32Lane { .. } => regular_op_cost,
                V128Load64Lane { .. } => regular_op_cost,
                V128Store8Lane { .. } => regular_op_cost,
                V128Store16Lane { .. } => regular_op_cost,
                V128Store32Lane { .. } => regular_op_cost,
                V128Store64Lane { .. } => regular_op_cost,
                V128Const { .. } => regular_op_cost,
                I8x16Shuffle { .. } => regular_op_cost,
                I8x16ExtractLaneS { .. } => regular_op_cost,
                I8x16ExtractLaneU { .. } => regular_op_cost,
                I8x16ReplaceLane { .. } => regular_op_cost,
                I16x8ExtractLaneS { .. } => regular_op_cost,
                I16x8ExtractLaneU { .. } => regular_op_cost,
                I16x8ReplaceLane { .. } => regular_op_cost,
                I32x4ExtractLane { .. } => regular_op_cost,
                I32x4ReplaceLane { .. } => regular_op_cost,
                I64x2ExtractLane { .. } => regular_op_cost,
                I64x2ReplaceLane { .. } => regular_op_cost,
                F32x4ExtractLane { .. } => regular_op_cost,
                F32x4ReplaceLane { .. } => regular_op_cost,
                F64x2ExtractLane { .. } => regular_op_cost,
                F64x2ReplaceLane { .. } => regular_op_cost,
                I8x16Swizzle => regular_op_cost,
                I8x16Splat => regular_op_cost,
                I16x8Splat => regular_op_cost,
                I32x4Splat => regular_op_cost,
                I64x2Splat => regular_op_cost,
                F32x4Splat => regular_op_cost,
                F64x2Splat => regular_op_cost,
                I8x16Eq => regular_op_cost,
                I8x16Ne => regular_op_cost,
                I8x16LtS => regular_op_cost,
                I8x16LtU => regular_op_cost,
                I8x16GtS => regular_op_cost,
                I8x16GtU => regular_op_cost,
                I8x16LeS => regular_op_cost,
                I8x16LeU => regular_op_cost,
                I8x16GeS => regular_op_cost,
                I8x16GeU => regular_op_cost,
                I16x8Eq => regular_op_cost,
                I16x8Ne => regular_op_cost,
                I16x8LtS => regular_op_cost,
                I16x8LtU => regular_op_cost,
                I16x8GtS => regular_op_cost,
                I16x8GtU => regular_op_cost,
                I16x8LeS => regular_op_cost,
                I16x8LeU => regular_op_cost,
                I16x8GeS => regular_op_cost,
                I16x8GeU => regular_op_cost,
                I32x4Eq => regular_op_cost,
                I32x4Ne => regular_op_cost,
                I32x4LtS => regular_op_cost,
                I32x4LtU => regular_op_cost,
                I32x4GtS => regular_op_cost,
                I32x4GtU => regular_op_cost,
                I32x4LeS => regular_op_cost,
                I32x4LeU => regular_op_cost,
                I32x4GeS => regular_op_cost,
                I32x4GeU => regular_op_cost,
                I64x2Eq => regular_op_cost,
                I64x2Ne => regular_op_cost,
                I64x2LtS => regular_op_cost,
                I64x2GtS => regular_op_cost,
                I64x2LeS => regular_op_cost,
                I64x2GeS => regular_op_cost,
                F32x4Eq => regular_op_cost,
                F32x4Ne => regular_op_cost,
                F32x4Lt => regular_op_cost,
                F32x4Gt => regular_op_cost,
                F32x4Le => regular_op_cost,
                F32x4Ge => regular_op_cost,
                F64x2Eq => regular_op_cost,
                F64x2Ne => regular_op_cost,
                F64x2Lt => regular_op_cost,
                F64x2Gt => regular_op_cost,
                F64x2Le => regular_op_cost,
                F64x2Ge => regular_op_cost,
                V128Not => regular_op_cost,
                V128And => regular_op_cost,
                V128AndNot => regular_op_cost,
                V128Or => regular_op_cost,
                V128Xor => regular_op_cost,
                V128Bitselect => regular_op_cost,
                V128AnyTrue => regular_op_cost,
                I8x16Abs => regular_op_cost,
                I8x16Neg => regular_op_cost,
                I8x16Popcnt => regular_op_cost,
                I8x16AllTrue => regular_op_cost,
                I8x16Bitmask => regular_op_cost,
                I8x16NarrowI16x8S => regular_op_cost,
                I8x16NarrowI16x8U => regular_op_cost,
                I8x16Shl => regular_op_cost,
                I8x16ShrS => regular_op_cost,
                I8x16ShrU => regular_op_cost,
                I8x16Add => regular_op_cost,
                I8x16AddSatS => regular_op_cost,
                I8x16AddSatU => regular_op_cost,
                I8x16Sub => regular_op_cost,
                I8x16SubSatS => regular_op_cost,
                I8x16SubSatU => regular_op_cost,
                I8x16MinS => regular_op_cost,
                I8x16MinU => regular_op_cost,
                I8x16MaxS => regular_op_cost,
                I8x16MaxU => regular_op_cost,
                I8x16RoundingAverageU => regular_op_cost,
                I16x8ExtAddPairwiseI8x16S => regular_op_cost,
                I16x8ExtAddPairwiseI8x16U => regular_op_cost,
                I16x8Abs => regular_op_cost,
                I16x8Neg => regular_op_cost,
                I16x8Q15MulrSatS => regular_op_cost,
                I16x8AllTrue => regular_op_cost,
                I16x8Bitmask => regular_op_cost,
                I16x8NarrowI32x4S => regular_op_cost,
                I16x8NarrowI32x4U => regular_op_cost,
                I16x8ExtendLowI8x16S => regular_op_cost,
                I16x8ExtendHighI8x16S => regular_op_cost,
                I16x8ExtendLowI8x16U => regular_op_cost,
                I16x8ExtendHighI8x16U => regular_op_cost,
                I16x8Shl => regular_op_cost,
                I16x8ShrS => regular_op_cost,
                I16x8ShrU => regular_op_cost,
                I16x8Add => regular_op_cost,
                I16x8AddSatS => regular_op_cost,
                I16x8AddSatU => regular_op_cost,
                I16x8Sub => regular_op_cost,
                I16x8SubSatS => regular_op_cost,
                I16x8SubSatU => regular_op_cost,
                I16x8Mul => regular_op_cost,
                I16x8MinS => regular_op_cost,
                I16x8MinU => regular_op_cost,
                I16x8MaxS => regular_op_cost,
                I16x8MaxU => regular_op_cost,
                I16x8RoundingAverageU => regular_op_cost,
                I16x8ExtMulLowI8x16S => regular_op_cost,
                I16x8ExtMulHighI8x16S => regular_op_cost,
                I16x8ExtMulLowI8x16U => regular_op_cost,
                I16x8ExtMulHighI8x16U => regular_op_cost,
                I32x4ExtAddPairwiseI16x8S => regular_op_cost,
                I32x4ExtAddPairwiseI16x8U => regular_op_cost,
                I32x4Abs => regular_op_cost,
                I32x4Neg => regular_op_cost,
                I32x4AllTrue => regular_op_cost,
                I32x4Bitmask => regular_op_cost,
                I32x4ExtendLowI16x8S => regular_op_cost,
                I32x4ExtendHighI16x8S => regular_op_cost,
                I32x4ExtendLowI16x8U => regular_op_cost,
                I32x4ExtendHighI16x8U => regular_op_cost,
                I32x4Shl => regular_op_cost,
                I32x4ShrS => regular_op_cost,
                I32x4ShrU => regular_op_cost,
                I32x4Add => regular_op_cost,
                I32x4Sub => regular_op_cost,
                I32x4Mul => regular_op_cost,
                I32x4MinS => regular_op_cost,
                I32x4MinU => regular_op_cost,
                I32x4MaxS => regular_op_cost,
                I32x4MaxU => regular_op_cost,
                I32x4DotI16x8S => regular_op_cost,
                I32x4ExtMulLowI16x8S => regular_op_cost,
                I32x4ExtMulHighI16x8S => regular_op_cost,
                I32x4ExtMulLowI16x8U => regular_op_cost,
                I32x4ExtMulHighI16x8U => regular_op_cost,
                I64x2Abs => regular_op_cost,
                I64x2Neg => regular_op_cost,
                I64x2AllTrue => regular_op_cost,
                I64x2Bitmask => regular_op_cost,
                I64x2ExtendLowI32x4S => regular_op_cost,
                I64x2ExtendHighI32x4S => regular_op_cost,
                I64x2ExtendLowI32x4U => regular_op_cost,
                I64x2ExtendHighI32x4U => regular_op_cost,
                I64x2Shl => regular_op_cost,
                I64x2ShrS => regular_op_cost,
                I64x2ShrU => regular_op_cost,
                I64x2Add => regular_op_cost,
                I64x2Sub => regular_op_cost,
                I64x2Mul => regular_op_cost,
                I64x2ExtMulLowI32x4S => regular_op_cost,
                I64x2ExtMulHighI32x4S => regular_op_cost,
                I64x2ExtMulLowI32x4U => regular_op_cost,
                I64x2ExtMulHighI32x4U => regular_op_cost,
                F32x4Ceil => regular_op_cost,
                F32x4Floor => regular_op_cost,
                F32x4Trunc => regular_op_cost,
                F32x4Nearest => regular_op_cost,
                F32x4Abs => regular_op_cost,
                F32x4Neg => regular_op_cost,
                F32x4Sqrt => regular_op_cost,
                F32x4Add => regular_op_cost,
                F32x4Sub => regular_op_cost,
                F32x4Mul => regular_op_cost,
                F32x4Div => regular_op_cost,
                F32x4Min => regular_op_cost,
                F32x4Max => regular_op_cost,
                F32x4PMin => regular_op_cost,
                F32x4PMax => regular_op_cost,
                F64x2Ceil => regular_op_cost,
                F64x2Floor => regular_op_cost,
                F64x2Trunc => regular_op_cost,
                F64x2Nearest => regular_op_cost,
                F64x2Abs => regular_op_cost,
                F64x2Neg => regular_op_cost,
                F64x2Sqrt => regular_op_cost,
                F64x2Add => regular_op_cost,
                F64x2Sub => regular_op_cost,
                F64x2Mul => regular_op_cost,
                F64x2Div => regular_op_cost,
                F64x2Min => regular_op_cost,
                F64x2Max => regular_op_cost,
                F64x2PMin => regular_op_cost,
                F64x2PMax => regular_op_cost,
                I32x4TruncSatF32x4S => regular_op_cost,
                I32x4TruncSatF32x4U => regular_op_cost,
                F32x4ConvertI32x4S => regular_op_cost,
                F32x4ConvertI32x4U => regular_op_cost,
                I32x4TruncSatF64x2SZero => regular_op_cost,
                I32x4TruncSatF64x2UZero => regular_op_cost,
                F64x2ConvertLowI32x4S => regular_op_cost,
                F64x2ConvertLowI32x4U => regular_op_cost,
                F32x4DemoteF64x2Zero => regular_op_cost,
                F64x2PromoteLowF32x4 => regular_op_cost,
            }
        };

        let mut compiler_config = Compiler::default();
        if module_config.has_metering {
            let metering = Arc::new(Metering::new(0, cost_function));
            compiler_config.push_middleware(metering);
        } else {
            let metering = Arc::new(Metering::new(0, |_| 0));
            compiler_config.push_middleware(metering);
        }
        Ok(compiler_config)
    }
}
