// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//
// Copyright (c) DUSK NETWORK. All rights reserved.

extern crate alloc;

use crate::RuskModule;
use alloc::vec::Vec;
use dusk_abi::ContractId;
use dusk_abi::Module;
use dusk_bls12_381::BlsScalar;
use dusk_bls12_381_sign::{
    PublicKey as BlsPublicKey, Signature as BlsSignature,
};
use dusk_bytes::Serializable;
use dusk_pki::PublicKey;
use dusk_schnorr::Signature;

use crate::{PaymentInfo, PublicInput, PAYMENT_INFO};

pub fn hash(bytes: Vec<u8>) -> BlsScalar {
    dusk_abi::query(&RuskModule::id(), &(RuskModule::HASH, bytes), 0)
        .expect("query RuskModule for Hash should not fail")
}

pub fn contract_to_scalar(address: &ContractId) -> BlsScalar {
    // TODO provisory fn until native ContractId -> BlsScalar conversion is
    // implemented
    // https://github.com/dusk-network/cargo-bake/issues/1
    let mut bls_address = [0u8; 32];

    bls_address.copy_from_slice(address.as_bytes());

    // Infallible conversion
    BlsScalar::from_bytes(&bls_address).unwrap_or_default()
}

pub fn poseidon_hash(scalars: Vec<BlsScalar>) -> BlsScalar {
    dusk_abi::query(&RuskModule::id(), &(RuskModule::POSEIDON_HASH, scalars), 0)
        .expect("query RuskModule for Poseidon Hash should not fail")
}

pub fn verify_proof(
    proof: Vec<u8>,
    verifier_data: Vec<u8>,
    pi: Vec<PublicInput>,
) -> bool {
    dusk_abi::query(
        &RuskModule::id(),
        &(RuskModule::VERIFY_PROOF, proof, verifier_data, pi),
        0,
    )
    .expect("query RuskModule for verify a proof should not fail")
}

pub fn verify_schnorr_sign(
    sign: Signature,
    pk: PublicKey,
    message: BlsScalar,
) -> bool {
    dusk_abi::query(
        &RuskModule::id(),
        &(RuskModule::VERIFY_SCHNORR_SIGN, sign, pk, message),
        0,
    )
    .expect("query RuskModule for verifying schnorr signature should not fail")
}

pub fn verify_bls_sign(
    sign: BlsSignature,
    pk: BlsPublicKey,
    message: Vec<u8>,
) -> bool {
    dusk_abi::query(
        &RuskModule::id(),
        &(RuskModule::VERIFY_BLS_SIGN, sign, pk, message),
        0,
    )
    .expect("query RuskModule for verifying bls signature should not fail")
}

/// FIXME: Until this is not moved to be part of Cake! we will reserve the 0
/// Query idx for this payable info.
pub fn payment_info(addr: ContractId) -> PaymentInfo {
    dusk_abi::query(&addr, &PAYMENT_INFO, 0)
        .expect("Failed to retrieve payment info from the specified address")
}
