use crate::{visitor, ExitCode, SharedFlags};
use anyhow::{Context, Result};
use rune::compile::FileSourceLoader;
use rune::termcolor::StandardStream;
use rune::{Diagnostics, Options, Source, Sources};
use std::io::Write;
use std::path::Path;
use structopt::StructOpt;

#[derive(StructOpt, Debug, Clone)]
pub(crate) struct Flags {
    /// Exit with a non-zero exit-code even for warnings
    #[structopt(long)]
    warnings_are_errors: bool,

    #[structopt(flatten)]
    pub(crate) shared: SharedFlags,
}

pub(crate) fn run(
    o: &mut StandardStream,
    flags: &Flags,
    options: &Options,
    path: &Path,
) -> Result<ExitCode> {
    writeln!(o, "Checking: {}", path.display())?;

    let context = flags.shared.context()?;

    let source =
        Source::from_path(path).with_context(|| format!("reading file: {}", path.display()))?;

    let mut sources = Sources::new();

    sources.insert(source);

    let mut diagnostics = if flags.shared.warnings || flags.warnings_are_errors {
        Diagnostics::new()
    } else {
        Diagnostics::without_warnings()
    };

    let mut test_finder = visitor::FunctionVisitor::new(visitor::Attribute::None);
    let mut source_loader = FileSourceLoader::new();

    let _ = rune::prepare(&mut sources)
        .with_context(&context)
        .with_diagnostics(&mut diagnostics)
        .with_options(options)
        .with_visitor(&mut test_finder)
        .with_source_loader(&mut source_loader)
        .build();

    diagnostics.emit(o, &sources)?;

    if diagnostics.has_error() || flags.warnings_are_errors && diagnostics.has_warning() {
        Ok(ExitCode::Failure)
    } else {
        Ok(ExitCode::Success)
    }
}
