//! `POST /_matrix/identity/*/3pid/bind`
//!
//! Endpoint to bind a session to a Matrix user ID.

pub mod v2 {
    //! `/v2/` ([spec])
    //!
    //! [spec]: https://spec.matrix.org/v1.2/identity-service-api/#post_matrixidentityv23pidbind

    use ruma_common::{
        api::ruma_api, thirdparty::Medium, ClientSecret, MilliSecondsSinceUnixEpoch, OwnedUserId,
        ServerSignatures, SessionId, UserId,
    };

    ruma_api! {
        metadata: {
            description: "Publish an association between a session and a Matrix user ID.",
            method: POST,
            name: "bind_3pid",
            stable_path: "/_matrix/identity/v2/3pid/bind",
            rate_limited: false,
            authentication: AccessToken,
            added: 1.0,
        }

        request: {
            /// The session ID generated by the `requestToken` call.
            pub sid: &'a SessionId,

            /// The client secret passed to the `requestToken` call.
            pub client_secret: &'a ClientSecret,

            /// The Matrix user ID to associate with the 3PIDs.
            pub mxid: &'a UserId,
        }

        response: {
            /// The 3PID address of the user being looked up.
            pub address: String,

            /// The medium type of the 3PID.
            pub medium: Medium,

            /// The Matrix user ID associated with the 3PID.
            pub mxid: OwnedUserId,

            /// A UNIX timestamp before which the association is not known to be valid.
            pub not_before: MilliSecondsSinceUnixEpoch,

            /// A UNIX timestamp after which the association is not known to be valid.
            pub not_after: MilliSecondsSinceUnixEpoch,

            /// The UNIX timestamp at which the association was verified.
            pub ts: MilliSecondsSinceUnixEpoch,

            /// The signatures of the verifiying identity servers which show that the
            /// association should be trusted, if you trust the verifying identity services.
            pub signatures: ServerSignatures,
        }
    }

    impl<'a> Request<'a> {
        /// Creates a `Request` with the given session ID, client secret and Matrix user ID.
        pub fn new(sid: &'a SessionId, client_secret: &'a ClientSecret, mxid: &'a UserId) -> Self {
            Self { sid, client_secret, mxid }
        }
    }

    impl Response {
        /// Creates a `Response` with the given 3PID address, medium, Matrix user ID, timestamps and
        /// signatures.
        pub fn new(
            address: String,
            medium: Medium,
            mxid: OwnedUserId,
            not_before: MilliSecondsSinceUnixEpoch,
            not_after: MilliSecondsSinceUnixEpoch,
            ts: MilliSecondsSinceUnixEpoch,
            signatures: ServerSignatures,
        ) -> Self {
            Self { address, medium, mxid, not_before, not_after, ts, signatures }
        }
    }
}
