// Copyright 2022 tison <wander4096@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use std::io::Cursor;

use bytes::Buf;

use crate::{Error, Result};

#[derive(Debug)]
pub enum Model {
    Error(String),
    Status(String),
    Integer(i64),
    // Double,
    Nil,
    String(Vec<u8>),
    Array(Vec<Model>),
    Map(Vec<(Model, Model)>),
    Set(Vec<Model>),
    Bool(bool),
    // Verb,
    // Push,
    // BigNum,
}

pub fn check(cursor: &mut Cursor<&[u8]>) -> Result<()> {
    if !cursor.has_remaining() {
        return Err(Error::Incomplete);
    }

    match cursor.get_u8() {
        b'-' | b'+' | b':' | b'_' | b'#' => {
            readline(cursor)?;
        }
        b'$' => {
            let len = readline(cursor)?;
            let len = atoi::atoi::<i64>(len).ok_or_else(|| Error::unknown("malformed"))?;
            if len >= 0 {
                readline(cursor)?;
            }
        }
        b'~' | b'*' => {
            let len = readline(cursor)?;
            let len = atoi::atoi::<i64>(len).ok_or_else(|| Error::unknown("malformed"))?;
            for _ in 0..len {
                check(cursor)?;
            }
        }
        b'%' => {
            let len = readline(cursor)?;
            let len = atoi::atoi::<i64>(len).ok_or_else(|| Error::unknown("malformed"))?;
            let len = len * 2;
            for _ in 0..len {
                check(cursor)?;
            }
        }
        b => return Err(Error::unknown(format!("unknown type: {}", b))),
    }
    Ok(())
}

pub fn parse(cursor: &mut Cursor<&[u8]>) -> Result<Model> {
    let model = match cursor.get_u8() {
        b'-' => {
            let error = readline(cursor)?;
            let error = String::from_utf8(error.to_vec()).map_err(Error::unknown)?;
            Ok(Model::Error(error))
        }
        b'+' => {
            let status = readline(cursor)?;
            let status = String::from_utf8(status.to_vec()).map_err(Error::unknown)?;
            Ok(Model::Status(status))
        }
        b':' => {
            let integer = readline(cursor)?;
            match atoi::atoi::<i64>(integer) {
                None => Err(Error::unknown(format!("malformed integer: {:?}", integer))),
                Some(i) => Ok(Model::Integer(i)),
            }
        }
        b'_' => {
            let remaining = readline(cursor)?;
            if remaining.is_empty() {
                Ok(Model::Nil)
            } else {
                Err(Error::unknown(format!("malformed nil: {:?}", remaining)))
            }
        }
        b'#' => {
            let remaining = readline(cursor)?;
            if remaining.len() != 1 {
                Err(Error::unknown(format!("malformed bool: {:?}", remaining)))
            } else if remaining[0].to_ascii_lowercase() == b't' {
                Ok(Model::Bool(true))
            } else if remaining[0].to_ascii_lowercase() == b'f' {
                Ok(Model::Bool(false))
            } else {
                Err(Error::unknown(format!("malformed bool: {:?}", remaining)))
            }
        }
        b'$' => {
            let len = readline(cursor)?;
            match atoi::atoi::<i64>(len) {
                None => Err(Error::unknown(format!("malformed len: {:?}", len))),
                Some(len) if len < -1 => Err(Error::unknown(format!("malformed len: {:?}", len))),
                Some(len) if len == -1 => Ok(Model::Nil),
                Some(_) => {
                    let next = readline(cursor)?;
                    Ok(Model::String(next.to_vec()))
                }
            }
        }
        t @ (b'~' | b'*') => {
            let len = readline(cursor)?;
            match atoi::atoi::<i64>(len) {
                None => Err(Error::unknown(format!("malformed len: {:?}", len))),
                Some(len) if len < -1 => Err(Error::unknown(format!("malformed len: {:?}", len))),
                Some(len) if len == -1 => Ok(Model::Nil),
                Some(len) => {
                    let mut vec = Vec::with_capacity(len as usize);
                    for _ in 0..len {
                        vec.push(parse(cursor)?);
                    }
                    match t {
                        b'~' => Ok(Model::Set(vec)),
                        b'*' => Ok(Model::Array(vec)),
                        _ => unreachable!(),
                    }
                }
            }
        }
        b'%' => {
            let len = readline(cursor)?;
            match atoi::atoi::<i64>(len) {
                None => Err(Error::unknown(format!("malformed len: {:?}", len))),
                Some(len) if len < -1 => Err(Error::unknown(format!("malformed len: {:?}", len))),
                Some(len) if len == -1 => Ok(Model::Nil),
                Some(len) => {
                    let mut vec = Vec::with_capacity((2 * len) as usize);
                    for _ in 0..len {
                        let k = parse(cursor)?;
                        let v = parse(cursor)?;
                        vec.push((k, v));
                    }
                    Ok(Model::Map(vec))
                }
            }
        }
        b => Err(Error::unknown(format!("unknown type: {}", b))),
    }?;

    Ok(model)
}

fn readline<'a>(cursor: &mut Cursor<&'a [u8]>) -> Result<&'a [u8]> {
    let start = cursor.position() as usize;
    let end = cursor.get_ref().len() - 1;

    for i in start..end {
        let inner = cursor.get_ref();
        if inner[i] == b'\r' && inner[i + 1] == b'\n' {
            cursor.set_position((i + 2) as u64);
            return Ok(&cursor.get_ref()[start..i]);
        }
    }

    Err(Error::Incomplete)
}
