// Copyright 2022 tison <wander4096@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use tokio::net::{TcpStream, ToSocketAddrs};

use crate::{
    command::{FlushAll, Get, Ping, Set},
    Connection, Error, Model, Result,
};

pub struct Client {
    connection: Connection,
}

impl Client {
    pub async fn connect<A: ToSocketAddrs>(addr: A) -> Result<Self> {
        let socket = TcpStream::connect(addr).await?;
        let connection = Connection::new(socket);
        Ok(Self { connection })
    }

    pub async fn ping(&mut self) -> Result<()> {
        self.connection.send(Ping).await?;
        match self.connection.recv().await? {
            Some(Model::Status(status)) if status == "PONG" => Ok(()),
            Some(Model::Error(e)) => Err(Error::Redis(e)),
            m => Err(Error::unknown(format!("unknown model: {:?}", m))),
        }
    }

    pub async fn get(&mut self, key: Vec<u8>) -> Result<Option<Vec<u8>>> {
        self.connection.send(Get::new(key)).await?;
        match self.connection.recv().await? {
            Some(Model::String(result)) => Ok(Some(result)),
            Some(Model::Nil) => Ok(None),
            Some(Model::Error(e)) => Err(Error::Redis(e)),
            m => Err(Error::unknown(format!("unknown model: {:?}", m))),
        }
    }

    pub async fn set(&mut self, set: Set) -> Result<Option<Vec<u8>>> {
        self.connection.send(set).await?;
        match self.connection.recv().await? {
            Some(Model::Status(status)) if status == "OK" => Ok(Some(vec![])),
            Some(Model::Nil) => Ok(None),
            Some(Model::String(result)) => Ok(Some(result)),
            Some(Model::Error(e)) => Err(Error::Redis(e)),
            m => Err(Error::unknown(format!("unknown model: {:?}", m))),
        }
    }

    pub async fn flush_all(&mut self, sync: bool) -> Result<()> {
        self.connection.send(FlushAll::new(sync)).await?;
        match self.connection.recv().await? {
            Some(Model::Error(e)) => Err(Error::Redis(e)),
            Some(Model::Status(status)) if status == "OK" => Ok(()),
            m => Err(Error::unknown(format!("unknown model: {:?}", m))),
        }
    }
}
