//! Company identifier implementation

use core::fmt;

/// Company identifier for use in link layer Control PDUs.
#[derive(Copy, Clone)]
pub struct CompanyId(u16);

impl fmt::Debug for CompanyId {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "CompanyId(0x{:X})", self.as_u16())
    }
}

impl CompanyId {
    /// Create a company ID from a raw `u16`.
    pub fn from_raw(raw: u16) -> Self {
        Self(raw)
    }

    /// Get the raw `u16` representing this company identifier.
    pub fn as_u16(&self) -> u16 {
        self.0
    }

    /// Get the full company name from this identifer.
    ///
    /// **WARNING**: Using this method anywhere will add roughly 70KB to your binary size!
    pub fn name(&self) -> Option<&'static str> {
        match self.0 {
            0x0000 => Some("Ericsson Technology Licensing"),
            0x0001 => Some("Nokia Mobile Phones"),
            0x0002 => Some("Intel Corp."),
            0x0003 => Some("IBM Corp."),
            0x0004 => Some("Toshiba Corp."),
            0x0005 => Some("3Com"),
            0x0006 => Some("Microsoft"),
            0x0007 => Some("Lucent"),
            0x0008 => Some("Motorola"),
            0x0009 => Some("Infineon Technologies AG"),
            0x000A => Some("Qualcomm Technologies International, Ltd. (QTIL)"),
            0x000B => Some("Silicon Wave"),
            0x000C => Some("Digianswer A/S"),
            0x000D => Some("Texas Instruments Inc."),
            0x000E => Some("Parthus Technologies Inc."),
            0x000F => Some("Broadcom Corporation"),
            0x0010 => Some("Mitel Semiconductor"),
            0x0011 => Some("Widcomm, Inc."),
            0x0012 => Some("Zeevo, Inc."),
            0x0013 => Some("Atmel Corporation"),
            0x0014 => Some("Mitsubishi Electric Corporation"),
            0x0015 => Some("RTX Telecom A/S"),
            0x0016 => Some("KC Technology Inc."),
            0x0017 => Some("Newlogic"),
            0x0018 => Some("Transilica, Inc."),
            0x0019 => Some("Rohde & Schwarz GmbH & Co. KG"),
            0x001A => Some("TTPCom Limited"),
            0x001B => Some("Signia Technologies, Inc."),
            0x001C => Some("Conexant Systems Inc."),
            0x001D => Some("Qualcomm"),
            0x001E => Some("Inventel"),
            0x001F => Some("AVM Berlin"),
            0x0020 => Some("BandSpeed, Inc."),
            0x0021 => Some("Mansella Ltd"),
            0x0022 => Some("NEC Corporation"),
            0x0023 => Some("WavePlus Technology Co., Ltd."),
            0x0024 => Some("Alcatel"),
            0x0025 => Some("NXP Semiconductors (formerly Philips Semiconductors)"),
            0x0026 => Some("C Technologies"),
            0x0027 => Some("Open Interface"),
            0x0028 => Some("R F Micro Devices"),
            0x0029 => Some("Hitachi Ltd"),
            0x002A => Some("Symbol Technologies, Inc."),
            0x002B => Some("Tenovis"),
            0x002C => Some("Macronix International Co. Ltd."),
            0x002D => Some("GCT Semiconductor"),
            0x002E => Some("Norwood Systems"),
            0x002F => Some("MewTel Technology Inc."),
            0x0030 => Some("ST Microelectronics"),
            0x0031 => Some("Synopsys, Inc."),
            0x0032 => Some("Red-M (Communications) Ltd"),
            0x0033 => Some("Commil Ltd"),
            0x0034 => Some("Computer Access Technology Corporation (CATC)"),
            0x0035 => Some("Eclipse (HQ Espana) S.L."),
            0x0036 => Some("Renesas Electronics Corporation"),
            0x0037 => Some("Mobilian Corporation"),
            0x0038 => Some("Syntronix Corporation"),
            0x0039 => Some("Integrated System Solution Corp."),
            0x003A => Some("Matsushita Electric Industrial Co., Ltd."),
            0x003B => Some("Gennum Corporation"),
            0x003C => Some("BlackBerry Limited (formerly Research In Motion)"),
            0x003D => Some("IPextreme, Inc."),
            0x003E => Some("Systems and Chips, Inc"),
            0x003F => Some("Bluetooth SIG, Inc"),
            0x0040 => Some("Seiko Epson Corporation"),
            0x0041 => Some("Integrated Silicon Solution Taiwan, Inc."),
            0x0042 => Some("CONWISE Technology Corporation Ltd"),
            0x0043 => Some("PARROT AUTOMOTIVE SAS"),
            0x0044 => Some("Socket Mobile"),
            0x0045 => Some("Atheros Communications, Inc."),
            0x0046 => Some("MediaTek, Inc."),
            0x0047 => Some("Bluegiga"),
            0x0048 => Some("Marvell Technology Group Ltd."),
            0x0049 => Some("3DSP Corporation"),
            0x004A => Some("Accel Semiconductor Ltd."),
            0x004B => Some("Continental Automotive Systems"),
            0x004C => Some("Apple, Inc."),
            0x004D => Some("Staccato Communications, Inc."),
            0x004E => Some("Avago Technologies"),
            0x004F => Some("APT Ltd."),
            0x0050 => Some("SiRF Technology, Inc."),
            0x0051 => Some("Tzero Technologies, Inc."),
            0x0052 => Some("J&M Corporation"),
            0x0053 => Some("Free2move AB"),
            0x0054 => Some("3DiJoy Corporation"),
            0x0055 => Some("Plantronics, Inc."),
            0x0056 => Some("Sony Ericsson Mobile Communications"),
            0x0057 => Some("Harman International Industries, Inc."),
            0x0058 => Some("Vizio, Inc."),
            0x0059 => Some("Nordic Semiconductor ASA"),
            0x005A => Some("EM Microelectronic-Marin SA"),
            0x005B => Some("Ralink Technology Corporation"),
            0x005C => Some("Belkin International, Inc."),
            0x005D => Some("Realtek Semiconductor Corporation"),
            0x005E => Some("Stonestreet One, LLC"),
            0x005F => Some("Wicentric, Inc."),
            0x0060 => Some("RivieraWaves S.A.S"),
            0x0061 => Some("RDA Microelectronics"),
            0x0062 => Some("Gibson Guitars"),
            0x0063 => Some("MiCommand Inc."),
            0x0064 => Some("Band XI International, LLC"),
            0x0065 => Some("Hewlett-Packard Company"),
            0x0066 => Some("9Solutions Oy"),
            0x0067 => Some("GN Netcom A/S"),
            0x0068 => Some("General Motors"),
            0x0069 => Some("A&D Engineering, Inc."),
            0x006A => Some("MindTree Ltd."),
            0x006B => Some("Polar Electro OY"),
            0x006C => Some("Beautiful Enterprise Co., Ltd."),
            0x006D => Some("BriarTek, Inc"),
            0x006E => Some("Summit Data Communications, Inc."),
            0x006F => Some("Sound ID"),
            0x0070 => Some("Monster, LLC"),
            0x0071 => Some("connectBlue AB"),
            0x0072 => Some("ShangHai Super Smart Electronics Co. Ltd."),
            0x0073 => Some("Group Sense Ltd."),
            0x0074 => Some("Zomm, LLC"),
            0x0075 => Some("Samsung Electronics Co. Ltd."),
            0x0076 => Some("Creative Technology Ltd."),
            0x0077 => Some("Laird Technologies"),
            0x0078 => Some("Nike, Inc."),
            0x0079 => Some("lesswire AG"),
            0x007A => Some("MStar Semiconductor, Inc."),
            0x007B => Some("Hanlynn Technologies"),
            0x007C => Some("A & R Cambridge"),
            0x007D => Some("Seers Technology Co., Ltd."),
            0x007E => Some("Sports Tracking Technologies Ltd."),
            0x007F => Some("Autonet Mobile"),
            0x0080 => Some("DeLorme Publishing Company, Inc."),
            0x0081 => Some("WuXi Vimicro"),
            0x0082 => Some("Sennheiser Communications A/S"),
            0x0083 => Some("TimeKeeping Systems, Inc."),
            0x0084 => Some("Ludus Helsinki Ltd."),
            0x0085 => Some("BlueRadios, Inc."),
            0x0086 => Some("Equinux AG"),
            0x0087 => Some("Garmin International, Inc."),
            0x0088 => Some("Ecotest"),
            0x0089 => Some("GN ReSound A/S"),
            0x008A => Some("Jawbone"),
            0x008B => Some("Topcon Positioning Systems, LLC"),
            0x008C => Some(
                "Gimbal Inc. (formerly Qualcomm Labs, Inc. and Qualcomm Retail Solutions, Inc.)",
            ),
            0x008D => Some("Zscan Software"),
            0x008E => Some("Quintic Corp"),
            0x008F => Some("Telit Wireless Solutions GmbH (formerly Stollmann E+V GmbH)"),
            0x0090 => Some("Funai Electric Co., Ltd."),
            0x0091 => Some("Advanced PANMOBIL systems GmbH & Co. KG"),
            0x0092 => Some("ThinkOptics, Inc."),
            0x0093 => Some("Universal Electronics, Inc."),
            0x0094 => Some("Airoha Technology Corp."),
            0x0095 => Some("NEC Lighting, Ltd."),
            0x0096 => Some("ODM Technology, Inc."),
            0x0097 => Some("ConnecteDevice Ltd."),
            0x0098 => Some("zero1.tv GmbH"),
            0x0099 => Some("i.Tech Dynamic Global Distribution Ltd."),
            0x009A => Some("Alpwise"),
            0x009B => Some("Jiangsu Toppower Automotive Electronics Co., Ltd."),
            0x009C => Some("Colorfy, Inc."),
            0x009D => Some("Geoforce Inc."),
            0x009E => Some("Bose Corporation"),
            0x009F => Some("Suunto Oy"),
            0x00A0 => Some("Kensington Computer Products Group"),
            0x00A1 => Some("SR-Medizinelektronik"),
            0x00A2 => Some("Vertu Corporation Limited"),
            0x00A3 => Some("Meta Watch Ltd."),
            0x00A4 => Some("LINAK A/S"),
            0x00A5 => Some("OTL Dynamics LLC"),
            0x00A6 => Some("Panda Ocean Inc."),
            0x00A7 => Some("Visteon Corporation"),
            0x00A8 => Some("ARP Devices Limited"),
            0x00A9 => Some("Magneti Marelli S.p.A"),
            0x00AA => Some("CAEN RFID srl"),
            0x00AB => Some("Ingenieur-Systemgruppe Zahn GmbH"),
            0x00AC => Some("Green Throttle Games"),
            0x00AD => Some("Peter Systemtechnik GmbH"),
            0x00AE => Some("Omegawave Oy"),
            0x00AF => Some("Cinetix"),
            0x00B0 => Some("Passif Semiconductor Corp"),
            0x00B1 => Some("Saris Cycling Group, Inc"),
            0x00B2 => Some("Bekey A/S"),
            0x00B3 => Some("Clarinox Technologies Pty. Ltd."),
            0x00B4 => Some("BDE Technology Co., Ltd."),
            0x00B5 => Some("Swirl Networks"),
            0x00B6 => Some("Meso international"),
            0x00B7 => Some("TreLab Ltd"),
            0x00B8 => Some("Qualcomm Innovation Center, Inc. (QuIC)"),
            0x00B9 => Some("Johnson Controls, Inc."),
            0x00BA => Some("Starkey Laboratories Inc."),
            0x00BB => Some("S-Power Electronics Limited"),
            0x00BC => Some("Ace Sensor Inc"),
            0x00BD => Some("Aplix Corporation"),
            0x00BE => Some("AAMP of America"),
            0x00BF => Some("Stalmart Technology Limited"),
            0x00C0 => Some("AMICCOM Electronics Corporation"),
            0x00C1 => Some("Shenzhen Excelsecu Data Technology Co.,Ltd"),
            0x00C2 => Some("Geneq Inc."),
            0x00C3 => Some("adidas AG"),
            0x00C4 => Some("LG Electronics"),
            0x00C5 => Some("Onset Computer Corporation"),
            0x00C6 => Some("Selfly BV"),
            0x00C7 => Some("Quuppa Oy."),
            0x00C8 => Some("GeLo Inc"),
            0x00C9 => Some("Evluma"),
            0x00CA => Some("MC10"),
            0x00CB => Some("Binauric SE"),
            0x00CC => Some("Beats Electronics"),
            0x00CD => Some("Microchip Technology Inc."),
            0x00CE => Some("Elgato Systems GmbH"),
            0x00CF => Some("ARCHOS SA"),
            0x00D0 => Some("Dexcom, Inc."),
            0x00D1 => Some("Polar Electro Europe B.V."),
            0x00D2 => Some("Dialog Semiconductor B.V."),
            0x00D3 => Some("Taixingbang Technology (HK) Co,. LTD."),
            0x00D4 => Some("Kawantech"),
            0x00D5 => Some("Austco Communication Systems"),
            0x00D6 => Some("Timex Group USA, Inc."),
            0x00D7 => Some("Qualcomm Technologies, Inc."),
            0x00D8 => Some("Qualcomm Connected Experiences, Inc."),
            0x00D9 => Some("Voyetra Turtle Beach"),
            0x00DA => Some("txtr GmbH"),
            0x00DB => Some("Biosentronics"),
            0x00DC => Some("Procter & Gamble"),
            0x00DD => Some("Hosiden Corporation"),
            0x00DE => Some("Muzik LLC"),
            0x00DF => Some("Misfit Wearables Corp"),
            0x00E0 => Some("Google"),
            0x00E1 => Some("Danlers Ltd"),
            0x00E2 => Some("Semilink Inc"),
            0x00E3 => Some("inMusic Brands, Inc"),
            0x00E4 => Some("L.S. Research Inc."),
            0x00E5 => Some("Eden Software Consultants Ltd."),
            0x00E6 => Some("Freshtemp"),
            0x00E7 => Some("KS Technologies"),
            0x00E8 => Some("ACTS Technologies"),
            0x00E9 => Some("Vtrack Systems"),
            0x00EA => Some("Nielsen-Kellerman Company"),
            0x00EB => Some("Server Technology Inc."),
            0x00EC => Some("BioResearch Associates"),
            0x00ED => Some("Jolly Logic, LLC"),
            0x00EE => Some("Above Average Outcomes, Inc."),
            0x00EF => Some("Bitsplitters GmbH"),
            0x00F0 => Some("PayPal, Inc."),
            0x00F1 => Some("Witron Technology Limited"),
            0x00F2 => Some("Morse Project Inc."),
            0x00F3 => Some("Kent Displays Inc."),
            0x00F4 => Some("Nautilus Inc."),
            0x00F5 => Some("Smartifier Oy"),
            0x00F6 => Some("Elcometer Limited"),
            0x00F7 => Some("VSN Technologies, Inc."),
            0x00F8 => Some("AceUni Corp., Ltd."),
            0x00F9 => Some("StickNFind"),
            0x00FA => Some("Crystal Code AB"),
            0x00FB => Some("KOUKAAM a.s."),
            0x00FC => Some("Delphi Corporation"),
            0x00FD => Some("ValenceTech Limited"),
            0x00FE => Some("Stanley Black and Decker"),
            0x00FF => Some("Typo Products, LLC"),
            0x0100 => Some("TomTom International BV"),
            0x0101 => Some("Fugoo, Inc."),
            0x0102 => Some("Keiser Corporation"),
            0x0103 => Some("Bang & Olufsen A/S"),
            0x0104 => Some("PLUS Location Systems Pty Ltd"),
            0x0105 => Some("Ubiquitous Computing Technology Corporation"),
            0x0106 => Some("Innovative Yachtter Solutions"),
            0x0107 => Some("William Demant Holding A/S"),
            0x0108 => Some("Chicony Electronics Co., Ltd."),
            0x0109 => Some("Atus BV"),
            0x010A => Some("Codegate Ltd"),
            0x010B => Some("ERi, Inc"),
            0x010C => Some("Transducers Direct, LLC"),
            0x010D => Some("Fujitsu Ten LImited"),
            0x010E => Some("Audi AG"),
            0x010F => Some("HiSilicon Technologies Col, Ltd."),
            0x0110 => Some("Nippon Seiki Co., Ltd."),
            0x0111 => Some("Steelseries ApS"),
            0x0112 => Some("Visybl Inc."),
            0x0113 => Some("Openbrain Technologies, Co., Ltd."),
            0x0114 => Some("Xensr"),
            0x0115 => Some("e.solutions"),
            0x0116 => Some("10AK Technologies"),
            0x0117 => Some("Wimoto Technologies Inc"),
            0x0118 => Some("Radius Networks, Inc."),
            0x0119 => Some("Wize Technology Co., Ltd."),
            0x011A => Some("Qualcomm Labs, Inc."),
            0x011B => Some("Hewlett Packard Enterprise"),
            0x011C => Some("Baidu"),
            0x011D => Some("Arendi AG"),
            0x011E => Some("Skoda Auto a.s."),
            0x011F => Some("Volkswagen AG"),
            0x0120 => Some("Porsche AG"),
            0x0121 => Some("Sino Wealth Electronic Ltd."),
            0x0122 => Some("AirTurn, Inc."),
            0x0123 => Some("Kinsa, Inc"),
            0x0124 => Some("HID Global"),
            0x0125 => Some("SEAT es"),
            0x0126 => Some("Promethean Ltd."),
            0x0127 => Some("Salutica Allied Solutions"),
            0x0128 => Some("GPSI Group Pty Ltd"),
            0x0129 => Some("Nimble Devices Oy"),
            0x012A => Some("Changzhou Yongse Infotech Co., Ltd."),
            0x012B => Some("SportIQ"),
            0x012C => Some("TEMEC Instruments B.V."),
            0x012D => Some("Sony Corporation"),
            0x012E => Some("ASSA ABLOY"),
            0x012F => Some("Clarion Co. Inc."),
            0x0130 => Some("Warehouse Innovations"),
            0x0131 => Some("Cypress Semiconductor"),
            0x0132 => Some("MADS Inc"),
            0x0133 => Some("Blue Maestro Limited"),
            0x0134 => Some("Resolution Products, Ltd."),
            0x0135 => Some("Aireware LLC"),
            0x0136 => Some("Silvair, Inc."),
            0x0137 => Some("Prestigio Plaza Ltd."),
            0x0138 => Some("NTEO Inc."),
            0x0139 => Some("Focus Systems Corporation"),
            0x013A => Some("Tencent Holdings Ltd."),
            0x013B => Some("Allegion"),
            0x013C => Some("Murata Manufacturing Co., Ltd."),
            0x013D => Some("WirelessWERX"),
            0x013E => Some("Nod, Inc."),
            0x013F => Some("B&B Manufacturing Company"),
            0x0140 => Some("Alpine Electronics (China) Co., Ltd"),
            0x0141 => Some("FedEx Services"),
            0x0142 => Some("Grape Systems Inc."),
            0x0143 => Some("Bkon Connect"),
            0x0144 => Some("Lintech GmbH"),
            0x0145 => Some("Novatel Wireless"),
            0x0146 => Some("Ciright"),
            0x0147 => Some("Mighty Cast, Inc."),
            0x0148 => Some("Ambimat Electronics"),
            0x0149 => Some("Perytons Ltd."),
            0x014A => Some("Tivoli Audio, LLC"),
            0x014B => Some("Master Lock"),
            0x014C => Some("Mesh-Net Ltd"),
            0x014D => Some("HUIZHOU DESAY SV AUTOMOTIVE CO., LTD."),
            0x014E => Some("Tangerine, Inc."),
            0x014F => Some("B&W Group Ltd."),
            0x0150 => Some("Pioneer Corporation"),
            0x0151 => Some("OnBeep"),
            0x0152 => Some("Vernier Software & Technology"),
            0x0153 => Some("ROL Ergo"),
            0x0154 => Some("Pebble Technology"),
            0x0155 => Some("NETATMO"),
            0x0156 => Some("Accumulate AB"),
            0x0157 => Some("Anhui Huami Information Technology Co., Ltd."),
            0x0158 => Some("Inmite s.r.o."),
            0x0159 => Some("ChefSteps, Inc."),
            0x015A => Some("micas AG"),
            0x015B => Some("Biomedical Research Ltd."),
            0x015C => Some("Pitius Tec S.L."),
            0x015D => Some("Estimote, Inc."),
            0x015E => Some("Unikey Technologies, Inc."),
            0x015F => Some("Timer Cap Co."),
            0x0160 => Some("AwoX"),
            0x0161 => Some("yikes"),
            0x0162 => Some("MADSGlobalNZ Ltd."),
            0x0163 => Some("PCH International"),
            0x0164 => Some("Qingdao Yeelink Information Technology Co., Ltd."),
            0x0165 => Some("Milwaukee Tool (Formally Milwaukee Electric Tools)"),
            0x0166 => Some("MISHIK Pte Ltd"),
            0x0167 => Some("Ascensia Diabetes Care US Inc."),
            0x0168 => Some("Spicebox LLC"),
            0x0169 => Some("emberlight"),
            0x016A => Some("Cooper-Atkins Corporation"),
            0x016B => Some("Qblinks"),
            0x016C => Some("MYSPHERA"),
            0x016D => Some("LifeScan Inc"),
            0x016E => Some("Volantic AB"),
            0x016F => Some("Podo Labs, Inc"),
            0x0170 => Some("Roche Diabetes Care AG"),
            0x0171 => Some("Amazon Fulfillment Service"),
            0x0172 => Some("Connovate Technology Private Limited"),
            0x0173 => Some("Kocomojo, LLC"),
            0x0174 => Some("Everykey Inc."),
            0x0175 => Some("Dynamic Controls"),
            0x0176 => Some("SentriLock"),
            0x0177 => Some("I-SYST inc."),
            0x0178 => Some("CASIO COMPUTER CO., LTD."),
            0x0179 => Some("LAPIS Semiconductor Co., Ltd."),
            0x017A => Some("Telemonitor, Inc."),
            0x017B => Some("taskit GmbH"),
            0x017C => Some("Daimler AG"),
            0x017D => Some("BatAndCat"),
            0x017E => Some("BluDotz Ltd"),
            0x017F => Some("XTel Wireless ApS"),
            0x0180 => Some("Gigaset Communications GmbH"),
            0x0181 => Some("Gecko Health Innovations, Inc."),
            0x0182 => Some("HOP Ubiquitous"),
            0x0183 => Some("Walt Disney"),
            0x0184 => Some("Nectar"),
            0x0185 => Some("bel'apps LLC"),
            0x0186 => Some("CORE Lighting Ltd"),
            0x0187 => Some("Seraphim Sense Ltd"),
            0x0188 => Some("Unico RBC"),
            0x0189 => Some("Physical Enterprises Inc."),
            0x018A => Some("Able Trend Technology Limited"),
            0x018B => Some("Konica Minolta, Inc."),
            0x018C => Some("Wilo SE"),
            0x018D => Some("Extron Design Services"),
            0x018E => Some("Fitbit, Inc."),
            0x018F => Some("Fireflies Systems"),
            0x0190 => Some("Intelletto Technologies Inc."),
            0x0191 => Some("FDK CORPORATION"),
            0x0192 => Some("Cloudleaf, Inc"),
            0x0193 => Some("Maveric Automation LLC"),
            0x0194 => Some("Acoustic Stream Corporation"),
            0x0195 => Some("Zuli"),
            0x0196 => Some("Paxton Access Ltd"),
            0x0197 => Some("WiSilica Inc."),
            0x0198 => Some("VENGIT Korlatolt Felelossegu Tarsasag"),
            0x0199 => Some("SALTO SYSTEMS S.L."),
            0x019A => Some("TRON Forum (formerly T-Engine Forum)"),
            0x019B => Some("CUBETECH s.r.o."),
            0x019C => Some("Cokiya Incorporated"),
            0x019D => Some("CVS Health"),
            0x019E => Some("Ceruus"),
            0x019F => Some("Strainstall Ltd"),
            0x01A0 => Some("Channel Enterprises (HK) Ltd."),
            0x01A1 => Some("FIAMM"),
            0x01A2 => Some("GIGALANE.CO.,LTD"),
            0x01A3 => Some("EROAD"),
            0x01A4 => Some("Mine Safety Appliances"),
            0x01A5 => Some("Icon Health and Fitness"),
            0x01A6 => Some("Wille Engineering (formely as Asandoo GmbH)"),
            0x01A7 => Some("ENERGOUS CORPORATION"),
            0x01A8 => Some("Taobao"),
            0x01A9 => Some("Canon Inc."),
            0x01AA => Some("Geophysical Technology Inc."),
            0x01AB => Some("Facebook, Inc."),
            0x01AC => Some("Trividia Health, Inc."),
            0x01AD => Some("FlightSafety International"),
            0x01AE => Some("Earlens Corporation"),
            0x01AF => Some("Sunrise Micro Devices, Inc."),
            0x01B0 => Some("Star Micronics Co., Ltd."),
            0x01B1 => Some("Netizens Sp. z o.o."),
            0x01B2 => Some("Nymi Inc."),
            0x01B3 => Some("Nytec, Inc."),
            0x01B4 => Some("Trineo Sp. z o.o."),
            0x01B5 => Some("Nest Labs Inc."),
            0x01B6 => Some("LM Technologies Ltd"),
            0x01B7 => Some("General Electric Company"),
            0x01B8 => Some("i+D3 S.L."),
            0x01B9 => Some("HANA Micron"),
            0x01BA => Some("Stages Cycling LLC"),
            0x01BB => Some("Cochlear Bone Anchored Solutions AB"),
            0x01BC => Some("SenionLab AB"),
            0x01BD => Some("Syszone Co., Ltd"),
            0x01BE => Some("Pulsate Mobile Ltd."),
            0x01BF => Some("Hong Kong HunterSun Electronic Limited"),
            0x01C0 => Some("pironex GmbH"),
            0x01C1 => Some("BRADATECH Corp."),
            0x01C2 => Some("Transenergooil AG"),
            0x01C3 => Some("Bunch"),
            0x01C4 => Some("DME Microelectronics"),
            0x01C5 => Some("Bitcraze AB"),
            0x01C6 => Some("HASWARE Inc."),
            0x01C7 => Some("Abiogenix Inc."),
            0x01C8 => Some("Poly-Control ApS"),
            0x01C9 => Some("Avi-on"),
            0x01CA => Some("Laerdal Medical AS"),
            0x01CB => Some("Fetch My Pet"),
            0x01CC => Some("Sam Labs Ltd."),
            0x01CD => Some("Chengdu Synwing Technology Ltd"),
            0x01CE => Some("HOUWA SYSTEM DESIGN, k.k."),
            0x01CF => Some("BSH"),
            0x01D0 => Some("Primus Inter Pares Ltd"),
            0x01D1 => Some("August Home, Inc"),
            0x01D2 => Some("Gill Electronics"),
            0x01D3 => Some("Sky Wave Design"),
            0x01D4 => Some("Newlab S.r.l."),
            0x01D5 => Some("ELAD srl"),
            0x01D6 => Some("G-wearables inc."),
            0x01D7 => Some("Squadrone Systems Inc."),
            0x01D8 => Some("Code Corporation"),
            0x01D9 => Some("Savant Systems LLC"),
            0x01DA => Some("Logitech International SA"),
            0x01DB => Some("Innblue Consulting"),
            0x01DC => Some("iParking Ltd."),
            0x01DD => Some("Koninklijke Philips Electronics N.V."),
            0x01DE => Some("Minelab Electronics Pty Limited"),
            0x01DF => Some("Bison Group Ltd."),
            0x01E0 => Some("Widex A/S"),
            0x01E1 => Some("Jolla Ltd"),
            0x01E2 => Some("Lectronix, Inc."),
            0x01E3 => Some("Caterpillar Inc"),
            0x01E4 => Some("Freedom Innovations"),
            0x01E5 => Some("Dynamic Devices Ltd"),
            0x01E6 => Some("Technology Solutions (UK) Ltd"),
            0x01E7 => Some("IPS Group Inc."),
            0x01E8 => Some("STIR"),
            0x01E9 => Some("Sano, Inc."),
            0x01EA => Some("Advanced Application Design, Inc."),
            0x01EB => Some("AutoMap LLC"),
            0x01EC => Some("Spreadtrum Communications Shanghai Ltd"),
            0x01ED => Some("CuteCircuit LTD"),
            0x01EE => Some("Valeo Service"),
            0x01EF => Some("Fullpower Technologies, Inc."),
            0x01F0 => Some("KloudNation"),
            0x01F1 => Some("Zebra Technologies Corporation"),
            0x01F2 => Some("Itron, Inc."),
            0x01F3 => Some("The University of Tokyo"),
            0x01F4 => Some("UTC Fire and Security"),
            0x01F5 => Some("Cool Webthings Limited"),
            0x01F6 => Some("DJO Global"),
            0x01F7 => Some("Gelliner Limited"),
            0x01F8 => Some("Anyka (Guangzhou) Microelectronics Technology Co, LTD"),
            0x01F9 => Some("Medtronic Inc."),
            0x01FA => Some("Gozio Inc."),
            0x01FB => Some("Form Lifting, LLC"),
            0x01FC => Some("Wahoo Fitness, LLC"),
            0x01FD => Some("Kontakt Micro-Location Sp. z o.o."),
            0x01FE => Some("Radio Systems Corporation"),
            0x01FF => Some("Freescale Semiconductor, Inc."),
            0x0200 => Some("Verifone Systems Pte Ltd. Taiwan Branch"),
            0x0201 => Some("AR Timing"),
            0x0202 => Some("Rigado LLC"),
            0x0203 => Some("Kemppi Oy"),
            0x0204 => Some("Tapcentive Inc."),
            0x0205 => Some("Smartbotics Inc."),
            0x0206 => Some("Otter Products, LLC"),
            0x0207 => Some("STEMP Inc."),
            0x0208 => Some("LumiGeek LLC"),
            0x0209 => Some("InvisionHeart Inc."),
            0x020A => Some("Macnica Inc."),
            0x020B => Some("Jaguar Land Rover Limited"),
            0x020C => Some("CoroWare Technologies, Inc"),
            0x020D => Some("Simplo Technology Co., LTD"),
            0x020E => Some("Omron Healthcare Co., LTD"),
            0x020F => Some("Comodule GMBH"),
            0x0210 => Some("ikeGPS"),
            0x0211 => Some("Telink Semiconductor Co. Ltd"),
            0x0212 => Some("Interplan Co., Ltd"),
            0x0213 => Some("Wyler AG"),
            0x0214 => Some("IK Multimedia Production srl"),
            0x0215 => Some("Lukoton Experience Oy"),
            0x0216 => Some("MTI Ltd"),
            0x0217 => Some("Tech4home, Lda"),
            0x0218 => Some("Hiotech AB"),
            0x0219 => Some("DOTT Limited"),
            0x021A => Some("Blue Speck Labs, LLC"),
            0x021B => Some("Cisco Systems, Inc"),
            0x021C => Some("Mobicomm Inc"),
            0x021D => Some("Edamic"),
            0x021E => Some("Goodnet, Ltd"),
            0x021F => Some("Luster Leaf Products Inc"),
            0x0220 => Some("Manus Machina BV"),
            0x0221 => Some("Mobiquity Networks Inc"),
            0x0222 => Some("Praxis Dynamics"),
            0x0223 => Some("Philip Morris Products S.A."),
            0x0224 => Some("Comarch SA"),
            0x0225 => Some("Nestl Nespresso S.A."),
            0x0226 => Some("Merlinia A/S"),
            0x0227 => Some("LifeBEAM Technologies"),
            0x0228 => Some("Twocanoes Labs, LLC"),
            0x0229 => Some("Muoverti Limited"),
            0x022A => Some("Stamer Musikanlagen GMBH"),
            0x022B => Some("Tesla Motors"),
            0x022C => Some("Pharynks Corporation"),
            0x022D => Some("Lupine"),
            0x022E => Some("Siemens AG"),
            0x022F => Some("Huami (Shanghai) Culture Communication CO., LTD"),
            0x0230 => Some("Foster Electric Company, Ltd"),
            0x0231 => Some("ETA SA"),
            0x0232 => Some("x-Senso Solutions Kft"),
            0x0233 => Some("Shenzhen SuLong Communication Ltd"),
            0x0234 => Some("FengFan (BeiJing) Technology Co, Ltd"),
            0x0235 => Some("Qrio Inc"),
            0x0236 => Some("Pitpatpet Ltd"),
            0x0237 => Some("MSHeli s.r.l."),
            0x0238 => Some("Trakm8 Ltd"),
            0x0239 => Some("JIN CO, Ltd"),
            0x023A => Some("Alatech Tehnology"),
            0x023B => Some("Beijing CarePulse Electronic Technology Co, Ltd"),
            0x023C => Some("Awarepoint"),
            0x023D => Some("ViCentra B.V."),
            0x023E => Some("Raven Industries"),
            0x023F => Some("WaveWare Technologies Inc."),
            0x0240 => Some("Argenox Technologies"),
            0x0241 => Some("Bragi GmbH"),
            0x0242 => Some("16Lab Inc"),
            0x0243 => Some("Masimo Corp"),
            0x0244 => Some("Iotera Inc"),
            0x0245 => Some("Endress+Hauser"),
            0x0246 => Some("ACKme Networks, Inc."),
            0x0247 => Some("FiftyThree Inc."),
            0x0248 => Some("Parker Hannifin Corp"),
            0x0249 => Some("Transcranial Ltd"),
            0x024A => Some("Uwatec AG"),
            0x024B => Some("Orlan LLC"),
            0x024C => Some("Blue Clover Devices"),
            0x024D => Some("M-Way Solutions GmbH"),
            0x024E => Some("Microtronics Engineering GmbH"),
            0x024F => Some("Schneider Schreibgerte GmbH"),
            0x0250 => Some("Sapphire Circuits LLC"),
            0x0251 => Some("Lumo Bodytech Inc."),
            0x0252 => Some("UKC Technosolution"),
            0x0253 => Some("Xicato Inc."),
            0x0254 => Some("Playbrush"),
            0x0255 => Some("Dai Nippon Printing Co., Ltd."),
            0x0256 => Some("G24 Power Limited"),
            0x0257 => Some("AdBabble Local Commerce Inc."),
            0x0258 => Some("Devialet SA"),
            0x0259 => Some("ALTYOR"),
            0x025A => Some("University of Applied Sciences Valais/Haute Ecole Valaisanne"),
            0x025B => Some("Five Interactive, LLC dba Zendo"),
            0x025C => Some("NetEaseHangzhouNetwork co.Ltd."),
            0x025D => Some("Lexmark International Inc."),
            0x025E => Some("Fluke Corporation"),
            0x025F => Some("Yardarm Technologies"),
            0x0260 => Some("SensaRx"),
            0x0261 => Some("SECVRE GmbH"),
            0x0262 => Some("Glacial Ridge Technologies"),
            0x0263 => Some("Identiv, Inc."),
            0x0264 => Some("DDS, Inc."),
            0x0265 => Some("SMK Corporation"),
            0x0266 => Some("Schawbel Technologies LLC"),
            0x0267 => Some("XMI Systems SA"),
            0x0268 => Some("Cerevo"),
            0x0269 => Some("Torrox GmbH & Co KG"),
            0x026A => Some("Gemalto"),
            0x026B => Some("DEKA Research & Development Corp."),
            0x026C => Some("Domster Tadeusz Szydlowski"),
            0x026D => Some("Technogym SPA"),
            0x026E => Some("FLEURBAEY BVBA"),
            0x026F => Some("Aptcode Solutions"),
            0x0270 => Some("LSI ADL Technology"),
            0x0271 => Some("Animas Corp"),
            0x0272 => Some("Alps Electric Co., Ltd."),
            0x0273 => Some("OCEASOFT"),
            0x0274 => Some("Motsai Research"),
            0x0275 => Some("Geotab"),
            0x0276 => Some("E.G.O. Elektro-Gertebau GmbH"),
            0x0277 => Some("bewhere inc"),
            0x0278 => Some("Johnson Outdoors Inc"),
            0x0279 => Some("steute Schaltgerate GmbH & Co. KG"),
            0x027A => Some("Ekomini inc."),
            0x027B => Some("DEFA AS"),
            0x027C => Some("Aseptika Ltd"),
            0x027D => Some("HUAWEI Technologies Co., Ltd. ( )"),
            0x027E => Some("HabitAware, LLC"),
            0x027F => Some("ruwido austria gmbh"),
            0x0280 => Some("ITEC corporation"),
            0x0281 => Some("StoneL"),
            0x0282 => Some("Sonova AG"),
            0x0283 => Some("Maven Machines, Inc."),
            0x0284 => Some("Synapse Electronics"),
            0x0285 => Some("Standard Innovation Inc."),
            0x0286 => Some("RF Code, Inc."),
            0x0287 => Some("Wally Ventures S.L."),
            0x0288 => Some("Willowbank Electronics Ltd"),
            0x0289 => Some("SK Telecom"),
            0x028A => Some("Jetro AS"),
            0x028B => Some("Code Gears LTD"),
            0x028C => Some("NANOLINK APS"),
            0x028D => Some("IF, LLC"),
            0x028E => Some("RF Digital Corp"),
            0x028F => Some("Church & Dwight Co., Inc"),
            0x0290 => Some("Multibit Oy"),
            0x0291 => Some("CliniCloud Inc"),
            0x0292 => Some("SwiftSensors"),
            0x0293 => Some("Blue Bite"),
            0x0294 => Some("ELIAS GmbH"),
            0x0295 => Some("Sivantos GmbH"),
            0x0296 => Some("Petzl"),
            0x0297 => Some("storm power ltd"),
            0x0298 => Some("EISST Ltd"),
            0x0299 => Some("Inexess Technology Simma KG"),
            0x029A => Some("Currant, Inc."),
            0x029B => Some("C2 Development, Inc."),
            0x029C => Some("Blue Sky Scientific, LLC"),
            0x029D => Some("ALOTTAZS LABS, LLC"),
            0x029E => Some("Kupson spol. s r.o."),
            0x029F => Some("Areus Engineering GmbH"),
            0x02A0 => Some("Impossible Camera GmbH"),
            0x02A1 => Some("InventureTrack Systems"),
            0x02A2 => Some("LockedUp"),
            0x02A3 => Some("Itude"),
            0x02A4 => Some("Pacific Lock Company"),
            0x02A5 => Some("Tendyron Corporation ( )"),
            0x02A6 => Some("Robert Bosch GmbH"),
            0x02A7 => Some("Illuxtron international B.V."),
            0x02A8 => Some("miSport Ltd."),
            0x02A9 => Some("Chargelib"),
            0x02AA => Some("Doppler Lab"),
            0x02AB => Some("BBPOS Limited"),
            0x02AC => Some("RTB Elektronik GmbH & Co. KG"),
            0x02AD => Some("Rx Networks, Inc."),
            0x02AE => Some("WeatherFlow, Inc."),
            0x02AF => Some("Technicolor USA Inc."),
            0x02B0 => Some("Bestechnic(Shanghai),Ltd"),
            0x02B1 => Some("Raden Inc"),
            0x02B2 => Some("JouZen Oy"),
            0x02B3 => Some("CLABER S.P.A."),
            0x02B4 => Some("Hyginex, Inc."),
            0x02B5 => Some("HANSHIN ELECTRIC RAILWAY CO.,LTD."),
            0x02B6 => Some("Schneider Electric"),
            0x02B7 => Some("Oort Technologies LLC"),
            0x02B8 => Some("Chrono Therapeutics"),
            0x02B9 => Some("Rinnai Corporation"),
            0x02BA => Some("Swissprime Technologies AG"),
            0x02BB => Some("Koha.,Co.Ltd"),
            0x02BC => Some("Genevac Ltd"),
            0x02BD => Some("Chemtronics"),
            0x02BE => Some("Seguro Technology Sp. z o.o."),
            0x02BF => Some("Redbird Flight Simulations"),
            0x02C0 => Some("Dash Robotics"),
            0x02C1 => Some("LINE Corporation"),
            0x02C2 => Some("Guillemot Corporation"),
            0x02C3 => Some("Techtronic Power Tools Technology Limited"),
            0x02C4 => Some("Wilson Sporting Goods"),
            0x02C5 => Some("Lenovo (Singapore) Pte Ltd. ( )"),
            0x02C6 => Some("Ayatan Sensors"),
            0x02C7 => Some("Electronics Tomorrow Limited"),
            0x02C8 => Some("VASCO Data Security International, Inc."),
            0x02C9 => Some("PayRange Inc."),
            0x02CA => Some("ABOV Semiconductor"),
            0x02CB => Some("AINA-Wireless Inc."),
            0x02CC => Some("Eijkelkamp Soil & Water"),
            0x02CD => Some("BMA ergonomics b.v."),
            0x02CE => Some("Teva Branded Pharmaceutical Products R&D, Inc."),
            0x02CF => Some("Anima"),
            0x02D0 => Some("3M"),
            0x02D1 => Some("Empatica Srl"),
            0x02D2 => Some("Afero, Inc."),
            0x02D3 => Some("Powercast Corporation"),
            0x02D4 => Some("Secuyou ApS"),
            0x02D5 => Some("OMRON Corporation"),
            0x02D6 => Some("Send Solutions"),
            0x02D7 => Some("NIPPON SYSTEMWARE CO.,LTD."),
            0x02D8 => Some("Neosfar"),
            0x02D9 => Some("Fliegl Agrartechnik GmbH"),
            0x02DA => Some("Gilvader"),
            0x02DB => Some("Digi International Inc (R)"),
            0x02DC => Some("DeWalch Technologies, Inc."),
            0x02DD => Some("Flint Rehabilitation Devices, LLC"),
            0x02DE => Some("Samsung SDS Co., Ltd."),
            0x02DF => Some("Blur Product Development"),
            0x02E0 => Some("University of Michigan"),
            0x02E1 => Some("Victron Energy BV"),
            0x02E2 => Some("NTT docomo"),
            0x02E3 => Some("Carmanah Technologies Corp."),
            0x02E4 => Some("Bytestorm Ltd."),
            0x02E5 => Some("Espressif Incorporated ( () )"),
            0x02E6 => Some("Unwire"),
            0x02E7 => Some("Connected Yard, Inc."),
            0x02E8 => Some("American Music Environments"),
            0x02E9 => Some("Sensogram Technologies, Inc."),
            0x02EA => Some("Fujitsu Limited"),
            0x02EB => Some("Ardic Technology"),
            0x02EC => Some("Delta Systems, Inc"),
            0x02ED => Some("HTC Corporation"),
            0x02EE => Some("Citizen Holdings Co., Ltd."),
            0x02EF => Some("SMART-INNOVATION.inc"),
            0x02F0 => Some("Blackrat Software"),
            0x02F1 => Some("The Idea Cave, LLC"),
            0x02F2 => Some("GoPro, Inc."),
            0x02F3 => Some("AuthAir, Inc"),
            0x02F4 => Some("Vensi, Inc."),
            0x02F5 => Some("Indagem Tech LLC"),
            0x02F6 => Some("Intemo Technologies"),
            0x02F7 => Some("DreamVisions co., Ltd."),
            0x02F8 => Some("Runteq Oy Ltd"),
            0x02F9 => Some("IMAGINATION TECHNOLOGIES LTD"),
            0x02FA => Some("CoSTAR TEchnologies"),
            0x02FB => Some("Clarius Mobile Health Corp."),
            0x02FC => Some("Shanghai Frequen Microelectronics Co., Ltd."),
            0x02FD => Some("Uwanna, Inc."),
            0x02FE => Some("Lierda Science & Technology Group Co., Ltd."),
            0x02FF => Some("Silicon Laboratories"),
            0x0300 => Some("World Moto Inc."),
            0x0301 => Some("Giatec Scientific Inc."),
            0x0302 => Some("Loop Devices, Inc"),
            0x0303 => Some("IACA electronique"),
            0x0304 => Some("Proxy Technologies, Inc."),
            0x0305 => Some("Swipp ApS"),
            0x0306 => Some("Life Laboratory Inc."),
            0x0307 => Some("FUJI INDUSTRIAL CO.,LTD."),
            0x0308 => Some("Surefire, LLC"),
            0x0309 => Some("Dolby Labs"),
            0x030A => Some("Ellisys"),
            0x030B => Some("Magnitude Lighting Converters"),
            0x030C => Some("Hilti AG"),
            0x030D => Some("Devdata S.r.l."),
            0x030E => Some("Deviceworx"),
            0x030F => Some("Shortcut Labs"),
            0x0310 => Some("SGL Italia S.r.l."),
            0x0311 => Some("PEEQ DATA"),
            0x0312 => Some("Ducere Technologies Pvt Ltd"),
            0x0313 => Some("DiveNav, Inc."),
            0x0314 => Some("RIIG AI Sp. z o.o."),
            0x0315 => Some("Thermo Fisher Scientific"),
            0x0316 => Some("AG Measurematics Pvt. Ltd."),
            0x0317 => Some("CHUO Electronics CO., LTD."),
            0x0318 => Some("Aspenta International"),
            0x0319 => Some("Eugster Frismag AG"),
            0x031A => Some("Amber wireless GmbH"),
            0x031B => Some("HQ Inc"),
            0x031C => Some("Lab Sensor Solutions"),
            0x031D => Some("Enterlab ApS"),
            0x031E => Some("Eyefi, Inc."),
            0x031F => Some("MetaSystem S.p.A."),
            0x0320 => Some("SONO ELECTRONICS. CO., LTD"),
            0x0321 => Some("Jewelbots"),
            0x0322 => Some("Compumedics Limited"),
            0x0323 => Some("Rotor Bike Components"),
            0x0324 => Some("Astro, Inc."),
            0x0325 => Some("Amotus Solutions"),
            0x0326 => Some("Healthwear Technologies (Changzhou)Ltd"),
            0x0327 => Some("Essex Electronics"),
            0x0328 => Some("Grundfos A/S"),
            0x0329 => Some("Eargo, Inc."),
            0x032A => Some("Electronic Design Lab"),
            0x032B => Some("ESYLUX"),
            0x032C => Some("NIPPON SMT.CO.,Ltd"),
            0x032D => Some("BM innovations GmbH"),
            0x032E => Some("indoormap"),
            0x032F => Some("OttoQ Inc"),
            0x0330 => Some("North Pole Engineering"),
            0x0331 => Some("3flares Technologies Inc."),
            0x0332 => Some("Electrocompaniet A.S."),
            0x0333 => Some("Mul-T-Lock"),
            0x0334 => Some("Corentium AS"),
            0x0335 => Some("Enlighted Inc"),
            0x0336 => Some("GISTIC"),
            0x0337 => Some("AJP2 Holdings, LLC"),
            0x0338 => Some("COBI GmbH"),
            0x0339 => Some("Blue Sky Scientific, LLC"),
            0x033A => Some("Appception, Inc."),
            0x033B => Some("Courtney Thorne Limited"),
            0x033C => Some("Virtuosys"),
            0x033D => Some("TPV Technology Limited"),
            0x033E => Some("Monitra SA"),
            0x033F => Some("Automation Components, Inc."),
            0x0340 => Some("Letsense s.r.l."),
            0x0341 => Some("Etesian Technologies LLC"),
            0x0342 => Some("GERTEC BRASIL LTDA."),
            0x0343 => Some("Drekker Development Pty. Ltd."),
            0x0344 => Some("Whirl Inc"),
            0x0345 => Some("Locus Positioning"),
            0x0346 => Some("Acuity Brands Lighting, Inc"),
            0x0347 => Some("Prevent Biometrics"),
            0x0348 => Some("Arioneo"),
            0x0349 => Some("VersaMe"),
            0x034A => Some("Vaddio"),
            0x034B => Some("Libratone A/S"),
            0x034C => Some("HM Electronics, Inc."),
            0x034D => Some("TASER International, Inc."),
            0x034E => Some("SafeTrust Inc."),
            0x034F => Some("Heartland Payment Systems"),
            0x0350 => Some("Bitstrata Systems Inc."),
            0x0351 => Some("Pieps GmbH"),
            0x0352 => Some("iRiding(Xiamen)Technology Co.,Ltd."),
            0x0353 => Some("Alpha Audiotronics, Inc."),
            0x0354 => Some("TOPPAN FORMS CO.,LTD."),
            0x0355 => Some("Sigma Designs, Inc."),
            0x0356 => Some("Spectrum Brands, Inc."),
            0x0357 => Some("Polymap Wireless"),
            0x0358 => Some("MagniWare Ltd."),
            0x0359 => Some("Novotec Medical GmbH"),
            0x035A => Some("Medicom Innovation Partner a/s"),
            0x035B => Some("Matrix Inc."),
            0x035C => Some("Eaton Corporation"),
            0x035D => Some("KYS"),
            0x035E => Some("Naya Health, Inc."),
            0x035F => Some("Acromag"),
            0x0360 => Some("Insulet Corporation"),
            0x0361 => Some("Wellinks Inc."),
            0x0362 => Some("ON Semiconductor"),
            0x0363 => Some("FREELAP SA"),
            0x0364 => Some("Favero Electronics Srl"),
            0x0365 => Some("BioMech Sensor LLC"),
            0x0366 => Some("BOLTT Sports technologies Private limited"),
            0x0367 => Some("Saphe International"),
            0x0368 => Some("Metormote AB"),
            0x0369 => Some("littleBits"),
            0x036A => Some("SetPoint Medical"),
            0x036B => Some("BRControls Products BV"),
            0x036C => Some("Zipcar"),
            0x036D => Some("AirBolt Pty Ltd"),
            0x036E => Some("KeepTruckin Inc"),
            0x036F => Some("Motiv, Inc."),
            0x0370 => Some("Wazombi Labs O"),
            0x0371 => Some("ORBCOMM"),
            0x0372 => Some("Nixie Labs, Inc."),
            0x0373 => Some("AppNearMe Ltd"),
            0x0374 => Some("Holman Industries"),
            0x0375 => Some("Expain AS"),
            0x0376 => Some("Electronic Temperature Instruments Ltd"),
            0x0377 => Some("Plejd AB"),
            0x0378 => Some("Propeller Health"),
            0x0379 => Some("Shenzhen iMCO Electronic Technology Co.,Ltd"),
            0x037A => Some("Algoria"),
            0x037B => Some("Apption Labs Inc."),
            0x037C => Some("Cronologics Corporation"),
            0x037D => Some("MICRODIA Ltd."),
            0x037E => Some("lulabytes S.L."),
            0x037F => Some("Nestec S.A."),
            0x0380 => Some("LLC \"MEGA-F service\""),
            0x0381 => Some("Sharp Corporation"),
            0x0382 => Some("Precision Outcomes Ltd"),
            0x0383 => Some("Kronos Incorporated"),
            0x0384 => Some("OCOSMOS Co., Ltd."),
            0x0385 => Some("Embedded Electronic Solutions Ltd. dba e2Solutions"),
            0x0386 => Some("Aterica Inc."),
            0x0387 => Some("BluStor PMC, Inc."),
            0x0388 => Some("Kapsch TrafficCom AB"),
            0x0389 => Some("ActiveBlu Corporation"),
            0x038A => Some("Kohler Mira Limited"),
            0x038B => Some("Noke"),
            0x038C => Some("Appion Inc."),
            0x038D => Some("Resmed Ltd"),
            0x038E => Some("Crownstone B.V."),
            0x038F => Some("Xiaomi Inc."),
            0x0390 => Some("INFOTECH s.r.o."),
            0x0391 => Some("Thingsquare AB"),
            0x0392 => Some("T&D"),
            0x0393 => Some("LAVAZZA S.p.A."),
            0x0394 => Some("Netclearance Systems, Inc."),
            0x0395 => Some("SDATAWAY"),
            0x0396 => Some("BLOKS GmbH"),
            0x0397 => Some("LEGO System A/S"),
            0x0398 => Some("Thetatronics Ltd"),
            0x0399 => Some("Nikon Corporation"),
            0x039A => Some("NeST"),
            0x039B => Some("South Silicon Valley Microelectronics"),
            0x039C => Some("ALE International"),
            0x039D => Some("CareView Communications, Inc."),
            0x039E => Some("SchoolBoard Limited"),
            0x039F => Some("Molex Corporation"),
            0x03A0 => Some("BARROT TECHNOLOGY LIMITED (formerly IVT Wireless Limited)"),
            0x03A1 => Some("Alpine Labs LLC"),
            0x03A2 => Some("Candura Instruments"),
            0x03A3 => Some("SmartMovt Technology Co., Ltd"),
            0x03A4 => Some("Token Zero Ltd"),
            0x03A5 => Some("ACE CAD Enterprise Co., Ltd. (ACECAD)"),
            0x03A6 => Some("Medela, Inc"),
            0x03A7 => Some("AeroScout"),
            0x03A8 => Some("Esrille Inc."),
            0x03A9 => Some("THINKERLY SRL"),
            0x03AA => Some("Exon Sp. z o.o."),
            0x03AB => Some("Meizu Technology Co., Ltd."),
            0x03AC => Some("Smablo LTD"),
            0x03AD => Some("XiQ"),
            0x03AE => Some("Allswell Inc."),
            0x03AF => Some("Comm-N-Sense Corp DBA Verigo"),
            0x03B0 => Some("VIBRADORM GmbH"),
            0x03B1 => Some("Otodata Wireless Network Inc."),
            0x03B2 => Some("Propagation Systems Limited"),
            0x03B3 => Some("Midwest Instruments & Controls"),
            0x03B4 => Some("Alpha Nodus, inc."),
            0x03B5 => Some("petPOMM, Inc"),
            0x03B6 => Some("Mattel"),
            0x03B7 => Some("Airbly Inc."),
            0x03B8 => Some("A-Safe Limited"),
            0x03B9 => Some("FREDERIQUE CONSTANT SA"),
            0x03BA => Some("Maxscend Microelectronics Company Limited"),
            0x03BB => Some("Abbott Diabetes Care"),
            0x03BC => Some("ASB Bank Ltd"),
            0x03BD => Some("amadas"),
            0x03BE => Some("Applied Science, Inc."),
            0x03BF => Some("iLumi Solutions Inc."),
            0x03C0 => Some("Arch Systems Inc."),
            0x03C1 => Some("Ember Technologies, Inc."),
            0x03C2 => Some("Snapchat Inc"),
            0x03C3 => Some("Casambi Technologies Oy"),
            0x03C4 => Some("Pico Technology Inc."),
            0x03C5 => Some("St. Jude Medical, Inc."),
            0x03C6 => Some("Intricon"),
            0x03C7 => Some("Structural Health Systems, Inc."),
            0x03C8 => Some("Avvel International"),
            0x03C9 => Some("Gallagher Group"),
            0x03CA => Some("In2things Automation Pvt. Ltd."),
            0x03CB => Some("SYSDEV Srl"),
            0x03CC => Some("Vonkil Technologies Ltd"),
            0x03CD => Some("Wynd Technologies, Inc."),
            0x03CE => Some("CONTRINEX S.A."),
            0x03CF => Some("MIRA, Inc."),
            0x03D0 => Some("Watteam Ltd"),
            0x03D1 => Some("Density Inc."),
            0x03D2 => Some("IOT Pot India Private Limited"),
            0x03D3 => Some("Sigma Connectivity AB"),
            0x03D4 => Some("PEG PEREGO SPA"),
            0x03D5 => Some("Wyzelink Systems Inc."),
            0x03D6 => Some("Yota Devices LTD"),
            0x03D7 => Some("FINSECUR"),
            0x03D8 => Some("Zen-Me Labs Ltd"),
            0x03D9 => Some("3IWare Co., Ltd."),
            0x03DA => Some("EnOcean GmbH"),
            0x03DB => Some("Instabeat, Inc"),
            0x03DC => Some("Nima Labs"),
            0x03DD => Some("Andreas Stihl AG & Co. KG"),
            0x03DE => Some("Nathan Rhoades LLC"),
            0x03DF => Some("Grob Technologies, LLC"),
            0x03E0 => Some("Actions (Zhuhai) Technology Co., Limited"),
            0x03E1 => Some("SPD Development Company Ltd"),
            0x03E2 => Some("Sensoan Oy"),
            0x03E3 => Some("Qualcomm Life Inc"),
            0x03E4 => Some("Chip-ing AG"),
            0x03E5 => Some("ffly4u"),
            0x03E6 => Some("IoT Instruments Oy"),
            0x03E7 => Some("TRUE Fitness Technology"),
            0x03E8 => Some("Reiner Kartengeraete GmbH & Co. KG."),
            0x03E9 => Some("SHENZHEN LEMONJOY TECHNOLOGY CO., LTD."),
            0x03EA => Some("Hello Inc."),
            0x03EB => Some("Evollve Inc."),
            0x03EC => Some("Jigowatts Inc."),
            0x03ED => Some("BASIC MICRO.COM,INC."),
            0x03EE => Some("CUBE TECHNOLOGIES"),
            0x03EF => Some("foolography GmbH"),
            0x03F0 => Some("CLINK"),
            0x03F1 => Some("Hestan Smart Cooking Inc."),
            0x03F2 => Some("WindowMaster A/S"),
            0x03F3 => Some("Flowscape AB"),
            0x03F4 => Some("PAL Technologies Ltd"),
            0x03F5 => Some("WHERE, Inc."),
            0x03F6 => Some("Iton Technology Corp."),
            0x03F7 => Some("Owl Labs Inc."),
            0x03F8 => Some("Rockford Corp."),
            0x03F9 => Some("Becon Technologies Co.,Ltd."),
            0x03FA => Some("Vyassoft Technologies Inc"),
            0x03FB => Some("Nox Medical"),
            0x03FC => Some("Kimberly-Clark"),
            0x03FD => Some("Trimble Navigation Ltd."),
            0x03FE => Some("Littelfuse"),
            0x03FF => Some("Withings"),
            0x0400 => Some("i-developer IT Beratung UG"),
            0x0401 => Some(""),
            0x0402 => Some("Sears Holdings Corporation"),
            0x0403 => Some("Gantner Electronic GmbH"),
            0x0404 => Some("Authomate Inc"),
            0x0405 => Some("Vertex International, Inc."),
            0x0406 => Some("Airtago"),
            0x0407 => Some("Swiss Audio SA"),
            0x0408 => Some("ToGetHome Inc."),
            0x0409 => Some("AXIS"),
            0x040A => Some("Openmatics"),
            0x040B => Some("Jana Care Inc."),
            0x040C => Some("Senix Corporation"),
            0x040D => Some("NorthStar Battery Company, LLC"),
            0x040E => Some("SKF (U.K.) Limited"),
            0x040F => Some("CO-AX Technology, Inc."),
            0x0410 => Some("Fender Musical Instruments"),
            0x0411 => Some("Luidia Inc"),
            0x0412 => Some("SEFAM"),
            0x0413 => Some("Wireless Cables Inc"),
            0x0414 => Some("Lightning Protection International Pty Ltd"),
            0x0415 => Some("Uber Technologies Inc"),
            0x0416 => Some("SODA GmbH"),
            0x0417 => Some("Fatigue Science"),
            0x0418 => Some("Alpine Electronics Inc."),
            0x0419 => Some("Novalogy LTD"),
            0x041A => Some("Friday Labs Limited"),
            0x041B => Some("OrthoAccel Technologies"),
            0x041C => Some("WaterGuru, Inc."),
            0x041D => Some("Benning Elektrotechnik und Elektronik GmbH & Co. KG"),
            0x041E => Some("Dell Computer Corporation"),
            0x041F => Some("Kopin Corporation"),
            0x0420 => Some("TecBakery GmbH"),
            0x0421 => Some("Backbone Labs, Inc."),
            0x0422 => Some("DELSEY SA"),
            0x0423 => Some("Chargifi Limited"),
            0x0424 => Some("Trainesense Ltd."),
            0x0425 => Some("Unify Software and Solutions GmbH & Co. KG"),
            0x0426 => Some("Husqvarna AB"),
            0x0427 => Some("Focus fleet and fuel management inc"),
            0x0428 => Some("SmallLoop, LLC"),
            0x0429 => Some("Prolon Inc."),
            0x042A => Some("BD Medical"),
            0x042B => Some("iMicroMed Incorporated"),
            0x042C => Some("Ticto N.V."),
            0x042D => Some("Meshtech AS"),
            0x042E => Some("MemCachier Inc."),
            0x042F => Some("Danfoss A/S"),
            0x0430 => Some("SnapStyk Inc."),
            0x0431 => Some("Amway Corporation"),
            0x0432 => Some("Silk Labs, Inc."),
            0x0433 => Some("Pillsy Inc."),
            0x0434 => Some("Hatch Baby, Inc."),
            0x0435 => Some("Blocks Wearables Ltd."),
            0x0436 => Some("Drayson Technologies (Europe) Limited"),
            0x0437 => Some("eBest IOT Inc."),
            0x0438 => Some("Helvar Ltd"),
            0x0439 => Some("Radiance Technologies"),
            0x043A => Some("Nuheara Limited"),
            0x043B => Some("Appside co., ltd."),
            0x043C => Some("DeLaval"),
            0x043D => Some("Coiler Corporation"),
            0x043E => Some("Thermomedics, Inc."),
            0x043F => Some("Tentacle Sync GmbH"),
            0x0440 => Some("Valencell, Inc."),
            0x0441 => Some("iProtoXi Oy"),
            0x0442 => Some("SECOM CO., LTD."),
            0x0443 => Some("Tucker International LLC"),
            0x0444 => Some("Metanate Limited"),
            0x0445 => Some("Kobian Canada Inc."),
            0x0446 => Some("NETGEAR, Inc."),
            0x0447 => Some("Fabtronics Australia Pty Ltd"),
            0x0448 => Some("Grand Centrix GmbH"),
            0x0449 => Some("1UP USA.com llc"),
            0x044A => Some("SHIMANO INC."),
            0x044B => Some("Nain Inc."),
            0x044C => Some("LifeStyle Lock, LLC"),
            0x044D => Some("VEGA Grieshaber KG"),
            0x044E => Some("Xtrava Inc."),
            0x044F => Some("TTS Tooltechnic Systems AG & Co. KG"),
            0x0450 => Some("Teenage Engineering AB"),
            0x0451 => Some("Tunstall Nordic AB"),
            0x0452 => Some("Svep Design Center AB"),
            0x0453 => Some("GreenPeak Technologies BV"),
            0x0454 => Some("Sphinx Electronics GmbH & Co KG"),
            0x0455 => Some("Atomation"),
            0x0456 => Some("Nemik Consulting Inc"),
            0x0457 => Some("RF INNOVATION"),
            0x0458 => Some("Mini Solution Co., Ltd."),
            0x0459 => Some("Lumenetix, Inc"),
            0x045A => Some("2048450 Ontario Inc"),
            0x045B => Some("SPACEEK LTD"),
            0x045C => Some("Delta T Corporation"),
            0x045D => Some("Boston Scientific Corporation"),
            0x045E => Some("Nuviz, Inc."),
            0x045F => Some("Real Time Automation, Inc."),
            0x0460 => Some("Kolibree"),
            0x0461 => Some("vhf elektronik GmbH"),
            0x0462 => Some("Bonsai Systems GmbH"),
            0x0463 => Some("Fathom Systems Inc."),
            0x0464 => Some("Bellman & Symfon"),
            0x0465 => Some("International Forte Group LLC"),
            0x0466 => Some("CycleLabs Solutions inc."),
            0x0467 => Some("Codenex Oy"),
            0x0468 => Some("Kynesim Ltd"),
            0x0469 => Some("Palago AB"),
            0x046A => Some("INSIGMA INC."),
            0x046B => Some("PMD Solutions"),
            0x046C => Some("Qingdao Realtime Technology Co., Ltd."),
            0x046D => Some("BEGA Gantenbrink-Leuchten KG"),
            0x046E => Some("Pambor Ltd."),
            0x046F => Some("Develco Products A/S"),
            0x0470 => Some("iDesign s.r.l."),
            0x0471 => Some("TiVo Corp"),
            0x0472 => Some("Control-J Pty Ltd"),
            0x0473 => Some("Steelcase, Inc."),
            0x0474 => Some("iApartment co., ltd."),
            0x0475 => Some("Icom inc."),
            0x0476 => Some("Oxstren Wearable Technologies Private Limited"),
            0x0477 => Some("Blue Spark Technologies"),
            0x0478 => Some("FarSite Communications Limited"),
            0x0479 => Some("mywerk system GmbH"),
            0x047A => Some("Sinosun Technology Co., Ltd."),
            0x047B => Some("MIYOSHI ELECTRONICS CORPORATION"),
            0x047C => Some("POWERMAT LTD"),
            0x047D => Some("Occly LLC"),
            0x047E => Some("OurHub Dev IvS"),
            0x047F => Some("Pro-Mark, Inc."),
            0x0480 => Some("Dynometrics Inc."),
            0x0481 => Some("Quintrax Limited"),
            0x0482 => Some("POS Tuning Udo Vosshenrich GmbH & Co. KG"),
            0x0483 => Some("Multi Care Systems B.V."),
            0x0484 => Some("Revol Technologies Inc"),
            0x0485 => Some("SKIDATA AG"),
            0x0486 => {
                Some("DEV TECNOLOGIA INDUSTRIA, COMERCIO E MANUTENCAO DE EQUIPAMENTOS LTDA. - ME")
            }
            0x0487 => Some("Centrica Connected Home"),
            0x0488 => Some("Automotive Data Solutions Inc"),
            0x0489 => Some("Igarashi Engineering"),
            0x048A => Some("Taelek Oy"),
            0x048B => Some("CP Electronics Limited"),
            0x048C => Some("Vectronix AG"),
            0x048D => Some("S-Labs Sp. z o.o."),
            0x048E => Some("Companion Medical, Inc."),
            0x048F => Some("BlueKitchen GmbH"),
            0x0490 => Some("Matting AB"),
            0x0491 => Some("SOREX - Wireless Solutions GmbH"),
            0x0492 => Some("ADC Technology, Inc."),
            0x0493 => Some("Lynxemi Pte Ltd"),
            0x0494 => Some("SENNHEISER electronic GmbH & Co. KG"),
            0x0495 => Some("LMT Mercer Group, Inc"),
            0x0496 => Some("Polymorphic Labs LLC"),
            0x0497 => Some("Cochlear Limited"),
            0x0498 => Some("METER Group, Inc. USA"),
            0x0499 => Some("Ruuvi Innovations Ltd."),
            0x049A => Some("Situne AS"),
            0x049B => Some("nVisti, LLC"),
            0x049C => Some("DyOcean"),
            0x049D => Some("Uhlmann & Zacher GmbH"),
            0x049E => Some("AND!XOR LLC"),
            0x049F => Some("tictote AB"),
            0x04A0 => Some("Vypin, LLC"),
            0x04A1 => Some("PNI Sensor Corporation"),
            0x04A2 => Some("ovrEngineered, LLC"),
            0x04A3 => Some("GT-tronics HK Ltd"),
            0x04A4 => Some("Herbert Waldmann GmbH & Co. KG"),
            0x04A5 => Some("Guangzhou FiiO Electronics Technology Co.,Ltd"),
            0x04A6 => Some("Vinetech Co., Ltd"),
            0x04A7 => Some("Dallas Logic Corporation"),
            0x04A8 => Some("BioTex, Inc."),
            0x04A9 => Some("DISCOVERY SOUND TECHNOLOGY, LLC"),
            0x04AA => Some("LINKIO SAS"),
            0x04AB => Some("Harbortronics, Inc."),
            0x04AC => Some("Undagrid B.V."),
            0x04AD => Some("Shure Inc"),
            0x04AE => Some("ERM Electronic Systems LTD"),
            0x04AF => Some("BIOROWER Handelsagentur GmbH"),
            0x04B0 => Some("Weba Sport und Med. Artikel GmbH"),
            0x04B1 => Some("Kartographers Technologies Pvt. Ltd."),
            0x04B2 => Some("The Shadow on the Moon"),
            0x04B3 => Some("mobike (Hong Kong) Limited"),
            0x04B4 => Some("Inuheat Group AB"),
            0x04B5 => Some("Swiftronix AB"),
            0x04B6 => Some("Diagnoptics Technologies"),
            0x04B7 => Some("Analog Devices, Inc."),
            0x04B8 => Some("Soraa Inc."),
            0x04B9 => Some("CSR Building Products Limited"),
            0x04BA => Some("Crestron Electronics, Inc."),
            0x04BB => Some("Neatebox Ltd"),
            0x04BC => Some("Draegerwerk AG & Co. KGaA"),
            0x04BD => Some("AlbynMedical"),
            0x04BE => Some("Averos FZCO"),
            0x04BF => Some("VIT Initiative, LLC"),
            0x04C0 => Some("Statsports International"),
            0x04C1 => Some("Sospitas, s.r.o."),
            0x04C2 => Some("Dmet Products Corp."),
            0x04C3 => Some("Mantracourt Electronics Limited"),
            0x04C4 => Some("TeAM Hutchins AB"),
            0x04C5 => Some("Seibert Williams Glass, LLC"),
            0x04C6 => Some("Insta GmbH"),
            0x04C7 => Some("Svantek Sp. z o.o."),
            0x04C8 => Some("Shanghai Flyco Electrical Appliance Co., Ltd."),
            0x04C9 => Some("Thornwave Labs Inc"),
            0x04CA => Some("Steiner-Optik GmbH"),
            0x04CB => Some("Novo Nordisk A/S"),
            0x04CC => Some("Enflux Inc."),
            0x04CD => Some("Safetech Products LLC"),
            0x04CE => Some("GOOOLED S.R.L."),
            0x04CF => Some("DOM Sicherheitstechnik GmbH & Co. KG"),
            0x04D0 => Some("Olympus Corporation"),
            0x04D1 => Some("KTS GmbH"),
            0x04D2 => Some("Anloq Technologies Inc."),
            0x04D3 => Some("Queercon, Inc"),
            0x04D4 => Some("5th Element Ltd"),
            0x04D5 => Some("Gooee Limited"),
            0x04D6 => Some("LUGLOC LLC"),
            0x04D7 => Some("Blincam, Inc."),
            0x04D8 => Some("FUJIFILM Corporation"),
            0x04D9 => Some("RandMcNally"),
            0x04DA => Some("Franceschi Marina snc"),
            0x04DB => Some("Engineered Audio, LLC."),
            0x04DC => Some("IOTTIVE (OPC) PRIVATE LIMITED"),
            0x04DD => Some("4MOD Technology"),
            0x04DE => Some("Lutron Electronics Co., Inc."),
            0x04DF => Some("Emerson"),
            0x04E0 => Some("Guardtec, Inc."),
            0x04E1 => Some("REACTEC LIMITED"),
            0x04E2 => Some("EllieGrid"),
            0x04E3 => Some("Under Armour"),
            0x04E4 => Some("Woodenshark"),
            0x04E5 => Some("Avack Oy"),
            0x04E6 => Some("Smart Solution Technology, Inc."),
            0x04E7 => Some("REHABTRONICS INC."),
            0x04E8 => Some("STABILO International"),
            0x04E9 => Some("Busch Jaeger Elektro GmbH"),
            0x04EA => Some("Pacific Bioscience Laboratories, Inc"),
            0x04EB => Some("Bird Home Automation GmbH"),
            0x04EC => Some("Motorola Solutions"),
            0x04ED => Some("R9 Technology, Inc."),
            0x04EE => Some("Auxivia"),
            0x04EF => Some("DaisyWorks, Inc"),
            0x04F0 => Some("Kosi Limited"),
            0x04F1 => Some("Theben AG"),
            0x04F2 => Some("InDreamer Techsol Private Limited"),
            0x04F3 => Some("Cerevast Medical"),
            0x04F4 => Some("ZanCompute Inc."),
            0x04F5 => Some("Pirelli Tyre S.P.A."),
            0x04F6 => Some("McLear Limited"),
            0x04F7 => Some("Shenzhen Huiding Technology Co.,Ltd."),
            0x04F8 => Some("Convergence Systems Limited"),
            0x04F9 => Some("Interactio"),
            0x04FA => Some("Androtec GmbH"),
            0x04FB => Some("Benchmark Drives GmbH & Co. KG"),
            0x04FC => Some("SwingLync L. L. C."),
            0x04FD => Some("Tapkey GmbH"),
            0x04FE => Some("Woosim Systems Inc."),
            0x04FF => Some("Microsemi Corporation"),
            0x0500 => Some("Wiliot LTD."),
            0x0501 => Some("Polaris IND"),
            0x0502 => Some("Specifi-Kali LLC"),
            0x0503 => Some("Locoroll, Inc"),
            0x0504 => Some("PHYPLUS Inc"),
            0x0505 => Some("Inplay Technologies LLC"),
            0x0506 => Some("Hager"),
            0x0507 => Some("Yellowcog"),
            0x0508 => Some("Axes System sp. z o. o."),
            0x0509 => Some("myLIFTER Inc."),
            0x050A => Some("Shake-on B.V."),
            0x050B => Some("Vibrissa Inc."),
            0x050C => Some("OSRAM GmbH"),
            0x050D => Some("TRSystems GmbH"),
            0x050E => Some("Yichip Microelectronics (Hangzhou) Co.,Ltd."),
            0x050F => Some("Foundation Engineering LLC"),
            0x0510 => Some("UNI-ELECTRONICS, INC."),
            0x0511 => Some("Brookfield Equinox LLC"),
            0x0512 => Some("Soprod SA"),
            0x0513 => Some("9974091 Canada Inc."),
            0x0514 => Some("FIBRO GmbH"),
            0x0515 => Some("RB Controls Co., Ltd."),
            0x0516 => Some("Footmarks"),
            0x0517 => Some("Amtronic Sverige AB (formerly Amcore AB)"),
            0x0518 => Some("MAMORIO.inc"),
            0x0519 => Some("Tyto Life LLC"),
            0x051A => Some("Leica Camera AG"),
            0x051B => Some("Angee Technologies Ltd."),
            0x051C => Some("EDPS"),
            0x051D => Some("OFF Line Co., Ltd."),
            0x051E => Some("Detect Blue Limited"),
            0x051F => Some("Setec Pty Ltd"),
            0x0520 => Some("Target Corporation"),
            0x0521 => Some("IAI Corporation"),
            0x0522 => Some("NS Tech, Inc."),
            0x0523 => Some("MTG Co., Ltd."),
            0x0524 => Some("Hangzhou iMagic Technology Co., Ltd"),
            0x0525 => Some("HONGKONG NANO IC TECHNOLOGIES CO., LIMITED"),
            0x0526 => Some("Honeywell International Inc."),
            0x0527 => Some("Albrecht JUNG"),
            0x0528 => Some("Lunera Lighting Inc."),
            0x0529 => Some("Lumen UAB"),
            0x052A => Some("Keynes Controls Ltd"),
            0x052B => Some("Novartis AG"),
            0x052C => Some("Geosatis SA"),
            0x052D => Some("EXFO, Inc."),
            0x052E => Some("LEDVANCE GmbH"),
            0x052F => Some("Center ID Corp."),
            0x0530 => Some("Adolene, Inc."),
            0x0531 => Some("D&M Holdings Inc."),
            0x0532 => Some("CRESCO Wireless, Inc."),
            0x0533 => Some("Nura Operations Pty Ltd"),
            0x0534 => Some("Frontiergadget, Inc."),
            0x0535 => Some("Smart Component Technologies Limited"),
            0x0536 => Some("ZTR Control Systems LLC"),
            0x0537 => Some("MetaLogics Corporation"),
            0x0538 => Some("Medela AG"),
            0x0539 => Some("OPPLE Lighting Co., Ltd"),
            0x053A => Some("Savitech Corp.,"),
            0x053B => Some("prodigy"),
            0x053C => Some("Screenovate Technologies Ltd"),
            0x053D => Some("TESA SA"),
            0x053E => Some("CLIM8 LIMITED"),
            0x053F => Some("Silergy Corp"),
            0x0540 => Some("SilverPlus, Inc"),
            0x0541 => Some("Sharknet srl"),
            0x0542 => Some("Mist Systems, Inc."),
            0x0543 => Some("MIWA LOCK CO.,Ltd"),
            0x0544 => Some("OrthoSensor, Inc."),
            0x0545 => Some("Candy Hoover Group s.r.l"),
            0x0546 => Some("Apexar Technologies S.A."),
            0x0547 => Some("LOGICDATA d.o.o."),
            0x0548 => Some("Knick Elektronische Messgeraete GmbH & Co. KG"),
            0x0549 => Some("Smart Technologies and Investment Limited"),
            0x054A => Some("Linough Inc."),
            0x054B => Some("Advanced Electronic Designs, Inc."),
            0x054C => Some("Carefree Scott Fetzer Co Inc"),
            0x054D => Some("Sensome"),
            0x054E => Some("FORTRONIK storitve d.o.o."),
            0x054F => Some("Sinnoz"),
            0x0550 => Some("Versa Networks, Inc."),
            0x0551 => Some("Sylero"),
            0x0552 => Some("Avempace SARL"),
            0x0553 => Some("Nintendo Co., Ltd."),
            0x0554 => Some("National Instruments"),
            0x0555 => Some("KROHNE Messtechnik GmbH"),
            0x0556 => Some("Otodynamics Ltd"),
            0x0557 => Some("Arwin Technology Limited"),
            0x0558 => Some("benegear, inc."),
            0x0559 => Some("Newcon Optik"),
            0x055A => Some("CANDY HOUSE, Inc."),
            0x055B => Some("FRANKLIN TECHNOLOGY INC"),
            0x055C => Some("Lely"),
            0x055D => Some("Valve Corporation"),
            0x055E => Some("Hekatron Vertriebs GmbH"),
            0x055F => Some("PROTECH S.A.S. DI GIRARDI ANDREA & C."),
            0x0560 => Some("Sarita CareTech APS (formerly Sarita CareTech IVS)"),
            0x0561 => Some("Finder S.p.A."),
            0x0562 => Some("Thalmic Labs Inc."),
            0x0563 => Some("Steinel Vertrieb GmbH"),
            0x0564 => Some("Beghelli Spa"),
            0x0565 => Some("Beijing Smartspace Technologies Inc."),
            0x0566 => Some("CORE TRANSPORT TECHNOLOGIES NZ LIMITED"),
            0x0567 => Some("Xiamen Everesports Goods Co., Ltd"),
            0x0568 => Some("Bodyport Inc."),
            0x0569 => Some("Audionics System, INC."),
            0x056A => Some("Flipnavi Co.,Ltd."),
            0x056B => Some("Rion Co., Ltd."),
            0x056C => Some("Long Range Systems, LLC"),
            0x056D => Some("Redmond Industrial Group LLC"),
            0x056E => Some("VIZPIN INC."),
            0x056F => Some("BikeFinder AS"),
            0x0570 => Some("Consumer Sleep Solutions LLC"),
            0x0571 => Some("PSIKICK, INC."),
            0x0572 => Some("AntTail.com"),
            0x0573 => Some("Lighting Science Group Corp."),
            0x0574 => Some("AFFORDABLE ELECTRONICS INC"),
            0x0575 => Some("Integral Memroy Plc"),
            0x0576 => Some("Globalstar, Inc."),
            0x0577 => Some("True Wearables, Inc."),
            0x0578 => Some("Wellington Drive Technologies Ltd"),
            0x0579 => Some("Ensemble Tech Private Limited"),
            0x057A => Some("OMNI Remotes"),
            0x057B => Some("Duracell U.S. Operations Inc."),
            0x057C => Some("Toor Technologies LLC"),
            0x057D => Some("Instinct Performance"),
            0x057E => Some("Beco, Inc"),
            0x057F => Some("Scuf Gaming International, LLC"),
            0x0580 => Some("ARANZ Medical Limited"),
            0x0581 => Some("LYS TECHNOLOGIES LTD"),
            0x0582 => Some("Breakwall Analytics, LLC"),
            0x0583 => Some("Code Blue Communications"),
            0x0584 => Some("Gira Giersiepen GmbH & Co. KG"),
            0x0585 => Some("Hearing Lab Technology"),
            0x0586 => Some("LEGRAND"),
            0x0587 => Some("Derichs GmbH"),
            0x0588 => Some("ALT-TEKNIK LLC"),
            0x0589 => Some("Star Technologies"),
            0x058A => Some("START TODAY CO.,LTD."),
            0x058B => Some("Maxim Integrated Products"),
            0x058C => Some("MERCK Kommanditgesellschaft auf Aktien"),
            0x058D => Some("Jungheinrich Aktiengesellschaft"),
            0x058E => Some("Oculus VR, LLC"),
            0x058F => Some("HENDON SEMICONDUCTORS PTY LTD"),
            0x0590 => Some("Pur3 Ltd"),
            0x0591 => Some("Viasat Group S.p.A."),
            0x0592 => Some("IZITHERM"),
            0x0593 => Some("Spaulding Clinical Research"),
            0x0594 => Some("Kohler Company"),
            0x0595 => Some("Inor Process AB"),
            0x0596 => Some("My Smart Blinds"),
            0x0597 => Some("RadioPulse Inc"),
            0x0598 => Some("rapitag GmbH"),
            0x0599 => Some("Lazlo326, LLC."),
            0x059A => Some("Teledyne Lecroy, Inc."),
            0x059B => Some("Dataflow Systems Limited"),
            0x059C => Some("Macrogiga Electronics"),
            0x059D => Some("Tandem Diabetes Care"),
            0x059E => Some("Polycom, Inc."),
            0x059F => Some("Fisher & Paykel Healthcare"),
            0x05A0 => Some("RCP Software Oy"),
            0x05A1 => Some("Shanghai Xiaoyi Technology Co.,Ltd."),
            0x05A2 => Some("ADHERIUM(NZ) LIMITED"),
            0x05A3 => Some("Axiomware Systems Incorporated"),
            0x05A4 => Some("O. E. M. Controls, Inc."),
            0x05A5 => Some("Kiiroo BV"),
            0x05A6 => Some("Telecon Mobile Limited"),
            0x05A7 => Some("Sonos Inc"),
            0x05A8 => Some("Tom Allebrandi Consulting"),
            0x05A9 => Some("Monidor"),
            0x05AA => Some("Tramex Limited"),
            0x05AB => Some("Nofence AS"),
            0x05AC => Some("GoerTek Dynaudio Co., Ltd."),
            0x05AD => Some("INIA"),
            0x05AE => Some("CARMATE MFG.CO.,LTD"),
            0x05AF => Some("ONvocal"),
            0x05B0 => Some("NewTec GmbH"),
            0x05B1 => Some("Medallion Instrumentation Systems"),
            0x05B2 => Some("CAREL INDUSTRIES S.P.A."),
            0x05B3 => Some("Parabit Systems, Inc."),
            0x05B4 => Some("White Horse Scientific ltd"),
            0x05B5 => Some("verisilicon"),
            0x05B6 => Some("Elecs Industry Co.,Ltd."),
            0x05B7 => Some("Beijing Pinecone Electronics Co.,Ltd."),
            0x05B8 => Some("Ambystoma Labs Inc."),
            0x05B9 => Some("Suzhou Pairlink Network Technology"),
            0x05BA => Some("igloohome"),
            0x05BB => Some("Oxford Metrics plc"),
            0x05BC => Some("Leviton Mfg. Co., Inc."),
            0x05BD => Some("ULC Robotics Inc."),
            0x05BE => Some("RFID Global by Softwork SrL"),
            0x05BF => Some("Real-World-Systems Corporation"),
            0x05C0 => Some("Nalu Medical, Inc."),
            0x05C1 => Some("P.I.Engineering"),
            0x05C2 => Some("Grote Industries"),
            0x05C3 => Some("Runtime, Inc."),
            0x05C4 => Some("Codecoup sp. z o.o. sp. k."),
            0x05C5 => Some("SELVE GmbH & Co. KG"),
            0x05C6 => Some("Smart Animal Training Systems, LLC"),
            0x05C7 => Some("Lippert Components, INC"),
            0x05C8 => Some("SOMFY SAS"),
            0x05C9 => Some("TBS Electronics B.V."),
            0x05CA => Some("MHL Custom Inc"),
            0x05CB => Some("LucentWear LLC"),
            0x05CC => Some("WATTS ELECTRONICS"),
            0x05CD => Some("RJ Brands LLC"),
            0x05CE => Some("V-ZUG Ltd"),
            0x05CF => Some("Biowatch SA"),
            0x05D0 => Some("Anova Applied Electronics"),
            0x05D1 => Some("Lindab AB"),
            0x05D2 => Some("frogblue TECHNOLOGY GmbH"),
            0x05D3 => Some("Acurable Limited"),
            0x05D4 => Some("LAMPLIGHT Co., Ltd."),
            0x05D5 => Some("TEGAM, Inc."),
            0x05D6 => Some("Zhuhai Jieli technology Co.,Ltd"),
            0x05D7 => Some("modum.io AG"),
            0x05D8 => Some("Farm Jenny LLC"),
            0x05D9 => Some("Toyo Electronics Corporation"),
            0x05DA => Some("Applied Neural Research Corp"),
            0x05DB => Some("Avid Identification Systems, Inc."),
            0x05DC => Some("Petronics Inc."),
            0x05DD => Some("essentim GmbH"),
            0x05DE => Some("QT Medical INC."),
            0x05DF => Some("VIRTUALCLINIC.DIRECT LIMITED"),
            0x05E0 => Some("Viper Design LLC"),
            0x05E1 => Some("Human, Incorporated"),
            0x05E2 => Some("stAPPtronics GmbH"),
            0x05E3 => Some("Elemental Machines, Inc."),
            0x05E4 => Some("Taiyo Yuden Co., Ltd"),
            0x05E5 => Some("INEO ENERGY& SYSTEMS"),
            0x05E6 => Some("Motion Instruments Inc."),
            0x05E7 => Some("PressurePro"),
            0x05E8 => Some("COWBOY"),
            0x05E9 => Some("iconmobile GmbH"),
            0x05EA => Some("ACS-Control-System GmbH"),
            0x05EB => Some("Bayerische Motoren Werke AG"),
            0x05EC => Some("Gycom Svenska AB"),
            0x05ED => Some("Fuji Xerox Co., Ltd"),
            0x05EE => Some("Glide Inc."),
            0x05EF => Some("SIKOM AS"),
            0x05F0 => Some("beken"),
            0x05F1 => Some("The Linux Foundation"),
            0x05F2 => Some("Try and E CO.,LTD."),
            0x05F3 => Some("SeeScan"),
            0x05F4 => Some("Clearity, LLC"),
            0x05F5 => Some("GS TAG"),
            0x05F6 => Some("DPTechnics"),
            0x05F7 => Some("TRACMO, INC."),
            0x05F8 => Some("Anki Inc."),
            0x05F9 => Some("Hagleitner Hygiene International GmbH"),
            0x05FA => Some("Konami Sports Life Co., Ltd."),
            0x05FB => Some("Arblet Inc."),
            0x05FC => Some("Masbando GmbH"),
            0x05FD => Some("Innoseis"),
            0x05FE => Some("Niko"),
            0x05FF => Some("Wellnomics Ltd"),
            0x0600 => Some("iRobot Corporation"),
            0x0601 => Some("Schrader Electronics"),
            0x0602 => Some("Geberit International AG"),
            0x0603 => Some("Fourth Evolution Inc"),
            0x0604 => Some("Cell2Jack LLC"),
            0x0605 => Some("FMW electronic Futterer u. Maier-Wolf OHG"),
            0x0606 => Some("John Deere"),
            0x0607 => Some("Rookery Technology Ltd"),
            0x0608 => Some("KeySafe-Cloud"),
            0x0609 => Some("BUCHI Labortechnik AG"),
            0x060A => Some("IQAir AG"),
            0x060B => Some("Triax Technologies Inc"),
            0x060C => Some("Vuzix Corporation"),
            0x060D => Some("TDK Corporation"),
            0x060E => Some("Blueair AB"),
            0x060F => Some("Signify Netherlands (formerlyPhilips Lighting B.V.)"),
            0x0610 => Some("ADH GUARDIAN USA LLC"),
            0x0611 => Some("Beurer GmbH"),
            0x0612 => Some("Playfinity AS"),
            0x0613 => Some("Hans Dinslage GmbH"),
            0x0614 => Some("OnAsset Intelligence, Inc."),
            0x0615 => Some("INTER ACTION Corporation"),
            0x0616 => Some("OS42 UG (haftungsbeschraenkt)"),
            0x0617 => Some("WIZCONNECTED COMPANY LIMITED"),
            0x0618 => Some("Audio-Technica Corporation"),
            0x0619 => Some("Six Guys Labs, s.r.o."),
            0x061A => Some("R.W. Beckett Corporation"),
            0x061B => Some("silex technology, inc."),
            0x061C => Some("Univations Limited"),
            0x061D => Some("SENS Innovation ApS"),
            0x061E => Some("Diamond Kinetics, Inc."),
            0x061F => Some("Phrame Inc."),
            0x0620 => Some("Forciot Oy"),
            0x0621 => Some("Noordung d.o.o."),
            0x0622 => Some("Beam Labs, LLC"),
            0x0623 => Some("Philadelphia Scientific (U.K.) Limited"),
            0x0624 => Some("Biovotion AG"),
            0x0625 => Some("Square Panda, Inc."),
            0x0626 => Some("Amplifico"),
            0x0627 => Some("WEG S.A."),
            0x0628 => Some("Ensto Oy"),
            0x0629 => Some("PHONEPE PVT LTD"),
            0x062A => Some("Lunatico Astronomia SL"),
            0x062B => Some("MinebeaMitsumi Inc."),
            0x062C => Some("ASPion GmbH"),
            0x062D => Some("Vossloh-Schwabe Deutschland GmbH"),
            0x062E => Some("Procept"),
            0x062F => Some("ONKYO Corporation"),
            0x0630 => Some("Asthrea D.O.O."),
            0x0631 => Some("Fortiori Design LLC"),
            0x0632 => Some("Hugo Muller GmbH & Co KG"),
            0x0633 => Some("Wangi Lai PLT"),
            0x0634 => Some("Fanstel Corp"),
            0x0635 => Some("Crookwood"),
            0x0636 => Some("ELECTRONICA INTEGRAL DE SONIDO S.A."),
            0x0637 => Some("GiP Innovation Tools GmbH"),
            0x0638 => Some("LX SOLUTIONS PTY LIMITED"),
            0x0639 => Some("Shenzhen Minew Technologies Co., Ltd."),
            0x063A => Some("Prolojik Limited"),
            0x063B => Some("Kromek Group Plc"),
            0x063C => Some("Contec Medical Systems Co., Ltd."),
            0x063D => Some("Xradio Technology Co.,Ltd."),
            0x063E => Some("The Indoor Lab, LLC"),
            0x063F => Some("LDL TECHNOLOGY"),
            0x0640 => Some("Parkifi"),
            0x0641 => Some("Revenue Collection Systems FRANCE SAS"),
            0x0642 => Some("Bluetrum Technology Co.,Ltd"),
            0x0643 => Some("makita corporation"),
            0x0644 => Some("Apogee Instruments"),
            0x0645 => Some("BM3"),
            0x0646 => Some("SGV Group Holding GmbH & Co. KG"),
            0x0647 => Some("MED-EL"),
            0x0648 => Some("Ultune Technologies"),
            0x0649 => Some("Ryeex Technology Co.,Ltd."),
            0x064A => Some("Open Research Institute, Inc."),
            0x064B => Some("Scale-Tec, Ltd"),
            0x064C => Some("Zumtobel Group AG"),
            0x064D => Some("iLOQ Oy"),
            0x064E => Some("KRUXWorks Technologies Private Limited"),
            0x064F => Some("Digital Matter Pty Ltd"),
            0x0650 => Some("Coravin, Inc."),
            0x0651 => Some("Stasis Labs, Inc."),
            0x0652 => Some("ITZ Innovations- und Technologiezentrum GmbH"),
            0x0653 => Some("Meggitt SA"),
            0x0654 => Some("Ledlenser GmbH & Co. KG"),
            0x0655 => Some("Renishaw PLC"),
            0x0656 => Some("ZhuHai AdvanPro Technology Company Limited"),
            0x0657 => Some("Meshtronix Limited"),
            0x0658 => Some("Payex Norge AS"),
            0x0659 => Some("UnSeen Technologies Oy"),
            0x065A => Some("Zound Industries International AB"),
            0x065B => Some("Sesam Solutions BV"),
            0x065C => Some("PixArt Imaging Inc."),
            0x065D => Some("Panduit Corp."),
            0x065E => Some("Alo AB"),
            0x065F => Some("Ricoh Company Ltd"),
            0x0660 => Some("RTC Industries, Inc."),
            0x0661 => Some("Mode Lighting Limited"),
            0x0662 => Some("Particle Industries, Inc."),
            0x0663 => Some("Advanced Telemetry Systems, Inc."),
            0x0664 => Some("RHA TECHNOLOGIES LTD"),
            0x0665 => Some("Pure International Limited"),
            0x0666 => Some("WTO Werkzeug-Einrichtungen GmbH"),
            0x0667 => Some("Spark Technology Labs Inc."),
            0x0668 => Some("Bleb Technology srl"),
            0x0669 => Some("Livanova USA, Inc."),
            0x066A => Some("Brady Worldwide Inc."),
            0x066B => Some("DewertOkin GmbH"),
            0x066C => Some("Ztove ApS"),
            0x066D => Some("Venso EcoSolutions AB"),
            0x066E => Some("Eurotronik Kranj d.o.o."),
            0x066F => Some("Hug Technology Ltd"),
            0x0670 => Some("Gema Switzerland GmbH"),
            0x0671 => Some("Buzz Products Ltd."),
            0x0672 => Some("Kopi"),
            0x0673 => Some("Innova Ideas Limited"),
            0x0674 => Some("BeSpoon"),
            0x0675 => Some("Deco Enterprises, Inc."),
            0x0676 => Some("Expai Solutions Private Limited"),
            0x0677 => Some("Innovation First, Inc."),
            0x0678 => Some("SABIK Offshore GmbH"),
            0x0679 => Some("4iiii Innovations Inc."),
            0x067A => Some("The Energy Conservatory, Inc."),
            0x067B => Some("I.FARM, INC."),
            0x067C => Some("Tile, Inc."),
            0x067D => Some("Form Athletica Inc."),
            0x067E => Some("MbientLab Inc"),
            0x067F => {
                Some("NETGRID S.N.C. DI BISSOLI MATTEO, CAMPOREALE SIMONE, TOGNETTI FEDERICO")
            }
            0x0680 => Some("Mannkind Corporation"),
            0x0681 => Some("Trade FIDES a.s."),
            0x0682 => Some("Photron Limited"),
            0x0683 => Some("Eltako GmbH"),
            0x0684 => Some("Dermalapps, LLC"),
            0x0685 => Some("Greenwald Industries"),
            0x0686 => Some("inQs Co., Ltd."),
            0x0687 => Some("Cherry GmbH"),
            0x0688 => Some("Amsted Digital Solutions Inc."),
            0x0689 => Some("Tacx b.v."),
            0x068A => Some("Raytac Corporation"),
            0x068B => Some("Jiangsu Teranovo Tech Co., Ltd."),
            0x068C => Some("Changzhou Sound Dragon Electronics and Acoustics Co., Ltd"),
            0x068D => Some("JetBeep Inc."),
            0x068E => Some("Razer Inc."),
            0x068F => Some("JRM Group Limited"),
            0x0690 => Some("Eccrine Systems, Inc."),
            0x0691 => Some("Curie Point AB"),
            0x0692 => Some("Georg Fischer AG"),
            0x0693 => Some("Hach - Danaher"),
            0x0694 => Some("T&A Laboratories LLC"),
            0x0695 => Some("Koki Holdings Co., Ltd."),
            0x0696 => Some("Gunakar Private Limited"),
            0x0697 => Some("Stemco Products Inc"),
            0x0698 => Some("Wood IT Security, LLC"),
            0x0699 => Some("RandomLab SAS"),
            0x069A => Some("Adero, Inc. (formerly as TrackR, Inc.)"),
            0x069B => Some("Dragonchip Limited"),
            0x069C => Some("Noomi AB"),
            0x069D => Some("Vakaros LLC"),
            0x069E => Some("Delta Electronics, Inc."),
            0x069F => Some("FlowMotion Technologies AS"),
            0x06A0 => Some("OBIQ Location Technology Inc."),
            0x06A1 => Some("Cardo Systems, Ltd"),
            0x06A2 => Some("Globalworx GmbH"),
            0x06A3 => Some("Nymbus, LLC"),
            0x06A4 => Some("Sanyo Techno Solutions Tottori Co., Ltd."),
            0x06A5 => Some("TEKZITEL PTY LTD"),
            0x06A6 => Some("Roambee Corporation"),
            0x06A7 => Some("Chipsea Technologies (ShenZhen) Corp."),
            0x06A8 => Some("GD Midea Air-Conditioning Equipment Co., Ltd."),
            0x06A9 => Some("Soundmax Electronics Limited"),
            0x06AA => Some("Produal Oy"),
            0x06AB => Some("HMS Industrial Networks AB"),
            0x06AC => Some("Ingchips Technology Co., Ltd."),
            0x06AD => Some("InnovaSea Systems Inc."),
            0x06AE => Some("SenseQ Inc."),
            0x06AF => Some("Shoof Technologies"),
            0x06B0 => Some("BRK Brands, Inc."),
            0x06B1 => Some("SimpliSafe, Inc."),
            0x06B2 => Some("Tussock Innovation 2013 Limited"),
            0x06B3 => Some("The Hablab ApS"),
            0x06B4 => Some("Sencilion Oy"),
            0x06B5 => Some("Wabilogic Ltd."),
            0x06B6 => Some("Sociometric Solutions, Inc."),
            0x06B7 => Some("iCOGNIZE GmbH"),
            0x06B8 => Some("ShadeCraft, Inc"),
            0x06B9 => Some("Beflex Inc."),
            0x06BA => Some("Beaconzone Ltd"),
            0x06BB => Some("Leaftronix Analogic Solutions Private Limited"),
            0x06BC => Some("TWS Srl"),
            0x06BD => Some("ABB Oy"),
            0x06BE => Some("HitSeed Oy"),
            0x06BF => Some("Delcom Products Inc."),
            0x06C0 => Some("CAME S.p.A."),
            0x06C1 => Some("Alarm.com Holdings, Inc"),
            0x06C2 => Some("Measurlogic Inc."),
            0x06C3 => Some("King I Electronics.Co.,Ltd"),
            0x06C4 => Some("Dream Labs GmbH"),
            0x06C5 => Some("Urban Compass, Inc"),
            0x06C6 => Some("Simm Tronic Limited"),
            0x06C7 => Some("Somatix Inc"),
            0x06C8 => Some("Storz & Bickel GmbH & Co. KG"),
            0x06C9 => Some("MYLAPS B.V."),
            0x06CA => Some("Shenzhen Zhongguang Infotech Technology Development Co., Ltd"),
            0x06CB => Some("Dyeware, LLC"),
            0x06CC => Some("Dongguan SmartAction Technology Co.,Ltd."),
            0x06CD => Some("DIG Corporation"),
            0x06CE => Some("FIOR & GENTZ"),
            0x06CF => Some("Belparts N.V."),
            0x06D0 => Some("Etekcity Corporation"),
            0x06D1 => Some("Meyer Sound Laboratories, Incorporated"),
            0x06D2 => Some("CeoTronics AG"),
            0x06D3 => Some("TriTeq Lock and Security, LLC"),
            0x06D4 => Some("DYNAKODE TECHNOLOGY PRIVATE LIMITED"),
            0x06D5 => Some("Sensirion AG"),
            0x06D6 => Some("JCT Healthcare Pty Ltd"),
            0x06D7 => Some("FUBA Automotive Electronics GmbH"),
            0x06D8 => Some("AW Company"),
            0x06D9 => Some("Shanghai Mountain View Silicon Co.,Ltd."),
            0x06DA => Some("Zliide Technologies ApS"),
            0x06DB => Some("Automatic Labs, Inc."),
            0x06DC => Some("Industrial Network Controls, LLC"),
            0x06DD => Some("Intellithings Ltd."),
            0x06DE => Some("Navcast, Inc."),
            0x06DF => Some("Hubbell Lighting, Inc."),
            0x06E0 => Some("Avaya"),
            0x06E1 => Some("Milestone AV Technologies LLC"),
            0x06E2 => Some("Alango Technologies Ltd"),
            0x06E3 => Some("Spinlock Ltd"),
            0x06E4 => Some("Aluna"),
            0x06E5 => Some("OPTEX CO.,LTD."),
            0x06E6 => Some("NIHON DENGYO KOUSAKU"),
            0x06E7 => Some("VELUX A/S"),
            0x06E8 => Some("Almendo Technologies GmbH"),
            0x06E9 => Some("Zmartfun Electronics, Inc."),
            0x06EA => Some("SafeLine Sweden AB"),
            0x06EB => Some("Houston Radar LLC"),
            0x06EC => Some("Sigur"),
            0x06ED => Some("J Neades Ltd"),
            0x06EE => Some("Avantis Systems Limited"),
            0x06EF => Some("ALCARE Co., Ltd."),
            0x06F0 => Some("Chargy Technologies, SL"),
            0x06F1 => Some("Shibutani Co., Ltd."),
            0x06F2 => Some("Trapper Data AB"),
            0x06F3 => Some("Alfred International Inc."),
            0x06F4 => Some("Near Field Solutions Ltd"),
            0x06F5 => Some("Vigil Technologies Inc."),
            0x06F6 => Some("Vitulo Plus BV"),
            0x06F7 => Some("WILKA Schliesstechnik GmbH"),
            0x06F8 => Some("BodyPlus Technology Co.,Ltd"),
            0x06F9 => Some("happybrush GmbH"),
            0x06FA => Some("Enequi AB"),
            0x06FB => Some("Sartorius AG"),
            0x06FC => Some("Tom Communication Industrial Co.,Ltd."),
            0x06FD => Some("ESS Embedded System Solutions Inc."),
            0x06FE => Some("Mahr GmbH"),
            0x06FF => Some("Redpine Signals Inc"),
            0x0700 => Some("TraqFreq LLC"),
            0x0701 => Some("PAFERS TECH"),
            0x0702 => Some("Akciju sabiedriba \"SAF TEHNIKA\""),
            0x0703 => Some("Beijing Jingdong Century Trading Co., Ltd."),
            0x0704 => Some("JBX Designs Inc."),
            0x0705 => Some("AB Electrolux"),
            0x0706 => Some("Wernher von Braun Center for ASdvanced Research"),
            0x0707 => Some("Essity Hygiene and Health Aktiebolag"),
            0x0708 => Some("Be Interactive Co., Ltd"),
            0x0709 => Some("Carewear Corp."),
            0x070A => Some("Huf Hlsbeck & Frst GmbH & Co. KG"),
            0x070B => Some("Element Products, Inc."),
            0x070C => Some("Beijing Winner Microelectronics Co.,Ltd"),
            0x070D => Some("SmartSnugg Pty Ltd"),
            0x070E => Some("FiveCo Sarl"),
            0x070F => Some("California Things Inc."),
            0x0710 => Some("Audiodo AB"),
            0x0711 => Some("ABAX AS"),
            0x0712 => Some("Bull Group Company Limited"),
            0x0713 => Some("Respiri Limited"),
            0x0714 => Some("MindPeace Safety LLC"),
            0x0715 => Some("MBARC LABS Inc (formerly Vgyan Solutions)"),
            0x0716 => Some("Altonics"),
            0x0717 => Some("iQsquare BV"),
            0x0718 => Some("IDIBAIX enginneering"),
            0x0719 => Some("ECSG"),
            0x071A => Some("REVSMART WEARABLE HK CO LTD"),
            0x071B => Some("Precor"),
            0x071C => Some("F5 Sports, Inc"),
            0x071D => Some("exoTIC Systems"),
            0x071E => Some("DONGGUAN HELE ELECTRONICS CO., LTD"),
            0x071F => Some("Dongguan Liesheng Electronic Co.Ltd"),
            0x0720 => Some("Oculeve, Inc."),
            0x0721 => Some("Clover Network, Inc."),
            0x0722 => Some("Xiamen Eholder Electronics Co.Ltd"),
            0x0723 => Some("Ford Motor Company"),
            0x0724 => Some("Guangzhou SuperSound Information Technology Co.,Ltd"),
            0x0725 => Some("Tedee Sp. z o.o."),
            0x0726 => Some("PHC Corporation"),
            0x0727 => Some("STALKIT AS"),
            0x0728 => Some("Eli Lilly and Company"),
            0x0729 => Some("SwaraLink Technologies"),
            0x072A => Some("JMR embedded systems GmbH"),
            0x072B => Some("Bitkey Inc."),
            0x072C => Some("GWA Hygiene GmbH"),
            0x072D => Some("Safera Oy"),
            0x072E => Some("Open Platform Systems LLC"),
            0x072F => Some("OnePlus Electronics (Shenzhen) Co., Ltd."),
            0x0730 => Some("Wildlife Acoustics, Inc."),
            0x0731 => Some("ABLIC Inc."),
            0x0732 => Some("Dairy Tech, Inc."),
            0x0733 => Some("Iguanavation, Inc."),
            0x0734 => Some("DiUS Computing Pty Ltd"),
            0x0735 => Some("UpRight Technologies LTD"),
            0x0736 => Some("FrancisFund, LLC"),
            0x0737 => Some("LLC Navitek"),
            0x0738 => Some("Glass Security Pte Ltd"),
            0x0739 => Some("Jiangsu Qinheng Co., Ltd."),
            0x073A => Some("Chandler Systems Inc."),
            0x073B => Some("Fantini Cosmi s.p.a."),
            0x073C => Some("Acubit ApS"),
            0x073D => Some("Beijing Hao Heng Tian Tech Co., Ltd."),
            0x073E => Some("Bluepack S.R.L."),
            0x073F => Some("Beijing Unisoc Technologies Co., Ltd."),
            0x0740 => Some("HITIQ LIMITED"),
            0x0741 => Some("MAC SRL"),
            0x0742 => Some("DML LLC"),
            0x0743 => Some("Sanofi"),
            0x0744 => Some("SOCOMEC"),
            0x0745 => Some("WIZNOVA, Inc."),
            0x0746 => Some("Seitec Elektronik GmbH"),
            0x0747 => Some("OR Technologies Pty Ltd"),
            0x0748 => Some("GuangZhou KuGou Computer Technology Co.Ltd"),
            0x0749 => Some("DIAODIAO (Beijing) Technology Co., Ltd."),
            0x074A => Some("Illusory Studios LLC"),
            0x074B => Some("Sarvavid Software Solutions LLP"),
            0x074C => Some("iopool s.a."),
            0x074D => Some("Amtech Systems, LLC"),
            0x074E => Some("EAGLE DETECTION SA"),
            0x074F => Some("MEDIATECH S.R.L."),
            0x0750 => Some("Hamilton Professional Services of Canada Incorporated"),
            0x0751 => Some("Changsha JEMO IC Design Co.,Ltd"),
            0x0752 => Some("Elatec GmbH"),
            0x0753 => Some("JLG Industries, Inc."),
            0x0754 => Some("Michael Parkin"),
            0x0755 => Some("Brother Industries, Ltd"),
            0x0756 => Some("Lumens For Less, Inc"),
            0x0757 => Some("ELA Innovation"),
            0x0758 => Some("umanSense AB"),
            0x0759 => Some("Shanghai InGeek Cyber Security Co., Ltd."),
            0x075A => Some("HARMAN CO.,LTD."),
            0x075B => Some("Smart Sensor Devices AB"),
            0x075C => Some("Antitronics Inc."),
            0x075D => Some("RHOMBUS SYSTEMS, INC."),
            0x075E => Some("Katerra Inc."),
            0x075F => Some("Remote Solution Co., LTD."),
            0x0760 => Some("Vimar SpA"),
            0x0761 => Some("Mantis Tech LLC"),
            0x0762 => Some("TerOpta Ltd"),
            0x0763 => Some("PIKOLIN S.L."),
            0x0764 => Some("WWZN Information Technology Company Limited"),
            0x0765 => Some("Voxx International"),
            0x0766 => Some("ART AND PROGRAM, INC."),
            0x0767 => Some("NITTO DENKO ASIA TECHNICAL CENTRE PTE. LTD."),
            0x0768 => Some("Peloton Interactive Inc."),
            0x0769 => Some("Force Impact Technologies"),
            0x076A => Some("Dmac Mobile Developments, LLC"),
            0x076B => Some("Engineered Medical Technologies"),
            0x076C => Some("Noodle Technology inc"),
            0x076D => Some("Graesslin GmbH"),
            0x076E => Some("WuQi technologies, Inc."),
            0x076F => Some("Successful Endeavours Pty Ltd"),
            0x0770 => Some("InnoCon Medical ApS"),
            0x0771 => Some("Corvex Connected Safety"),
            0x0772 => Some("Thirdwayv Inc."),
            0x0773 => Some("Echoflex Solutions Inc."),
            0x0774 => Some("C-MAX Asia Limited"),
            0x0775 => Some("4eBusiness GmbH"),
            0x0776 => Some("Cyber Transport Control GmbH"),
            0x0777 => Some("Cue"),
            0x0778 => Some("KOAMTAC INC."),
            0x0779 => Some("Loopshore Oy"),
            0x077A => Some("Niruha Systems Private Limited"),
            0x077B => Some("AmaterZ, Inc."),
            0x077C => Some("radius co., ltd."),
            0x077D => Some("Sensority, s.r.o."),
            0x077E => Some("Sparkage Inc."),
            0x077F => Some("Glenview Software Corporation"),
            0x0780 => Some("Finch Technologies Ltd."),
            0x0781 => Some("Qingping Technology (Beijing) Co., Ltd."),
            0x0782 => Some("DeviceDrive AS"),
            0x0783 => Some("ESEMBER LIMITED LIABILITY COMPANY"),
            0x0784 => Some("audifon GmbH & Co. KG"),
            0x0785 => Some("O2 Micro, Inc."),
            0x0786 => Some("HLP Controls Pty Limited"),
            0x0787 => Some("Pangaea Solution"),
            0x0788 => Some("BubblyNet, LLC"),
            _ => None,
        }
    }
}
