//! Represent Google Trend Categories.

use serde::Deserialize;
use serde_json::Value;
use std::fmt::{Display, Formatter, Result};

#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Default, Deserialize)]
pub struct Category {
    id: u16,
}

impl Category {
    /// Create a new Category.
    /// 
    /// Returns a Category instance.
    /// 
    /// # Example
    /// ```
    /// # use rtrend::Category;
    /// let category = Category::new(76);
    /// ```
    /// 
    /// # Panics
    /// An unsupported Category id will make the program to panic
    /// ```should_panic
    /// # use rtrend::Category;
    /// // 1800 is an unsupported category id
    /// let category = Category::new(1800);
    /// ```
    pub fn new(id: u16) -> Category {
        Self {
            id: Self::check_category(id),
        }
    }

    fn check_category(id: u16) -> u16 {
        let value: Value = serde_json::from_str(Self::CONTENT).unwrap();
        let ids = value
            .as_array()
            .unwrap()
            .into_iter()
            .filter(|ident| ident["id"] == id)
            .collect::<Vec<&Value>>();

        match ids.len() {
            0 => panic!("Unsupported category"),
            _ => ids[0],
        };

        id
    }

    /// Category supported by Google Trend
    const CONTENT: &'static str = r#"[
        { "name": "All", "id": 0 },
        { "name": "Celebrities & Entertainment News", "id": 184 },
        { "name": "Animated Films", "id": 1104 },
        { "name": "Anime & Manga", "id": 317 },
        { "name": "Cartoons", "id": 319 },
        { "name": "Comics", "id": 318 },
        { "name": "Comics & Animation", "id": 316 },
        { "name": "Film & TV Awards", "id": 1108 },
        { "name": "Film & TV Production", "id": 1117 },
        { "name": "Film & TV Industry", "id": 1116 },
        { "name": "Music Awards", "id": 1113 },
        { "name": "Record Labels", "id": 1114 },
        { "name": "Recording Industry", "id": 1115 },
        { "name": "Entertainment Industry", "id": 612 },
        { "name": "Clubs & Nightlife", "id": 188 },
        { "name": "Concerts & Music Festivals", "id": 891 },
        { "name": "Film Festivals", "id": 1086 },
        { "name": "Live Sporting Events", "id": 1273 },
        { "name": "Movie Listings & Theater Showtimes", "id": 1085 },
        { "name": "Ticket Sales", "id": 614 },
        { "name": "Events & Listings", "id": 569 },
        { "name": "Dress-Up & Fashion Games", "id": 1173 },
        { "name": "Flash-Based Entertainment", "id": 447 },
        { "name": "Fun Tests & Silly Surveys", "id": 1174 },
        { "name": "Fun & Trivia", "id": 539 },
        { "name": "Comedy Films", "id": 1095 },
        { "name": "Live Comedy", "id": 895 },
        { "name": "Political Humor", "id": 1180 },
        { "name": "Spoofs & Satire", "id": 1244 },
        { "name": "TV Comedies", "id": 1047 },
        { "name": "Humor", "id": 182 },
        { "name": "Martial Arts Films", "id": 1101 },
        { "name": "Superhero Films", "id": 1100 },
        { "name": "Western Films", "id": 1099 },
        { "name": "Action & Adventure Films", "id": 1097 },
        { "name": "Animated Films", "id": 1104 },
        { "name": "Bollywood & South Asian Film", "id": 360 },
        { "name": "Silent Films", "id": 1098 },
        { "name": "Classic Films", "id": 1102 },
        { "name": "Comedy Films", "id": 1095 },
        { "name": "Cult & Indie Films", "id": 1103 },
        { "name": "Documentary Films", "id": 1072 },
        { "name": "Drama Films", "id": 1094 },
        { "name": "DVD & Video Rentals", "id": 1145 },
        { "name": "DVD & Video Shopping", "id": 210 },
        { "name": "Family Films", "id": 1291 },
        { "name": "Film & TV Awards", "id": 1108 },
        { "name": "Film Festivals", "id": 1086 },
        { "name": "Horror Films", "id": 615 },
        { "name": "Movie Memorabilia", "id": 213 },
        { "name": "Movie Reviews & Previews", "id": 1107 },
        { "name": "Movie Reference", "id": 1106 },
        { "name": "Musical Films", "id": 1105 },
        { "name": "Romance Films", "id": 1310 },
        { "name": "Science Fiction & Fantasy Films", "id": 616 },
        { "name": "Thriller, Crime & Mystery Films", "id": 1096 },
        { "name": "Movies", "id": 34 },
        { "name": "CD & Audio Shopping", "id": 217 },
        { "name": "Opera", "id": 1185 },
        { "name": "Classical Music", "id": 586 },
        { "name": "Country Music", "id": 587 },
        { "name": "Dance & Electronic Music", "id": 588 },
        { "name": "Experimental & Industrial Music", "id": 1022 },
        { "name": "Folk & Traditional Music", "id": 1023 },
        { "name": "Blues", "id": 1040 },
        { "name": "Jazz", "id": 42 },
        { "name": "Jazz & Blues", "id": 589 },
        { "name": "Latin Pop", "id": 1285 },
        { "name": "Music Art & Memorabilia", "id": 218 },
        { "name": "Music Education & Instruction", "id": 1087 },
        { "name": "DJ Resources & Equipment", "id": 1025 },
        { "name": "Music Recording Technology", "id": 1026 },
        { "name": "Drums & Percussion", "id": 1327 },
        { "name": "Guitars", "id": 1325 },
        { "name": "Pianos & Keyboards", "id": 1326 },
        { "name": "Musical Instruments", "id": 216 },
        { "name": "Samples & Sound Libraries", "id": 1091 },
        { "name": "Music Equipment & Technology", "id": 1024 },
        { "name": "Music Composition & Theory", "id": 1028 },
        { "name": "Sheet Music", "id": 892 },
        { "name": "Song Lyrics & Tabs", "id": 617 },
        { "name": "Music Reference", "id": 1027 },
        { "name": "Music Streams & Downloads", "id": 220 },
        { "name": "Pop Music", "id": 1021 },
        { "name": "Podcasting", "id": 809 },
        { "name": "Talk Radio", "id": 1186 },
        { "name": "Radio", "id": 215 },
        { "name": "Christian & Gospel Music", "id": 585 },
        { "name": "Religious Music", "id": 1020 },
        { "name": "Classic Rock & Oldies", "id": 1037 },
        { "name": "Hard Rock & Progressive", "id": 1035 },
        { "name": "Indie & Alternative Music", "id": 1038 },
        { "name": "Metal (Music)", "id": 1036 },
        { "name": "Punk (Music)", "id": 1041 },
        { "name": "Rock Music", "id": 590 },
        { "name": "Soundtracks", "id": 893 },
        { "name": "Rap & Hip-Hop", "id": 1030 },
        { "name": "Reggaeton", "id": 1242 },
        { "name": "Soul & R&B", "id": 1039 },
        { "name": "Urban & Hip-Hop", "id": 592 },
        { "name": "Vocals & Show Tunes", "id": 618 },
        { "name": "African Music", "id": 1208 },
        { "name": "Arab & Middle Eastern Music", "id": 1034 },
        { "name": "East Asian Music", "id": 1033 },
        { "name": "Brazilian Music", "id": 1287 },
        { "name": "Latin Pop", "id": 1285 },
        { "name": "Reggaeton", "id": 1242 },
        { "name": "Salsa & Tropical Music", "id": 1286 },
        { "name": "Latin American Music", "id": 591 },
        { "name": "Reggaeton", "id": 1242 },
        { "name": "Reggae & Caribbean Music", "id": 1031 },
        { "name": "South Asian Music", "id": 1032 },
        { "name": "World Music", "id": 593 },
        { "name": "Music & Audio", "id": 35 },
        { "name": "Edgy & Bizarre", "id": 538 },
        { "name": "Occult & Paranormal", "id": 449 },
        { "name": "Offbeat", "id": 33 },
        { "name": "Flash-Based Entertainment", "id": 447 },
        { "name": "Music Streams & Downloads", "id": 220 },
        { "name": "Massive Multiplayer", "id": 935 },
        { "name": "Online Games", "id": 105 },
        { "name": "Photo & Image Sharing", "id": 978 },
        { "name": "Photo Rating Sites", "id": 320 },
        { "name": "Stock Photography", "id": 574 },
        { "name": "Online Image Galleries", "id": 1222 },
        { "name": "Video Sharing", "id": 979 },
        { "name": "Online Video", "id": 211 },
        { "name": "Web Portals", "id": 301 },
        { "name": "Webcams & Virtual Tours", "id": 575 },
        { "name": "Online Media", "id": 613 },
        { "name": "Acting & Theater", "id": 894 },
        { "name": "Broadway & Musical Theater", "id": 1243 },
        { "name": "Dance", "id": 581 },
        { "name": "Opera", "id": 1185 },
        { "name": "Performing Arts", "id": 23 },
        { "name": "Video Sharing", "id": 979 },
        { "name": "Online Video", "id": 211 },
        { "name": "TV Commercials", "id": 1055 },
        { "name": "TV Guides & Reference", "id": 1187 },
        { "name": "TV Networks & Stations", "id": 359 },
        { "name": "TV Comedies", "id": 1047 },
        { "name": "TV Crime & Legal Shows", "id": 1111 },
        { "name": "TV Medical Shows", "id": 1194 },
        { "name": "TV Soap Operas", "id": 357 },
        { "name": "TV Dramas", "id": 1193 },
        { "name": "TV Family-Oriented Shows", "id": 1110 },
        { "name": "TV Game Shows", "id": 1050 },
        { "name": "TV Reality Shows", "id": 1049 },
        { "name": "TV Sci-Fi & Fantasy Shows", "id": 1112 },
        { "name": "TV Talk Shows", "id": 1048 },
        { "name": "TV Shows & Programs", "id": 358 },
        { "name": "TV & Video", "id": 36 },
        { "name": "Architecture", "id": 477 },
        { "name": "Art & Craft Supplies", "id": 1361 },
        { "name": "Arts Education", "id": 1195 },
        { "name": "CAD & CAM", "id": 1300 },
        { "name": "Graphic Design", "id": 654 },
        { "name": "Industrial & Product Design", "id": 655 },
        { "name": "Interior Design", "id": 656 },
        { "name": "Design", "id": 653 },
        { "name": "Painting", "id": 1167 },
        { "name": "Binoculars, Telescopes & Optical Devices", "id": 1384 },
        { "name": "Camcorders", "id": 308 },
        { "name": "Camera Lenses", "id": 1383 },
        { "name": "Cameras", "id": 307 },
        { "name": "Cameras & Camcorders", "id": 306 },
        { "name": "Camera & Photo Equipment", "id": 573 },
        { "name": "Video File Formats & Codecs", "id": 1315 },
        { "name": "Photo & Video Software", "id": 577 },
        { "name": "Photographic & Digital Arts", "id": 439 },
        { "name": "Visual Art & Design", "id": 24 },
        { "name": "Arts & Entertainment", "id": 3 },
        { "name": "Automotive Industry", "id": 1190 },
        { "name": "Bicycles & Accessories", "id": 1191 },
        { "name": "Boats & Watercraft", "id": 1140 },
        { "name": "Campers & RVs", "id": 1213 },
        { "name": "Classic Vehicles", "id": 1013 },
        { "name": "Cargo Trucks & Trailers", "id": 1215 },
        { "name": "Commercial Vehicles", "id": 1214 },
        { "name": "Custom & Performance Vehicles", "id": 806 },
        { "name": "Electric & Plug-In Vehicles", "id": 1380 },
        { "name": "Hybrid & Alternative Vehicles", "id": 810 },
        { "name": "Microcars & City Cars", "id": 1317 },
        { "name": "Motorcycles", "id": 273 },
        { "name": "Off-Road Vehicles", "id": 148 },
        { "name": "Personal Aircraft", "id": 1147 },
        { "name": "Scooters & Mopeds", "id": 1212 },
        { "name": "SUVs", "id": 1057 },
        { "name": "Trucks", "id": 1056 },
        { "name": "Vans & Minivans", "id": 1058 },
        { "name": "Trucks & SUVs", "id": 610 },
        { "name": "Acura", "id": 820 },
        { "name": "Audi", "id": 821 },
        { "name": "Bentley", "id": 1059 },
        { "name": "BMW", "id": 822 },
        { "name": "Buick", "id": 1060 },
        { "name": "Cadillac", "id": 823 },
        { "name": "Chevrolet", "id": 826 },
        { "name": "Chrysler", "id": 833 },
        { "name": "Citroën", "id": 834 },
        { "name": "Dodge", "id": 836 },
        { "name": "Ferrari", "id": 1061 },
        { "name": "Fiat", "id": 838 },
        { "name": "Ford", "id": 840 },
        { "name": "GM-Daewoo", "id": 896 },
        { "name": "GMC", "id": 842 },
        { "name": "Honda", "id": 843 },
        { "name": "Hummer", "id": 1062 },
        { "name": "Hyundai", "id": 845 },
        { "name": "Isuzu", "id": 1378 },
        { "name": "Jaguar", "id": 1063 },
        { "name": "Jeep", "id": 846 },
        { "name": "Kia", "id": 848 },
        { "name": "Lamborghini", "id": 1064 },
        { "name": "Land Rover", "id": 1065 },
        { "name": "Lexus", "id": 849 },
        { "name": "Lincoln", "id": 850 },
        { "name": "Maserati", "id": 1066 },
        { "name": "Mazda", "id": 851 },
        { "name": "Mercedes-Benz", "id": 852 },
        { "name": "Mercury", "id": 853 },
        { "name": "Mini", "id": 1067 },
        { "name": "Mitsubishi", "id": 854 },
        { "name": "Infiniti", "id": 1377 },
        { "name": "Nissan", "id": 855 },
        { "name": "Peugeot", "id": 856 },
        { "name": "Pontiac", "id": 857 },
        { "name": "Porsche", "id": 858 },
        { "name": "Renault-Samsung", "id": 859 },
        { "name": "Rolls-Royce", "id": 1068 },
        { "name": "Saab", "id": 897 },
        { "name": "Saturn", "id": 860 },
        { "name": "Subaru", "id": 861 },
        { "name": "Suzuki", "id": 1070 },
        { "name": "Scion", "id": 1069 },
        { "name": "Toyota", "id": 863 },
        { "name": "Vauxhall-Opel", "id": 898 },
        { "name": "Volkswagen", "id": 865 },
        { "name": "Volvo", "id": 867 },
        { "name": "Vehicle Brands", "id": 815 },
        { "name": "Drunk Driving Law", "id": 968 },
        { "name": "Vehicle Codes & Driving Laws", "id": 1294 },
        { "name": "Vehicle Licensing & Registration", "id": 170 },
        { "name": "Vehicle Maintenance", "id": 138 },
        { "name": "Auto Exterior", "id": 1217 },
        { "name": "Auto Interior", "id": 1218 },
        { "name": "Car Audio", "id": 230 },
        { "name": "Car Video", "id": 1189 },
        { "name": "GPS & Navigation", "id": 794 },
        { "name": "Car Electronics", "id": 1188 },
        { "name": "Engine & Transmission", "id": 1216 },
        { "name": "Vehicle Fuels & Lubricants", "id": 1269 },
        { "name": "Vehicle Wheels & Tires", "id": 438 },
        { "name": "Vehicle Parts & Accessories", "id": 89 },
        { "name": "Fuel Economy & Gas Prices", "id": 1268 },
        { "name": "Vehicle Specs, Reviews & Comparisons", "id": 1267 },
        { "name": "Vehicle Shopping", "id": 473 },
        { "name": "Vehicle Shows", "id": 803 },
        { "name": "Autos & Vehicles", "id": 47 },
        { "name": "Beauty Pageants", "id": 1219 },
        { "name": "Body Art", "id": 239 },
        { "name": "Cosmetic Surgery", "id": 238 },
        { "name": "Cosmetic Procedures", "id": 1220 },
        { "name": "Cosmetology & Beauty Professionals", "id": 147 },
        { "name": "Hygiene & Toiletries", "id": 244 },
        { "name": "Make-Up & Cosmetics", "id": 234 },
        { "name": "Perfumes & Fragrances", "id": 242 },
        { "name": "Skin & Nail Care", "id": 93 },
        { "name": "Unwanted Body & Facial Hair Removal", "id": 144 },
        { "name": "Face & Body Care", "id": 143 },
        { "name": "Fashion Designers & Collections", "id": 98 },
        { "name": "Fashion Modeling", "id": 1155 },
        { "name": "Fashion & Style", "id": 185 },
        { "name": "Bodybuilding", "id": 241 },
        { "name": "Yoga & Pilates", "id": 611 },
        { "name": "Fitness", "id": 94 },
        { "name": "Hair Loss", "id": 235 },
        { "name": "Hair Care", "id": 146 },
        { "name": "Massage Therapy", "id": 557 },
        { "name": "Spas & Beauty Services", "id": 145 },
        { "name": "Weight Loss", "id": 236 },
        { "name": "Beauty & Fitness", "id": 44 },
        { "name": "Biographies & Quotations", "id": 690 },
        { "name": "Book Retailers", "id": 355 },
        { "name": "Children's Literature", "id": 1183 },
        { "name": "E-Books", "id": 608 },
        { "name": "Fan Fiction", "id": 540 },
        { "name": "Literary Classics", "id": 1184 },
        { "name": "Magazines", "id": 412 },
        { "name": "Poetry", "id": 565 },
        { "name": "Writers Resources", "id": 1177 },
        { "name": "Books & Literature", "id": 22 },
        { "name": "Loyalty Cards & Programs", "id": 1309 },
        { "name": "Marketing Services", "id": 83 },
        { "name": "Public Relations", "id": 327 },
        { "name": "Search Engine Optimization & Marketing", "id": 84 },
        { "name": "Telemarketing", "id": 328 },
        { "name": "Advertising & Marketing", "id": 25 },
        { "name": "Defense Industry", "id": 669 },
        { "name": "Space Technology", "id": 668 },
        { "name": "Aerospace & Defense", "id": 356 },
        { "name": "Agricultural Equipment", "id": 748 },
        { "name": "Aquaculture", "id": 747 },
        { "name": "Crops & Seed", "id": 749 },
        { "name": "Food Production", "id": 621 },
        { "name": "Forestry", "id": 750 },
        { "name": "Horticulture", "id": 751 },
        { "name": "Livestock", "id": 752 },
        { "name": "Agriculture & Forestry", "id": 46 },
        { "name": "Automotive Industry", "id": 1190 },
        { "name": "Business Education", "id": 799 },
        { "name": "Commercial Lending", "id": 1160 },
        { "name": "Investment Banking", "id": 1139 },
        { "name": "Risk Management", "id": 620 },
        { "name": "Venture Capital", "id": 905 },
        { "name": "Business Finance", "id": 1138 },
        { "name": "Company Earnings", "id": 1240 },
        { "name": "Mergers & Acquisitions", "id": 1241 },
        { "name": "Company News", "id": 1179 },
        { "name": "Economy News", "id": 1164 },
        { "name": "Financial Markets", "id": 1163 },
        { "name": "Fiscal Policy News", "id": 1165 },
        { "name": "Business News", "id": 784 },
        { "name": "Business Plans & Presentations", "id": 336 },
        { "name": "Compensation & Benefits", "id": 723 },
        { "name": "Corporate Training", "id": 331 },
        { "name": "Payroll Services", "id": 724 },
        { "name": "Recruitment & Staffing", "id": 330 },
        { "name": "Human Resources", "id": 157 },
        { "name": "Business Process", "id": 721 },
        { "name": "Project Management Software", "id": 1359 },
        { "name": "Project Management", "id": 1360 },
        { "name": "Strategic Planning", "id": 722 },
        { "name": "Supply Chain Management", "id": 801 },
        { "name": "Management", "id": 338 },
        { "name": "Business Operations", "id": 1159 },
        { "name": "Loyalty Cards & Programs", "id": 1309 },
        { "name": "Marketing Services", "id": 83 },
        { "name": "Public Relations", "id": 327 },
        { "name": "Search Engine Optimization & Marketing", "id": 84 },
        { "name": "Telemarketing", "id": 328 },
        { "name": "Advertising & Marketing", "id": 25 },
        { "name": "Consulting", "id": 1162 },
        { "name": "Trade Shows & Conventions", "id": 335 },
        { "name": "Corporate Events", "id": 334 },
        { "name": "Merchant Services & Payment Systems", "id": 280 },
        { "name": "E-Commerce Services", "id": 340 },
        { "name": "Fire & Security Services", "id": 726 },
        { "name": "Knowledge Management", "id": 800 },
        { "name": "Office & Facilities Management", "id": 337 },
        { "name": "Office Services", "id": 28 },
        { "name": "Business Cards & Stationary", "id": 1375 },
        { "name": "Office Furniture", "id": 333 },
        { "name": "Copiers", "id": 1331 },
        { "name": "Fax Machines", "id": 1332 },
        { "name": "Ink & Toner", "id": 1333 },
        { "name": "Printers", "id": 494 },
        { "name": "Scanners", "id": 495 },
        { "name": "Printers, Copiers & Fax", "id": 1330 },
        { "name": "Office Supplies", "id": 95 },
        { "name": "Outsourcing", "id": 718 },
        { "name": "Stock Photography", "id": 574 },
        { "name": "Photo & Video Services", "id": 576 },
        { "name": "Physical Asset Management", "id": 719 },
        { "name": "Quality Control & Tracking", "id": 720 },
        { "name": "Signage", "id": 1076 },
        { "name": "Writing & Editing Services", "id": 725 },
        { "name": "Business Services", "id": 329 },
        { "name": "Agrochemicals", "id": 670 },
        { "name": "Cleaning Agents", "id": 671 },
        { "name": "Coatings & Adhesives", "id": 672 },
        { "name": "Dyes & Pigments", "id": 673 },
        { "name": "Plastics & Polymers", "id": 674 },
        { "name": "Chemicals Industry", "id": 288 },
        { "name": "Doors & Windows", "id": 827 },
        { "name": "HVAC & Climate Control", "id": 828 },
        { "name": "Nails Screws & Fasteners", "id": 829 },
        { "name": "Plumbing Fixtures & Equipment", "id": 830 },
        { "name": "Wood & Plastics", "id": 831 },
        { "name": "Building Materials & Supplies", "id": 650 },
        { "name": "Civil Engineering", "id": 651 },
        { "name": "Construction Consulting & Contracting", "id": 652 },
        { "name": "Urban & Regional Planning", "id": 686 },
        { "name": "Construction & Maintenance", "id": 48 },
        { "name": "Electricity", "id": 658 },
        { "name": "Nuclear Energy", "id": 954 },
        { "name": "Vehicle Fuels & Lubricants", "id": 1269 },
        { "name": "Oil & Gas", "id": 659 },
        { "name": "Renewable & Alternative Energy", "id": 657 },
        { "name": "Recycling", "id": 1307 },
        { "name": "Waste Management", "id": 660 },
        { "name": "Water Supply & Treatment", "id": 1349 },
        { "name": "Energy & Utilities", "id": 233 },
        { "name": "Customer Relationship Management (CRM)", "id": 341 },
        { "name": "Data Management", "id": 343 },
        { "name": "Merchant Services & Payment Systems", "id": 280 },
        { "name": "E-Commerce Services", "id": 340 },
        { "name": "Enterprise Resource Planning (ERP)", "id": 342 },
        { "name": "Enterprise Technology", "id": 77 },
        { "name": "Film & TV Awards", "id": 1108 },
        { "name": "Film & TV Production", "id": 1117 },
        { "name": "Film & TV Industry", "id": 1116 },
        { "name": "Music Awards", "id": 1113 },
        { "name": "Record Labels", "id": 1114 },
        { "name": "Recording Industry", "id": 1115 },
        { "name": "Entertainment Industry", "id": 612 },
        { "name": "Event Planning", "id": 956 },
        { "name": "Grocery & Food Retailers", "id": 121 },
        { "name": "Restaurant Supply", "id": 816 },
        { "name": "Food Service", "id": 957 },
        { "name": "Hospitality Industry", "id": 955 },
        { "name": "Valves Hoses & Fittings", "id": 839 },
        { "name": "Fluid Handling", "id": 1152 },
        { "name": "Generators", "id": 835 },
        { "name": "Heavy Machinery", "id": 837 },
        { "name": "Industrial Materials & Equipment", "id": 287 },
        { "name": "Factory Automation", "id": 661 },
        { "name": "Manufacturing", "id": 49 },
        { "name": "Metals & Mining", "id": 606 },
        { "name": "Pharmaceuticals & Biotech", "id": 255 },
        { "name": "Business Cards & Stationary", "id": 1375 },
        { "name": "Document & Printing Services", "id": 332 },
        { "name": "Journalism & News Industry", "id": 1204 },
        { "name": "Printing & Publishing", "id": 1176 },
        { "name": "Professional & Trade Associations", "id": 1199 },
        { "name": "Retail Equipment & Technology", "id": 844 },
        { "name": "Retail Trade", "id": 841 },
        { "name": "Business Formation", "id": 1200 },
        { "name": "Home Office", "id": 727 },
        { "name": "MLM & Business Opportunities", "id": 552 },
        { "name": "Small Business", "id": 551 },
        { "name": "Textiles & Nonwovens", "id": 566 },
        { "name": "Aviation", "id": 662 },
        { "name": "Distribution & Logistics", "id": 664 },
        { "name": "Cargo Trucks & Trailers", "id": 1215 },
        { "name": "Freight & Trucking", "id": 289 },
        { "name": "Import & Export", "id": 354 },
        { "name": "Couriers & Messengers", "id": 663 },
        { "name": "Mail & Package Delivery", "id": 1150 },
        { "name": "Maritime Transport", "id": 665 },
        { "name": "Moving & Relocation", "id": 291 },
        { "name": "Packaging", "id": 290 },
        { "name": "Airport Parking & Transportation", "id": 1245 },
        { "name": "Parking", "id": 1306 },
        { "name": "Public Storage", "id": 1347 },
        { "name": "Rail Transport", "id": 666 },
        { "name": "Urban Transport", "id": 667 },
        { "name": "Transportation & Logistics", "id": 50 },
        { "name": "Business & Industrial", "id": 12 },
        { "name": "CAD & CAM", "id": 1300 },
        { "name": "Chips & Processors", "id": 741 },
        { "name": "Computer Memory", "id": 226 },
        { "name": "Sound & Video Cards", "id": 740 },
        { "name": "Computer Components", "id": 717 },
        { "name": "CD & DVD Drives & Burners", "id": 1321 },
        { "name": "CD & DVD Storage Media", "id": 1322 },
        { "name": "Computer Memory", "id": 226 },
        { "name": "Data Backup & Recovery", "id": 1323 },
        { "name": "Flash Drives & Memory Cards", "id": 1318 },
        { "name": "Hard Drives", "id": 1320 },
        { "name": "Memory Card Readers", "id": 1319 },
        { "name": "Network Storage", "id": 729 },
        { "name": "Computer Drives & Storage", "id": 496 },
        { "name": "Computer Monitors & Displays", "id": 487 },
        { "name": "Input Devices", "id": 493 },
        { "name": "Copiers", "id": 1331 },
        { "name": "Fax Machines", "id": 1332 },
        { "name": "Ink & Toner", "id": 1333 },
        { "name": "Printers", "id": 494 },
        { "name": "Scanners", "id": 495 },
        { "name": "Printers, Copiers & Fax", "id": 1330 },
        { "name": "Computer Peripherals", "id": 312 },
        { "name": "Computer Servers", "id": 728 },
        { "name": "Desktop Computers", "id": 309 },
        { "name": "Hardware Modding & Tuning", "id": 739 },
        { "name": "Tablet PCs", "id": 1277 },
        { "name": "Laptops & Notebooks", "id": 310 },
        { "name": "Computer Hardware", "id": 30 },
        { "name": "Antivirus & Malware", "id": 315 },
        { "name": "Network Security", "id": 344 },
        { "name": "Computer Security", "id": 314 },
        { "name": "Headphones", "id": 1396 },
        { "name": "MP3 & Portable Media Players", "id": 227 },
        { "name": "Speakers", "id": 1158 },
        { "name": "Stereo Systems & Components", "id": 91 },
        { "name": "Audio Equipment", "id": 361 },
        { "name": "Binoculars, Telescopes & Optical Devices", "id": 1384 },
        { "name": "Camcorders", "id": 308 },
        { "name": "Camera Lenses", "id": 1383 },
        { "name": "Cameras", "id": 307 },
        { "name": "Cameras & Camcorders", "id": 306 },
        { "name": "Camera & Photo Equipment", "id": 573 },
        { "name": "Car Audio", "id": 230 },
        { "name": "Car Video", "id": 1189 },
        { "name": "GPS & Navigation", "id": 794 },
        { "name": "Car Electronics", "id": 1188 },
        { "name": "Electronic Accessories", "id": 1192 },
        { "name": "E-Book Readers", "id": 1324 },
        { "name": "Handheld Game Consoles", "id": 1046 },
        { "name": "MP3 & Portable Media Players", "id": 227 },
        { "name": "PDAs & Handhelds", "id": 228 },
        { "name": "Gadgets & Portable Electronics", "id": 362 },
        { "name": "Handheld Game Consoles", "id": 1046 },
        { "name": "Nintendo", "id": 1043 },
        { "name": "Sony PlayStation", "id": 1044 },
        { "name": "Xbox", "id": 1045 },
        { "name": "Game Systems & Consoles", "id": 899 },
        { "name": "GPS & Navigation", "id": 794 },
        { "name": "DVRs & Set-Top Boxes", "id": 1393 },
        { "name": "Home Theater Systems", "id": 1157 },
        { "name": "Projectors & Screens", "id": 1334 },
        { "name": "HDTVs", "id": 1354 },
        { "name": "LCD TVs", "id": 1356 },
        { "name": "Plasma TVs", "id": 1355 },
        { "name": "Projection TVs", "id": 1357 },
        { "name": "Televisions", "id": 305 },
        { "name": "Blu-Ray Players & Recorders", "id": 1394 },
        { "name": "DVD Players & Recorders", "id": 1395 },
        { "name": "Video Players & Recorders", "id": 492 },
        { "name": "TV & Video Equipment", "id": 229 },
        { "name": "Consumer Electronics", "id": 78 },
        { "name": "Data Sheets & Electronics Reference", "id": 900 },
        { "name": "Electromechanical Devices", "id": 743 },
        { "name": "Electronic Components", "id": 742 },
        { "name": "Optoelectronics & Fiber", "id": 744 },
        { "name": "Power Supplies", "id": 745 },
        { "name": "Test & Measurement", "id": 746 },
        { "name": "Electronics & Electrical", "id": 434 },
        { "name": "Customer Relationship Management (CRM)", "id": 341 },
        { "name": "Data Management", "id": 343 },
        { "name": "Merchant Services & Payment Systems", "id": 280 },
        { "name": "E-Commerce Services", "id": 340 },
        { "name": "Enterprise Resource Planning (ERP)", "id": 342 },
        { "name": "Enterprise Technology", "id": 77 },
        { "name": "Data Formats & Protocols", "id": 488 },
        { "name": "Distributed & Parallel Computing", "id": 1298 },
        { "name": "Network Monitoring & Management", "id": 347 },
        { "name": "Networking Equipment", "id": 346 },
        { "name": "VPN & Remote Access", "id": 1279 },
        { "name": "Networking", "id": 311 },
        { "name": "C & C++", "id": 731 },
        { "name": "Developer Jobs", "id": 802 },
        { "name": "Development Tools", "id": 730 },
        { "name": "Java", "id": 732 },
        { "name": "Scripting Languages", "id": 733 },
        { "name": "Windows & .NET", "id": 734 },
        { "name": "Programming", "id": 31 },
        { "name": "Accounting & Financial Software", "id": 1341 },
        { "name": "Calendar & Scheduling Software", "id": 1358 },
        { "name": "Presentation Software", "id": 1346 },
        { "name": "Project Management Software", "id": 1359 },
        { "name": "Spreadsheet Software", "id": 1344 },
        { "name": "Word Processing Software", "id": 1345 },
        { "name": "Business & Productivity Software", "id": 498 },
        { "name": "Device Drivers", "id": 225 },
        { "name": "Educational Software", "id": 804 },
        { "name": "Freeware & Shareware", "id": 901 },
        { "name": "Content Management", "id": 808 },
        { "name": "Internet Clients & Browsers", "id": 304 },
        { "name": "Proxying & Filtering", "id": 902 },
        { "name": "Internet Software", "id": 807 },
        { "name": "Ringtones & Mobile Goodies", "id": 532 },
        { "name": "Mobile Apps & Add-Ons", "id": 1109 },
        { "name": "Audio Files Formats & Codecs", "id": 1092 },
        { "name": "Audio & Music Software", "id": 1089 },
        { "name": "Fonts", "id": 805 },
        { "name": "Desktop Publishing", "id": 1088 },
        { "name": "Graphics & Animation Software", "id": 486 },
        { "name": "Media Players", "id": 1090 },
        { "name": "Video File Formats & Codecs", "id": 1315 },
        { "name": "Photo & Video Software", "id": 577 },
        { "name": "Multimedia Software", "id": 497 },
        { "name": "Open Source", "id": 313 },
        { "name": "Linux & Unix", "id": 736 },
        { "name": "Mac OS", "id": 735 },
        { "name": "Mobile OS", "id": 1382 },
        { "name": "Windows OS", "id": 737 },
        { "name": "Operating Systems", "id": 303 },
        { "name": "Software Utilities", "id": 224 },
        { "name": "Web Apps & Online Tools", "id": 1142 },
        { "name": "Software", "id": 32 },
        { "name": "Technical Support", "id": 567 },
        { "name": "Technology News", "id": 785 },
        { "name": "Computers & Electronics", "id": 5 },
        { "name": "Accounting & Financial Software", "id": 1341 },
        { "name": "Tax Preparation & Planning", "id": 1283 },
        { "name": "Accounting & Auditing", "id": 278 },
        { "name": "Banking", "id": 37 },
        { "name": "Auto Financing", "id": 468 },
        { "name": "College Financing", "id": 813 },
        { "name": "Credit Cards", "id": 811 },
        { "name": "Debt Management", "id": 812 },
        { "name": "Home Financing", "id": 466 },
        { "name": "Credit & Lending", "id": 279 },
        { "name": "Currencies & Foreign Exchange", "id": 814 },
        { "name": "Financial Planning", "id": 903 },
        { "name": "College Financing", "id": 813 },
        { "name": "Grants & Financial Assistance", "id": 1282 },
        { "name": "Auto Insurance", "id": 467 },
        { "name": "Health Insurance", "id": 249 },
        { "name": "Home Insurance", "id": 465 },
        { "name": "Insurance", "id": 38 },
        { "name": "Commodities & Futures Trading", "id": 904 },
        { "name": "Investing", "id": 107 },
        { "name": "Retirement & Pension", "id": 619 },
        { "name": "Finance", "id": 7 },
        { "name": "Beer", "id": 404 },
        { "name": "Liquor", "id": 406 },
        { "name": "Wine", "id": 405 },
        { "name": "Alcoholic Beverages", "id": 277 },
        { "name": "Candy & Sweets", "id": 906 },
        { "name": "Baked Goods", "id": 907 },
        { "name": "Fruits & Vegetables", "id": 908 },
        { "name": "Meat & Seafood", "id": 909 },
        { "name": "Soups & Stews", "id": 910 },
        { "name": "Vegetarian Cuisine", "id": 825 },
        { "name": "Asian Cuisine", "id": 912 },
        { "name": "Latin American Cuisine", "id": 913 },
        { "name": "Mediterranean Cuisine", "id": 914 },
        { "name": "North American Cuisine", "id": 915 },
        { "name": "World Cuisines", "id": 911 },
        { "name": "Cooking & Recipes", "id": 122 },
        { "name": "Culinary Training", "id": 297 },
        { "name": "Grocery & Food Retailers", "id": 121 },
        { "name": "Coffee & Tea", "id": 916 },
        { "name": "Non-Alcoholic Beverages", "id": 560 },
        { "name": "Dining Guides", "id": 917 },
        { "name": "Fast Food", "id": 918 },
        { "name": "Restaurant Supply", "id": 816 },
        { "name": "Restaurants", "id": 276 },
        { "name": "Food & Drink", "id": 71 },
        { "name": "Arcade & Coin-Op Games", "id": 919 },
        { "name": "Chess & Abstract Strategy Games", "id": 921 },
        { "name": "Miniatures & Wargaming", "id": 922 },
        { "name": "Board Games", "id": 920 },
        { "name": "Collectible Card Games", "id": 923 },
        { "name": "Poker & Casino Games", "id": 924 },
        { "name": "Card Games", "id": 39 },
        { "name": "Action & Platform Games", "id": 1311 },
        { "name": "Adventure Games", "id": 925 },
        { "name": "Casual Games", "id": 926 },
        { "name": "Driving & Racing Games", "id": 927 },
        { "name": "Fighting Games", "id": 928 },
        { "name": "Handheld Game Consoles", "id": 1046 },
        { "name": "Nintendo", "id": 1043 },
        { "name": "Sony PlayStation", "id": 1044 },
        { "name": "Xbox", "id": 1045 },
        { "name": "Game Systems & Consoles", "id": 899 },
        { "name": "Game Cheats & Hints", "id": 381 },
        { "name": "Gaming Media & Reference", "id": 1343 },
        { "name": "Music & Dance Games", "id": 929 },
        { "name": "Shooter Games", "id": 930 },
        { "name": "Simulation Games", "id": 931 },
        { "name": "Sports Games", "id": 932 },
        { "name": "Strategy Games", "id": 933 },
        { "name": "Video Game Emulation", "id": 1342 },
        { "name": "Video Game Retailers", "id": 1146 },
        { "name": "Computer & Video Games", "id": 41 },
        { "name": "Drawing & Coloring", "id": 1397 },
        { "name": "Dress-Up & Fashion Games", "id": 1173 },
        { "name": "Family-Oriented Games & Activities", "id": 1290 },
        { "name": "Massive Multiplayer", "id": 935 },
        { "name": "Online Games", "id": 105 },
        { "name": "Party Games", "id": 936 },
        { "name": "Puzzles & Brainteasers", "id": 937 },
        { "name": "Roleplaying Games", "id": 622 },
        { "name": "Billiards", "id": 939 },
        { "name": "Table Tennis", "id": 940 },
        { "name": "Table Games", "id": 938 },
        { "name": "Games", "id": 8 },
        { "name": "Alzheimer's Disease", "id": 624 },
        { "name": "Aging & Geriatrics", "id": 623 },
        { "name": "Acupuncture & Chinese Medicine", "id": 1239 },
        { "name": "Cleansing & Detoxification", "id": 1238 },
        { "name": "Alternative & Natural Medicine", "id": 499 },
        { "name": "AIDS & HIV", "id": 625 },
        { "name": "Allergies", "id": 626 },
        { "name": "Arthritis", "id": 628 },
        { "name": "Cancer", "id": 429 },
        { "name": "Cold & Flu", "id": 629 },
        { "name": "Diabetes", "id": 630 },
        { "name": "Ear Nose & Throat", "id": 1211 },
        { "name": "Eating Disorders", "id": 571 },
        { "name": "Diabetes", "id": 630 },
        { "name": "Thyroid Conditions", "id": 1329 },
        { "name": "Endocrine Conditions", "id": 1328 },
        { "name": "Genetic Disorders", "id": 941 },
        { "name": "GERD & Digestive Disorders", "id": 638 },
        { "name": "Heart & Hypertension", "id": 559 },
        { "name": "Cold & Flu", "id": 629 },
        { "name": "Parasites & Parasitic Diseases", "id": 1262 },
        { "name": "AIDS & HIV", "id": 625 },
        { "name": "Sexually Transmitted Diseases", "id": 421 },
        { "name": "Vaccines & Immunizations", "id": 1263 },
        { "name": "Infectious Diseases", "id": 632 },
        { "name": "Injury", "id": 817 },
        { "name": "Alzheimer's Disease", "id": 624 },
        { "name": "Neurological Disorders", "id": 942 },
        { "name": "Obesity", "id": 818 },
        { "name": "Headaches & Migraines", "id": 631 },
        { "name": "Pain Management", "id": 819 },
        { "name": "Asthma", "id": 627 },
        { "name": "Respiratory Conditions", "id": 824 },
        { "name": "Skin Conditions", "id": 420 },
        { "name": "Sleep Disorders", "id": 633 },
        { "name": "Health Conditions", "id": 419 },
        { "name": "Health Education & Medical Training", "id": 254 },
        { "name": "Health Foundations & Medical Research", "id": 252 },
        { "name": "Health Policy", "id": 1256 },
        { "name": "Health News", "id": 1253 },
        { "name": "Mobility Equipment & Accessories", "id": 1353 },
        { "name": "Assistive Technology", "id": 1352 },
        { "name": "Medical Devices & Equipment", "id": 251 },
        { "name": "Doctors' Offices", "id": 634 },
        { "name": "Hospitals & Treatment Centers", "id": 250 },
        { "name": "Medical Tests & Exams", "id": 943 },
        { "name": "Cosmetic Surgery", "id": 238 },
        { "name": "Surgery", "id": 944 },
        { "name": "Vaccines & Immunizations", "id": 1263 },
        { "name": "Medical Procedures", "id": 635 },
        { "name": "Physical Therapy", "id": 500 },
        { "name": "Medical Facilities & Services", "id": 256 },
        { "name": "Medical Photos & Illustration", "id": 945 },
        { "name": "Medical Literature & Resources", "id": 253 },
        { "name": "Erectile Dysfunction", "id": 202 },
        { "name": "Men's Health", "id": 636 },
        { "name": "Anxiety & Stress", "id": 639 },
        { "name": "Depression", "id": 640 },
        { "name": "ADD & ADHD", "id": 642 },
        { "name": "Learning & Developmental Disabilities", "id": 641 },
        { "name": "Mental Health", "id": 437 },
        { "name": "Assisted Living & Long Term Care", "id": 649 },
        { "name": "Nursing", "id": 418 },
        { "name": "Cholesterol Issues", "id": 643 },
        { "name": "Special & Restricted Diets", "id": 457 },
        { "name": "Vitamins & Supplements", "id": 237 },
        { "name": "Nutrition", "id": 456 },
        { "name": "Oral & Dental Care", "id": 245 },
        { "name": "Pediatrics", "id": 645 },
        { "name": "Drugs & Medications", "id": 646 },
        { "name": "Pharmacy", "id": 248 },
        { "name": "Health Policy", "id": 1256 },
        { "name": "Occupational Health & Safety", "id": 644 },
        { "name": "Poisons & Overdoses", "id": 946 },
        { "name": "Vaccines & Immunizations", "id": 1263 },
        { "name": "Public Health", "id": 947 },
        { "name": "Birth Control", "id": 198 },
        { "name": "Erectile Dysfunction", "id": 202 },
        { "name": "Infertility", "id": 647 },
        { "name": "Pregnancy & Maternity", "id": 401 },
        { "name": "OBGYN", "id": 558 },
        { "name": "Sex Education & Counseling", "id": 536 },
        { "name": "Sexual Enhancement", "id": 1236 },
        { "name": "AIDS & HIV", "id": 625 },
        { "name": "Sexually Transmitted Diseases", "id": 421 },
        { "name": "Reproductive Health", "id": 195 },
        { "name": "Drug & Alcohol Testing", "id": 1351 },
        { "name": "Drug & Alcohol Treatment", "id": 1350 },
        { "name": "Smoking & Smoking Cessation", "id": 1237 },
        { "name": "Steroids & Performance-Enhancing Drugs", "id": 1235 },
        { "name": "Substance Abuse", "id": 257 },
        { "name": "Eyeglasses & Contacts", "id": 1224 },
        { "name": "Vision Care", "id": 246 },
        { "name": "Pregnancy & Maternity", "id": 401 },
        { "name": "OBGYN", "id": 558 },
        { "name": "Women's Health", "id": 648 },
        { "name": "Health", "id": 45 },
        { "name": "Antiques & Collectibles", "id": 64 },
        { "name": "Bowling", "id": 1016 },
        { "name": "Clubs & Nightlife", "id": 188 },
        { "name": "Clubs & Organizations", "id": 189 },
        { "name": "Film & TV Awards", "id": 1108 },
        { "name": "Lottery & Sweepstakes", "id": 364 },
        { "name": "Contests, Awards & Prizes", "id": 1276 },
        { "name": "Fiber & Textile Arts", "id": 1230 },
        { "name": "Crafts", "id": 284 },
        { "name": "Bicycles & Accessories", "id": 1191 },
        { "name": "Cycling", "id": 458 },
        { "name": "Equestrian", "id": 568 },
        { "name": "Fishing", "id": 462 },
        { "name": "Hiking & Camping", "id": 542 },
        { "name": "Hunting & Shooting", "id": 461 },
        { "name": "Outdoors", "id": 688 },
        { "name": "Paintball", "id": 786 },
        { "name": "Animal Welfare", "id": 883 },
        { "name": "Pet Food & Supplies", "id": 379 },
        { "name": "Veterinarians", "id": 380 },
        { "name": "Animal Products & Services", "id": 882 },
        { "name": "Birds", "id": 884 },
        { "name": "Cats", "id": 885 },
        { "name": "Dogs", "id": 886 },
        { "name": "Exotic Pets", "id": 607 },
        { "name": "Fish & Aquaria", "id": 887 },
        { "name": "Horses", "id": 888 },
        { "name": "Rabbits & Rodents", "id": 889 },
        { "name": "Reptiles & Amphibians", "id": 890 },
        { "name": "Pets", "id": 563 },
        { "name": "Insects & Entomology", "id": 1278 },
        { "name": "Zoos-Aquariums-Preserves", "id": 1009 },
        { "name": "Wildlife", "id": 119 },
        { "name": "Pets & Animals", "id": 66 },
        { "name": "Binoculars, Telescopes & Optical Devices", "id": 1384 },
        { "name": "Camcorders", "id": 308 },
        { "name": "Camera Lenses", "id": 1383 },
        { "name": "Cameras", "id": 307 },
        { "name": "Cameras & Camcorders", "id": 306 },
        { "name": "Camera & Photo Equipment", "id": 573 },
        { "name": "Video File Formats & Codecs", "id": 1315 },
        { "name": "Photo & Video Software", "id": 577 },
        { "name": "Photographic & Digital Arts", "id": 439 },
        { "name": "Radio Control & Modeling", "id": 787 },
        { "name": "Personal Aircraft", "id": 1147 },
        { "name": "Recreational Aviation", "id": 999 },
        { "name": "Running & Walking", "id": 541 },
        { "name": "Cards & Greetings", "id": 100 },
        { "name": "Flowers", "id": 323 },
        { "name": "Gifts", "id": 99 },
        { "name": "Party & Holiday Supplies", "id": 324 },
        { "name": "Gifts & Special Event Items", "id": 70 },
        { "name": "Birthdays & Name Days", "id": 1270 },
        { "name": "Carnival & Mardi Gras", "id": 1246 },
        { "name": "Christmas", "id": 1078 },
        { "name": "Easter", "id": 1123 },
        { "name": "Christian Holidays", "id": 1274 },
        { "name": "Halloween & October 31st", "id": 1079 },
        { "name": "Islamic Holidays", "id": 1275 },
        { "name": "Jewish Holidays", "id": 1124 },
        { "name": "New Year", "id": 1271 },
        { "name": "Thanksgiving", "id": 1125 },
        { "name": "Valentine's Day", "id": 1122 },
        { "name": "Holidays & Seasonal Events", "id": 678 },
        { "name": "Weddings", "id": 293 },
        { "name": "Special Occasions", "id": 977 },
        { "name": "Goth Subculture", "id": 503 },
        { "name": "Science Fiction & Fantasy", "id": 676 },
        { "name": "Subcultures & Niche Interests", "id": 502 },
        { "name": "Boats & Watercraft", "id": 1140 },
        { "name": "Boating", "id": 459 },
        { "name": "Diving & Underwater Activities", "id": 1305 },
        { "name": "Surf & Swim", "id": 689 },
        { "name": "Water Sports", "id": 118 },
        { "name": "Water Activities", "id": 1002 },
        { "name": "Hobbies & Leisure", "id": 65 },
        { "name": "Bathroom", "id": 1365 },
        { "name": "Bedding & Bed Linens", "id": 1369 },
        { "name": "Beds & Headboards", "id": 1367 },
        { "name": "Mattresses", "id": 1368 },
        { "name": "Bedroom", "id": 1366 },
        { "name": "Bed & Bath", "id": 948 },
        { "name": "Cleaning Supplies & Services", "id": 949 },
        { "name": "Domestic Services", "id": 472 },
        { "name": "Gardening & Landscaping", "id": 269 },
        { "name": "Major Kitchen Appliances", "id": 1293 },
        { "name": "Small Kitchen Appliances", "id": 1292 },
        { "name": "Water Filters & Purifiers", "id": 1371 },
        { "name": "Home Appliances", "id": 271 },
        { "name": "Clocks", "id": 1363 },
        { "name": "Lamps & Lighting", "id": 272 },
        { "name": "Rugs & Carpets", "id": 1362 },
        { "name": "Sofas & Chairs", "id": 1370 },
        { "name": "Home Furnishings", "id": 270 },
        { "name": "Construction & Power Tools", "id": 950 },
        { "name": "Doors & Windows", "id": 827 },
        { "name": "Rugs & Carpets", "id": 1362 },
        { "name": "Flooring", "id": 832 },
        { "name": "House Painting & Finishing", "id": 1232 },
        { "name": "HVAC & Climate Control", "id": 828 },
        { "name": "Plumbing", "id": 1153 },
        { "name": "Roofing", "id": 1175 },
        { "name": "Home Improvement", "id": 158 },
        { "name": "Home Storage & Shelving", "id": 1348 },
        { "name": "Homemaking & Interior Decor", "id": 137 },
        { "name": "HVAC & Climate Control", "id": 828 },
        { "name": "Cutlery & Cutting Accessories", "id": 1373 },
        { "name": "Cookware & Diningware", "id": 120 },
        { "name": "Major Kitchen Appliances", "id": 1293 },
        { "name": "Small Kitchen Appliances", "id": 1292 },
        { "name": "Kitchen & Dining", "id": 951 },
        { "name": "Laundry", "id": 1364 },
        { "name": "Nursery & Playroom", "id": 1372 },
        { "name": "Pest Control", "id": 471 },
        { "name": "Swimming Pools & Spas", "id": 952 },
        { "name": "Yard & Patio", "id": 953 },
        { "name": "Home & Garden", "id": 11 },
        { "name": "Radio Equipment", "id": 1182 },
        { "name": "Communications Equipment", "id": 385 },
        { "name": "Microblogging", "id": 1381 },
        { "name": "Text & Instant Messaging", "id": 1379 },
        { "name": "Voice & Video Chat", "id": 386 },
        { "name": "Email & Messaging", "id": 394 },
        { "name": "Bluetooth Accessories", "id": 1170 },
        { "name": "Mobile & Wireless Accessories", "id": 1171 },
        { "name": "Ringtones & Mobile Goodies", "id": 532 },
        { "name": "Mobile Apps & Add-Ons", "id": 1109 },
        { "name": "Mobile OS", "id": 1382 },
        { "name": "Smart Phones", "id": 1071 },
        { "name": "Mobile Phones", "id": 390 },
        { "name": "Mobile & Wireless", "id": 382 },
        { "name": "People Search", "id": 1234 },
        { "name": "Search Engines", "id": 485 },
        { "name": "Cable & Satellite Providers", "id": 501 },
        { "name": "ISPs", "id": 104 },
        { "name": "Calling Cards", "id": 389 },
        { "name": "Phone Service Providers", "id": 384 },
        { "name": "Service Providers", "id": 383 },
        { "name": "Teleconferencing", "id": 392 },
        { "name": "Web Apps & Online Tools", "id": 1142 },
        { "name": "Web Portals", "id": 301 },
        { "name": "Affiliate Programs", "id": 326 },
        { "name": "Search Engine Optimization & Marketing", "id": 84 },
        { "name": "Web Design & Development", "id": 422 },
        { "name": "Web Hosting & Domain Registration", "id": 53 },
        { "name": "Web Stats & Analytics", "id": 675 },
        { "name": "Web Services", "id": 302 },
        { "name": "Internet & Telecom", "id": 13 },
        { "name": "Academic Conferences & Publications", "id": 1289 },
        { "name": "Alumni & Reunions", "id": 1015 },
        { "name": "Arts Education", "id": 1195 },
        { "name": "Business Education", "id": 799 },
        { "name": "Colleges & Universities", "id": 372 },
        { "name": "Distance Learning", "id": 367 },
        { "name": "Early Childhood Education", "id": 1012 },
        { "name": "Foreign Language Study", "id": 1266 },
        { "name": "Health Education & Medical Training", "id": 254 },
        { "name": "Homeschooling", "id": 791 },
        { "name": "Legal Education", "id": 792 },
        { "name": "Music Education & Instruction", "id": 1087 },
        { "name": "Primary & Secondary Schooling (K-12)", "id": 371 },
        { "name": "Special Education", "id": 1118 },
        { "name": "Standardized & Admissions Tests", "id": 373 },
        { "name": "Study Abroad", "id": 1308 },
        { "name": "Teaching & Classroom Resources", "id": 700 },
        { "name": "Training & Certification", "id": 1388 },
        { "name": "Computer Education", "id": 1229 },
        { "name": "Vocational & Continuing Education", "id": 369 },
        { "name": "Education", "id": 74 },
        { "name": "Career Resources & Planning", "id": 959 },
        { "name": "Developer Jobs", "id": 802 },
        { "name": "Job Listings", "id": 960 },
        { "name": "Resumes & Portfolios", "id": 961 },
        { "name": "Jobs", "id": 60 },
        { "name": "Jobs & Education", "id": 958 },
        { "name": "Courts & Judiciary", "id": 1075 },
        { "name": "Embassies & Consulates", "id": 962 },
        { "name": "Executive Branch", "id": 963 },
        { "name": "Government Agencies", "id": 1387 },
        { "name": "Government Contracting & Procurement", "id": 1385 },
        { "name": "Intelligence & Counterterrorism", "id": 1221 },
        { "name": "Legislative Branch", "id": 964 },
        { "name": "Lobbying", "id": 1386 },
        { "name": "Multilateral Organizations", "id": 965 },
        { "name": "Public Finance", "id": 1161 },
        { "name": "Drug Laws & Policy", "id": 1314 },
        { "name": "Fiscal Policy News", "id": 1165 },
        { "name": "Health Policy", "id": 1256 },
        { "name": "Immigration Policy & Border Issues", "id": 1313 },
        { "name": "International Relations", "id": 521 },
        { "name": "Public Policy", "id": 1316 },
        { "name": "Royalty", "id": 702 },
        { "name": "State & Local Government", "id": 966 },
        { "name": "Visa & Immigration", "id": 555 },
        { "name": "Government", "id": 76 },
        { "name": "Accident & Personal Injury Law", "id": 427 },
        { "name": "Bankruptcy", "id": 423 },
        { "name": "Business & Corporate Law", "id": 1272 },
        { "name": "Constitutional Law & Civil Rights", "id": 967 },
        { "name": "Criminal Law", "id": 424 },
        { "name": "Family Law", "id": 522 },
        { "name": "Intellectual Property", "id": 426 },
        { "name": "Labor & Employment Law", "id": 701 },
        { "name": "Legal Education", "id": 792 },
        { "name": "Legal Services", "id": 969 },
        { "name": "Product Liability", "id": 970 },
        { "name": "Drunk Driving Law", "id": 968 },
        { "name": "Vehicle Codes & Driving Laws", "id": 1294 },
        { "name": "Legal", "id": 75 },
        { "name": "Air Force", "id": 1247 },
        { "name": "Army", "id": 1248 },
        { "name": "Marines", "id": 1250 },
        { "name": "Military History", "id": 1288 },
        { "name": "Navy", "id": 1249 },
        { "name": "Veterans", "id": 793 },
        { "name": "Military", "id": 366 },
        { "name": "Corporate & Financial Crime", "id": 1181 },
        { "name": "Gangs & Organized Crime", "id": 1312 },
        { "name": "Prisons & Corrections", "id": 1284 },
        { "name": "Crime & Justice", "id": 704 },
        { "name": "Emergency Services", "id": 168 },
        { "name": "Intelligence & Counterterrorism", "id": 1221 },
        { "name": "Law Enforcement", "id": 535 },
        { "name": "Health Policy", "id": 1256 },
        { "name": "Occupational Health & Safety", "id": 644 },
        { "name": "Poisons & Overdoses", "id": 946 },
        { "name": "Vaccines & Immunizations", "id": 1263 },
        { "name": "Public Health", "id": 947 },
        { "name": "Security Products & Services", "id": 705 },
        { "name": "Public Safety", "id": 166 },
        { "name": "Counseling Services", "id": 511 },
        { "name": "Welfare & Unemployment", "id": 706 },
        { "name": "Social Services", "id": 508 },
        { "name": "Law & Government", "id": 19 },
        { "name": "Broadcast & Network News", "id": 112 },
        { "name": "Company Earnings", "id": 1240 },
        { "name": "Mergers & Acquisitions", "id": 1241 },
        { "name": "Company News", "id": 1179 },
        { "name": "Economy News", "id": 1164 },
        { "name": "Financial Markets", "id": 1163 },
        { "name": "Fiscal Policy News", "id": 1165 },
        { "name": "Business News", "id": 784 },
        { "name": "Celebrities & Entertainment News", "id": 184 },
        { "name": "Scandals & Investigations", "id": 1259 },
        { "name": "Gossip & Tabloid News", "id": 507 },
        { "name": "Health Policy", "id": 1256 },
        { "name": "Health News", "id": 1253 },
        { "name": "Journalism & News Industry", "id": 1204 },
        { "name": "Local News", "id": 572 },
        { "name": "Newspapers", "id": 408 },
        { "name": "Campaigns & Elections", "id": 398 },
        { "name": "Left-Wing Politics", "id": 410 },
        { "name": "Media Critics & Watchdogs", "id": 1203 },
        { "name": "Opinion & Commentary", "id": 1201 },
        { "name": "Political Polls & Surveys", "id": 1202 },
        { "name": "Right-Wing Politics", "id": 409 },
        { "name": "Politics", "id": 396 },
        { "name": "Sports News", "id": 1077 },
        { "name": "Technology News", "id": 785 },
        { "name": "Weather", "id": 63 },
        { "name": "World News", "id": 1209 },
        { "name": "News", "id": 16 },
        { "name": "Microblogging", "id": 1381 },
        { "name": "Blogging Resources & Services", "id": 504 },
        { "name": "Matrimonial Services", "id": 546 },
        { "name": "Personals", "id": 102 },
        { "name": "Photo Rating Sites", "id": 320 },
        { "name": "Dating & Personals", "id": 55 },
        { "name": "File Sharing & Hosting", "id": 321 },
        { "name": "Forum & Chat Providers", "id": 191 },
        { "name": "Clip Art & Animated GIFs", "id": 1223 },
        { "name": "Skins Themes & Wallpapers", "id": 578 },
        { "name": "Social Network Apps & Add-Ons", "id": 847 },
        { "name": "Online Goodies", "id": 43 },
        { "name": "Online Journals & Personal Sites", "id": 582 },
        { "name": "Photo & Image Sharing", "id": 978 },
        { "name": "Video Sharing", "id": 979 },
        { "name": "Photo & Video Sharing", "id": 275 },
        { "name": "Social Network Apps & Add-Ons", "id": 847 },
        { "name": "Social Networks", "id": 529 },
        { "name": "Virtual Worlds", "id": 972 },
        { "name": "Online Communities", "id": 299 },
        { "name": "Mobility Equipment & Accessories", "id": 1353 },
        { "name": "Assistive Technology", "id": 1352 },
        { "name": "Disabled & Special Needs", "id": 677 },
        { "name": "African-Americans", "id": 547 },
        { "name": "Africans & Diaspora", "id": 579 },
        { "name": "Arabs & Middle Easterners", "id": 556 },
        { "name": "East Asians & Diaspora", "id": 549 },
        { "name": "South Asians & Diaspora", "id": 528 },
        { "name": "Southeast Asians & Pacific Islanders", "id": 580 },
        { "name": "Asians & Diaspora", "id": 1257 },
        { "name": "Discrimination & Identity Relations", "id": 1205 },
        { "name": "Eastern Europeans", "id": 682 },
        { "name": "Expatriate Communities", "id": 973 },
        { "name": "Gay-Lesbian-Bisexual-Transgender", "id": 113 },
        { "name": "Native Americans", "id": 171 },
        { "name": "Indigenous Peoples", "id": 681 },
        { "name": "Jewish Holidays", "id": 1124 },
        { "name": "Jewish Culture", "id": 550 },
        { "name": "Latinos & Latin-Americans", "id": 548 },
        { "name": "Western Europeans", "id": 683 },
        { "name": "Ethnic & Identity Groups", "id": 56 },
        { "name": "Etiquette", "id": 1304 },
        { "name": "Ancestry & Genealogy", "id": 400 },
        { "name": "Baby & Pet Names", "id": 1231 },
        { "name": "Adoption", "id": 974 },
        { "name": "Baby Care & Hygiene", "id": 115 },
        { "name": "Babies & Toddlers", "id": 1374 },
        { "name": "Child Care", "id": 403 },
        { "name": "Pregnancy & Maternity", "id": 401 },
        { "name": "Youth Camps", "id": 402 },
        { "name": "Parenting", "id": 58 },
        { "name": "Family", "id": 1132 },
        { "name": "Friendship", "id": 1134 },
        { "name": "Divorce & Separation", "id": 1261 },
        { "name": "Weddings", "id": 293 },
        { "name": "Marriage", "id": 1133 },
        { "name": "Romance", "id": 1135 },
        { "name": "Divorce & Separation", "id": 1261 },
        { "name": "Troubled Relationships", "id": 1260 },
        { "name": "Family & Relationships", "id": 1131 },
        { "name": "Children's Literature", "id": 1183 },
        { "name": "Family Films", "id": 1291 },
        { "name": "Drawing & Coloring", "id": 1397 },
        { "name": "Dress-Up & Fashion Games", "id": 1173 },
        { "name": "Family-Oriented Games & Activities", "id": 1290 },
        { "name": "TV Family-Oriented Shows", "id": 1110 },
        { "name": "Children's Interests", "id": 679 },
        { "name": "Teen Interests", "id": 680 },
        { "name": "Kids & Teens", "id": 154 },
        { "name": "Astrology & Divination", "id": 448 },
        { "name": "Buddhism", "id": 862 },
        { "name": "Christmas", "id": 1078 },
        { "name": "Easter", "id": 1123 },
        { "name": "Christian Holidays", "id": 1274 },
        { "name": "Christianity", "id": 864 },
        { "name": "Hinduism", "id": 866 },
        { "name": "Islamic Holidays", "id": 1275 },
        { "name": "Islam", "id": 868 },
        { "name": "Judaism", "id": 869 },
        { "name": "Occult & Paranormal", "id": 449 },
        { "name": "Pagan & Esoteric Traditions", "id": 1258 },
        { "name": "Places of Worship", "id": 1296 },
        { "name": "Scientology", "id": 1251 },
        { "name": "Self-Help & Motivational", "id": 870 },
        { "name": "Skeptics & Non-Believers", "id": 975 },
        { "name": "Spirituality", "id": 101 },
        { "name": "Theology & Religious Study", "id": 1340 },
        { "name": "Religion & Belief", "id": 59 },
        { "name": "Seniors & Retirement", "id": 298 },
        { "name": "Animal Welfare", "id": 883 },
        { "name": "Charity & Philanthropy", "id": 57 },
        { "name": "Discrimination & Identity Relations", "id": 1205 },
        { "name": "Drug Laws & Policy", "id": 1314 },
        { "name": "Climate Change & Global Warming", "id": 1255 },
        { "name": "Environmental Issues", "id": 82 },
        { "name": "Health Policy", "id": 1256 },
        { "name": "Housing & Development", "id": 1166 },
        { "name": "Human Rights & Liberties", "id": 1280 },
        { "name": "Immigration Policy & Border Issues", "id": 1313 },
        { "name": "Media Critics & Watchdogs", "id": 1203 },
        { "name": "Poverty & Hunger", "id": 1127 },
        { "name": "Privacy Issues", "id": 1281 },
        { "name": "Reproductive Rights", "id": 976 },
        { "name": "Same-Sex Marriage", "id": 1301 },
        { "name": "Labor & Employment Law", "id": 701 },
        { "name": "Unions & Labor Movement", "id": 1121 },
        { "name": "Work & Labor Issues", "id": 703 },
        { "name": "Social Issues & Advocacy", "id": 54 },
        { "name": "Public Speaking", "id": 1303 },
        { "name": "Communications & Media Studies", "id": 1302 },
        { "name": "Demographics", "id": 510 },
        { "name": "Economics", "id": 520 },
        { "name": "International Relations", "id": 521 },
        { "name": "Psychology", "id": 543 },
        { "name": "Social Sciences", "id": 509 },
        { "name": "Goth Subculture", "id": 503 },
        { "name": "Science Fiction & Fantasy", "id": 676 },
        { "name": "Subcultures & Niche Interests", "id": 502 },
        { "name": "People & Society", "id": 14 },
        { "name": "Animal Welfare", "id": 883 },
        { "name": "Pet Food & Supplies", "id": 379 },
        { "name": "Veterinarians", "id": 380 },
        { "name": "Animal Products & Services", "id": 882 },
        { "name": "Birds", "id": 884 },
        { "name": "Cats", "id": 885 },
        { "name": "Dogs", "id": 886 },
        { "name": "Exotic Pets", "id": 607 },
        { "name": "Fish & Aquaria", "id": 887 },
        { "name": "Horses", "id": 888 },
        { "name": "Rabbits & Rodents", "id": 889 },
        { "name": "Reptiles & Amphibians", "id": 890 },
        { "name": "Pets", "id": 563 },
        { "name": "Insects & Entomology", "id": 1278 },
        { "name": "Zoos-Aquariums-Preserves", "id": 1009 },
        { "name": "Wildlife", "id": 119 },
        { "name": "Pets & Animals", "id": 66 },
        { "name": "Apartments & Residential Rentals", "id": 378 },
        { "name": "Commercial & Investment Real Estate", "id": 1178 },
        { "name": "Property Development", "id": 687 },
        { "name": "Property Inspections & Appraisals", "id": 463 },
        { "name": "Property Management", "id": 425 },
        { "name": "Real Estate Agencies", "id": 96 },
        { "name": "Real Estate Listings", "id": 1080 },
        { "name": "Timeshares & Vacation Properties", "id": 1081 },
        { "name": "Real Estate", "id": 29 },
        { "name": "Business & Personal Listings", "id": 377 },
        { "name": "Directories & Listings", "id": 527 },
        { "name": "Biographies & Quotations", "id": 690 },
        { "name": "Calculators & Reference Tools", "id": 691 },
        { "name": "Dictionaries & Encyclopedias", "id": 692 },
        { "name": "Educational Resources", "id": 374 },
        { "name": "Legal Forms", "id": 1137 },
        { "name": "Forms Guides & Templates", "id": 693 },
        { "name": "How-To, DIY & Expert Content", "id": 694 },
        { "name": "Public Records", "id": 1136 },
        { "name": "Time & Calendars", "id": 695 },
        { "name": "General Reference", "id": 980 },
        { "name": "City & Local Guides", "id": 1014 },
        { "name": "Traffic & Public Transit", "id": 685 },
        { "name": "Maps", "id": 268 },
        { "name": "Geographic Reference", "id": 1084 },
        { "name": "Military History", "id": 1288 },
        { "name": "History", "id": 433 },
        { "name": "Myth & Folklore", "id": 609 },
        { "name": "Philosophy", "id": 1093 },
        { "name": "Humanities", "id": 474 },
        { "name": "Dictionaries & Encyclopedias", "id": 692 },
        { "name": "Foreign Language Study", "id": 1266 },
        { "name": "Translation Tools & Resources", "id": 1265 },
        { "name": "Foreign Language Resources", "id": 1264 },
        { "name": "Language Resources", "id": 108 },
        { "name": "Libraries & Museums", "id": 375 },
        { "name": "Public Speaking", "id": 1303 },
        { "name": "Communications & Media Studies", "id": 1302 },
        { "name": "Demographics", "id": 510 },
        { "name": "Economics", "id": 520 },
        { "name": "International Relations", "id": 521 },
        { "name": "Psychology", "id": 543 },
        { "name": "Social Sciences", "id": 509 },
        { "name": "Data Sheets & Electronics Reference", "id": 900 },
        { "name": "Technical Support", "id": 567 },
        { "name": "Technical Reference", "id": 1233 },
        { "name": "Reference", "id": 533 },
        { "name": "Astronomy", "id": 435 },
        { "name": "Anatomy", "id": 788 },
        { "name": "Insects & Entomology", "id": 1278 },
        { "name": "Flora & Fauna", "id": 981 },
        { "name": "Genetics", "id": 982 },
        { "name": "Neuroscience", "id": 1226 },
        { "name": "Biological Sciences", "id": 440 },
        { "name": "Chemistry", "id": 505 },
        { "name": "Computer Education", "id": 1229 },
        { "name": "Distributed & Parallel Computing", "id": 1298 },
        { "name": "Machine Learning & Artificial Intelligence", "id": 1299 },
        { "name": "C & C++", "id": 731 },
        { "name": "Developer Jobs", "id": 802 },
        { "name": "Development Tools", "id": 730 },
        { "name": "Java", "id": 732 },
        { "name": "Scripting Languages", "id": 733 },
        { "name": "Windows & .NET", "id": 734 },
        { "name": "Programming", "id": 31 },
        { "name": "Computer Science", "id": 1227 },
        { "name": "Climate Change & Global Warming", "id": 1255 },
        { "name": "Atmospheric Science", "id": 1254 },
        { "name": "Geology", "id": 443 },
        { "name": "Paleontology", "id": 1169 },
        { "name": "Water & Marine Sciences", "id": 441 },
        { "name": "Earth Sciences", "id": 1168 },
        { "name": "Climate Change & Global Warming", "id": 1255 },
        { "name": "Ecology & Environment", "id": 442 },
        { "name": "CAD & CAM", "id": 1300 },
        { "name": "Robotics", "id": 1141 },
        { "name": "Technology News", "id": 785 },
        { "name": "Engineering & Technology", "id": 231 },
        { "name": "Statistics", "id": 1252 },
        { "name": "Mathematics", "id": 436 },
        { "name": "Physics", "id": 444 },
        { "name": "Scientific Equipment", "id": 445 },
        { "name": "Scientific Institutions", "id": 446 },
        { "name": "Science", "id": 174 },
        { "name": "Antiques & Collectibles", "id": 64 },
        { "name": "Apparel Services", "id": 1228 },
        { "name": "Athletic Apparel", "id": 983 },
        { "name": "T-Shirts", "id": 428 },
        { "name": "Casual Apparel", "id": 984 },
        { "name": "Children's Clothing", "id": 985 },
        { "name": "Gems & Jewelry", "id": 350 },
        { "name": "Handbags & Purses", "id": 986 },
        { "name": "Watches", "id": 987 },
        { "name": "Clothing Accessories", "id": 124 },
        { "name": "Costumes", "id": 988 },
        { "name": "Eyeglasses & Contacts", "id": 1224 },
        { "name": "Eyewear", "id": 989 },
        { "name": "Footwear", "id": 697 },
        { "name": "Formal Wear", "id": 990 },
        { "name": "Headwear", "id": 991 },
        { "name": "Men's Clothing", "id": 992 },
        { "name": "Outerwear", "id": 993 },
        { "name": "Sleepwear", "id": 994 },
        { "name": "Swimwear", "id": 995 },
        { "name": "Undergarments", "id": 530 },
        { "name": "Uniforms & Workwear", "id": 996 },
        { "name": "Women's Clothing", "id": 997 },
        { "name": "Apparel", "id": 68 },
        { "name": "Auctions", "id": 292 },
        { "name": "Classifieds", "id": 61 },
        { "name": "Headphones", "id": 1396 },
        { "name": "MP3 & Portable Media Players", "id": 227 },
        { "name": "Speakers", "id": 1158 },
        { "name": "Stereo Systems & Components", "id": 91 },
        { "name": "Audio Equipment", "id": 361 },
        { "name": "Binoculars, Telescopes & Optical Devices", "id": 1384 },
        { "name": "Camcorders", "id": 308 },
        { "name": "Camera Lenses", "id": 1383 },
        { "name": "Cameras", "id": 307 },
        { "name": "Cameras & Camcorders", "id": 306 },
        { "name": "Camera & Photo Equipment", "id": 573 },
        { "name": "Car Audio", "id": 230 },
        { "name": "Car Video", "id": 1189 },
        { "name": "GPS & Navigation", "id": 794 },
        { "name": "Car Electronics", "id": 1188 },
        { "name": "Electronic Accessories", "id": 1192 },
        { "name": "E-Book Readers", "id": 1324 },
        { "name": "Handheld Game Consoles", "id": 1046 },
        { "name": "MP3 & Portable Media Players", "id": 227 },
        { "name": "PDAs & Handhelds", "id": 228 },
        { "name": "Gadgets & Portable Electronics", "id": 362 },
        { "name": "Handheld Game Consoles", "id": 1046 },
        { "name": "Nintendo", "id": 1043 },
        { "name": "Sony PlayStation", "id": 1044 },
        { "name": "Xbox", "id": 1045 },
        { "name": "Game Systems & Consoles", "id": 899 },
        { "name": "GPS & Navigation", "id": 794 },
        { "name": "DVRs & Set-Top Boxes", "id": 1393 },
        { "name": "Home Theater Systems", "id": 1157 },
        { "name": "Projectors & Screens", "id": 1334 },
        { "name": "HDTVs", "id": 1354 },
        { "name": "LCD TVs", "id": 1356 },
        { "name": "Plasma TVs", "id": 1355 },
        { "name": "Projection TVs", "id": 1357 },
        { "name": "Televisions", "id": 305 },
        { "name": "Blu-Ray Players & Recorders", "id": 1394 },
        { "name": "DVD Players & Recorders", "id": 1395 },
        { "name": "Video Players & Recorders", "id": 492 },
        { "name": "TV & Video Equipment", "id": 229 },
        { "name": "Consumer Electronics", "id": 78 },
        { "name": "Consumer Advocacy & Protection", "id": 97 },
        { "name": "Coupons & Discount Offers", "id": 365 },
        { "name": "Loyalty Cards & Programs", "id": 1309 },
        { "name": "Technical Support", "id": 567 },
        { "name": "Warranties & Service Contracts", "id": 451 },
        { "name": "Customer Services", "id": 450 },
        { "name": "Price Comparisons", "id": 352 },
        { "name": "Vehicle Specs, Reviews & Comparisons", "id": 1267 },
        { "name": "Product Reviews & Price Comparisons", "id": 353 },
        { "name": "Consumer Resources", "id": 69 },
        { "name": "Book Retailers", "id": 355 },
        { "name": "CD & Audio Shopping", "id": 217 },
        { "name": "DVD & Video Rentals", "id": 1145 },
        { "name": "DVD & Video Shopping", "id": 210 },
        { "name": "DVD & Video Rentals", "id": 1145 },
        { "name": "Entertainment Media Rentals", "id": 1144 },
        { "name": "Video Game Retailers", "id": 1146 },
        { "name": "Entertainment Media", "id": 1143 },
        { "name": "Cards & Greetings", "id": 100 },
        { "name": "Flowers", "id": 323 },
        { "name": "Gifts", "id": 99 },
        { "name": "Party & Holiday Supplies", "id": 324 },
        { "name": "Gifts & Special Event Items", "id": 70 },
        { "name": "Luxury Goods", "id": 696 },
        { "name": "Mass Merchants & Department Stores", "id": 73 },
        { "name": "Stock Photography", "id": 574 },
        { "name": "Photo & Video Services", "id": 576 },
        { "name": "Shopping Portals & Search Engines", "id": 531 },
        { "name": "Bicycles & Accessories", "id": 1191 },
        { "name": "Sports Memorabilia", "id": 1083 },
        { "name": "Sporting Goods", "id": 263 },
        { "name": "Swap Meets & Outdoor Markets", "id": 1210 },
        { "name": "Ticket Sales", "id": 614 },
        { "name": "Tobacco Products", "id": 123 },
        { "name": "Toys", "id": 432 },
        { "name": "Wholesalers & Liquidators", "id": 1225 },
        { "name": "Shopping", "id": 18 },
        { "name": "College Sports", "id": 1073 },
        { "name": "Boxing", "id": 515 },
        { "name": "Martial Arts", "id": 516 },
        { "name": "Wrestling", "id": 512 },
        { "name": "Combat Sports", "id": 514 },
        { "name": "Drag & Street Racing", "id": 1206 },
        { "name": "Stunts & Dangerous Feats", "id": 1207 },
        { "name": "Extreme Sports", "id": 554 },
        { "name": "Fantasy Sports", "id": 998 },
        { "name": "Bowling", "id": 1016 },
        { "name": "Bicycles & Accessories", "id": 1191 },
        { "name": "Cycling", "id": 458 },
        { "name": "Golf", "id": 261 },
        { "name": "Gymnastics", "id": 519 },
        { "name": "Tennis", "id": 1376 },
        { "name": "Racquet Sports", "id": 262 },
        { "name": "Running & Walking", "id": 541 },
        { "name": "Skate Sports", "id": 1126 },
        { "name": "Track & Field", "id": 518 },
        { "name": "Individual Sports", "id": 1000 },
        { "name": "Live Sporting Events", "id": 1273 },
        { "name": "Drag & Street Racing", "id": 1206 },
        { "name": "Motor Sports", "id": 180 },
        { "name": "Bicycles & Accessories", "id": 1191 },
        { "name": "Sports Memorabilia", "id": 1083 },
        { "name": "Sporting Goods", "id": 263 },
        { "name": "Sports Coaching & Training", "id": 1082 },
        { "name": "Sports News", "id": 1077 },
        { "name": "American Football", "id": 258 },
        { "name": "Baseball", "id": 259 },
        { "name": "Basketball", "id": 264 },
        { "name": "Cheerleading", "id": 534 },
        { "name": "Cricket", "id": 296 },
        { "name": "Handball", "id": 1017 },
        { "name": "Hockey", "id": 260 },
        { "name": "Rugby", "id": 517 },
        { "name": "Soccer", "id": 294 },
        { "name": "Volleyball", "id": 699 },
        { "name": "Team Sports", "id": 1001 },
        { "name": "Water Sports", "id": 118 },
        { "name": "Ice Skating", "id": 1149 },
        { "name": "Skiing & Snowboarding", "id": 1148 },
        { "name": "Winter Sports", "id": 265 },
        { "name": "Olympics", "id": 513 },
        { "name": "World Sports Competitions", "id": 1198 },
        { "name": "Sports", "id": 20 },
        { "name": "Airport Parking & Transportation", "id": 1245 },
        { "name": "Personal Aircraft", "id": 1147 },
        { "name": "Recreational Aviation", "id": 999 },
        { "name": "Air Travel", "id": 203 },
        { "name": "Bus & Rail", "id": 708 },
        { "name": "Car Rental & Taxi Services", "id": 205 },
        { "name": "Carpooling & Ridesharing", "id": 1339 },
        { "name": "Cruises & Charters", "id": 206 },
        { "name": "Hotels & Accommodations", "id": 179 },
        { "name": "Luggage & Travel Accessories", "id": 1003 },
        { "name": "Adventure Travel", "id": 707 },
        { "name": "Agritourism", "id": 1389 },
        { "name": "Ecotourism", "id": 1005 },
        { "name": "Sightseeing Tours", "id": 1390 },
        { "name": "Vineyards & Wine Tourism", "id": 1391 },
        { "name": "Specialty Travel", "id": 1004 },
        { "name": "Beaches & Islands", "id": 1074 },
        { "name": "Historical Sites & Buildings", "id": 1006 },
        { "name": "Lakes & Rivers", "id": 1120 },
        { "name": "Mountain & Ski Resorts", "id": 1119 },
        { "name": "Regional Parks & Gardens", "id": 1007 },
        { "name": "Theme Parks", "id": 1008 },
        { "name": "Zoos-Aquariums-Preserves", "id": 1009 },
        { "name": "Tourist Destinations", "id": 208 },
        { "name": "Tourist Boards & Visitor Centers", "id": 1392 },
        { "name": "Vacation Offers", "id": 1019 },
        { "name": "Travel Agencies & Services", "id": 1010 },
        { "name": "Travel Guides & Travelogues", "id": 1011 },
        { "name": "Travel", "id": 67 }
      ]"#;
}

impl Display for Category {
    fn fmt(&self, f: &mut Formatter) -> Result {
        write!(f, "{}", self.id)
    }
}
