#![allow(dead_code)]
#![allow(unused_variables)]
#![allow(unused_imports)]

use super::*;

fn point2(x: f64, y: f64) -> super::Rect<2, f64> {
    Rect::new_point([x, y])
}

extern crate rand;

use std::fmt::Display;
use std::fs::File;
use std::io::{Error, Write};

fn test_rtree(count: usize, points_only: bool) {
    let mut tr = RTree::new();
    let mut pts = vec![];
    for _ in 0..count {
        let xy = [
            rand::random::<f64>() * 360.0 - 180.0,
            rand::random::<f64>() * 180.0 - 90.0,
        ];
        if points_only {
            pts.push(Rect::new_point(xy));
        } else {
            pts.push(Rect::new(
                xy,
                [xy[0] + rand::random::<f64>(), xy[1] + rand::random::<f64>()],
            ));
        }
    }
    // insert each point
    for i in 0..pts.len() {
        tr.insert(pts[i], i);
        // check the length
        assert_eq!(tr.len(), i + 1);
        // search for this item
        assert_eq!(tr.search(pts[i]).filter(|x| x.data == &i).count(), 1);
    }
    // scan all rects and compare
    let mut all: Vec<IterItem<2, f64, usize>> = tr.scan().collect();
    all.sort_by(|a, b| a.data.cmp(b.data));
    assert_eq!(all.len(), pts.len());
    for i in 0..pts.len() {
        if pts[i] != all[i].rect {
            panic!("not equal");
        }
    }

    // search for each point again
    for i in 0..pts.len() {
        assert_eq!(tr.search(pts[i]).filter(|x| x.data == &i).count(), 1);
    }

    // scan kNN
    let mut dist = 0.0;
    let mut count = 0;
    let target = Rect::new([-112.0, -33.0], [-112.0, -33.0]);
    for item in tr.nearby(|rect, _| rect.box_dist(&target)) {
        if item.dist < dist {
            panic!("out or order")
        }
        dist = item.dist;
        count += 1;
    }
    assert_eq!(count, pts.len());

    // remove each point
    for i in 0..pts.len() {
        let res = tr.remove(pts[i], &i).unwrap();
        // check the rect
        if pts[i] != res.0 {
            panic!("not equal");
        }
        // check the data
        assert_eq!(res.1, i);
        // check the length
        assert_eq!(tr.len(), pts.len() - i - 1);
        // search for this item
        assert_eq!(tr.search(pts[i]).filter(|x| x.data == &i).count(), 0);
    }
}

#[test]
fn points() {
    test_rtree(100_000, true);
}

#[test]
fn rects() {
    test_rtree(100_000, false);
}

impl<const D: usize, C, T: PartialEq> Node<D, C, T>
where
    C: PartialOrd + Copy + Sub<Output = C> + Mul<Output = C> + Display + Default,
{
    fn svg(&self, depth: usize) -> String {
        let mut out = String::new();
        if let Data::Nodes(nodes) = &self.data {
            out += &format!(
                "<rect x=\"{}\" y=\"{}\" width=\"{}\" \
                       height=\"{}\" stroke=\"{}\" fill-opacity=\"0\" \
                       stroke-opacity=\"1\" stroke-width=\"0.4\"/>\n",
                self.rect.min[0],
                self.rect.min[1],
                self.rect.max[0] - self.rect.min[0],
                self.rect.max[1] - self.rect.min[1],
                ["red", "#009900", "#cccc00", "purple"][depth % 4]
            );
            for i in 0..nodes.len() {
                out += &nodes[i].svg(depth + 1);
            }
        } else {
            out += &format!(
                "<circle cx=\"{}\" cy=\"{}\" r=\"0.2\" \
                         fill=\"{}\" stroke-width=\"0\"/>\n",
                self.rect.min[0], self.rect.min[1], "black"
            );
        }
        return out;
    }
}

impl<const D: usize, C, T: PartialEq> RTree<D, C, T>
where
    C: PartialOrd + Copy + Sub<Output = C> + Mul<Output = C> + Display + Default,
{
    pub fn svg(&self) -> String {
        let mut out = String::new();
        out += &format!(
            "<svg viewBox=\"{} {} {} {}\" \
                  xmlns =\"http://www.w3.org/2000/svg\">\n",
            -190.0, -100.0, 380.0, 190.0,
        );
        out += "<g transform=\"scale(1,-1)\">\n";
        if let Some(root) = &self.root {
            out += &root.svg(0);
        }
        out += "</g>\n";
        out += "</svg>\n";
        out
    }
}

#[test]
fn default_rect() {
    let mut tr: RTree<2, f64, usize> = RTree::new();
    tr.insert(Rect::default(), 1);
}

#[test]
fn svg() {
    to_svg(PREDEF_PTS, "predefined.svg");
    to_svg(CITIES_PTS, "cities.svg");
}

fn to_pts(pts: &str) -> Vec<[f64; 2]> {
    pts.split(";")
        .map(|x| {
            let v: Vec<f64> = x.split(",").map(|x| x.parse().unwrap()).collect();
            [v[0], v[1]]
        })
        .collect()
}

fn to_svg(pts: &str, path: &str) {
    let pts = to_pts(pts);
    let mut tr = RTree::new();
    for i in 0..pts.len() {
        tr.insert(point2(pts[i][0], pts[i][1]), i);
    }
    let svg = tr.svg();
    let mut output = File::create(path).unwrap();
    write!(output, "{}", svg).unwrap();
}

// These are hardcoded points that are for the purpose of filling an RTree
// and outputing an SVG for comparing against other implementations of this
// RTree.
const PREDEF_PTS: &str = "\
    -52.9434,2.3502;79.7989,0.0965;-70.7779,57.1756;36.2933,77.0761;21.4716,3.4453;-14.5109,-18.9968;-33.9442,-11.1449;10.3230,-66.1787;-76.7850,-68.3149;48.2775,-57.6251;1.8490,32.8058;-6.3306,-50.2694;-11.0860,-26.7247;-71.1707,-77.4811;-40.9573,-81.1828;13.3053,26.7539;-15.3284,-43.5700;-16.2263,30.5950;53.7956,42.5554;-17.4207,-45.7420;28.6247,-73.9760;-47.9121,-24.0529;20.3135,81.6178;-75.7848,-61.4280;60.6492,45.6531;32.6774,-1.8117;6.7576,-30.7179;36.9515,84.9250;22.8975,23.5716;\
    -1.0784,-66.1536;80.9594,-43.7432;47.2764,-59.5494;16.6996,7.8703;-67.1000,-64.3771;-5.6678,45.6234;15.5682,11.4618;-70.3210,36.6239;-27.7830,21.0051;-32.8146,0.4812;-80.1441,-61.4365;-9.8833,-60.8014;-70.2467,-37.1890;-40.2480,-35.0243;0.3481,-59.1803;-1.3824,11.9884;38.1801,-86.4870;41.1855,-43.7043;-49.6251,-46.7210;70.0259,82.8820;2.0714,-57.8324;-32.0670,61.0331;57.2634,-77.8351;22.5605,30.6484;13.3030,-74.1716;-51.8567,-38.8966;65.7489,0.6785;-43.7919,40.2813;13.5196,52.1255;\
    86.9304,18.5351;53.8486,-52.5475;-41.7540,-78.1031;63.3798,51.8127;-35.8394,-46.5872;-74.4762,36.0596;-13.9605,-63.7214;-29.0363,8.0101;-10.1690,-11.2793;-44.8196,35.2034;88.2495,-49.9427;86.3409,67.8455;-7.6916,48.4385;29.9678,88.1157;-31.7521,21.9599;3.7198,-86.8286;22.4988,-34.8358;-60.3207,-8.1582;-34.6868,-84.3874;-71.1294,-49.3260;16.9321,53.0747;33.2197,-28.7784;-76.2441,0.3074;51.9192,-12.5396;-71.4122,67.0280;52.8581,24.6432;-36.9683,-36.4627;76.0643,15.6494;29.3803,86.3542;\
    32.0771,-71.2889;-58.4152,21.6894;-85.8911,-36.1723;-24.5700,29.0935;76.6989,7.8303;-46.9513,-77.7443;-88.0968,-79.8746;19.8507,27.8617;69.4082,-67.3788;-40.2571,80.8320;29.0660,33.6331;-53.9672,88.9871;30.1764,-7.7115;-67.8264,15.3220;-67.5435,-14.0896;-13.2111,53.6434;67.0665,-62.0404;-8.1534,73.0098;1.7675,-38.0446;-81.8682,-86.1640;-77.2129,41.7893;76.0977,43.2557;61.2522,36.3448;-40.7277,20.7127;-49.9203,44.0072;-63.8978,44.6034;-15.0568,77.9749;1.2336,-19.8776;-1.4258,10.8275;\
    54.2571,-84.9135;-4.4290,-86.3580;-6.6614,-82.4708;37.4052,-26.0952;-83.3688,39.4285;53.9972,59.2077;-87.2161,-31.3833;59.1044,-58.0511;-10.6633,-63.1487;8.9783,-10.6305;-12.5184,-61.9062;-57.5105,-13.5986;-74.4164,66.2898;40.6505,-63.8591;38.1709,-56.7642;-80.1483,72.1291;-54.6147,-48.3416;-19.3687,40.2425;87.0986,-28.9486;72.9299,-5.0667;-82.1215,-28.9288;-73.2589,61.3906;71.1194,-44.5480;3.9836,14.1080;60.7343,-14.7402;-81.9011,81.8563;3.1987,-51.5167;-28.2084,-14.8615;77.2357,-22.4988;\
    4.4660,8.3415;5.0761,49.3795;28.4371,37.7285;3.8474,77.9208;41.0683,14.1515;-84.6480,33.4903;-33.0685,-72.0830;-28.1757,-43.6106;21.7882,9.2081;-75.6114,8.8944;23.2134,-24.5230;-20.0532,52.9221;30.5061,-88.3147;-82.0978,-81.9650;-12.5273,8.8273;-6.6512,-40.3822;65.7155,-50.7077;-18.4567,26.9775;89.0259,58.9477;-39.1066,-48.3658;85.2529,-58.3033;16.9372,-72.1637;38.9981,-35.0735;4.2960,5.8720;-69.0029,6.6783;-74.5165,-9.2155;77.0172,-15.8982;77.2111,-20.9122;10.5617,50.5628;36.7159,13.7970;\
    -88.7977,44.7608;-40.9536,-48.0693;86.3579,71.5294;70.9395,-37.3347;-8.4910,68.6508;-18.0749,51.4585;32.9062,-25.1475;-2.2470,-5.8002;-35.1355,-77.5106;-21.1463,-40.4981;-14.6883,-0.1885;20.8950,-76.0406;-20.0832,-71.5254;-77.1039,-52.1065;-30.8174,-44.5264;-6.3434,-45.8011;-50.7452,-82.0047;26.5532,76.9169;-47.4995,-3.8571;-15.9241,-82.6100;75.9835,33.2700;-40.4169,28.1756;-80.4825,-86.9009;-67.9610,-51.3600;-44.9310,20.6217;29.3389,38.5861;-17.1868,-61.4062;10.2863,80.6114;-85.5774,3.8388;\
    -25.9579,45.2343;-45.6040,-51.6530;-45.0599,50.5700;-82.3293,-47.7250;-15.6298,15.4391;-31.0017,61.9690;-19.6140,54.3091;-17.2402,19.6268;-42.5112,-46.7835;-70.2621,-44.9183;31.1295,62.7480;60.5247,-83.9339;13.9924,10.5003;48.8592,-77.3547;77.7701,-73.3667;-1.0778,28.7263;77.2253,84.3070;-75.7620,52.7147;-47.7578,65.1181;-6.4685,-72.7622;-54.9103,-68.9114;-56.5110,60.4853;-30.6803,-53.9745;-39.6891,-78.4086;56.9182,-23.3755;34.8054,67.7611;-60.1925,-79.8757;55.6025,-11.5118;57.4923,-43.8925;\
    -69.2271,-80.8509;-87.2129,-39.3090;13.2397,-52.9905;7.8343,-47.8217;60.1905,36.0755;-46.2321,-86.9192;9.6910,-52.1751;82.1073,3.2804;62.6113,84.5029;0.3937,-18.7198;-10.5888,41.4950;-24.9596,25.0843;26.7151,-84.5413;22.8757,22.4057;-58.2860,12.5250;-59.9110,-85.8419;6.0928,-24.3420;30.9450,-63.3300;-33.7900,-58.1274;26.1678,57.8704;29.2159,16.7054;-27.8575,-31.1036;5.6148,-10.6074;74.7245,-65.5407;-0.5136,28.0763;-26.0114,51.9623;12.6901,71.6695;37.5503,-11.2884;85.9142,15.5730;\
    39.0081,-22.6366;23.2022,-46.9197;72.5890,28.2488;-27.0062,68.1462;-25.5885,80.3128;7.1205,41.3212;-34.1489,-22.8620;-48.7464,-12.6946;74.5176,-78.7932;-51.9404,80.0213;-88.2574,-65.7252;66.1371,25.6997;56.7606,-38.6798;-12.9869,32.4605;43.5826,33.7614;-77.7009,-67.2829;66.8542,50.6482;-44.4890,55.9245;36.8936,36.1455;4.9570,87.0211;89.8758,45.1632;17.8615,-8.4198;-78.9756,-32.5195;-60.7864,-76.8903;-10.2154,82.2405;83.4223,28.9638;-80.2242,42.4906;-16.5073,11.7500;33.4220,33.2088;\
    32.4872,25.6165;-71.8052,85.5684;54.9529,17.3122;-68.2498,39.6814;-14.7118,-19.8332;55.1996,-88.6808;-49.1521,-80.9897;-55.6756,0.7200;-55.1793,-74.8804;81.6077,-31.7803;-66.1399,81.8779;-30.1113,11.4333;5.7586,76.0547;57.2768,-5.9051;-84.0314,-49.7809;-26.1461,51.7768;-22.1395,8.4888;77.5471,-27.2998;50.0910,66.6174;-50.5820,-42.9611;37.6183,-75.5827;66.4459,-82.6761;-15.6944,-1.7931;14.3352,-25.3912;2.3959,70.4230;-48.0739,0.7544;63.9774,81.9489;-16.3303,-78.9417;-69.0464,39.5361;\
    -31.0807,52.6274;80.0969,-50.8927;-47.6331,-86.5258;-31.7826,-81.4586;-55.5014,-35.0160;13.8407,79.4961;-52.9087,7.8719;-17.2291,-80.0960;53.6484,-30.7694;-14.8948,49.2581;55.4250,-68.9726;-87.6850,8.8810;-33.1584,48.5048;52.0220,-89.5231;-40.6441,69.3154;-20.9094,61.7686;-61.5080,52.2388;64.2091,67.1454;74.2173,-64.1455;2.0732,-65.8985;-49.5373,36.2591;-23.7607,11.5814;38.8282,-35.2040;79.1334,-2.2126;-18.5264,4.1235;68.2501,18.7503;-63.0554,70.7341;85.1288,71.3774;-58.3849,85.6689;\
    -43.2705,60.9234;1.3773,71.5044;-73.9729,40.5506;88.9481,-73.9361;42.6679,-73.2078;73.6783,45.0335;-61.2018,-68.5173;-30.6254,0.5283;-79.8646,-37.6187;-76.1819,-72.0089;-24.2034,39.7396;57.5190,48.6093;73.2141,-59.5807;-10.9796,-66.8317;-16.5220,8.5569;-66.5117,30.2980;-61.5627,88.9660;-68.8261,-44.6686;46.1838,-48.3580;14.7668,-61.5439;54.1177,10.6038;-2.1301,-42.0325;85.8943,11.4893;-12.9333,4.1751;-69.9762,-37.5011;-5.4861,73.7358;-88.8035,67.9389;59.7848,-4.1730;6.4518,-4.8899;\
    -40.3329,30.0730;63.1220,44.7245;47.1077,-6.0890;65.7583,-46.3528;58.5334,-33.9046;83.6273,-44.5301;-77.1676,82.0083;-8.3815,24.6806;-47.3139,-6.1970;51.3132,19.2416;-4.5901,12.2321;-16.4801,-31.4038;13.0595,-85.7804;-12.3926,37.1996;-78.9179,78.8333;-11.7130,64.9164;-32.8302,-39.3124;-30.2074,55.7388;83.9253,39.4534;-1.1630,15.7650;-46.5653,18.4163;-0.0154,72.9451;-56.0877,11.6550;-28.1180,9.0266;67.1333,-85.4121;-17.3986,29.2941;-33.3568,-11.6923;53.0484,28.0425;-49.0512,51.5841;\
    -50.8336,29.9062;63.0311,3.5101;-13.7699,74.2688;29.5358,69.5477;-86.8905,-26.8892;-14.2838,14.8348;-19.8938,-28.4286;43.4298,-73.0661;4.6554,-18.3869;10.5280,-68.1632;24.1204,-83.8095;-22.8565,-1.5500;-34.0093,-17.8814;-3.3333,6.5955;-28.8775,22.5522;-69.3611,-62.8103;-33.8547,-89.9466;-21.6078,63.7730;11.6983,-13.7234;70.3667,2.5264;65.1577,-68.1761;52.4679,27.1731;-15.0621,-47.5655;45.4060,-35.7604;-63.9172,0.8718;-43.5434,5.8501;-62.5399,79.0188;-47.7538,-45.8239;-17.1508,-19.7940;\
    -33.2220,-27.7610;-81.5378,-69.0413;-37.0707,-47.1813;-83.6335,13.6370;-42.8741,74.0179;-34.5145,77.0880;-4.3289,67.8375;-24.1152,-51.4052;83.7485,85.6460;-35.4435,-36.7103;-74.8992,11.2095;32.1774,24.8540;-8.0815,-19.0882;15.6655,79.3817;-42.4136,-47.2277;-77.6627,-83.7567;58.3438,82.9944;-75.0941,69.3294;25.4392,40.6012;-61.0246,-7.2221;85.9255,80.7075;-28.1645,-85.5145;48.0017,65.5803;-21.0488,38.1642;61.4635,65.0047;-33.3308,21.7429;87.5311,24.5640;-73.3008,66.1197;-30.4783,29.3639;\
    8.5229,56.6486;-13.7644,54.4509;79.9751,-36.7915;50.5152,-37.9054;80.2015,-21.0408;-71.6869,88.3540;-55.7580,73.3866;-22.3594,-40.5109;-31.0829,79.6926;-32.0702,61.2345;9.9563,-10.7157;3.1893,51.0244;-3.1781,-77.0447;-64.6216,40.0305;-80.4432,-51.8202;39.7182,-22.1685;-14.3471,49.1895;71.4156,18.4536;49.3849,-22.9461;32.8360,64.5213;-29.4786,44.4825;-34.8768,56.2419;83.9938,-5.1193;-89.9475,-15.7402;15.8753,-26.4371;-59.2485,-36.4049;63.5081,-48.3057;-30.6237,28.5572;46.0018,3.7920;\
    6.5802,-36.9271;-64.0136,-87.6502;26.6565,-79.9805;-45.2478,-32.7100;-52.1907,-24.9961;56.4383,33.0611;-53.2675,-56.9942;33.3971,-39.5237;-52.3940,-23.7571;65.9531,-14.4773;80.2281,87.5800;-32.9501,-25.5961;1.2152,2.2925;51.8063,-30.0636;-33.2765,31.4628;42.5221,-25.0361;-76.9311,2.1847;-63.9892,-14.4333;-6.6220,-61.2169;-32.2040,76.8378;-40.5808,32.4269;-40.3002,-69.3892;87.9410,66.8456;38.9926,6.8878;-73.6047,29.0581;-27.8385,7.6197;55.9702,61.6916;-85.3465,-20.6114;41.3139,41.6083;\
    -67.4707,38.5385;13.5486,9.6479;-25.0533,25.7621;-56.2857,-83.4090;-68.5426,22.6971;58.2882,-45.4290;49.3186,47.6067;-79.0582,17.4807;-27.7210,-63.2892;-57.6367,28.7307;17.2145,28.6445;25.8892,-80.9628;-83.0954,-32.5231;-60.9474,-35.3149;20.8042,-25.3309;59.7175,-9.6269;-48.4026,80.4356;71.4724,-63.1663;-65.6169,-83.1836;31.1233,50.6399;89.4090,-58.7263;-56.8134,35.6424;-1.9870,76.6238;-34.4510,-71.0530;21.0428,-21.8501;88.1951,-77.5687;20.2935,-87.0181;83.9771,75.1439;29.4589,34.7445;\
    -14.2287,-73.3792;-2.5052,-34.9347;-57.1395,33.3535;30.3226,82.1585;-50.0110,-64.1174;-66.6991,33.2650;81.4486,26.2537;16.5824,-49.1302;-65.4873,67.8786;1.8493,75.6869;-51.7611,-6.9432;-15.8041,12.7080;10.5355,48.3821;-47.2539,-66.0540;19.7980,-21.3918;7.3200,-81.3282;-60.8794,19.4504;-16.0037,1.8716;69.3316,49.3667;85.4973,38.7003;47.3440,8.0316;49.5444,82.9798;-57.8345,0.9889;-57.1600,-81.8909;37.9118,-66.7371;9.7176,-67.7539;-51.5919,72.6842;10.2980,-70.5326;72.3562,-83.9524;\
    -71.6369,-58.5893;-76.0621,-29.3883;-22.8659,-20.9290;-29.0006,86.8755;-22.0670,15.5413;34.5449,65.4479;34.3524,-66.6382;-26.2762,87.5574;-83.8959,4.6905;61.5699,18.8614;-20.4350,-51.2488;-54.7315,-20.5817;2.4575,-84.7425;37.6943,-9.5111;1.1969,-2.8540;33.7326,26.4164;-7.6831,-11.5806;-28.5758,-17.1808;-85.2692,71.8988;67.4390,10.3697;38.3742,42.8474;-32.6314,21.0563;69.5705,71.5829;21.4281,74.3753;-46.0275,-39.9097;80.3925,24.0717;74.4087,-88.4813;-22.5185,-16.3908;-0.2918,32.7710;\
    41.4694,35.7904;-20.2767,-2.4967;51.9845,-36.7197;27.6418,36.2399;-30.0058,62.9967;54.4478,76.0027;-44.4962,-65.9762;66.0384,89.9881;-66.2017,-16.1885;-1.1234,-88.6890;67.8084,6.8134;56.4702,44.1911;55.5408,-13.9232;5.7780,-37.0869;-42.4919,80.1494;23.3671,58.2422;-22.9629,-1.0311;-27.8635,-81.9934;6.6783,73.2169;18.0533,83.0931;-78.3361,79.3511;-27.0232,79.6349;-28.4196,-78.2382;59.2776,-79.8229;-11.6825,-49.8756;73.8849,-66.7200;-8.4341,88.7545;-56.5173,82.7530;-60.2083,16.6967;\
    -51.9885,-85.5549;-52.2384,54.4372;64.7254,-48.7328;0.2584,-86.7116;26.3940,-87.7310;-83.8045,32.9629;-46.9558,-8.6496;29.7599,-87.7977;89.9721,7.4844;79.6903,-28.3720;47.5334,-5.2594;57.5719,-45.4520;13.8043,-8.6851;-73.7204,-45.3295;37.7690,-49.6300;-11.6078,-57.5777;-20.3913,-28.8905;23.6836,8.6381;78.4838,78.3319;-65.5779,70.1120;63.0910,-8.0961;-50.9332,23.6840;62.0292,-11.2352;43.7532,9.5109;46.3306,78.9764;-18.6936,41.5023;-15.0097,-51.6610;70.7750,-5.5802;-63.9624,-4.2327;\
    57.0088,-29.1994;27.6967,38.3884;73.9759,-17.9269;-39.3208,21.6714;-4.0670,83.1703;65.2890,-71.3259;22.6366,55.6119;-88.1377,28.6641;-5.3892,-88.5038;-10.6674,18.8970;80.5521,32.3199;28.5787,59.1938;27.1008,39.6384;-59.2475,47.6801;16.6968,-61.3290;63.6715,-21.8496;62.0609,63.9534;-33.9215,-11.1731;-30.9228,77.6044;-59.3863,42.2914;26.7527,67.3897;41.3014,40.1809;48.8683,84.1737;-37.4049,-64.2675;4.3905,23.5088;-5.5212,36.3604;37.4847,-20.9440;-26.5467,27.6644;-60.2969,12.7998;42.7950,59.0191;\
    -53.6255,8.0315;-41.2044,-21.8788;-37.8386,-80.6275;-25.8529,-3.8191;42.3623,-70.2175;-38.4467,41.2322;20.3268,1.0026;-61.5230,-38.8704;6.4595,-38.0670;-11.3392,-37.8595;-62.9920,10.3925;-53.9259,-89.5068;44.5639,-18.6681;-59.1049,-84.1289;11.6192,-65.8581;-82.4682,-50.7466;-88.2061,-48.4824;-44.2050,86.0141;34.3022,-37.5376;-22.3810,61.7929;-41.6554,18.5933;42.2249,77.6620;0.9078,85.1198;-86.9339,45.0765;-76.0087,-89.4644;-42.4336,-58.9809;29.9573,34.8463;20.1096,-63.2571;31.8802,-62.2847;\
    18.9149,47.9573;-77.4758,46.9668;55.6949,-17.1758;7.7838,12.8521;12.3128,-36.3352;-65.6810,-71.3295;68.8509,-52.2772;42.6591,-2.4930;73.6910,-38.8126;-12.2511,-85.1009;-0.6116,80.6204;87.0235,40.2178;-88.7377,-53.6516;-46.1781,29.8903;-84.6660,6.9018;2.9089,86.7590;-52.2912,-58.4654;75.5366,32.9670;-29.3029,85.3793;10.6085,-72.6458;74.3852,-26.5282;-6.8507,-73.4591;16.5700,42.7426;-42.5506,-87.8284;-2.1494,-43.6336;-33.0855,61.1692;-76.5945,-66.3752;-47.2396,-53.8643;-9.9452,-32.7796;\
    84.3054,82.7143;-13.8572,-8.0906;-5.2076,84.5162;68.8547,1.7035;-46.3794,19.3436;84.6349,-18.6312;33.8226,45.9883;-52.8669,-56.4000;73.0114,63.3475;4.9598,15.5844;17.6571,-61.6369;55.4708,32.4986;18.9987,70.1033;68.4103,-44.4867;64.6280,-21.7074;3.3328,34.4001;-14.6254,-18.2326;-22.2863,-38.2042;81.1070,-25.9663;0.0833,15.6521;-36.0658,6.6787;44.1297,-57.3448;-77.4983,-79.7854;-7.4273,-60.4167;53.2934,3.0174;45.9182,-21.5774;-58.6191,39.3167;-49.6179,39.6903;75.3159,-24.6568;-69.1825,28.9490;\
    -76.8058,16.9568;21.6020,-43.3597;-34.1638,-13.1989;89.1530,-87.1078;-83.3882,-17.6140;27.3030,-20.3388;45.3693,-50.4411;72.2010,-70.7647;-48.0730,87.9302;-68.1937,-57.5327;-69.5389,85.0686;-62.5240,-21.3907;41.8377,69.7396;11.8607,-17.0128;88.9959,26.0586;-81.2077,22.7997;13.3429,37.2134;-62.5191,-27.2299;28.7731,0.4058;21.2769,65.3027;-75.3686,49.9290;57.5958,-49.0927;19.2514,52.9829;-32.4258,-23.4958;20.9665,-15.5571;-64.4517,-31.4142;-29.8731,-52.8671;-23.5399,2.4014;-68.9183,-73.4371;\
    -72.1134,-68.0433;87.9769,84.8968;26.4762,-9.8283;-9.8930,66.3966;-40.7183,61.6124;-45.0040,11.0535;-19.1588,-17.5263;-72.1150,31.2387;-4.7759,86.5840;13.9830,31.4063;-43.2216,-7.3847;-8.2109,71.3976;32.7122,66.1437;35.3804,69.3373;-40.7291,28.1931;42.9514,58.9420;-88.2469,-58.2586;72.2355,79.0218;-29.5935,53.7369;-13.5002,8.6535;34.7423,-8.8411;-83.1481,69.3759;38.0170,-89.9637;5.2181,49.2668;-12.7405,-9.3940;-84.0826,-48.8690;65.6717,88.1945;41.5027,48.0569;-47.6314,-29.8668;\
    -11.0258,-38.6089;17.2088,67.1974;51.1331,43.9729;12.6056,-25.9183;41.5135,-80.3678;-79.2343,-26.9192;-31.7996,22.8366;19.6675,-46.0099;-81.5546,-35.4043;-34.5557,-28.4040;46.9367,77.8035;-36.6622,-62.0039;-85.7172,88.4026;-39.8149,-54.0734;-11.5700,44.3138;82.6940,-2.4446;-33.0984,53.2581;-21.8887,11.5581;52.9966,-41.5374;-21.4149,43.4276;25.8359,14.3735;32.6537,-86.0662;65.8811,81.9798;-64.4399,-67.1517;-77.9531,-22.9348;-47.3836,-1.3511;16.3311,-40.0210;6.1788,-77.1883;70.6378,61.7452;\
    6.7707,-21.2799;39.9871,82.5208;-11.0819,48.6063;44.3800,-86.6337;22.8701,-52.5005;-82.1093,35.7490;-85.1319,75.6206;-35.2025,-4.7458;55.2131,9.1053;68.0635,-69.0935;-53.1784,56.7414;-20.4806,11.5124;-42.0484,30.5079;-76.6457,24.5834;-18.7574,58.8501;-46.7581,34.8475;-13.2895,47.9504;89.0726,77.1128;-30.5025,53.7786;-35.2222,-89.5906;-2.5952,-42.2872;-79.2291,64.3574;-56.7326,-75.8767;71.1218,46.0781;24.6011,32.3293;61.8576,-54.9805;-46.8657,70.8657;60.0164,-24.6718;-38.2331,66.3840;\
    88.4036,22.3879;-79.7381,32.9098;-47.1318,-32.6270;22.9507,-54.2211;66.5696,44.1985;-40.1245,-65.2391;18.7754,29.3471;31.8577,-31.8939;75.8675,65.5157;-24.4475,67.5648;89.2613,53.6424;-50.5503,-4.4009;68.4456,-7.7750;7.7180,-45.4429;-7.1364,-9.0648;-50.7427,-74.0877;-48.5807,85.8287;25.3194,-4.1141;63.7140,51.3543;-24.6687,-45.3841;-77.2395,12.0709;18.2678,-0.5897;53.4537,-67.6936;-59.0078,54.6938;-11.0310,12.3898;35.2991,74.8112;-30.1894,-21.2636;-82.1988,-40.7620;18.2043,14.0635;\
    -84.0631,28.4846;-74.3157,38.5144;78.1617,-50.8601;-32.0396,-75.1855;59.3745,73.4411;-5.2510,58.2736;36.2357,-2.8902;62.9240,-2.7722;60.4933,19.0468;-28.4754,62.2558;-6.8411,51.0686;-8.0006,-37.3571;54.3741,-84.6426;-84.6556,-6.0232;-85.7499,57.8326;-20.2074,-42.6395;-13.2881,-58.2562;-88.4513,13.1711;28.4833,-83.7800;4.6037,-68.9462;24.0473,-82.4231;-6.4153,45.5103;-10.3626,-5.9519;34.7197,-86.0886;-32.2885,-24.4386;-73.7171,82.7650;51.4235,59.2335;14.8961,-2.5063;-87.6518,-71.0923;\
    5.2221,2.4668;19.7382,-18.7995;-27.2727,-35.1198;-26.6985,71.7790;-0.1723,7.4095;-31.8979,-86.9759;-52.2959,-0.7179;47.9184,-16.9405;73.2437,-32.7742;-62.6392,-48.8185;-41.8433,12.5374;-85.5792,1.7511;51.9595,34.5486;17.7644,21.9929;-17.8906,55.8973;-19.0618,73.1925;-16.2237,71.2063;-30.1949,-1.6917;53.8063,43.3650;-13.2790,36.9794;-28.3707,68.3068;70.2881,7.7610;9.5173,-10.2882;13.1667,-50.2936;-76.2800,72.2660;57.3123,27.4626;1.7518,-58.1282;-89.0129,-8.2716;-72.0565,-25.4761;43.4226,24.5215;\
    -40.7996,77.5505;30.0949,80.0905;-84.6141,45.5842;38.5835,-80.9466;-59.8997,41.0251;70.0463,-2.9155;-68.9357,51.3804;-0.3815,-74.4040;80.6141,-63.5501;76.1270,8.2882\
";

const CITIES_PTS: &str = "\
    27.0500,57.4167;13.2000,47.3500;103.4167,4.2500;-71.0495,42.3334;-67.5500,-16.9500;21.6622,41.6506;22.3850,41.4958;-87.9167,15.3333;8.8219,45.8291;27.0539,55.6339;23.5894,58.9394;144.9633,-37.8140;-9.0833,38.8500;23.5000,54.7667;17.3831,45.2550;119.5883,39.9317;13.2667,50.3833;32.4264,35.0367;4.0333,49.2500;57.5950,-20.2975;-119.2691,50.2581;42.8844,11.7853;21.0375,42.0678;38.4000,9.0333;101.3167,3.2167;104.1667,-4.1333;20.1814,39.6589;-80.9833,8.1000;22.2897,55.2522;11.9333,51.4833;6.2267,49.5819;-86.7000,14.2167;-3.8650,40.3223;36.5333,0.2667;-88.2000,13.6667;-55.5833,-34.5000;48.9203,39.9319;13.5000,43.5833;14.4333,7.0500;34.1833,12.4500;-56.3500,-27.3167;35.2833,-0.3667;57.4719,-20.2644;34.8044,39.8200;123.8556,9.6475;69.7444,40.3136;-86.5167,11.9833;-46.1883,-23.5228;36.7333,34.9167;23.8333,38.0167;105.4167,10.3833;26.2500,55.7333;21.4078,38.6214;8.6333,47.7000;139.4700,35.3419;79.9000,7.0333;34.5833,31.6667;15.0000,47.4167;61.2669,34.5953;50.6075,35.9521;43.6353,-12.2967;38.7333,9.8000;-88.5514,15.3433;-7.9333,37.0167;32.9042,34.9917;-68.7372,10.3406;103.7922,36.0564;-57.6667,-36.3333;-66.8500,-29.4333;13.9408,56.8332;-89.1514,13.7347;22.9758,55.0792;-64.8667,-24.2333;44.5069,-12.1703;-56.6833,5.7667;14.6968,57.6531;70.8778,31.9103;-84.0833,9.9500;-71.2542,-29.9078;39.3333,-10.9000;31.2500,-25.9667;143.6333,-3.5500;73.7614,43.5983;-71.1450,8.5983;26.5000,5.4000;7.1000,50.7333;46.0189,40.8297;-57.2333,-34.3000;35.3667,32.8625;45.4364,40.1350;26.7333,47.2500;7.5693,47.5402;22.9000,56.3167;-57.7167,-25.2833;-85.8833,11.5000;8.5333,7.7333;-90.6442,14.6842;-15.4167,13.5333;173.2667,2.0500;-98.2833,26.0833;36.6500,5.6500;-9.1667,54.1167;91.4667,27.5833;105.4500,12.0000;25.3167,57.9000;93.6833,22.9167;60.6125,56.8575;-83.5167,10.1000;-57.6333,-25.3833;14.7997,45.4008;4.7760,51.5866;126.4350,36.9389;24.7667,60.7500;169.9695,-46.1208;8.5246,47.3100;41.7325,42.6261;51.7833,48.5500;37.4500,9.0167;3.6500,6.8500;41.9308,34.4667;36.5167,36.1167;32.8506,35.1106;-6.9000,11.0833;\
    21.4333,47.8500;103.6531,16.0567;15.4500,47.0667;-70.9000,-33.2833;32.5000,13.7333;10.9236,43.9213;18.6708,49.6776;-71.6667,-35.4333;13.6333,32.4333;23.7333,38.0833;-104.6667,24.0333;6.0950,49.8544;11.7901,55.4426;100.1167,13.5833;44.4858,42.1242;3.6333,32.2667;-91.2290,14.6382;6.4833,50.8000;-43.3117,-22.7856;-72.2519,8.0378;-21.9167,64.1000;32.9014,35.0414;-9.0611,52.4492;33.8833,-4.2833;147.8500,-6.6000;-87.1500,12.6167;6.0733,49.6206;100.1625,49.6342;-67.7922,10.2625;1.7000,46.8167;23.1500,53.1333;-89.1333,13.6408;-9.7100,29.7100;130.5167,33.3167;20.9136,41.9236;-87.6167,14.7500;-54.0500,5.5000;-20.2333,63.7500;16.6373,57.7584;39.7269,41.0050;22.2167,45.4167;57.3244,37.4775;69.1192,34.7489;17.4500,6.4833;17.6706,44.2267;-88.5500,18.0667;127.4889,36.1031;35.5833,6.9833;49.6583,58.5969;138.3833,34.9667;23.0167,40.5500;19.0833,47.3500;-79.8889,9.3314;-111.8990,33.5092;35.3222,32.6786;69.4017,37.4922;-73.0498,-36.8270;32.5000,35.0644;40.9344,57.7708;153.3333,-27.9833;-88.6000,13.4333;48.1017,29.2253;100.5667,5.3667;-9.2000,38.6667;41.4339,52.7317;-171.5167,-14.0167;35.7450,31.3144;-69.7894,9.7861;34.9503,32.2328;48.2167,11.1667;34.6167,-1.1833;153.2773,-28.8135;8.5167,11.9964;43.7828,33.3561;96.2167,19.7333;39.1700,51.6664;26.1333,44.4667;19.8883,41.2864;123.4328,41.7922;-73.0833,-40.2833;-14.1167,13.5333;129.6700,41.4339;-83.8500,9.8000;-86.0500,15.0500;-9.4161,38.7028;24.3333,54.3667;-88.4000,15.3833;174.0102,-41.2883;153.1167,-26.6833;5.8961,35.3767;-72.0147,-17.0231;-79.7167,8.9167;22.0892,41.4839;91.6667,25.0333;-72.2279,-39.2857;77.2167,28.6667;31.5333,-16.9167;67.6150,39.5031;-63.1667,-17.8000;10.5366,57.4407;33.7094,1.1450;-15.9500,66.4500;28.6572,46.3003;6.1539,49.5628;125.3228,43.8800;-67.6236,5.6639;44.4883,40.8128;-84.9667,10.4667;52.9950,51.1678;24.8333,54.5167;83.5133,50.3550;6.2811,49.5050;11.9160,57.6919;-172.5833,-13.6500;65.3792,35.9783;139.0500,37.9167;-110.6683,50.0501;44.4372,-12.1556;-14.5333,13.3833;-13.3167,15.6000;49.7667,-16.1667;13.1333,47.9000;\
    -77.1167,18.2667;36.8667,36.5167;-82.3000,8.5333;45.3475,40.5953;-71.9658,46.0501;28.0472,47.6564;-110.9667,29.0667;23.7333,37.9833;30.7167,-27.9833;21.7864,40.3011;37.2825,41.1314;29.5133,40.0781;26.2000,-29.1333;21.4831,55.3489;7.0333,5.4833;39.5000,-4.4167;28.9078,64.8868;79.9042,6.7133;44.5936,40.3147;91.3667,24.5167;-89.3000,14.1333;4.3667,51.0667;44.6917,41.6608;-89.1172,13.5697;-64.1000,-32.1833;134.7000,34.8167;13.5706,45.5278;27.6500,-26.3167;-51.0664,0.0389;-90.3003,14.2764;-172.6000,-13.4667;48.4153,38.7753;16.2900,45.4375;101.0833,13.7000;-89.8275,13.5928;17.5500,8.9167;65.5092,44.8528;4.3417,51.9125;17.6454,59.8585;-76.6456,-10.2411;10.2333,6.3333;-57.1500,-25.9167;6.2686,49.6206;5.8097,51.0317;64.8339,32.0733;16.2833,48.0167;13.8722,45.7069;9.6167,36.6500;33.4625,34.9500;49.6964,33.3836;25.6061,45.6486;73.8500,31.0167;-78.6328,21.6967;12.0167,31.9500;25.5833,55.9667;152.1833,-4.2000;92.7917,56.0097;16.2667,48.1167;120.8969,14.9547;33.8667,-11.0167;-171.5500,-13.8833;88.1667,24.6833;-8.9167,38.6333;10.5939,35.8589;-68.6333,-15.7667;-72.7706,18.4289;5.5000,50.9667;-77.0364,38.8951;77.0833,43.8833;101.0167,4.0333;68.5108,43.5219;4.3292,51.8450;7.7500,49.4500;151.8733,7.3861;70.7903,48.0078;-66.8167,10.4333;24.6167,43.4167;-0.6417,35.6911;33.4500,-8.9000;72.9333,0.5333;-0.3333,51.7500;79.9333,6.9333;-6.5600,32.8600;22.1142,37.0389;107.4389,-6.5569;144.4500,-37.6833;41.6925,13.9297;7.0741,6.2101;-21.3000,64.6667;44.6456,31.5822;5.6500,50.8833;32.4589,-25.9622;123.5833,-10.1667;46.7333,-19.1667;48.8189,40.0986;-76.3333,-13.0833;48.5256,5.3514;-83.7000,9.3833;14.2102,42.4602;19.1014,44.3869;80.0000,44.1628;-5.9333,54.5833;107.3000,21.0167;-18.9167,66.1500;-67.3833,-14.3167;33.8828,-24.7117;-87.1667,12.5333;43.4461,-11.8258;-71.5525,-33.0229;21.3667,40.9833;16.0142,45.0847;23.3500,54.5667;174.4333,-36.6500;135.5683,34.8164;46.1167,-22.4000;-16.0150,13.3289;10.1017,35.6744;-22.5667,64.0167;-88.2500,13.5333;74.8656,31.6331;18.1333,47.2000;5.4697,10.4003;19.0500,47.3500;\
    43.3617,39.0239;44.3803,14.2972;27.5144,47.7708;24.3106,59.1142;38.4167,6.6000;23.7167,37.9500;89.4167,27.4333;29.0011,47.7664;13.6642,-8.5800;-80.9061,22.5333;27.7958,47.2042;43.2592,-11.8081;-84.1000,9.8500;50.0781,33.6436;21.0667,51.0667;10.5683,35.8708;19.6833,47.6667;31.3833,-10.1667;26.8667,-18.6167;-112.8186,49.7000;18.3167,47.6500;23.1333,-15.2500;21.7167,63.1167;-79.3022,-7.7333;-77.6444,0.8303;6.0422,49.4758;-72.5491,46.3501;6.3028,49.6625;-92.2833,14.9000;21.0647,55.3719;-88.6000,15.7167;73.7333,32.9333;25.8500,56.4833;16.9769,45.3406;91.2500,26.7833;26.2500,50.6167;51.8833,47.1167;10.2753,36.7681;19.8222,41.3953;36.3667,33.5667;18.4631,44.8672;24.5308,-32.2522;-8.6231,52.6647;73.1069,51.6214;-56.4667,-30.4000;-89.0278,13.9381;12.4833,-1.1333;38.2833,34.5500;72.9667,19.2000;31.4064,-0.7425;26.4211,58.2225;34.2861,0.0600;171.6500,-43.6333;-74.7036,7.0833;25.3833,54.3000;14.2833,27.5333;9.6099,47.1014;16.5000,43.5500;-8.1833,53.6333;9.1895,45.4643;-63.7333,-35.6667;89.8833,27.0667;-58.6167,6.4000;19.1167,17.9167;-46.7917,-23.5325;-57.6333,-32.6833;-6.2181,53.4597;26.6333,44.0833;49.1000,-18.9500;45.3881,36.9564;35.5333,8.1500;-72.9500,-36.6167;6.0561,49.5092;12.4537,55.6806;153.1000,-30.3833;139.3597,35.4389;100.5500,7.2000;-1.6500,5.0000;8.7995,46.1709;43.9064,40.1775;100.1333,18.1500;-79.9000,8.7500;-72.5133,7.8383;70.1333,28.3000;12.1179,57.6589;20.1894,40.3131;8.8333,3.7167;34.1417,35.4708;147.0000,-6.7333;140.7667,-37.8333;79.9667,14.4333;72.5553,29.7122;-88.0167,13.7000;-53.7833,-33.2667;77.9822,45.2422;121.1739,13.9394;7.3667,5.1167;-125.2446,50.0163;-0.9833,5.9167;-14.8333,65.7500;-72.9833,-41.3167;4.6555,52.1292;-68.3333,-34.6000;106.1478,-6.1128;16.5987,38.8908;27.4261,38.6131;-74.9403,20.6569;27.8636,52.2286;18.9553,69.6492;28.0408,59.4589;23.9167,4.9500;38.9333,15.3333;2.0833,14.3167;23.7000,52.1000;123.4797,7.8264;15.1500,4.1333;50.8400,24.7483;106.0667,30.8000;74.1167,32.4500;-88.2000,15.6667;-7.4131,52.3492;39.0167,35.9500;43.3244,14.5161;6.1347,49.8519;\
    -11.3686,6.7533;33.8917,35.2875;99.1225,18.7422;43.3067,-11.8331;-69.3228,10.0739;-77.2644,21.5453;42.3567,42.2042;67.7833,26.7333;12.5167,55.9333;100.3333,5.4167;-6.1519,53.4872;-171.6167,-14.0167;150.8000,-2.5667;9.2500,-1.5667;25.3000,46.3000;22.9719,44.8033;33.8833,13.3167;45.5464,33.7436;7.0964,13.4917;19.1323,50.3237;-91.8667,14.7000;38.0167,11.8500;35.5333,32.9667;16.5000,50.8500;29.1614,47.2144;46.6167,4.6833;28.4833,49.2333;18.4833,-33.5667;33.1206,48.6697;39.0333,9.1500;-16.7000,14.7667;102.3667,3.9333;-88.2333,15.1333;103.3333,3.8000;33.2458,35.0542;-86.1500,11.9167;24.8000,-13.6833;19.4500,51.0667;66.7889,37.0356;-172.1167,-13.6667;-72.7167,19.9333;104.7458,-2.9167;116.0500,5.8833;-89.4989,13.7436;17.8597,44.6064;20.9808,41.9672;4.8833,52.0850;24.3664,52.2161;44.6089,39.9458;15.7903,44.9733;10.5333,52.2667;21.4275,41.9547;-94.5622,39.1300;90.9733,23.8917;78.6833,10.8167;-81.3667,8.0667;-8.5669,53.1969;121.0453,14.8139;-79.3500,-3.9833;46.6550,39.0822;18.2997,59.4794;4.3833,50.9333;-72.6333,-39.3667;25.4244,57.5359;39.4333,10.6000;17.5667,47.1000;136.6500,36.5667;-8.5833,41.1167;3.5333,50.9833;105.2181,21.8233;16.3119,46.3858;15.4428,12.4742;34.8500,-4.5500;12.2425,45.4903;27.2167,52.0500;8.8667,54.9331;89.1167,23.9167;-87.4875,13.5361;9.5000,51.3167;26.3333,55.5333;-6.3478,53.7189;6.5003,49.7100;9.6667,56.3833;32.7333,-5.0667;12.3333,55.7500;87.2167,27.3667;50.2450,30.5983;10.3167,5.4000;16.8992,44.8900;5.6833,36.1500;101.4500,3.0333;34.5833,8.2500;23.7000,54.1000;18.3833,9.1500;16.9706,49.9653;-8.8882,38.5244;47.8458,40.9814;17.6092,43.8222;-82.8333,8.7833;42.1903,37.3272;-85.9333,11.4333;8.2000,53.1667;-82.8667,8.8167;36.2500,50.0000;104.3167,10.4833;22.0472,40.8006;45.8358,13.8383;6.1495,62.4722;35.1000,33.0667;106.7914,-6.5897;-63.2654,45.3668;113.5500,52.0333;19.1333,47.6333;101.5333,3.1500;-104.3188,19.0501;6.0358,49.6683;20.9603,41.7058;43.2833,15.0667;32.3167,14.0000;58.1825,23.6803;14.2833,48.2000;13.1600,11.8450;100.4833,5.1667;44.2142,40.1689;46.2383,38.9022;\
    4.5167,50.4000;-74.4167,18.5667;75.7667,11.2500;-96.6389,32.9126;-78.1667,18.4500;-73.8119,18.1667;110.4928,-7.3319;6.0294,49.5364;34.8333,-8.8500;72.9667,3.1000;9.6000,0.9833;150.6667,-10.6167;115.2167,-8.6500;10.2833,6.2333;69.1667,41.4500;112.1500,-8.1000;-87.0333,12.5667;34.6167,31.2833;-0.6167,44.8000;21.5306,41.9158;18.6914,44.2264;34.7833,31.2333;102.1167,2.3500;27.8333,44.3833;23.7833,46.5667;-89.7456,13.8414;24.6667,14.2000;70.0839,36.4397;-22.7000,64.0500;-103.2333,20.6167;-82.4333,8.5000;68.2303,38.5108;18.1497,44.4458;14.0400,57.1860;-80.2731,-3.5031;7.5167,51.6167;23.9167,46.1833;72.1500,21.7667;20.1706,41.2781;23.7000,43.1500;18.0922,42.6481;23.2833,45.0500;33.6442,-25.0519;-76.4272,20.3667;-80.5167,8.0000;36.5833,34.5167;25.4022,-33.7653;57.6953,-20.4189;-90.2667,15.4833;9.5451,56.1697;30.1833,-19.3333;80.0000,9.6667;42.8225,33.6367;16.2000,51.4000;-87.7000,14.3667;7.0444,13.5706;0.5167,51.2667;69.2200,33.5900;104.8667,21.7000;29.6267,-4.8769;-117.0167,32.5333;36.4000,33.5833;17.8167,51.6500;-21.9500,64.1500;6.2556,49.5853;23.9333,43.4667;171.2500,-44.4000;6.1169,49.7583;51.6500,43.2000;27.7333,11.0333;17.7028,45.7033;44.2067,15.3547;-72.5053,7.8833;90.4058,23.8900;19.6653,41.0764;10.7667,3.6500;113.3064,33.7408;85.1167,25.6000;-90.5317,14.4814;-7.9217,52.3769;129.7508,42.4428;5.9564,49.5825;21.0122,41.0889;-85.5167,12.3833;35.8331,40.6533;35.6000,-0.1667;-85.3833,13.2333;120.9425,14.4578;153.5667,-28.8667;57.6431,-20.0350;22.4333,38.9000;14.1667,8.4000;-9.1500,38.6333;-72.2250,18.5750;73.3833,5.7667;60.8728,29.4978;-78.1167,0.3500;33.4000,34.9917;8.4772,44.3090;128.6597,35.1494;47.6061,33.5350;-58.0192,-33.2558;-78.6167,-1.2491;80.4167,5.9667;-85.1289,41.1306;15.5497,38.1933;43.4811,-11.9208;-72.0947,18.6519;37.4833,11.2667;24.6093,59.3520;29.4500,-23.9000;24.8211,59.5114;18.8528,44.2986;-84.4333,10.3333;-79.2042,-3.9931;-63.5497,8.1222;68.4269,33.5536;-14.3167,13.2333;-71.9533,18.6747;34.3833,11.8500;95.4000,16.2833;-84.1000,10.0167;23.0322,54.6517;72.7644,42.9472;\
    -172.6667,-13.4667;139.4853,35.9086;128.3972,35.9889;43.7408,37.5744;8.5804,47.3733;-16.8333,14.1667;-55.7000,-24.5167;-7.7500,41.3000;8.5438,47.3922;-99.3514,19.3603;-87.9667,15.2333;-66.9167,10.5000;-68.5631,10.1550;143.6500,-6.1500;3.1227,50.9465;125.6900,39.1983;12.6004,55.6303;88.2833,24.6000;-0.6167,5.3333;96.0017,16.7053;34.4217,0.6769;5.4292,51.8867;37.8789,40.9847;-6.9667,38.8833;-79.7217,22.1039;23.5994,38.4636;51.6583,29.6183;-7.8506,51.9500;6.1292,49.5403;98.9817,18.7903;14.0450,37.4877;38.3500,-6.6000;-80.2167,-0.9333;16.1200,46.2231;100.4833,13.8000;44.9936,41.1725;59.6119,36.2958;-55.8000,-25.3667;16.7667,60.6167;-68.0667,-38.9500;-59.5833,13.2167;-89.7147,13.7342;73.3000,28.0167;28.8500,-19.6833;0.7143,51.5378;-99.3131,19.6219;123.8494,10.2447;89.6000,27.4833;-51.2300,-30.0331;45.5667,3.8500;29.7833,-19.0333;29.4711,46.8306;22.2000,13.4667;-67.3333,10.2333;23.1333,52.0333;-8.7425,51.7469;57.7122,-20.1111;40.5389,-2.5228;104.6333,15.8500;-2.9667,56.5000;24.5328,53.4078;17.2297,40.4761;7.6000,47.5667;-123.6860,48.8413;-88.0167,15.2833;128.7378,35.8233;3.3167,14.3500;-0.4663,52.1346;-7.2333,40.5500;-67.4672,7.8967;-15.3333,13.5333;144.5833,-37.6833;9.7833,34.8667;34.5667,49.5833;-55.6000,-32.7167;-14.9500,12.8833;30.9500,-17.3833;22.4442,40.6267;105.7244,9.2850;31.0167,-26.4500;25.3500,65.1833;13.5833,-1.6333;16.0756,45.7125;16.2167,48.3833;-85.8667,11.2500;-55.5500,-27.0500;-57.0500,-26.0167;11.6500,-1.6333;13.1569,55.3751;7.0833,51.6500;-80.3903,22.2417;34.7122,38.6250;153.0333,-27.2000;-103.5000,25.5667;18.2667,46.2000;39.2833,-6.8000;106.1333,-2.1333;-72.0161,19.6908;-91.4781,15.8319;-16.5233,13.4928;140.1167,39.7167;11.1500,54.8333;-84.9833,12.0500;51.3669,26.0217;136.8000,35.3000;101.4542,47.4750;-82.9667,8.8333;-1.5122,52.4066;101.9678,6.0267;3.2250,35.6350;-72.3500,-35.9667;132.4500,34.4000;68.4939,35.9075;27.9833,-25.6167;-78.1000,18.4000;-70.5889,-14.8864;-89.9833,14.6333;44.6244,40.2692;21.4410,37.6751;122.9789,10.6311;45.0500,-25.1667;-51.9386,-23.4253;\
    126.8889,35.8017;4.0713,50.5790;17.0333,51.1000;35.4828,33.4075;-71.7500,-35.6000;-83.5333,8.9667;-84.0333,9.8833;101.8500,2.9500;2.2500,48.9167;8.7667,3.3833;-76.9161,20.5603;9.4174,55.0443;106.2708,45.7625;18.9333,47.3333;-75.2322,4.4389;35.3794,32.9653;34.9533,40.5489;-72.5000,19.2833;-60.2833,-38.3833;-171.4833,-14.0167;-14.8833,13.6667;25.0500,54.6500;-6.2925,52.6747;26.8000,60.9500;57.4656,-20.4872;57.6119,-20.0133;-54.1035,3.6721;22.1814,55.4906;57.7311,-20.3631;-88.0333,15.7667;29.9167,50.0833;0.9833,8.5667;54.5444,28.7519;-87.7000,15.3167;49.1314,14.5300;17.8500,59.6167;57.5211,-20.1350;5.4497,43.5283;145.9167,-41.0667;20.8508,39.6675;-87.1667,12.6667;-69.6672,11.4092;-15.5167,13.4667;21.7675,39.5553;40.3981,56.1428;105.3067,-5.1131;15.8140,50.4317;31.9333,-26.8167;6.2889,51.9650;96.4333,18.4833;25.8500,45.0333;15.9772,45.7576;35.6408,31.1333;-77.9500,18.3000;-87.1333,15.1333;28.8042,55.4856;-72.5450,10.0644;-72.2853,18.5125;5.8914,49.5242;20.9333,47.3167;34.9333,32.3167;47.3833,-20.2333;7.1136,35.8775;-72.4667,19.0500;28.4167,64.1333;-88.8000,13.6333;-4.1942,14.4930;-89.2064,13.7456;25.3211,59.3361;149.5500,-6.2167;150.1500,-5.5500;27.0667,-18.3167;-90.0751,29.9546;-15.5667,13.4833;4.5333,36.4611;57.5333,22.9333;14.6870,49.7816;-4.2500,55.8333;11.0680,60.7945;38.1333,10.1667;44.7758,32.9256;31.8747,-3.0236;135.3570,34.7994;34.0289,38.3742;-73.0500,19.6333;26.8675,59.4506;69.6000,42.3000;20.5961,41.2686;-72.4824,45.8834;-8.1000,37.0667;175.8500,-37.0167;155.7333,-6.8333;23.5650,37.9681;-82.2500,9.3333;99.8833,13.8167;19.4667,51.7500;117.0950,-0.5830;26.5667,48.6667;23.5500,42.3333;-59.2667,-29.1333;95.8000,17.6500;37.2333,34.2333;-7.8000,37.0667;-101.8552,33.5779;-5.2100,33.4300;68.8311,37.7108;119.3061,26.0614;0.9136,35.9633;35.7317,32.5703;14.6589,45.9556;-61.9167,-37.4667;11.5333,4.1667;126.8219,37.3236;45.3303,41.7364;5.4861,52.2200;41.4500,9.3167;-57.2667,-25.3833;21.9428,41.5775;39.1000,-5.0667;9.4722,55.4904;120.8117,14.8419;44.2925,40.1728;30.4167,19.6333;\
    20.0222,41.4381;20.7833,48.1000;2.1732,41.3802;10.3247,36.8782;174.7667,-36.8667;57.6122,-20.2447;43.9367,42.3997;4.0833,48.3000;102.5842,17.8450;17.8081,43.3433;31.8833,51.0500;37.4500,-0.5333;15.8314,46.5761;-79.6500,-3.6667;74.1833,32.1500;144.2833,-36.7667;-16.2667,14.2333;7.0167,51.4500;24.6667,47.6500;16.3783,45.4661;16.0488,48.8555;43.8964,40.7814;32.5892,-25.9653;71.6117,36.7272;80.4500,16.3000;-80.1625,-5.0925;2.2000,48.8833;27.4500,55.7833;19.5672,41.3456;-81.1144,-5.0892;25.3333,42.2000;90.2333,24.7667;26.0833,41.8333;105.5967,21.3100;25.3497,58.1253;30.6000,-7.4167;8.5833,10.3833;28.0833,-12.8333;152.9167,-31.4333;29.0611,40.1917;100.0500,13.8167;24.5153,58.6058;21.0778,42.1033;88.3103,22.5892;30.2919,54.2108;-72.4333,19.7500;-88.3000,15.0333;16.2000,44.0439;45.2803,31.3094;129.6689,41.6697;16.9667,52.4167;-60.2833,-36.9000;13.1000,47.6833;9.3681,4.0786;22.8333,62.8000;15.8697,46.4214;-1.4000,50.9000;5.4750,52.5083;24.1833,65.8500;4.2667,50.8500;27.2667,44.1167;25.4785,42.9790;49.1064,39.4267;14.4400,10.4028;-0.4667,51.7500;-10.8333,13.8000;-8.6333,40.8667;49.6686,40.5897;23.0333,42.8500;-73.4000,18.2833;48.2958,30.3525;30.0000,0.2000;27.2889,36.8933;66.4200,35.9489;32.7333,13.1500;19.7511,41.0461;102.0333,15.8000;8.5217,47.4233;47.8000,-18.9167;14.2258,45.9144;51.5689,35.4394;-88.4667,13.4833;-50.9919,-29.9444;-57.3500,-25.4833;96.5631,16.9056;123.2856,10.9506;-78.5333,-1.3167;97.7164,16.2578;127.0042,36.7836;24.7142,57.5103;-14.8833,13.7000;80.1271,6.4809;133.7667,34.5833;-83.0458,42.3314;-56.4022,-34.4117;17.6483,43.0542;-6.1333,36.4167;16.9910,46.4535;-100.3000,25.7500;13.8481,44.8683;32.7667,-9.3333;-1.9333,7.5833;30.9667,-17.5167;26.2167,56.8500;103.4167,3.5000;4.4667,51.0333;36.1500,32.8167;14.2667,32.6500;-79.9833,-3.2667;20.1556,41.4661;23.2333,54.4000;30.6286,-0.5917;1.4983,35.1914;-103.4333,25.5500;68.6167,38.7333;152.2667,-4.3500;-85.8000,12.4167;91.4167,24.3833;17.1324,48.8489;13.7667,50.5500;-74.0036,45.7804;13.2167,47.4167;6.1392,46.1810;-88.6667,15.3000;\
    33.2667,-9.7167;36.6333,35.4333;-56.2200,-34.7264;69.0431,38.3131;-41.3000,-21.7500;25.4133,53.4881;134.9833,34.6333;25.8333,-13.4500;42.3739,42.4153;32.5167,34.7333;20.9267,41.9444;77.2328,43.3044;25.3167,42.9000;53.0586,36.5678;-84.0333,9.9167;151.9000,7.3667;7.6257,51.9624;85.9333,27.9667;-8.2758,52.1358;29.0875,37.7742;23.8667,55.9667;5.1167,7.6500;16.0833,8.5667;18.7153,63.2909;65.9833,35.4167;22.9508,40.6494;15.7828,43.7608;15.8611,11.8094;31.7167,-25.9833;120.9389,14.6753;9.0217,33.4575;35.9333,32.6500;27.5333,52.8000;25.0250,59.4767;9.0803,35.5456;36.6500,-1.1000;23.1000,42.0167;7.8209,51.6803;-56.4000,-25.1833;44.3592,40.3017;5.6333,50.6667;12.7000,-0.8167;3.7167,51.0500;79.8947,7.0844;25.5900,58.3639;72.1758,29.8003;32.9500,14.3500;126.2158,6.9528;-87.8333,14.6000;55.6717,29.4514;24.5486,57.0936;-55.0500,5.8833;20.4833,53.7833;49.8167,7.9833;-46.8356,-23.5225;33.3667,35.1667;-70.7500,-33.8167;19.9167,50.0833;2.3000,48.9000;20.7808,40.6186;9.2500,36.4667;-78.2333,9.3000;30.1333,-9.3833;13.6181,50.6042;44.1464,34.0164;-44.0867,-19.7669;18.5814,45.5828;-21.2000,64.0000;10.8167,35.5667;14.4500,48.3500;5.7181,52.5250;-8.8000,39.7500;-15.7000,13.5833;-78.0833,18.4500;26.5333,55.8833;27.8167,-26.7000;13.6411,45.9275;15.7333,-17.0667;-78.8047,-5.7081;4.6667,7.9167;17.9103,44.9772;35.2131,32.8667;8.6494,49.8706;24.3333,60.4167;90.7167,24.8833;2.3000,48.7500;-77.6167,21.4833;140.4667,36.3667;-7.8500,42.3333;31.8500,-18.1500;12.3109,55.9267;19.7022,40.7375;24.8000,-14.7833;125.0828,39.6875;140.3375,38.2528;5.7806,14.7572;26.4833,-18.3667;14.9633,46.3394;15.4371,60.4858;39.7631,37.2353;-82.8667,8.2833;73.0000,5.6667;4.5488,50.4814;35.4208,32.6889;33.6131,0.7697;13.4477,50.4998;105.8603,20.0781;-2.9253,43.2627;17.9479,58.9034;45.1492,40.8756;151.1833,-32.5667;-61.6167,12.1167;57.6467,-20.1639;35.6500,-15.8000;29.1608,47.2631;29.2667,52.0500;14.1475,49.3088;31.8500,-26.0333;-16.9667,14.7833;18.6667,54.3500;26.3292,42.6858;-80.5572,22.3408;-172.6167,-13.4667;55.7500,24.2500;\
    12.1405,54.0887;-66.2333,-17.7167;-55.2250,-34.3700;57.6633,-20.2369;-77.1048,18.4025;68.8167,37.6000;22.5039,58.2481;71.7314,34.1453;-99.5000,17.5500;19.4167,51.8500;5.4667,7.4500;-75.9000,-11.5333;-8.5667,41.1833;-101.3500,20.6833;17.5167,49.2167;37.0693,-1.0333;106.1333,11.2833;4.9722,52.3075;16.3000,9.4000;35.0000,1.0167;42.8628,41.6186;21.6150,41.9389;43.1156,42.0958;-121.2908,37.9577;51.4244,25.2919;21.6333,47.5333;24.9483,59.4364;90.4086,23.7231;39.7167,-5.0667;32.4564,1.3089;-72.3000,-36.7333;149.1000,-33.2833;20.6233,41.1642;128.7489,35.4933;-90.4000,15.1000;44.5192,-12.3322;15.4894,45.9531;50.8385,28.9684;121.4156,14.2786;-71.2261,-14.2694;14.3617,46.2411;-89.1094,13.7017;-76.2631,20.8872;6.6833,51.2000;44.0869,41.5914;96.5833,17.3000;20.6667,46.5667;24.9500,44.9500;14.2861,48.3064;-0.5833,32.7500;0.1500,34.8333;35.9375,32.3506;-79.6333,8.8833;108.8000,15.1167;14.5667,46.1333;2.9108,36.5706;-172.1333,-13.6000;-81.2514,23.1536;25.4500,44.9333;10.2045,59.7441;29.2000,-19.7833;126.9156,35.1547;-7.7333,54.9500;149.2000,-21.1500;57.5072,37.0733;-113.8020,52.2668;101.7500,3.0333;32.9097,2.2350;23.5795,47.6533;103.6167,-1.6000;16.0333,47.7167;134.0500,34.3333;14.4197,46.2456;21.5619,42.0264;84.9667,56.5000;-80.4167,-0.6000;128.1597,40.0247;-71.9164,9.0042;-96.6670,40.8000;-97.1000,18.8500;-54.3833,-33.2333;-80.9667,-2.2167;36.2500,33.4500;87.5833,43.8000;-9.0489,53.2719;18.3442,42.7119;10.0333,37.1667;16.3167,48.0667;-2.0000,50.7167;24.0233,52.2006;-81.5190,41.0814;174.9500,-37.2000;44.7675,42.1158;-15.1833,13.6167;14.6182,50.9115;24.3000,42.0333;-8.5000,41.1833;-72.4500,-38.5167;33.3000,34.7667;139.7172,35.5206;-82.8333,8.5333;4.8333,50.9833;13.6550,45.5322;48.7086,38.9278;68.3667,25.3667;98.6667,3.5833;48.0014,29.3650;20.1981,41.4056;112.9667,28.2000;-57.9489,-34.9314;-84.3880,33.7490;13.8451,58.3912;-69.1261,10.0800;35.7000,32.4000;5.6325,-1.4014;14.0458,46.6161;-76.3028,3.9022;4.3000,50.7667;-56.5006,-33.4131;28.9775,46.9131;35.1450,32.9900;103.8833,1.4500;21.1169,42.1433;\
    -87.2833,15.1667;-35.2094,-5.7950;34.5333,0.1000;-84.1500,9.4500;14.0833,5.6000;-0.1255,51.5084;131.1539,46.6361;-6.4625,53.5081;122.9558,14.1128;-6.9000,32.8800;-55.1833,5.5833;79.9083,6.9028;16.6147,46.2519;101.6142,56.1325;-119.0187,35.3733;9.5420,47.1182;123.4178,13.4250;47.9725,34.0734;-75.6214,4.8681;57.5442,-20.4189;129.6061,41.5878;4.6306,36.0667;34.5333,-4.7500;101.0833,5.7500;78.8239,44.8300;116.8547,-20.7291;15.2333,52.7333;-49.0661,-26.9194;14.8721,48.1229;-80.3833,-1.2167;-21.7667,64.5667;49.3833,-18.1667;101.7000,3.1667;26.7500,57.1833;17.6471,49.0251;28.1383,38.4833;6.5569,9.6139;116.9972,36.6683;-23.5333,66.1167;-9.7600,31.5100;-171.5667,-13.9000;26.1358,38.3678;30.3911,-5.1036;28.1075,-25.5072;69.1814,43.7650;34.7500,31.7833;-16.0911,13.4894;120.5792,15.2233;80.6167,16.5167;-77.6000,17.8833;12.5038,55.7704;-82.8000,21.8833;-16.6333,15.1167;-10.6125,6.4461;-115.1467,36.0972;-14.2167,65.0333;-63.4667,-18.8333;-76.4958,3.5850;37.7167,48.5333;21.3136,42.0117;34.7500,-4.8167;23.9683,54.9153;47.0650,34.3142;-1.4261,35.0611;57.5453,-20.0547;44.6836,39.8406;99.9667,12.8000;-16.9167,65.6500;-66.1500,-17.3833;28.3081,47.2544;44.5853,42.6072;10.7896,55.3127;41.1200,44.9800;34.8942,50.3086;20.7000,-11.0667;12.1522,-15.1961;-80.0997,43.9168;-6.9667,54.2500;114.3483,34.7911;106.2500,22.6667;91.5667,27.3167;-14.0833,13.4667;36.4758,45.3583;1.4531,14.2117;34.7667,-9.3333;-9.2300,32.3000;128.6553,40.3231;-14.1000,16.2500;36.5500,33.8333;2.4000,47.0833;3.2833,7.4333;-73.1333,-41.7667;-9.2500,38.7500;-76.2833,18.0333;73.0667,30.2833;69.6400,34.8500;153.1333,-30.3000;10.1500,56.1000;12.0833,50.8667;24.0333,54.6167;128.4733,37.1872;5.2141,52.3703;14.2500,40.8333;120.5936,18.1989;34.0467,37.5133;-2.7500,47.6667;-171.8167,-13.8333;2.3833,48.8167;53.3167,47.6500;-79.5500,9.1167;13.7664,56.1591;103.3833,1.4833;20.0167,51.5333;4.7167,51.0833;-0.9147,41.6477;33.4686,0.6092;-86.1833,13.3000;39.7667,9.8500;22.6833,63.6500;57.5439,-20.2283;20.5833,47.8167;129.8683,32.7550;-80.0461,22.6450;\
    20.8064,52.1716;143.2000,-9.0833;163.0239,5.2886;72.9469,44.2914;15.2286,45.2661;85.2667,26.7667;14.4519,12.5744;-2.5833,7.4500;-84.2667,10.0333;30.0281,52.8922;-123.9360,49.1663;-84.0833,9.8833;-75.9356,20.6439;10.6667,35.6333;23.3667,45.8333;48.8167,38.5328;71.3000,29.8833;69.1833,34.5167;11.8818,45.4152;30.0833,-20.2500;-86.0000,15.9167;26.9372,54.8686;39.3500,8.1500;23.1217,55.3736;35.3167,29.8000;12.1972,55.7683;21.6786,40.5147;12.4000,55.9667;7.4833,6.4333;9.9410,57.2702;-5.6667,11.3167;25.3500,13.6333;-79.6167,-1.0500;-13.1667,15.4833;-0.4667,46.3167;24.8050,56.7225;44.4833,-12.1833;16.3378,46.3044;9.5754,47.1127;-91.2333,14.4333;15.6000,48.4000;37.7067,36.1536;23.1278,57.0422;-4.8858,36.5154;33.5333,14.4000;-172.1333,-13.6167;-14.0167,64.9333;4.3500,51.1667;15.7375,12.9981;-77.9333,18.4000;25.3219,43.2428;106.5564,-6.2658;120.1253,33.3856;-16.7339,13.2019;127.0694,37.1450;-4.1430,50.3715;-58.4500,-34.5833;-34.8631,-7.1150;-63.2333,-17.1167;18.3283,44.1644;100.5167,5.7333;36.8167,-1.2833;18.5833,4.3667;127.4197,36.3214;24.5136,58.3747;-79.9667,-1.9333;14.3093,48.0434;36.9833,36.3667;20.3833,47.9000;40.4500,35.0167;12.2690,57.7705;21.0833,47.9333;-5.7000,34.2100;22.2500,32.7667;127.4058,39.6503;-44.0536,-19.9317;21.9217,39.3656;-7.7169,4.3750;148.2333,-8.7667;-57.1333,5.8833;37.8667,6.9667;12.4279,55.8476;3.3858,11.8614;29.6833,-16.8167;-83.9167,9.8667;15.4258,46.3367;140.8847,38.2547;15.5606,-12.8525;31.0447,-17.8178;30.8500,-17.0333;21.0000,52.2500;35.8861,32.5950;12.3267,45.4386;-171.5667,-14.0000;35.8167,31.9500;-73.1667,-37.0833;120.6856,15.0303;146.6500,-7.2000;15.6486,45.6683;43.2600,-11.6283;146.5333,-38.1833;74.5167,32.5000;-63.8333,-17.4000;16.0442,46.6472;11.9668,57.7072;107.6167,51.8333;73.1185,6.7018;-7.9133,53.0914;-8.6667,40.6000;26.2614,59.4336;89.7167,25.5500;-87.9065,43.0389;33.1667,-11.2000;44.3425,-12.2319;5.9108,49.5775;18.1781,43.9889;6.0314,50.0547;7.0833,4.7333;21.4295,38.3702;9.7500,37.1667;28.2269,-25.9989;22.5833,57.0500;-72.2000,19.4333;\
    6.2233,14.5539;-1.9100,34.6800;35.2333,31.7667;12.6944,56.0467;15.0308,12.0783;47.6000,-23.3500;-56.2500,-26.6000;5.0000,34.3833;43.2806,-11.6536;64.6667,45.0833;68.8600,36.7200;-84.8384,9.9762;68.7667,27.5333;105.3383,20.8133;13.8204,55.4297;44.3703,-12.1917;20.1375,41.4892;21.5667,56.0667;51.5058,25.6839;62.9297,53.3906;121.0600,13.7594;91.9667,23.1000;7.6279,47.0590;43.2953,-11.4181;33.7833,-13.9833;62.6000,30.1500;29.8167,-18.9167;-0.4500,6.0333;13.3776,49.7475;139.9833,35.6931;21.4000,47.4500;174.5167,-36.2833;102.8667,-3.3000;-23.1500,66.0833;-20.3000,65.6667;120.6181,31.3114;11.9805,57.8710;18.2067,43.8222;33.2000,35.1542;120.5908,15.3275;136.7604,35.4229;114.4942,37.0631;176.7000,-38.4667;26.0333,53.1333;-75.2958,9.3178;15.0667,-13.7333;48.4167,-17.8333;89.6167,24.0833;17.2267,43.7181;108.5667,-6.7333;8.7147,36.1822;26.1450,59.3347;100.3833,5.3833;-2.3656,51.3794;22.2014,-33.5907;33.0000,50.0167;26.9500,-27.2167;30.3667,13.7000;25.8333,53.6000;-114.0353,51.3001;-89.3500,15.5333;36.6667,33.9667;11.5833,1.6167;36.8333,7.6667;102.1000,16.5333;5.6208,52.3417;44.7564,41.3494;4.5667,51.9167;-66.9346,10.5990;22.1333,47.8333;-56.8886,-33.5389;-2.9300,35.1700;-8.2681,52.2658;25.0361,58.6481;-67.1667,10.2500;20.0731,42.3572;2.8833,33.8000;37.7650,12.9842;10.1728,24.9647;28.4442,-14.4357;9.2500,47.4167;-83.7833,10.2167;21.7756,41.7156;-56.8167,-25.8833;-0.5589,51.3190;95.1000,21.3333;83.9833,28.2333;103.3000,16.1833;80.5833,29.3000;146.8000,-19.2500;9.6167,1.4167;-1.9833,5.3000;-22.9167,16.1833;95.7333,16.4333;36.3167,21.0667;17.3008,44.9081;160.7000,-8.7667;-15.0667,13.8000;-80.2497,43.5501;11.4905,6.0830;-76.6433,20.3792;12.1667,47.5833;21.4000,50.9333;21.5869,41.8867;6.0453,14.4222;-79.4992,22.0789;21.7344,38.2444;175.2750,-40.6333;127.9558,40.0253;-89.3400,13.9756;32.7958,34.7792;-95.9378,41.2586;129.2117,35.8428;35.1653,32.9536;28.6978,48.0319;140.8833,37.0500;49.7131,40.4047;-88.9500,13.5167;-77.9000,18.3167;34.9892,32.8156;24.0500,54.4000;33.1731,0.4186;4.1833,7.3500;\
    -56.2778,-34.5228;50.0764,40.4539;48.1303,29.0825;33.1250,35.0458;53.2333,56.8500;-78.3162,44.3001;-78.6386,35.7721;128.1606,36.4153;33.4167,35.1000;25.6419,42.4328;34.3667,-4.2667;38.4667,7.0500;144.9667,-37.7667;-79.8011,9.3422;26.9000,46.5667;32.8417,34.7708;-76.6122,39.2904;9.0170,46.1928;0.2000,48.0000;14.5500,5.8833;20.9333,52.4000;5.6500,50.7000;25.5833,-33.9667;18.8667,46.6333;-171.5833,-13.8667;57.7306,-20.1631;42.5333,-0.3667;79.5833,18.0000;58.5700,51.2253;-15.5833,13.4167;100.5000,5.4833;-62.8167,-17.2333;24.7167,43.1333;24.4667,42.0500;151.2073,-33.8678;39.5228,12.5058;-55.0000,-25.4500;33.9333,13.1500;44.4378,-12.3067;2.1618,41.3890;3.0853,36.5406;-2.5833,51.4500;26.6333,60.9000;25.5572,58.8856;13.5667,4.3833;-99.9531,49.8469;65.5678,35.5483;-72.3167,18.2339;83.4500,27.5000;33.2000,35.1667;-79.4500,-7.2167;31.0500,-7.5167;-107.3897,24.7994;-87.9631,13.4314;44.5225,36.6114;13.6136,45.4964;-60.7000,-31.6333;72.6711,32.0836;127.6186,39.8317;90.8333,22.9500;-58.2694,-34.7203;26.3000,55.9667;24.4714,35.3647;102.9333,1.8500;17.9509,59.4280;2.4333,48.8000;103.8500,13.3667;46.2735,36.2499;57.4300,-20.2072;115.0667,4.7167;-77.6156,43.1548;19.6922,41.5775;-54.7500,-34.2000;-84.7167,11.0333;16.0319,45.0547;-14.1833,13.2667;13.5458,50.3272;36.8333,-1.1667;-59.6333,13.3167;83.7600,53.3600;46.7897,40.6103;45.8858,40.8994;35.1333,-16.0667;-6.9367,52.3967;15.9333,29.1167;27.7333,46.6333;6.1650,36.1594;23.8167,42.4333;23.7667,38.0667;57.7144,-20.1897;26.0473,57.7778;43.0333,4.5500;-79.4163,44.6168;23.2000,61.6333;34.7167,31.7833;80.6000,28.6833;5.9406,49.5256;-67.5833,-39.0333;120.4853,14.4336;29.6631,46.5153;36.2181,34.0058;139.6500,35.7333;3.8833,51.0000;8.4971,47.4031;3.4467,11.8878;90.7667,23.0500;20.0867,41.5069;19.8858,41.3072;24.4833,47.1333;23.3241,42.6975;26.2333,43.3500;-17.0603,14.7739;14.3000,6.5167;-8.6000,41.2000;12.9167,50.8333;7.8736,8.8486;-70.2000,-22.0833;39.2647,-15.1197;-77.7000,18.0500;-96.7167,17.0500;-91.6500,14.5667;-171.8000,-14.0167;43.4875,34.9250;\
    73.1000,30.6667;79.8667,6.8333;-90.5875,14.5269;48.9356,40.0433;12.0792,32.9344;-3.9432,51.6208;18.7856,44.5011;-74.7964,10.9639;32.4000,50.6000;-61.9669,-33.7497;5.3333,50.9333;67.2481,43.9150;42.0772,42.6044;38.2000,10.4500;9.7500,55.0500;177.2833,-38.0833;25.9706,58.6525;-105.7677,53.2001;8.7584,47.2053;80.0456,7.4708;63.5833,53.1667;35.7272,32.0392;15.2682,49.9484;13.7332,46.1830;119.4342,32.2092;35.0081,32.2647;-16.2546,28.4682;-76.5386,20.2550;28.1500,-13.5333;-70.6667,-34.1833;19.8522,46.7136;-80.2833,7.7667;-71.8569,18.3914;-78.8333,-2.7333;102.9833,13.5833;10.7833,34.7500;92.0667,49.9811;44.2225,32.5433;22.8169,54.6267;47.1167,-18.3167;-59.5333,13.0667;-57.6667,-25.2667;-82.4333,8.7833;71.5261,39.2422;-38.5108,-12.9711;8.5167,47.1667;-9.0000,29.7100;98.9500,3.5667;-6.2330,36.5939;34.5333,10.0667;19.7156,41.6839;-83.3333,10.0833;-74.7326,45.0334;19.5667,40.7167;37.9000,59.1333;24.1186,56.8761;49.6597,40.5408;102.5667,2.0333;26.5611,59.4842;25.9000,57.4333;19.4875,46.4314;74.8833,32.1000;110.3608,-7.7828;6.4985,46.5113;15.0533,46.1550;74.9950,46.8481;24.4833,64.6833;-77.8167,-0.9833;-87.0833,14.4000;-54.6167,-24.1333;120.8817,14.3869;-57.0597,-23.3428;22.5025,40.2719;-79.7911,9.3764;69.5967,40.6450;35.9167,35.3500;44.7231,40.1194;2.6502,39.5694;80.6611,6.9367;8.5372,47.3953;172.4833,-43.6500;91.4561,22.4678;7.2000,51.6167;-171.3667,-13.9833;-78.1333,0.0500;-85.6667,12.3833;-80.6833,-5.2667;11.0364,35.1667;-66.8833,-14.1667;18.3481,43.9019;-77.7500,18.0667;-171.5667,-13.9000;28.2000,-12.8167;73.9667,32.7000;19.5939,41.8136;2.2500,48.8333;8.5253,47.3950;-72.8814,-13.6339;96.1500,20.4333;9.5442,47.2181;-6.0931,53.5223;-77.2078,-11.4950;90.2333,22.9667;3.4500,51.2167;176.8492,-39.6381;-79.8167,-3.3333;-56.7833,-27.1500;48.3558,33.4878;70.4169,52.6292;-50.1619,-25.0950;2.9058,35.4514;9.5167,47.2333;5.9147,49.6886;-44.3028,-2.5297;10.1349,54.3213;-70.7081,-30.0319;31.7669,-2.6378;39.4667,-3.8667;-24.3000,16.6167;112.7203,-7.4539;21.2489,41.3689;-57.8500,-34.4667;-76.7519,18.0138;\
    22.7664,55.0778;45.6722,39.8417;27.2962,53.7627;17.0581,12.3767;43.3153,14.1950;-73.2506,10.4769;-16.4167,14.3500;125.2447,39.9811;-75.9361,4.2689;14.3250,45.3617;44.0020,56.3287;-7.9000,38.5667;25.0403,48.5306;23.8000,44.3167;32.4667,34.9167;-90.3586,14.6647;-9.2667,38.7667;23.8147,53.6814;-57.0667,-25.2500;-172.2167,-13.7833;25.3539,43.6231;68.8500,37.8167;35.0167,-10.9333;69.6400,34.1700;16.8489,45.8986;24.0833,51.7833;116.1167,-8.5833;-79.4663,46.3168;24.8667,60.6333;15.2264,46.6142;-88.8347,17.3347;-2.4833,53.7500;12.9391,50.3059;34.8417,32.1472;-4.5533,13.9061;104.0667,30.6667;36.1333,15.8167;44.7500,-24.7000;-2.3333,6.4667;43.4022,-11.4292;-171.8833,-13.8000;-100.9167,29.3000;6.2783,15.4522;-55.2500,-26.4833;-0.5500,47.4667;6.2000,48.6833;10.0607,44.0629;-85.6667,12.4667;4.6028,51.8725;20.6000,32.1833;8.9167,52.2833;-77.5828,20.0431;14.2167,12.3667;25.5461,52.1325;98.9167,8.0667;46.9998,-25.0318;3.2876,49.8489;-72.1253,19.6697;34.8367,32.1658;77.4833,10.0000;45.8028,41.7431;14.8633,45.6433;9.9333,51.5333;-171.8000,-13.9667;-3.0000,53.6667;7.9167,47.3500;-68.7328,10.3417;0.1167,52.2000;-79.6953,9.2747;22.4667,47.6833;67.7667,47.7833;-53.8069,-29.6842;10.5000,36.0333;32.9000,34.9958;1.8833,6.4000;44.2803,32.7147;-66.0500,-20.9167;-6.2000,36.4667;-171.9333,-13.9500;-86.3442,12.1589;126.3183,33.4644;2.4000,48.7833;9.2058,35.8606;30.6786,4.0914;-55.0833,5.8833;144.7500,-36.1333;108.3350,15.8794;-55.9500,-25.5333;6.5625,52.9967;12.5341,55.9618;26.9711,59.3531;103.8167,1.6000;17.7347,49.5480;29.4667,-25.7833;-74.1867,10.5214;35.7167,32.6000;4.0167,50.8333;-54.9500,-34.9667;-87.6333,14.4500;-15.5667,13.4333;-70.5833,-33.6167;-84.5333,9.9000;44.8819,13.0567;17.9575,43.0844;14.3625,45.7931;18.9667,50.3000;44.3744,-12.2442;26.6725,53.2181;99.9167,12.3833;-16.3667,13.8833;102.5500,2.2667;-5.8100,35.7800;-77.6500,18.5000;37.6667,-6.8167;127.1378,37.4386;-84.0488,9.9616;26.3000,47.4500;8.4000,34.3333;103.9000,21.3167;32.8886,0.7025;126.4522,36.7817;96.0833,22.0000;137.2167,36.6833;\
    -9.1333,38.8000;8.3167,4.9500;2.3488,48.8534;-97.8500,22.2167;-88.1833,14.3000;8.8667,4.9500;5.9883,49.5331;-71.9767,18.5058;27.4667,42.5000;-84.5000,9.9500;109.1833,12.2500;30.1944,55.1925;26.3333,46.9167;-78.6639,35.7868;48.0392,29.3181;57.6419,-20.0697;-71.8333,19.3833;-88.1667,14.3167;28.6667,50.2500;38.9833,5.8833;22.8075,37.5636;23.2036,57.2028;-2.0833,52.5000;13.5000,-14.9167;57.5572,-20.1911;33.6292,34.9542;49.1825,11.2847;7.6451,47.5227;95.7500,17.7833;-73.7732,-42.4721;57.4386,23.8494;17.1889,45.4364;23.7333,43.7833;43.2761,-11.6075;-40.2925,-20.3297;5.2333,52.0900;-75.2092,20.1444;-67.9197,10.0900;30.6106,3.8894;-97.3964,27.8006;31.0597,-17.3606;-73.0897,7.0647;39.4928,39.7522;-79.5333,9.0833;142.4833,-38.3833;14.1029,50.1473;-15.2000,13.7000;104.8833,15.2000;2.0333,41.4833;103.0667,2.0167;-2.9383,35.2937;4.3833,50.7167;12.7224,41.9936;-23.2167,15.1333;39.5333,9.6833;-65.3167,9.3500;9.9272,56.0399;112.7019,-7.8264;68.2500,43.3333;-62.7500,-17.6500;-23.0000,66.0333;-70.4111,8.7622;-171.8833,-13.8000;44.3833,-23.7167;-87.2333,15.5833;64.0781,45.4878;69.0064,39.9108;11.9639,11.7486;-8.5833,41.2000;-61.4833,-31.2667;9.5144,56.8040;135.4167,34.7167;101.2836,6.5428;49.7164,40.5322;-80.1000,-0.6833;25.7833,55.0000;1.2667,9.3500;45.3917,40.8808;43.3883,-11.4869;6.1061,49.7489;2.6000,7.3667;18.1167,47.7333;24.7167,46.7833;34.4333,-13.7833;28.4914,54.2256;-8.9667,38.7000;12.3234,58.3807;9.1833,47.6500;26.1153,59.1586;34.1500,-3.0833;14.3667,4.4333;27.7833,62.4833;19.0903,44.1711;-65.4833,-23.7333;-79.3833,9.0833;121.4581,31.2222;15.1878,60.1496;12.6946,57.5073;3.5736,51.4425;2.4667,48.7833;77.7000,28.9833;-84.4569,39.1620;-86.2000,11.8500;-70.5983,-32.8337;40.2058,47.7122;15.0906,32.3783;10.8782,56.4158;171.3833,7.1000;79.9501,23.1670;28.6833,-13.9667;47.0699,38.4774;123.9672,47.3408;95.6500,16.7333;69.0197,38.5531;106.6000,17.4833;26.0167,44.9500;-46.5650,-23.6939;28.2475,47.8686;9.8739,37.2744;40.4531,-2.3906;-72.7491,46.5668;15.0562,50.7671;12.1167,55.7500;12.3833,55.7833;\
    5.8167,49.6833;36.3175,35.8156;6.0822,49.6583;90.8500,23.1000;-65.7167,-21.4500;20.0053,39.8756;-1.1167,37.9833;-2.0333,7.0833;-76.9000,18.3667;73.3167,33.2667;89.6000,24.1667;-8.4333,41.5500;-72.7324,45.4001;77.4333,28.6667;-57.9484,48.9667;5.3767,35.3897;-89.5000,14.1333;14.5978,46.1381;18.8047,45.2883;115.0053,35.7028;150.9333,-31.1000;174.5000,-35.6167;66.0614,35.7017;14.3349,41.0832;26.7333,-27.9833;9.6831,47.4831;-15.6000,13.5667;44.4183,-12.2558;88.9167,25.6500;18.9667,50.3500;-128.6035,54.5163;89.8333,23.6000;81.7000,7.7167;10.7579,59.9115;175.8667,-38.2333;-79.9667,22.4000;-67.7792,10.0853;-85.7500,15.2000;16.2717,-0.0447;22.0119,41.4331;21.4819,41.9197;145.8000,-5.2167;125.1572,1.4411;17.8000,46.3667;134.2525,7.0394;56.5408,23.5856;153.0281,-27.4679;109.8222,40.6522;17.9379,62.6323;12.5750,32.7564;-71.2667,-33.0167;130.1528,42.8136;-72.2483,8.2189;127.5892,34.9753;15.0500,3.5167;11.2167,33.1333;-99.2614,19.5583;13.9833,31.7667;99.8000,13.5333;-84.1500,10.0667;-78.9500,-8.1000;139.4690,35.7992;33.0167,34.9167;34.3750,35.6000;32.4731,0.8492;25.9350,58.0028;108.2000,-7.3333;23.7500,38.0333;-9.1000,38.7833;27.1833,63.5667;-79.7867,8.8422;176.3167,-6.2833;45.9719,41.6169;22.5422,41.6139;-51.1628,-23.3103;-9.1500,38.6667;23.1214,57.3400;-56.0522,-34.8603;-67.5278,-14.4412;23.9764,32.0836;70.5500,40.0500;26.5375,58.2756;-16.9000,32.6333;23.8667,47.1500;10.7822,36.4692;-90.8061,14.9950;89.9167,22.9000;46.3167,-15.7167;24.5000,43.7833;5.6889,50.8483;103.5347,48.8125;6.9500,50.9333;-55.7500,-34.7667;4.9597,52.5050;-96.9167,19.5333;124.2464,7.2236;30.0000,-22.2167;14.3631,45.9158;9.7305,55.5059;23.5497,41.0856;6.2089,49.9350;-81.5375,22.7953;21.9833,56.9667;23.5875,56.8417;44.2900,32.7786;134.4544,7.3519;20.4289,41.6850;-74.1667,18.2667;-7.6253,52.0881;13.8972,25.9044;97.3714,16.9206;-1.4000,7.0667;5.0708,51.6825;43.2844,-11.3847;10.9333,2.0333;13.5472,-12.4314;29.3667,-19.7833;-86.9789,12.5431;22.6653,41.4053;-1.0167,5.9000;21.3331,41.3361;0.0903,35.9333;-120.3192,50.6665;\
    -55.5333,-23.2667;-72.3942,5.3394;14.2500,48.2667;20.1661,41.2861;31.6061,40.7358;-80.2667,8.4000;57.6767,-20.1458;-6.2489,53.3331;11.3833,-2.3833;39.0900,48.1192;-88.6333,13.8667;19.3769,45.2222;31.6561,9.5369;-78.8369,1.2883;35.4444,33.0247;16.4667,48.1333;130.5500,31.6000;20.2231,30.7592;5.0913,51.5555;30.6500,12.7167;-72.1833,19.5833;41.1206,37.8828;10.3965,43.7175;12.4847,32.7919;-73.0000,18.1833;5.5833,7.1833;-14.6333,13.4667;116.0833,5.9167;-77.2667,18.3833;-92.6333,16.7500;-82.3869,22.9700;-81.0333,8.1333;7.5333,9.0833;27.7266,64.2282;-72.4742,7.8339;15.6333,48.2000;21.6886,65.8252;57.5542,-20.2444;68.6333,27.9500;9.4500,36.5500;72.9000,2.6667;10.9886,49.4759;43.8603,40.2183;-43.3503,-21.7642;118.6158,4.4811;58.5097,37.1069;6.1300,49.6117;-86.8025,33.5207;-74.2500,18.5667;13.5000,46.8000;10.3883,55.3959;40.5167,8.8167;-97.5000,25.8833;1.6000,6.2333;-80.8875,-4.2608;-76.7832,18.0075;4.2325,50.7410;-89.9297,13.9011;34.6825,37.9692;22.5000,56.6667;-7.1119,52.2583;28.7967,47.8153;46.4083,41.7258;58.4347,50.2503;17.4275,21.8375;22.7383,54.6356;49.8039,40.3394;-86.8167,14.5333;17.3063,62.3913;-171.5667,-14.0000;130.7167,32.8000;22.8167,4.7333;-8.5378,37.1386;81.6308,47.0883;152.9500,-27.0833;27.5694,43.2222;-14.2833,64.6500;-63.1267,46.2352;26.4375,59.3611;70.4167,40.2833;18.0878,45.4886;33.1958,35.0458;38.7333,-6.7000;-6.1544,53.4508;118.7353,9.7392;113.5833,24.8000;-84.1833,9.9333;-88.1386,13.2203;-66.8567,10.2483;14.0323,50.6607;24.7925,58.5781;78.0167,27.1833;32.6672,0.3558;-61.7068,12.1902;128.5725,35.2081;121.0489,14.9106;-71.6000,-35.8500;120.3194,16.6186;-70.3333,-18.4833;8.4901,47.3740;16.5833,51.8500;-97.3375,37.6922;-71.1653,-31.6308;13.6500,48.0167;19.8803,41.3222;10.1919,11.6783;120.8833,14.8333;91.7167,26.8167;1.6728,36.2247;23.7333,41.5667;-6.1819,53.6128;25.5000,42.8667;36.3989,15.4558;25.1167,43.6500;-55.4833,-34.6000;175.6167,-40.3500;5.9111,52.5550;26.7000,60.8667;12.3921,47.4464;30.4333,-17.0333;-86.1333,13.9167;32.7667,-5.6333;30.1333,8.3000;-76.6667,18.2333;\
    -82.7000,8.5167;107.5425,-6.8722;13.9024,49.2614;23.8500,37.9500;54.9478,36.4300;-15.2667,13.4667;25.7333,65.1333;5.9164,49.6464;-76.5667,17.8667;-86.8794,12.4356;-71.6491,48.5501;51.0922,25.6094;-3.4833,40.4500;-8.1958,52.8817;-57.6167,-25.3667;51.5319,32.6861;37.0161,39.7483;23.7833,42.9000;22.9000,62.8333;149.7167,-34.7500;13.1108,14.2528;39.9833,7.0167;-90.8442,14.6200;-7.6100,33.5900;28.7833,-31.5833;-74.0060,40.7143;20.5667,32.5333;-8.8600,30.4800;25.6833,8.4667;38.8333,58.0500;46.0367,40.8628;20.2597,63.8284;19.4714,40.3253;35.8000,31.7167;9.5617,4.5025;38.4500,6.8667;27.2500,61.6833;5.3875,52.1550;73.5333,3.4833;72.6167,23.0333;37.1928,38.2019;45.4408,35.5617;-90.3767,14.0858;97.7500,22.9333;-57.1500,-26.3833;-62.6528,8.3533;125.5019,8.9442;-14.8000,66.0333;-73.6386,18.4825;85.5667,26.8667;73.3833,1.7833;19.0000,50.1500;33.0333,34.6750;7.4833,5.5333;-68.2003,10.4889;81.8500,25.4500;8.5603,40.7272;-1.2833,5.7000;30.1500,-18.1333;9.0147,33.5778;-72.2333,19.5333;-99.1333,23.7333;-70.1992,11.6997;15.6397,46.2375;121.0556,14.3647;-71.2614,10.1408;69.5881,38.0458;-76.9667,18.1667;-6.9261,52.8408;-100.9500,22.2000;24.8781,58.9303;-7.4889,53.2739;32.5656,0.3156;72.6500,30.1667;5.5333,50.6333;-81.6556,30.3322;-7.0497,53.3453;17.9500,50.6667;21.9089,55.1381;3.4833,34.9000;120.9822,14.6042;28.2333,-12.5500;2.2181,36.2622;-60.6733,2.8197;-88.0500,15.5333;23.9033,58.9703;37.6111,54.2044;23.4671,41.8863;5.9586,14.0758;-89.2131,13.7403;15.0030,49.1440;-77.0167,18.3000;10.4076,59.2676;135.6000,34.6167;126.9997,37.5664;33.5306,-24.6867;32.0228,0.4175;140.1167,35.6000;30.8333,-20.0833;23.9667,54.0167;-5.2700,35.6100;4.1833,7.6000;176.9167,-39.4833;-63.0000,-40.8000;11.9333,-1.8833;-80.5500,7.7333;-6.3500,31.7000;44.6875,40.7381;34.9500,-0.7833;115.8333,-31.9333;16.2502,39.3100;-91.4708,15.3197;22.9439,40.6403;27.0667,-14.9833;139.0667,36.3833;11.8769,43.4614;39.4833,-4.4667;-55.8833,-27.3833;2.2833,48.9000;24.1667,43.2000;114.4678,36.6006;-89.5042,13.9286;-74.8119,7.5942;-72.2333,-37.7167;\
    -14.9167,13.5500;-62.9500,-15.3833;125.0064,11.2511;44.4883,-12.1175;34.8303,32.0014;13.6342,45.9125;4.9167,51.0833;68.9125,37.1822;35.7169,32.5417;-72.1053,18.8333;-13.7000,65.1500;120.9675,15.4869;95.2167,18.8167;41.7786,41.9956;32.6167,-20.2000;20.6439,41.2408;-8.4167,39.6000;23.0139,57.3864;10.4514,12.8775;-85.8000,11.4500;46.2919,38.0800;10.7167,35.3000;-106.4869,31.7587;127.9453,37.3514;57.6136,-20.4086;-171.6667,-13.8500;35.3000,-16.0167;35.6000,3.1167;-86.0500,11.8667;25.9333,42.3000;73.8000,31.7500;-0.2167,51.9167;69.9856,40.6486;-75.8219,20.0247;89.0167,25.1000;175.6667,-39.4833;23.3636,57.0408;12.7833,47.3167;9.3081,4.0994;78.1667,11.6500;6.2589,49.6058;27.4667,-17.2500;47.7081,30.3892;35.9325,32.3847;6.0333,47.2500;172.3000,-43.7667;-79.7333,-1.9333;-8.6000,51.9167;-89.8450,13.9214;12.8712,50.2327;7.0833,51.1833;15.3142,45.6472;128.6308,36.8217;11.3667,8.8833;10.3833,5.2000;-0.5450,35.7200;-7.0300,33.8500;-54.9667,-32.9167;88.8917,25.7777;10.8333,35.7833;6.5667,51.3333;23.9853,56.4267;20.2500,31.6686;-57.0667,-34.3667;7.1607,62.7372;40.5078,-12.9608;20.9042,41.8961;28.1903,59.3772;-106.0833,28.6333;24.2667,44.6500;120.5675,15.1622;122.0739,6.9103;36.3000,33.6000;-65.9333,-17.6000;-58.9833,-27.4500;29.2333,53.1500;6.5856,36.9292;2.1000,41.5500;-65.8333,-17.5500;8.6161,56.3601;29.9000,-23.0500;18.2333,46.0833;35.0000,-15.7833;24.3667,44.4333;24.7500,42.1500;26.4500,-15.7500;14.1833,8.6667;27.5150,40.9806;20.2631,40.0311;90.7333,22.2167;-6.5936,53.3850;-25.0647,17.0197;14.8225,46.0586;7.8522,47.9959;25.7167,66.5000;-6.2667,13.4500;23.7979,38.0306;-66.8333,10.6167;85.3333,27.6667;-69.2167,-51.6333;3.8822,35.6758;83.6000,28.2667;16.1826,58.5942;16.2000,47.9667;24.2114,59.3197;95.7000,22.5667;-23.1333,66.1167;-78.5003,-7.1638;43.6500,3.1167;-6.1083,53.5828;-82.4333,8.3667;-86.2167,11.7333;-56.3853,-34.7633;6.6936,36.2633;18.0967,44.5492;-171.7000,-14.0167;17.3775,44.4908;-171.9333,-13.9333;-5.4667,12.3833;35.5000,9.5000;101.9333,2.7167;28.6689,45.9000;-87.4000,14.2667;\
    16.6000,47.6833;100.7778,13.9774;-17.3500,66.0500;30.2167,13.1833;-87.1908,13.3003;35.7386,31.2703;47.8192,30.4942;-81.3167,8.0167;25.6853,57.6257;118.0633,36.7906;-74.0833,4.6000;28.2008,46.2781;19.8756,41.7675;-83.0165,42.3001;18.4103,47.1899;-57.3500,-34.3167;-77.9169,21.3808;-89.4500,14.3333;20.7833,-11.5500;48.2667,-13.4000;12.3829,43.0967;27.1419,53.6850;31.1572,-0.4031;-55.5167,-30.9000;-72.3167,19.6833;127.5983,38.7103;106.8186,-6.4000;11.5555,55.4318;29.6667,-8.4667;14.5797,50.1048;16.6886,43.4447;-72.9297,5.7206;14.5289,45.9603;-87.8500,15.7667;22.1786,42.0031;77.4633,43.3550;-59.0167,-33.0333;153.4300,-28.0027;-70.4333,9.3667;47.1267,40.3744;-2.2374,53.4809;23.5833,55.0500;8.5838,56.4912;-72.4167,-35.3333;35.8953,32.2806;35.5978,32.2792;19.9803,40.3517;-75.9989,20.2172;23.1553,56.9669;30.9836,53.8128;14.6357,63.1792;31.7897,41.4556;4.2500,51.2167;107.6000,16.4667;25.6167,56.0833;-74.1167,18.6500;12.0138,57.6554;44.7681,38.1975;23.9833,38.0833;8.9106,12.9975;25.9333,48.3000;94.8833,20.1833;4.3528,50.8466;5.9908,49.6450;45.1600,42.6583;-79.9619,9.1472;112.0333,22.9333;2.2500,48.9500;-2.0833,51.9000;4.0000,6.9667;-73.3333,19.7000;14.0306,46.0028;-65.6500,-24.9167;-8.7833,41.5333;-23.6000,65.6833;8.6003,47.2558;24.9739,56.6775;-117.1573,32.7153;-3.7833,37.7667;20.4833,46.2167;35.6908,32.5722;5.9111,51.9800;-2.2200,52.1894;35.8500,31.8833;35.6997,32.4369;72.6342,49.8233;-54.0849,3.6741;33.1417,35.1083;16.8500,8.6500;20.0192,40.2958;16.2833,50.7667;-79.5842,-6.9222;8.6722,35.5744;130.3967,42.3422;28.4494,46.3333;-59.6500,13.2500;-7.3092,54.9972;30.4833,-20.0500;-90.4408,14.4653;43.3093,33.4235;23.2500,50.7167;-99.1103,19.6317;16.1544,46.4114;22.5833,57.7500;-80.7833,7.9500;-77.4350,20.1419;17.2294,45.9506;-75.4083,10.3294;16.5767,48.5700;24.9667,63.9167;-67.7161,10.2983;173.0167,-41.1333;-79.0333,-1.7000;7.7772,13.5081;-77.0766,18.1808;36.2833,-1.9000;9.9823,57.4642;68.5514,38.5247;47.8667,-18.4000;-79.2333,9.1333;-57.9667,-31.3833;-79.7800,9.3378;6.0292,52.0050;\
    142.8333,-4.2333;-2.7333,53.4500;-13.9333,13.5000;-1.3467,34.8650;5.3000,51.2333;-104.9847,39.7392;-1.4659,53.3830;-6.8792,53.7264;81.9056,50.6328;78.1167,9.9333;35.9119,32.5694;46.2150,36.5219;53.6500,28.9394;-91.4500,14.9500;5.2680,59.4136;139.6500,35.4500;39.9000,59.2167;117.1917,34.2669;31.7131,-0.3128;33.2625,35.0792;-0.8333,9.4000;-66.6636,10.2314;-14.8000,13.9833;66.9975,30.1958;-89.1792,13.8072;-106.4167,23.2167;28.1500,53.5167;-9.4833,13.0500;19.7050,40.9419;57.7453,-20.2542;90.1000,24.1000;33.1833,-4.2167;32.5000,34.8875;126.9269,37.3925;139.9247,35.7197;34.7431,-19.6094;19.6833,13.3000;5.3833,34.7167;4.3333,50.6000;19.9547,41.4786;-55.7500,-27.0833;36.9833,-7.6000;-82.5167,9.4333;128.1833,41.4000;16.4000,47.9500;39.0692,3.5164;7.7167,5.1667;42.3111,41.6464;174.6667,-36.4000;-100.0000,20.3833;-9.7167,52.2667;14.1318,50.5335;25.7500,56.1667;-56.3342,-34.8017;36.7000,1.1000;32.4750,34.8417;50.5478,26.1736;-9.3000,53.8500;33.8000,-1.5000;21.7144,42.1322;-58.2856,-33.9892;-77.9667,18.2333;69.0969,37.2383;47.4761,40.6500;36.6667,-1.2500;42.8956,12.4217;5.2000,50.8000;-72.3500,19.5167;14.8178,45.9525;28.1833,61.0667;2.5000,48.8000;129.7117,42.0606;43.8200,-12.3428;12.6664,55.5945;96.7333,17.9500;42.1581,41.6531;-79.7272,8.9989;-77.2789,20.2342;69.5345,36.7361;0.6833,47.3833;44.5333,1.7833;-8.4897,40.9023;-8.0333,13.5833;-89.1831,13.6589;5.8833,43.1000;58.3894,23.5317;35.7811,32.2200;-1.1505,52.9536;-172.6333,-13.6333;33.9833,35.0375;61.7775,34.9406;88.6333,25.6333;168.6627,-45.0302;-61.1167,13.2500;24.7997,59.5353;18.4667,50.0000;9.5444,47.1852;32.0461,53.3389;115.1000,30.2167;137.1667,34.9500;29.7078,46.7281;32.0667,49.4333;35.7667,32.3333;-3.8737,40.4929;-0.2500,52.5833;-67.0333,-18.1667;23.8000,38.0500;22.9172,40.7308;107.2000,12.4500;24.6000,56.8167;-87.8647,13.3078;113.2914,40.0936;15.7253,50.9008;51.6069,25.1678;118.3167,5.0333;44.4417,-12.1714;44.5286,-12.3464;25.0138,60.2967;6.4814,49.6961;25.7614,59.1417;-110.9265,32.2217;-15.2167,64.2500;34.4667,-1.0667;\
    127.2056,37.5400;-86.9667,14.1667;-87.7000,14.3833;-79.6833,-2.1167;19.3667,51.6667;80.1667,9.8167;9.2218,45.5571;27.8861,39.6492;-8.6167,41.5333;9.6502,59.6689;-6.3722,39.4765;-64.7167,-13.2833;81.3500,6.8667;101.4500,1.6833;57.3839,-20.4225;35.2833,0.5167;121.5419,29.8750;19.1167,50.8000;-171.9167,-13.8167;8.6187,47.3972;-63.2167,-19.8000;106.6381,-6.0989;71.0400,35.0200;51.8794,32.0339;47.8333,-22.8167;0.9194,14.7314;115.6333,-33.3333;-77.5333,-9.5333;-86.2500,15.4667;172.8000,-40.8500;39.2000,7.1000;-8.7167,39.4667;31.3333,-14.2500;50.0169,36.2622;102.4167,3.4500;-1.0939,10.8950;36.6000,33.7500;17.9083,59.5204;126.5603,43.8508;6.2181,49.8767;18.6497,44.4494;-3.1667,40.6333;31.0667,0.8000;47.3167,-19.6500;90.1469,23.5883;44.3097,15.3156;42.7125,11.1558;8.7794,36.5011;-56.6192,-34.6333;19.8333,47.9667;-77.4000,18.4667;32.1667,-18.7167;21.3500,47.6167;38.9997,-13.1256;22.0167,56.6833;8.5005,47.4077;21.7022,55.3592;34.1167,-10.6000;126.5219,33.5097;108.0000,13.9833;80.9167,26.8500;16.3616,56.6616;6.2247,49.5228;-82.7536,22.9875;18.6831,45.9931;18.5500,53.9667;3.8000,50.8667;45.4711,40.8336;34.0583,35.0125;-171.7833,-13.8000;98.5000,3.6000;45.4122,39.2089;-1.2200,32.1100;68.5311,37.4936;49.8222,40.4758;-64.2667,-27.7833;18.8000,54.1000;-1.6833,48.0833;141.3539,43.0547;42.8500,41.7000;36.9317,37.5875;31.5175,-0.6156;73.1786,42.8728;34.8917,36.9178;-8.6333,52.1333;2.0167,41.5667;17.6000,52.5500;23.3667,45.4167;-70.4000,-23.6500;19.2333,47.3833;129.0386,35.3386;37.8417,51.2967;0.4667,7.1500;5.8961,49.5136;36.2478,37.0742;31.3806,53.5722;2.2667,48.8833;122.7511,11.5853;7.5994,12.9978;3.8667,50.7667;34.8011,30.6094;15.4769,46.0236;12.9192,18.6867;135.4693,34.7824;7.3167,11.5333;38.7000,9.0333;6.2211,49.7769;3.7711,35.8914;-99.0331,19.4136;57.7758,-20.2861;20.1844,40.8697;135.4667,34.5833;-19.6500,65.7500;6.5975,36.4131;8.5500,3.4500;35.1000,31.8000;-72.7000,-37.2667;-99.5333,18.3500;-71.8500,18.2500;-58.0500,-32.9833;-67.2167,-16.9833;29.8333,9.2333;-117.3962,33.9533;-5.9242,37.2844;\
    24.8647,57.1519;-0.1000,5.5833;127.8711,39.9475;-85.0333,10.9000;19.4500,47.3500;18.1167,4.8000;26.9167,60.4667;-76.1333,-13.4500;73.0833,-0.6000;17.3278,44.7228;30.0000,-19.6667;5.8758,49.5967;12.4896,37.8212;-90.3167,-0.7500;57.5531,-20.1406;1.1069,41.1561;8.8289,35.1672;45.7749,38.4329;8.8917,14.9703;-70.9167,-33.6167;23.5833,38.0667;91.7833,24.4833;33.1583,34.7458;70.5797,37.1175;2.8833,50.8500;20.1106,41.3219;-88.8333,14.8500;35.8667,48.5167;27.0694,58.0603;-75.2333,-12.0667;-8.5833,41.7667;33.9500,35.1250;75.3667,51.6667;87.2833,26.4833;32.4214,0.6894;36.7333,33.8167;-79.7000,8.9333;49.8081,40.4314;6.6393,49.7557;44.9522,40.6331;24.0531,59.3567;104.9500,11.4833;-15.4167,28.0000;8.9833,13.8000;66.9100,32.1000;144.0667,-4.0667;129.8667,42.9500;47.4614,40.0528;127.3075,37.6525;36.1939,51.7303;9.6819,4.6364;-82.7619,22.8131;17.9500,-33.0167;-6.3734,53.2859;-16.2333,14.6333;79.9000,6.8833;21.5325,41.3572;48.0761,29.3339;152.0167,-28.2333;33.6583,34.9833;-54.6167,-25.5333;4.8639,52.3008;89.2833,27.3667;28.3333,43.4333;57.6161,-20.2147;34.7500,-0.1000;44.5797,40.3211;48.5125,41.3597;35.7500,0.5000;-171.8500,-13.8000;5.1806,52.1100;-21.4500,65.6833;17.4833,-12.0167;37.3825,37.0594;5.9525,49.6167;-79.8294,8.8844;6.1681,51.3700;-75.9833,-9.3000;-61.5319,-34.8650;5.2333,7.5000;-22.6833,63.9333;9.4500,0.3833;-79.4163,43.7001;158.2083,6.9639;6.3669,49.5450;-88.2667,13.6333;124.2167,8.1667;28.8167,-28.5333;-63.1767,9.7500;86.4833,26.7167;-77.0697,3.8933;171.7000,6.9833;-9.1333,38.7167;33.2344,0.4797;44.9625,40.7781;-66.8500,-18.2667;10.2167,-0.7000;89.7167,25.9667;109.1591,11.9214;14.1167,48.1833;47.1833,-19.0167;44.3147,31.9961;-112.0740,33.4484;-84.1667,9.9833;-74.3000,18.6333;10.4000,5.3500;25.1733,52.4711;35.7783,32.6453;1.1667,6.9500;20.6672,5.7619;45.7003,43.3083;12.8302,55.8708;-1.6667,6.2000;-70.2483,-18.0056;-2.2431,51.8657;38.9325,48.1331;26.8167,43.1667;57.7075,-20.2200;24.4181,57.2636;101.2833,4.1167;15.2006,50.0281;40.6083,34.8544;-5.7500,41.5000;179.2167,-8.5167;\
    -49.3794,-20.8197;45.5167,2.7667;-97.7431,30.2672;12.3739,50.0796;-8.9833,38.9500;24.7500,56.2000;3.8189,50.4482;-8.4808,42.0789;-71.1333,8.4167;33.9167,18.2667;76.5667,60.9333;-59.1500,-37.3167;48.3333,-21.2167;68.1333,37.5833;-79.2378,22.3272;-82.9500,8.6333;57.6531,-20.4286;44.5333,-22.9000;124.2928,7.9986;6.0833,33.1167;-91.3158,15.8036;10.2500,5.6333;36.0069,32.5592;175.0667,-1.4833;-66.8500,10.5000;-76.5333,17.9500;14.4096,50.0112;-86.3706,12.8011;27.8500,-12.5333;69.8333,42.4167;-99.2467,19.6469;118.7778,32.0617;22.4647,41.6383;-113.4687,53.5501;-6.7700,53.5561;-76.0833,-5.9000;3.4103,36.7306;31.0167,-26.5500;-55.4667,-31.5833;33.0792,35.1333;3.1947,13.0444;23.0000,62.9500;22.3319,42.2019;61.4297,55.1544;-76.6000,17.9167;37.3300,19.1039;-18.3667,65.9500;13.2500,8.4833;129.3167,35.5372;34.9167,-0.1833;91.5069,23.0300;27.6000,47.1667;46.6786,32.4642;-8.7667,41.3833;-79.3892,-6.6406;33.8500,-9.5833;13.5101,43.5982;-15.5167,13.4333;27.2500,57.1333;42.1258,9.3094;-63.8167,-22.5333;25.9328,54.4169;65.3333,55.4500;84.8667,47.4667;-55.2167,-24.9333;-91.1489,15.0306;3.7000,51.1167;-8.7500,41.4167;-88.8333,15.0333;-4.8333,39.9500;121.5487,16.6881;1.5148,42.5450;49.5858,37.2794;41.9453,37.9272;68.8667,27.7000;38.3667,-3.4000;18.8083,44.8728;34.7819,50.9197;22.8811,41.4139;79.9167,6.9333;50.5650,30.7936;-16.4667,14.1333;24.6344,58.3792;99.8500,6.3167;-86.6167,13.1667;-77.5667,17.8667;20.5000,54.7100;-0.3833,6.3833;-1.2125,54.6861;35.3167,32.9333;-104.8319,39.7294;70.1556,39.0078;8.7686,47.5169;28.8333,45.3500;174.4500,-36.8167;57.5583,-20.3325;5.0167,47.3167;105.9750,20.2539;20.1981,41.3408;24.7500,55.9667;32.9833,14.2500;-82.1511,22.9614;-91.1545,30.4507;128.2975,36.2408;17.2167,43.4467;34.1667,44.5000;-15.3333,66.2000;44.9308,40.5203;-84.4667,10.0667;6.1144,46.1898;-77.6500,18.1667;15.3000,47.4333;28.6500,55.5333;20.0064,41.2847;18.2128,11.9817;69.9203,33.3381;-9.0833,38.8167;-64.4728,9.4389;37.9564,36.5267;117.3608,32.9408;-57.4344,-23.4064;35.3289,37.0017;-89.6500,14.1167;\
    100.1167,46.7167;37.3667,5.6500;9.4936,35.0403;11.8742,54.7691;47.4308,6.7697;45.6319,38.9500;21.3758,55.2964;32.5750,34.7083;18.9764,43.6669;-71.1779,46.8033;101.5000,2.8833;44.6678,43.0367;6.4419,49.6747;42.4300,42.3242;35.3833,45.0333;6.0834,50.7766;23.2861,43.1953;-77.7667,-10.7500;44.3781,41.6931;75.3333,19.8833;44.4317,-12.1356;-79.0288,48.2399;-57.5333,6.2833;-66.7833,10.4906;-46.5333,-23.4628;-1.8833,50.7167;-2.4833,10.0500;-3.5667,40.4333;126.6664,38.7819;18.4186,45.6608;-56.5833,-23.4500;-73.8922,18.5672;-1.1628,54.5728;-70.2150,11.6581;4.8250,52.4400;33.6500,-9.2500;4.4333,51.3167;-56.1325,-15.6467;46.0447,32.1769;-0.5333,53.2333;34.3053,0.3936;2.8954,42.6976;42.8419,11.5222;-63.1667,-17.5000;155.6333,-6.2167;24.9342,60.1756;28.3325,54.0983;4.5000,51.2500;11.1717,10.2894;-3.6067,37.1882;-75.6826,44.5834;-76.2852,36.8468;-3.8249,40.3458;79.8939,7.0478;-89.1500,14.8333;46.0333,-18.7667;69.3250,38.3883;78.0333,30.3167;-90.3611,14.3094;-1.7500,4.8833;44.6219,34.8831;123.8444,8.1458;17.8667,43.1289;17.7008,43.2286;-72.7000,19.1167;-90.1979,38.6273;32.7500,-18.2167;9.6664,37.0400;27.8000,41.1592;-72.5333,19.8500;36.0667,-0.2833;-46.2564,-23.9931;127.2431,39.4381;-97.3833,18.4500;8.2500,47.0500;6.3333,6.7000;32.5222,0.2081;18.7786,43.5050;-14.7500,65.1000;2.4833,8.0333;23.6197,37.9625;26.4333,46.4167;44.9608,34.6914;68.4361,28.2839;17.9026,49.9387;6.6667,12.1642;-67.5000,-45.8667;85.8793,20.4650;-88.9500,14.4000;12.6401,50.1813;68.5406,39.3975;-124.9861,49.6829;36.0044,31.8786;18.6000,3.6333;-16.9667,32.6333;2.9053,36.5183;152.7000,-25.5333;35.6961,31.0656;-87.9500,17.9167;119.9000,-8.4833;25.0000,54.1667;13.7833,-0.6833;33.3558,45.1972;18.0128,44.6731;26.2500,59.1264;17.9614,43.6519;6.2917,49.8358;4.5500,8.5000;156.8500,-8.1000;6.9500,46.2500;-0.8000,6.0500;-87.9000,14.9833;-76.5667,18.2000;98.2839,48.7567;-71.2833,-35.1167;129.1069,37.5439;34.1794,1.0644;18.0378,44.4267;69.0000,25.5333;10.7000,1.8667;22.1833,62.4333;-85.9167,12.9167;114.8500,-3.4167;13.5507,58.1735;\
    101.6500,12.7833;-7.9167,40.6500;-8.2333,43.4833;112.9000,-7.6333;76.2333,9.9667;70.0667,22.4667;6.0075,49.8339;23.3000,24.2167;2.9167,51.2167;69.1833,28.2333;34.7167,-0.8000;-79.4167,-2.2000;28.3333,57.8333;-57.6000,-25.3667;28.9622,47.7492;1.6667,9.0167;-89.1833,14.4333;-76.6308,-12.6581;69.0300,33.9800;16.8439,46.8400;133.5500,33.5500;-72.9167,19.8167;25.3500,53.0167;18.4881,44.7078;45.9861,35.1778;27.1667,55.9000;29.8500,-18.0667;35.7047,31.1847;-172.0333,-13.9000;-86.1667,13.5500;37.2667,-1.5167;-79.6500,8.9500;-6.7000,32.5000;114.5833,-3.3333;8.9738,56.1393;28.4569,47.1886;-89.0053,13.7892;35.4417,32.7167;28.4833,-27.2667;-22.5167,63.9667;39.8475,64.5722;-46.6228,-23.6861;-88.2667,13.7667;33.9333,-13.6500;35.8450,32.3697;36.5372,-14.8031;-81.1286,22.5269;146.7167,-7.3333;37.7708,44.7239;44.3922,35.4681;-6.7189,53.9728;42.0792,42.2689;127.1428,37.5947;-172.2667,-13.4833;143.7667,-5.2833;-79.7333,-1.9500;36.0803,52.9658;6.4333,51.2000;96.5000,22.9167;-17.4439,14.6951;43.8475,40.7894;32.9583,34.6833;57.6564,-20.1708;-87.6833,14.3167;-86.1000,11.9667;44.3428,40.1019;-172.6000,-13.6333;38.0111,36.8175;23.9167,37.8333;15.1167,-11.3500;10.8586,36.5786;-86.9167,13.0333;23.5000,49.3500;19.9167,40.8003;-64.4833,-17.9000;15.2533,-12.9067;45.1833,54.1833;44.2736,40.3414;19.2000,50.3333;6.7500,51.4333;-98.8833,19.5167;78.0833,27.8833;68.6667,25.7667;29.1333,11.4000;57.5908,-20.1489;29.0000,-20.9333;-1.4333,46.6667;-76.2333,-9.9167;31.3436,1.4356;49.7064,40.5725;35.6581,34.2553;-0.5500,6.1667;16.3167,8.9833;41.4969,38.7486;99.7525,17.0306;36.6500,7.8500;10.8592,33.8747;99.9055,12.9577;27.4211,59.3592;27.0194,57.8339;-81.3792,28.5383;21.3403,41.0311;47.1500,40.6172;33.2333,14.9333;-71.7833,19.6167;-75.6436,4.5325;10.5833,35.7333;139.6672,35.2836;-2.6000,53.3833;54.3439,36.1667;-71.3436,-29.9533;44.4903,31.8017;0.7833,9.2500;101.7167,17.4833;22.4167,61.8000;32.3500,14.8667;26.7500,53.8833;-91.1667,14.4167;170.9667,-42.7167;-55.7667,-34.3500;102.2500,6.1333;-82.1665,42.4168;-77.2811,1.2136;\
    129.9933,42.9572;3.6139,51.5000;33.9592,17.5928;16.6156,46.3350;90.9667,24.0667;13.7333,9.0500;34.7647,32.0678;-80.2436,22.5867;100.8167,13.6667;5.6667,51.9700;87.3167,23.4833;43.1622,42.5211;6.2189,49.6133;-88.1976,17.4995;10.7117,35.7533;52.6789,36.5442;73.5000,32.5833;-93.1167,16.7500;31.3167,-26.9667;78.4744,17.3753;12.3667,55.6333;-123.7027,48.7829;-15.9080,14.7903;-75.8900,8.7575;5.2314,13.0514;43.2547,-11.6711;-118.8027,55.1667;1.4437,43.6043;-64.9000,-14.8333;9.1500,55.4833;16.4392,43.5089;36.3003,32.7333;-96.9333,18.8833;126.5989,34.5711;-65.7833,-28.4667;25.4500,45.7000;71.5333,36.6833;97.2333,24.2667;-66.7333,10.4833;21.1175,55.7172;39.7400,48.2950;34.8578,32.3336;-57.1500,-25.6333;69.3864,41.7589;172.3833,-43.5833;127.3433,34.8431;-6.7967,53.1819;26.2544,58.9817;23.6667,37.9500;-70.9167,-53.1500;-84.7833,11.1167;110.8317,-7.5561;-5.9869,37.3772;12.1950,10.6111;21.4794,65.3172;25.3964,41.1228;81.3333,28.2333;36.6575,15.1100;99.5483,14.0041;35.1167,-0.6833;-1.5478,53.7965;-2.0000,52.5167;7.2167,51.4833;-171.8000,-14.0167;36.7500,35.2833;26.1333,54.3167;10.9167,5.7167;-171.5667,-13.8667;-7.1500,33.7800;33.1417,34.7250;14.1967,45.6794;74.6333,26.4500;34.8883,32.1508;45.3586,8.9969;44.6114,41.0539;94.8167,20.8833;31.3000,51.5000;-56.7500,-26.2000;-117.8678,33.7456;33.3500,14.5667;72.4833,30.9667;99.9667,8.4333;17.4509,49.4551;32.7083,34.6708;28.6000,44.3167;6.0875,49.4806;-99.2217,19.5269;22.9875,40.6006;83.3739,26.7550;-20.3167,65.8333;-2.0000,52.6000;18.8414,45.3325;34.0586,0.3078;128.9861,37.1731;135.6491,34.8135;90.3711,22.7019;69.9000,42.2000;-76.7911,18.0754;-79.8833,8.6000;23.4833,37.9667;-97.1470,49.8844;47.5022,42.9753;-66.0000,9.2167;43.1481,11.5950;-46.3336,-23.9608;22.8833,56.4667;-9.2333,38.7000;128.7317,35.3042;-15.8455,13.2441;101.1606,16.4190;83.3000,17.7000;21.4033,55.7128;-72.6339,18.5108;21.0686,55.9175;18.2050,44.4492;36.3300,41.2867;-80.6667,8.1167;-77.4167,18.0333;-74.2167,18.3000;11.9000,4.4500;-54.7833,-34.0500;25.5628,58.1033;-76.7094,-11.9431;\
    18.9500,9.4500;-6.4575,52.3342;10.5667,2.7833;-80.5833,-1.3333;-90.7597,14.5264;47.8908,56.6388;57.6039,-20.3669;20.9156,42.0350;20.8333,46.9333;44.3356,-12.1858;8.9000,4.4333;94.7333,16.7833;-122.2708,37.8044;17.6351,47.6833;20.9900,40.6275;-57.2333,-25.4833;8.7000,48.8833;80.3992,6.6828;18.1133,43.2586;1.4167,9.0333;99.6114,7.5563;-80.5500,8.2500;-57.1667,-25.2500;-67.6667,-17.3667;168.3500,-46.4000;-89.8422,13.7503;9.1506,13.7142;41.7753,41.8200;42.5500,3.8000;9.9500,5.1167;0.6000,7.5833;21.9056,41.7839;43.4108,-11.8864;-99.2500,18.9167;29.4306,40.7978;-4.0100,34.2100;102.8238,47.1975;34.7667,-0.6833;35.4069,32.8869;43.8206,40.7239;32.9197,0.3772;48.5953,39.1167;50.1164,40.5161;9.5000,47.0667;90.7167,22.5000;35.0728,32.9236;16.5222,43.5025;5.9833,49.5883;-7.8022,52.6819;38.8000,-10.5667;-171.4333,-13.9333;18.2960,57.6409;48.0786,29.2958;18.2833,6.5000;123.1731,41.2719;-4.7167,41.6500;43.2531,-11.6114;118.5667,-20.3167;137.5833,-33.0333;-76.9667,-6.0500;45.3661,41.0933;44.6075,31.7353;-0.6333,44.8333;-61.2333,13.0167;35.7500,-4.2167;-54.5333,-33.4500;125.6450,38.7542;44.1833,13.9667;-89.7108,14.3308;3.5000,34.1667;71.8869,52.3544;48.4008,40.5692;33.2500,35.0333;-98.2000,19.0500;100.4333,6.6333;80.5000,8.7500;58.3619,29.1078;120.7333,14.8325;-66.8278,10.4667;80.3647,7.4867;-89.1814,13.7361;-6.3500,32.3300;-6.1359,53.2940;-3.5265,18.9048;-4.0000,15.9333;171.6037,-41.7526;35.8356,32.3175;23.6000,61.5500;35.1797,32.7006;6.9167,51.5167;-114.7767,32.4631;13.7967,50.3570;-72.3333,-38.2167;18.9333,46.9833;-108.9667,25.7667;38.9167,-6.7667;152.8667,-25.2833;19.9197,39.6200;-75.6036,4.9825;16.3721,48.2085;110.9000,-7.0833;6.0000,33.5333;8.8078,53.0752;-86.6753,12.3400;-66.7717,10.1217;-59.5667,13.2500;8.0333,6.3833;172.6333,-43.5333;-100.3833,20.6000;-65.2167,-26.8167;5.4833,50.6000;-7.4500,13.5500;-171.6000,-13.8500;124.5981,39.8933;-99.0525,19.6011;16.5537,40.8292;57.6908,-20.1117;41.0153,43.0033;52.4289,55.7561;18.6939,45.5511;49.8500,-16.9833;-83.8667,9.9000;36.4542,36.1372;\
    62.1900,34.2400;9.6500,47.2831;13.3598,38.1158;23.1167,63.8333;19.3167,47.1833;57.1044,23.9811;106.9592,-6.3947;18.6036,45.7700;80.0567,6.2250;-56.5167,-22.1333;-74.9833,-12.7667;-83.2486,22.5872;21.2153,41.5136;-86.5381,12.8869;23.7000,38.0167;-46.5383,-23.6639;26.0728,52.1214;48.5158,34.7961;18.5466,50.1019;107.6175,-6.8117;127.2703,37.0108;-171.9833,-13.9167;36.6833,-8.1333;-79.5500,-4.3333;-89.2744,13.8350;-18.0667,63.7833;23.2642,57.0828;-171.8000,-14.0167;34.8458,32.0311;-171.7667,-13.8167;-77.6000,17.9667;46.9756,40.6253;23.7871,61.4991;-9.4500,51.6833;-79.7000,9.3000;175.8333,-37.3833;-6.1500,35.1900;97.0333,20.7833;-6.5833,42.5500;-97.1081,32.7357;48.5500,34.8000;-69.2019,9.5597;107.4594,-6.4136;16.5667,8.6833;27.2731,59.3986;-57.8333,-31.0833;46.5472,39.2039;35.8833,32.6667;31.4181,41.2894;15.5158,46.5394;5.9694,52.2100;9.9167,2.9500;37.7833,12.1167;-82.6167,8.4833;23.4333,59.9667;17.3333,-12.1667;129.7331,62.0339;10.6500,-3.4167;24.0000,42.8333;14.1000,46.4000;123.3081,9.3103;-106.6511,35.0845;125.8000,38.8639;22.9350,41.8928;29.7667,62.6000;-54.3500,-24.0500;-64.3594,9.3106;-76.2386,3.3275;10.6916,60.7954;85.0000,24.7833;-68.8333,-16.2167;-56.8500,-25.4500;-85.4333,10.6333;89.8500,22.4667;-86.2000,14.6500;24.3000,55.5167;70.5769,39.8533;28.4667,-26.4167;49.4622,37.4711;-16.6667,14.1833;20.9167,14.5333;27.9333,-26.6667;6.8667,47.6333;138.1500,9.5167;-57.7833,-33.9167;35.3833,-23.8650;11.1386,55.3299;25.6667,60.4000;14.4891,40.7155;13.4655,59.3232;-43.8617,-16.7350;89.2500,25.7500;-115.1372,36.1750;-69.7500,9.0500;33.8500,-8.8500;-46.8761,-23.5106;-8.0333,37.1333;57.7644,-20.3142;-57.4667,-25.4833;100.2667,13.6500;24.8000,54.7667;-14.8833,13.6833;-172.1333,-13.6167;30.5206,39.7767;21.2508,55.8758;34.9167,32.4000;-71.8167,19.6500;31.2833,58.5167;-84.3000,10.2333;33.0833,-2.8500;70.2094,34.6714;-102.2667,19.9833;20.1217,30.9539;36.6317,35.9297;146.0333,-34.2833;-78.9136,22.1928;-5.9500,34.5600;73.2000,22.3000;-118.1892,33.7670;34.7400,31.8764;12.0690,55.8396;\
    38.1667,-2.6833;-15.6167,13.4000;174.4500,-0.6667;37.5333,36.3333;5.1000,50.4833;-84.5333,12.0667;-2.1000,7.7333;16.4333,7.2500;-122.7530,53.9166;103.6500,1.6000;36.5667,5.7833;14.4508,10.1092;-8.5378,7.5794;32.4825,37.8656;-73.2459,-39.8142;17.5883,43.9381;28.3500,-26.5000;23.7500,61.3167;127.5239,39.0425;20.2028,41.3144;44.4081,-12.1567;-72.8642,18.8317;32.1167,-18.5333;-78.7964,21.7353;-76.4667,17.9500;15.9922,46.6733;20.7500,38.9500;24.5667,54.2167;-6.0353,54.5234;10.9254,44.6478;-88.7833,13.6667;-14.3500,65.3667;79.9658,6.5761;-3.9836,22.6783;16.1164,45.8311;24.3739,58.4086;79.8167,8.0333;-15.6333,13.4167;19.9708,41.4719;-105.2307,20.6204;33.1833,34.7417;71.6433,48.6814;-60.5333,-31.7333;99.9333,13.7000;9.5357,55.7093;1.5333,6.3333;43.4842,-11.8617;26.6667,47.7500;-84.6500,10.4833;20.6997,40.7058;14.8167,47.2167;35.9500,31.9167;-54.4000,5.6167;-79.9500,-4.3833;23.6000,46.7667;-76.7317,8.0981;5.1222,52.0908;18.5242,44.4053;10.9978,45.4342;-47.2669,-22.8219;-3.7329,40.3057;100.2207,26.8688;-57.6000,-25.3167;-59.3292,-33.1444;33.1667,14.0333;-62.0833,-38.8833;14.3115,50.2411;-80.6333,-5.2000;7.7000,51.3667;32.3833,-18.9667;-79.4500,-1.0333;89.9167,24.2500;22.0833,48.2167;41.8981,42.3181;48.3658,34.1919;13.5456,-12.3481;36.7833,-1.8500;32.6000,-3.8333;70.3333,42.5333;80.1039,6.1403;-14.9333,13.5333;39.0475,15.0700;175.9333,-39.0000;28.0842,54.0333;56.4658,24.7433;61.2131,28.2197;6.1727,49.1191;43.2608,-11.7844;30.7667,50.5000;-3.9333,38.9833;75.6833,29.4500;57.7494,-20.1981;174.0833,-39.0667;-1.3500,53.4333;61.2006,52.1908;16.3333,48.3500;13.2082,55.6441;-3.3600,34.2300;12.2500,3.7667;14.4741,50.3505;47.0833,-21.4333;12.3518,55.6154;-58.1972,-33.5442;27.8667,-26.2667;-8.6333,39.9167;15.3931,46.5406;10.7667,1.8500;32.0000,46.9667;38.2161,48.2322;-67.5263,-16.4096;6.2600,49.6475;22.8631,54.7731;18.9669,45.3764;-59.6500,-29.1500;44.5500,-19.7000;-86.3667,11.7833;44.3197,-12.2331;126.8664,37.4772;-1.0833,51.2667;141.1500,39.7000;105.8500,21.0333;-15.8833,13.2500;9.8247,4.8453;\
    17.6844,43.1214;46.2594,31.0542;-90.7339,14.6464;23.2311,57.1147;-71.2145,46.8123;28.2833,44.2500;7.4667,51.3500;23.9725,56.0856;-48.9528,-16.3267;114.5000,48.0667;-85.1000,10.4333;-83.1592,22.9036;-172.3667,-13.4500;50.8808,34.6453;46.0833,-25.1667;45.2428,32.0625;4.2000,50.9167;9.7332,52.3705;5.0431,52.0200;44.3133,-12.2308;44.4747,-12.3117;-88.3667,13.3500;95.6833,16.2833;7.6868,45.0705;0.3333,46.5833;26.1667,-26.1500;-84.9000,11.4000;52.7756,29.2450;-89.7525,13.9769;49.7300,40.3272;48.4789,40.0128;44.2639,-12.1981;176.4333,-1.3500;-172.4000,-13.4500;31.7150,1.6744;23.2167,41.4000;14.1136,46.3692;18.5633,45.5236;13.1577,58.5052;38.2333,5.6333;17.2364,44.8597;68.7739,38.5600;125.0942,7.9064;90.5667,24.4333;-124.8028,49.2413;24.4644,52.5567;6.2597,49.5408;126.6047,36.3467;26.7333,-29.2333;68.8167,37.9500;35.9333,31.9500;18.5500,4.3000;-1.3681,34.9514;57.6014,-20.0758;25.2667,60.3667;0.4667,51.5667;18.9847,45.4900;24.4306,58.9039;-99.2396,19.4785;-56.0447,-34.8131;9.5222,47.2107;44.5167,4.0333;25.9000,41.9333;-82.5842,22.8058;76.9667,11.0000;-8.1281,6.0667;-15.8333,13.3500;-74.0667,4.8667;128.8961,37.7556;-86.8466,21.1743;101.8000,2.5167;2.1993,41.4181;-63.2500,-17.3333;-86.2000,14.9000;-70.7617,7.0903;100.3543,-0.9492;-58.3000,6.0000;56.2719,26.9581;-72.4667,19.6000;24.7928,59.0072;27.1592,58.6550;6.1239,49.6769;2.2081,41.4515;10.1833,37.1833;1.5333,45.1500;56.5157,23.2257;35.7333,32.4833;40.6525,37.0700;14.7275,45.7386;-91.1833,14.7667;46.7167,-16.4667;15.2222,46.8153;-94.4167,18.1500;175.2833,-37.7833;-87.4472,13.4242;144.3607,-38.1471;-78.0667,18.2667;19.9167,47.5000;125.7761,38.6703;29.7297,40.7767;-172.2833,-13.7333;16.9333,-12.3833;39.3167,-4.1333;68.7803,37.8364;-58.4797,-34.6148;-0.4815,38.3452;-80.7053,-4.8778;-16.2719,12.5833;10.0000,6.0167;-79.3333,-1.8000;-65.2167,-20.6500;13.5239,45.4314;-89.2408,13.6731;4.2667,36.8000;116.8289,-1.2675;46.8500,-19.8333;47.3578,8.4756;73.5667,2.9167;175.0500,-39.9333;4.5333,50.7667;19.1333,47.7833;32.7625,34.6958;14.1072,46.0422;\
    22.9578,42.5444;-61.6333,12.2167;-1.3667,7.3833;10.5667,1.4500;81.7333,28.8333;6.2531,49.7072;-89.6167,20.9667;69.7025,40.4928;120.6672,15.0883;-79.7500,22.4833;24.6578,59.3022;23.6333,60.8167;33.9500,-3.3667;34.7167,-0.5000;175.3833,-40.1667;17.5394,49.2093;-80.3447,-4.9269;49.8739,11.4711;-2.5000,34.5800;-0.0167,9.4333;-73.1000,18.4500;-172.3500,-13.4167;15.7181,45.8031;28.8000,45.1667;44.2356,-12.1883;0.9000,51.8833;10.2273,45.5248;24.2250,59.0831;89.1333,23.4167;-76.4810,44.2298;15.9058,43.7272;-60.0250,-3.1019;-79.8133,8.8558;68.7500,40.1667;178.6667,-7.5000;-73.1333,18.2833;11.0333,-2.8500;-81.0544,7.9922;26.5297,41.5031;40.1833,-10.2667;33.8667,-2.0500;-71.2667,-32.8833;9.9667,52.1500;46.3158,40.5869;-100.4667,25.6833;46.1772,31.4175;26.9833,42.6500;36.0811,40.3875;-71.5350,-16.3989;34.3667,-1.3500;2.6667,6.7167;6.1275,49.6569;58.7000,22.5667;68.8308,38.7758;138.7000,-35.0000;44.1167,13.8500;155.5000,-6.2500;100.9000,12.6667;-90.4114,14.5461;-94.5167,17.9833;15.0500,53.3333;43.2842,-11.6397;19.9972,41.4931;30.5167,50.4333;34.8833,32.2667;-87.9167,15.4333;10.6411,35.8256;32.4372,15.6361;15.8708,44.8169;-63.4000,-17.3500;18.7667,53.4833;95.3258,5.5617;-84.4833,15.7500;18.8017,43.9397;45.0033,41.2028;24.1833,42.5000;38.3000,4.0500;10.6000,-1.2167;80.0600,6.7167;-89.6811,13.9867;-88.3500,13.5333;-105.5344,50.4001;-72.3539,7.7072;33.8333,35.0375;-16.5217,12.7856;-84.0333,9.9500;12.8903,43.9036;139.7833,35.8833;10.6068,55.0598;28.4164,4.5706;-79.5458,22.4947;172.3167,-43.6167;17.4597,49.0697;6.0986,49.8306;-61.4008,-32.8169;-83.5136,22.5047;89.9833,22.5667;103.5000,16.4333;15.0806,46.5103;18.9997,45.3433;10.9000,35.6500;-55.6500,-34.4833;26.8833,-31.9000;-81.3331,48.4669;-66.9667,-18.3667;50.7858,25.4297;78.7833,28.8333;22.9458,39.3667;-172.6167,-13.6333;17.1036,61.7290;12.5580,56.1997;13.8000,47.9167;139.8667,36.5500;35.1000,9.8000;16.0833,47.7167;6.0933,50.0178;50.8122,61.6667;57.7667,22.9333;179.3833,-16.4167;5.8408,49.5464;30.0167,63.3167;-91.3667,14.8000;89.7833,22.3000;\
    34.7722,32.0114;29.5767,46.8319;24.8836,41.1414;139.2667,-35.1167;89.5167,22.7000;4.2333,50.7833;4.6333,51.8175;17.8786,43.4086;-86.0333,13.5667;68.5333,25.1333;14.6344,46.6622;23.5833,46.0667;-59.7833,8.2000;16.4506,46.5614;106.7331,-6.4214;17.6667,49.2167;38.2167,8.6667;1.3750,10.3067;-14.7500,13.4333;121.2256,14.1789;35.7536,32.3069;49.9525,40.4872;-89.1800,13.6128;45.6289,40.9922;5.5181,51.7650;4.3500,43.8333;-64.9167,-19.1667;49.5586,40.6233;20.2008,41.4333;45.0367,12.7794;16.1667,48.2000;9.2833,5.7667;31.2333,11.4500;-75.5361,6.2914;-4.6500,34.5500;27.1503,38.4072;61.7317,34.3456;-55.9000,-27.3333;21.3786,41.1083;46.5667,-18.4333;58.5933,23.6133;-57.1000,-25.0667;-76.3036,3.5394;45.0947,41.4547;102.8333,16.4333;79.9989,6.4211;-24.0000,65.5833;-79.1500,-0.2500;107.5878,-7.0453;14.9032,50.4114;-88.0667,14.8667;35.2047,32.8486;36.9500,-7.7000;14.6667,47.1667;21.6000,56.7167;33.5064,39.8453;25.6943,45.6174;-14.1167,13.1500;79.9167,6.9000;73.8667,32.8167;26.7251,58.3806;-79.5167,-1.8167;-25.1000,17.2000;33.9783,17.6972;16.2072,58.9959;18.0103,49.5944;42.7728,0.4958;123.1814,13.6192;17.8351,59.4227;135.3333,34.7167;4.8500,45.7500;11.7167,55.7167;-171.5333,-14.0167;6.7762,51.2217;74.8833,12.8667;169.1500,-44.7000;-0.2417,11.0581;-3.0094,43.3493;-6.6669,53.2158;-56.0967,-15.5961;-75.7256,-14.0681;23.2833,41.5667;-2.4597,36.8381;-72.5167,18.5500;17.2500,45.1411;-60.6664,-32.9511;-71.3375,-17.6394;7.3500,5.8167;30.6167,-29.8167;28.4000,11.8000;-1.7833,51.5167;-85.2500,10.5333;1.1333,8.9833;7.1575,13.3075;46.8128,40.2153;17.4000,44.1436;-78.5667,-1.0333;54.4361,36.8394;15.3722,10.2806;33.8917,35.0083;35.8175,37.0247;24.2667,42.9167;34.7667,31.6000;48.8000,-19.9000;5.0833,36.7500;-78.2667,0.2333;13.1713,50.3845;-78.6333,-1.5833;-99.8901,16.8634;17.5450,43.1969;-77.0317,18.1368;-63.3833,-17.9833;-96.1333,19.2000;121.6022,38.9122;16.1679,60.1454;9.6661,4.0844;34.8953,31.9514;39.0500,-7.6333;7.6000,50.3500;9.3761,47.4297;33.1292,35.1333;22.9488,40.6667;-81.9236,23.1539;35.2333,32.7500;\
    35.6933,31.9786;-59.5333,13.1833;95.2167,19.3667;-80.6167,7.9000;9.8167,2.3333;174.2833,-39.5917;-73.7500,18.2000;16.1833,54.2000;18.5333,46.3000;32.0025,36.5525;-87.5511,13.4689;23.2786,56.6250;92.3000,20.8667;12.5808,44.0633;19.9167,-11.7833;-117.9145,33.8353;48.6414,40.6303;90.5333,22.8167;23.5681,47.6573;34.0167,-8.8500;16.3875,46.3758;81.2333,8.5667;35.2750,32.9931;-91.4833,14.9000;16.5528,59.6162;-76.2667,-10.6833;31.1833,-26.5667;91.8450,22.8625;-100.3167,25.6667;3.3306,36.6486;-79.8417,-6.7736;-8.5100,32.8300;22.1000,56.1667;-80.2781,25.8576;65.7536,36.6672;-1.6167,6.6833;34.8386,31.8539;-171.8833,-13.8000;76.6000,8.8833;68.1347,37.2594;-14.2167,13.3167;-71.8333,19.0833;33.3917,34.9208;73.6500,30.6667;38.9500,-7.7333;-87.9833,14.4833;47.9833,-14.8667;30.8333,-14.5500;37.7667,8.9667;-76.2414,20.2978;22.1958,41.7458;4.2986,52.0767;16.3803,45.0458;103.8800,52.5300;130.4000,33.5833;-79.2167,-0.9333;-74.5539,-8.3791;94.4167,23.6333;120.4064,16.0692;45.9217,41.6189;18.3267,45.0111;-79.6167,-3.6833;-57.8000,6.5500;-79.9736,22.1494;-88.0000,15.6333;101.6167,3.3667;-3.7000,42.3500;108.5000,-6.7000;30.9500,18.0500;80.5500,29.8500;-99.6078,19.2536;9.9325,4.9533;32.9000,-2.5167;88.3697,22.5697;27.4000,8.7667;27.2000,43.3500;13.6494,45.9714;90.0667,22.4833;-57.5000,-25.6167;25.2000,61.8667;2.3189,35.2122;47.9617,29.3472;-85.4667,12.8333;37.0167,-0.1667;-65.9500,-24.4333;42.7506,0.0722;175.7000,-36.8333;12.8568,56.6745;2.0833,49.4333;-58.4167,-33.8833;20.9233,41.4842;15.5744,46.3928;8.4647,49.4883;23.2250,43.4125;26.0308,58.9631;-66.1000,-10.9833;97.2094,19.6742;-72.6525,7.3781;20.7500,-8.4167;-14.2667,13.2500;18.3167,47.4833;22.3000,48.6167;18.9667,46.1833;-68.9333,-22.4667;18.4000,54.5833;20.3356,41.1969;-55.6500,-34.0333;9.2043,48.4914;25.3342,42.8747;1.6669,9.7050;-83.7667,15.2667;33.9333,-9.9333;18.9858,46.5264;28.8833,55.4333;-76.9000,18.2333;90.6500,22.6833;30.1025,-4.5767;-74.7872,9.7919;22.4203,39.6372;50.2150,40.3639;-21.6833,65.7167;-73.3678,5.5353;22.3333,56.3167;\
    -72.3350,18.5392;-79.7833,8.8803;29.0756,47.0917;3.7667,32.8333;121.3258,14.0669;-86.2683,12.1508;-172.6833,-13.5333;34.4928,0.2094;20.0981,40.6122;87.1000,53.7500;29.1558,54.8589;11.0039,49.5897;68.7192,35.9425;45.5833,3.4000;100.4000,5.4167;-13.8767,64.8350;22.1228,42.1819;16.9353,47.2540;44.5244,33.8183;21.2833,46.6500;27.6286,55.6156;3.9333,7.8500;-3.3667,40.4833;44.2525,-12.1906;13.7383,51.0509;6.1000,46.1833;30.9158,52.5386;21.5500,29.0333;-4.9800,34.0500;-0.9833,37.6000;18.4269,44.8808;139.5000,-20.7333;-63.8628,11.0333;23.5000,51.1333;22.3000,52.1667;3.9523,50.4541;-54.9000,-24.9667;-63.0500,-17.8167;21.3267,55.5536;11.3875,59.1330;25.9333,60.9500;29.8333,52.2500;-80.0000,-2.1833;71.0600,34.9400;-100.4000,25.6667;31.3297,-17.2964;17.8064,45.2886;44.7278,39.9106;57.7575,-20.3428;3.7500,51.0000;26.5167,43.5333;23.8833,37.9000;-79.5333,9.1333;9.6769,35.6350;0.1077,49.4938;1.2228,6.1319;37.4667,12.6000;109.1333,-6.8667;22.1546,65.5842;5.4667,50.7833;-76.8500,17.9333;10.5333,57.3333;128.7989,41.2428;-75.6903,45.4209;36.2731,34.6211;-1.5833,6.4500;-23.7667,15.2833;-48.5044,-1.4558;57.4661,-20.4822;31.1808,52.5592;48.1303,29.1450;101.2833,12.6833;-71.7333,19.5500;43.2403,-11.7042;31.7742,0.9161;22.6000,57.2500;-2.1500,5.4333;24.1000,42.2167;21.0133,41.9317;-58.0333,-31.4000;16.6216,47.2309;34.9500,32.6000;8.0500,52.2667;134.5256,7.5275;72.0131,33.9967;4.3333,7.8500;23.2361,43.8139;20.0089,41.6103;-72.5667,-37.0833;-90.9269,14.6317;3.8964,7.3878;6.0003,50.1211;21.1833,56.8833;34.9314,30.9872;36.0833,32.9833;-71.6506,8.6219;127.4547,40.0183;23.2333,45.3500;15.7392,-12.7761;-54.5881,-25.5478;105.9800,9.6033;13.2713,50.3760;-171.8000,-14.0000;5.9333,43.1167;69.8719,38.7811;110.3333,1.5500;6.1469,49.5356;26.9833,-16.8167;127.2592,37.4092;28.1783,47.0814;-2.4333,53.5833;21.5000,61.1333;37.2000,-0.6667;8.6500,47.0167;12.0803,55.6415;21.1667,56.5500;56.7439,24.3689;-21.1500,63.8667;37.1000,9.5667;101.0500,4.4667;4.8625,52.2067;44.0706,40.1297;28.0594,46.6736;-56.2194,-34.6650;\
    -87.3167,15.6333;23.5333,55.8167;95.3833,21.4667;18.1583,44.4150;27.3333,-17.6167;-80.2667,22.3411;-57.8333,-33.5167;-88.9333,14.0333;19.3000,48.0833;22.6417,41.4761;102.1500,12.6000;21.3500,37.8000;8.5426,47.4082;40.9186,34.4500;139.7536,35.9764;153.0167,-27.4833;19.0833,5.7333;-62.0833,-31.4333;11.5000,3.5167;27.1500,-17.3667;58.5469,22.6903;-75.6908,-11.4197;-78.2167,0.3333;24.6667,60.2167;27.9167,62.3167;-75.5206,5.0700;9.5087,47.1650;28.0686,47.8161;-85.0833,12.0667;-67.4833,10.2333;-77.5994,-11.0700;106.7383,-6.3428;-75.6114,6.1719;27.8333,-12.3667;94.8667,21.1667;-77.8000,-0.9167;146.8833,-36.1167;168.0167,-46.3500;-22.1000,64.3167;48.4672,29.9681;-17.1211,14.9192;10.1833,5.1500;-71.3733,-33.0422;50.8167,9.4833;5.5228,22.7850;26.6167,-27.3667;57.6319,-20.2697;86.2000,26.6500;100.9333,4.7667;27.2000,60.5667;-122.8026,49.0164;33.9667,0.1000;35.8333,32.2833;10.2667,6.2500;169.5500,-0.9000;38.9769,45.0328;28.0000,7.7000;-76.2667,17.9167;8.9736,33.7019;5.7167,45.1667;13.2344,-8.8383;10.0208,35.9339;-171.4000,-14.0000;-113.9187,53.5334;23.5625,43.2100;11.1000,55.6833;134.6122,7.7172;58.1264,23.4064;37.3667,36.0667;71.2500,34.9800;25.7333,62.2333;39.9086,-16.2325;24.9422,59.4056;46.9242,41.4225;169.5406,-0.8692;-89.3168,48.4001;128.3236,34.9728;22.2500,15.1167;126.7319,35.7281;-73.9781,9.0047;-87.9530,15.6144;-77.3333,18.4667;168.3333,-46.6000;44.7908,41.7250;-86.6500,14.5500;-23.6833,15.1000;28.6333,-12.9667;126.6114,40.9872;-2.9500,15.0000;-57.5500,-25.1667;142.2000,-36.7167;-55.1333,-27.4833;51.2558,35.5644;-79.8150,8.8703;-74.6386,-11.2522;33.6455,49.0097;21.0642,41.9803;25.7500,57.6500;10.4667,5.0667;45.5333,9.5167;28.8833,-11.2000;102.3500,3.4833;-65.3167,-17.9333;-77.2500,17.7667;-76.4333,18.0667;18.4058,45.6803;37.5500,48.6167;43.2772,-11.5725;44.2950,40.2797;-81.8561,22.9114;135.8049,34.6851;-2.1183,53.5405;14.1100,40.8320;-91.7333,14.7167;127.4181,38.8558;2.9131,36.1367;95.3167,18.2833;-86.5833,14.0333;9.9187,57.0480;5.4778,51.4408;-122.9109,49.2068;3.2044,9.9339;\
    30.5522,37.7644;35.6000,30.8333;173.4667,-34.9833;-73.8203,-41.8697;46.0278,38.9081;21.5167,47.6667;35.2667,-14.4667;26.1378,53.5653;80.7889,6.0231;18.9585,47.4618;33.9250,35.0417;7.4474,46.9481;108.9833,11.5667;4.2167,51.1333;28.6833,-26.1500;-68.0039,10.1806;-35.0147,-8.1128;-56.3333,-33.7333;69.7061,37.6128;-69.1833,-12.6000;-85.3833,12.2667;124.8469,6.5031;49.5833,-16.9167;7.3500,46.2333;98.7500,3.6833;49.4167,-17.3667;36.4833,32.5167;-47.6492,-22.7253;33.3042,34.8667;0.2845,50.7687;9.1346,39.2074;-0.6833,37.9833;6.6194,52.5758;10.1797,36.8028;38.3119,38.3533;72.7081,42.9153;-84.1667,9.8000;10.2500,6.1333;-62.6486,45.5834;-73.2182,-39.8087;27.2833,54.0000;105.0772,14.9050;14.4667,48.2000;22.1814,42.0792;120.5711,15.9761;13.9111,45.8875;44.3939,33.3386;-53.8333,-34.1667;37.1814,55.9825;32.4333,-2.8500;-95.3633,29.7633;34.6333,13.4167;24.0000,42.0167;130.3953,42.5206;44.8631,40.7408;89.2167,23.1667;21.6833,47.8000;172.1167,-43.4833;0.4667,6.6000;176.8333,-37.9833;66.7500,44.1667;-79.8667,8.7833;-56.9000,-27.4000;91.7509,26.1862;73.0667,33.6000;9.2340,45.5345;82.1667,29.2833;-58.1667,6.8000;50.2833,-14.8833;106.9167,11.5333;99.7841,11.8210;44.0308,40.1522;16.4676,48.2945;121.4000,37.5333;61.6525,32.7622;-71.2311,-32.4524;-84.0667,9.9500;5.4000,43.3000;28.7350,47.0739;-66.6167,10.4667;-72.2214,7.8236;39.7167,-10.0000;79.9619,7.1733;-89.5597,13.9942;12.9401,57.7210;-172.5500,-13.6833;170.5036,-45.8742;-77.2333,17.8000;6.4222,49.8056;-7.5000,39.8167;-86.5833,13.0500;-75.5883,8.4147;-70.0806,10.1778;25.1000,31.7667;10.4731,35.9567;18.2333,54.6000;37.7942,37.0097;112.5603,37.8694;-49.2539,-16.6786;125.5567,38.0475;-14.0333,13.3833;-14.9667,13.5500;15.5333,10.9333;140.0833,35.5167;38.2844,14.1050;-86.8000,13.4333;23.1258,43.2361;-72.7167,-37.8000;47.7208,30.7186;20.7236,41.2572;-70.1667,-20.2167;42.7261,42.7856;-76.8500,18.3000;-57.2000,-26.2167;-15.9667,13.3500;-73.2389,10.0358;72.3500,30.0333;-79.2833,9.0833;33.6167,35.0333;-60.9500,-16.3667;79.7953,7.5758;28.5833,-20.1500;\
    27.4500,-26.9000;153.2667,-27.5333;136.2226,36.0644;14.3111,45.8133;31.7500,-19.0667;-70.0333,-15.8333;25.6333,-25.8500;-86.1167,11.9333;104.6500,14.6500;26.4000,-12.1833;-172.5000,-13.4833;80.7500,7.9500;73.4167,61.2500;-64.6167,10.2167;13.0333,48.2667;-77.6139,-11.0964;50.2333,10.2833;144.7333,-37.5833;48.8569,32.0436;25.9500,62.4167;34.4333,-1.2333;70.4667,43.1667;7.5281,36.1289;20.0000,47.1833;28.8000,-16.5167;-75.7967,8.8856;35.6481,34.1211;22.0000,50.0500;-43.9378,-19.9208;12.5667,56.0000;-1.8500,38.9833;6.1825,49.8669;125.6647,39.6178;-0.3333,5.7000;136.6167,34.9667;0.5486,51.3891;29.3858,47.2375;31.9500,-26.4500;-57.0500,-25.4833;-91.3667,14.9167;22.2597,58.3358;20.8853,41.8339;22.8833,47.8000;147.3167,-42.9667;43.0514,39.7194;7.3928,35.7964;45.8161,40.5656;9.8000,33.9000;17.8847,45.5400;-73.1667,-40.9167;-67.6325,10.3092;32.7667,-9.3000;43.4425,42.5794;16.4683,49.7559;16.5415,68.7986;67.0125,30.1872;35.6178,33.9808;39.2000,-7.1167;12.5000,50.7333;68.8036,38.4647;-57.5667,-33.6833;-65.6000,-22.1000;9.7331,47.4831;49.9644,40.4217;48.7333,-14.5500;22.7167,48.4500;-8.5300,32.2500;7.2661,43.7031;5.8708,51.7500;25.6500,44.9833;-109.9333,27.4833;91.8717,24.8967;27.1942,42.3489;-2.1000,7.1667;106.9167,47.9167;100.8833,12.9333;18.1637,59.3105;18.6167,7.6500;-47.4017,-22.5647;30.4500,-30.7500;18.0531,43.9667;11.3833,53.6333;46.0333,41.8167;-98.9667,19.3500;11.0606,35.5006;12.1833,55.6667;30.8000,-27.0000;121.1114,14.3122;63.3072,30.4347;-84.0833,9.9667;101.1125,15.6589;19.6913,46.9062;-15.7500,13.3500;126.2333,38.4167;44.9256,31.9878;81.6000,7.7667;2.8249,41.9831;47.6542,40.5183;16.6656,44.7653;15.5478,45.4872;49.2000,-17.4667;-5.6600,32.9300;90.2072,23.1753;-17.0422,20.9025;57.6797,36.2144;122.9900,41.1236;-56.4333,-25.7500;108.4833,15.5667;0.5646,50.8530;-79.7333,-1.4500;79.9842,6.4739;38.3667,8.1167;23.0333,42.6000;25.4414,59.4439;9.7464,4.7161;24.2833,55.0833;-64.8019,46.1159;19.8000,47.1667;24.8667,42.0167;3.0903,36.6044;126.4856,37.7472;48.2000,-18.9333;139.4833,35.6667;\
    3.2000,50.9167;-1.0667,5.2000;57.5611,-20.3706;-115.4683,32.6519;57.6447,-20.0186;47.2500,-20.5167;-16.6819,13.4781;121.6589,42.0156;-79.8333,-2.2000;-66.8500,-14.8167;31.4567,-0.0772;7.0167,43.5500;15.8817,49.2149;15.3792,46.3822;-72.9369,-41.4717;7.4500,51.5167;125.0961,38.2508;36.2706,54.5358;37.7000,11.9167;44.9478,38.5525;1.9667,7.9333;-91.5833,14.6500;80.5275,7.2536;-56.2833,-25.5667;18.3333,49.9167;-88.2167,16.9667;16.5975,41.2002;-0.2167,5.5500;35.6700,32.4183;3.3333,50.8000;-7.6100,33.0000;-3.7026,40.4165;33.7833,-3.1833;23.3675,52.3731;174.7500,-1.2167;69.6967,40.2386;46.0333,51.5667;-86.0500,11.8833;163.0036,5.3147;-90.6983,14.4056;20.6789,40.3378;-0.1667,5.6833;4.2500,51.9233;11.0167,-1.8667;27.7583,54.8961;-90.1922,14.7950;25.3333,-27.1833;29.2667,54.6667;128.3447,36.1272;-119.7724,36.7477;-79.5500,8.9667;10.0167,5.8833;-90.5517,14.5028;120.9261,14.7983;-72.2442,11.3842;36.6333,32.3167;89.6333,23.0500;-72.0662,-45.5752;24.3333,42.2000;24.8378,59.3069;-79.9842,21.8019;-88.9833,13.8333;43.8669,40.5722;5.0501,50.9834;17.8233,44.4864;105.1167,10.7000;15.8117,45.1894;40.7000,7.1333;114.3167,4.6167;73.4000,30.3500;102.6097,12.8506;113.8672,35.3089;-97.4595,20.5331;38.4158,48.9192;33.8625,35.1000;-67.7271,-16.1889;44.9000,2.6167;-57.2500,-25.2333;100.6333,4.9500;15.9926,59.5140;16.1792,44.9600;10.0000,-2.6500;42.1908,41.6300;69.1167,27.8500;70.6500,28.6500;57.5167,-20.5167;29.4081,38.6800;-44.1042,-22.5231;13.1800,32.8925;-76.1500,-13.6667;-68.7333,-11.0333;37.5928,15.1139;20.2667,46.6500;18.4167,-33.9167;24.9833,44.1167;-99.6672,19.2883;13.5883,45.5161;32.0500,-5.0667;-13.8667,13.4833;-72.8000,19.8000;30.1500,-16.9000;34.9139,32.0619;-172.0000,-13.9167;-84.1000,9.8667;61.6667,46.8000;-8.3400,33.2800;41.4394,11.5636;-54.5736,-25.5991;-67.3583,9.9111;35.9500,35.1833;49.8822,40.3953;31.1333,-26.3167;8.7195,47.4949;49.2833,-12.2667;77.5500,29.9667;-15.8167,13.2333;23.6667,37.9833;22.5833,-32.3500;15.1000,47.3833;45.9472,35.6203;25.8864,59.0386;-9.1167,38.6167;92.2197,22.1981;\
    -23.5167,14.9167;33.3167,35.3417;103.6667,1.5333;29.3289,52.1333;120.7700,14.9533;80.4000,9.4000;7.1500,46.8000;-75.2414,20.5819;57.5581,-20.0731;11.4500,4.3667;44.4647,-12.1950;36.2139,30.2933;35.5097,33.8719;20.8303,41.3028;-74.4958,20.3467;104.5333,11.4500;12.6333,6.4667;125.7097,39.2781;-83.9667,14.7333;46.6433,41.6336;2.6253,36.4669;-0.8773,41.6561;36.5000,34.8833;8.2667,4.8000;21.5544,41.3464;16.7131,44.9814;26.6167,44.0500;-85.6333,12.7667;173.9667,-35.2167;44.5319,-12.2569;-58.0833,-29.2000;18.7556,45.0503;12.6136,56.0361;35.5286,32.7956;120.9608,14.7369;21.1500,51.4167;35.6014,31.0558;-174.9500,-21.3333;115.0000,4.7500;25.1000,55.5333;-4.7667,37.8833;18.1720,40.3570;2.3500,6.4500;25.1053,59.2000;-75.5000,-11.8000;-9.3000,38.7667;11.8667,55.9667;17.5278,59.5671;90.2167,23.3167;-84.1500,10.0000;17.0839,44.2814;47.6500,-16.7833;-44.1983,-19.9678;127.1000,36.1294;32.3056,2.7667;70.5667,40.2000;46.0561,37.3404;4.7486,52.6317;106.7103,-6.5400;-91.4500,14.8167;20.5353,41.0667;34.8117,32.0697;114.2667,30.5833;43.8175,42.2708;174.7500,-36.8000;139.6333,35.8833;29.8167,-19.4500;36.5667,-0.9333;29.9333,-27.7500;-106.4833,31.7333;113.0333,3.1667;35.1525,32.5158;28.2833,-15.4167;-63.9039,-8.7619;8.8223,47.2256;106.2000,21.2667;57.5886,-20.3467;17.8332,59.1986;-8.0714,12.7467;105.9667,10.2500;-80.9058,22.7192;25.1333,43.3667;35.3089,33.0497;51.0500,11.8500;22.7694,41.9672;37.5014,41.0278;26.1667,55.1500;-6.5406,53.8597;16.8275,46.1628;12.6089,13.3156;35.4944,32.9667;-103.3167,20.6500;6.1917,49.5150;130.8418,-12.4611;6.1336,49.7158;16.9990,49.2775;3.2250,51.2098;23.1736,57.2464;4.4792,51.9225;39.8500,-3.6333;-73.5500,18.2667;32.5386,1.9756;96.4797,17.3367;-0.0167,7.8000;-0.3000,5.5833;27.6167,-16.5000;38.3167,6.4167;89.9333,24.9167;11.1167,33.5000;-1.3500,5.0833;26.1333,46.0000;-58.9208,-34.1769;25.2000,54.6000;49.9836,40.4264;43.0861,36.3292;47.8045,35.1664;43.3844,-11.4728;32.8667,-19.8000;25.5694,41.9403;-84.0833,10.0000;150.8833,-34.4333;14.9833,-10.7333;14.4746,50.1082;\
    39.0333,12.0333;127.1314,36.9156;28.8500,-16.0333;-84.2417,9.9149;-23.7167,64.8833;22.2667,4.6500;27.3500,53.9333;-77.1333,-0.7167;20.1500,46.7000;129.7000,42.1092;73.9783,31.7131;38.0500,48.3000;7.7667,36.9000;79.8358,7.2086;7.1333,50.9833;19.2750,50.2053;18.8500,-33.9333;34.2833,10.5500;-79.9014,9.3592;3.6892,36.1503;21.1236,42.0736;127.6733,26.2072;59.6508,55.1711;18.6261,49.7461;-56.2142,-34.0956;45.2064,31.5261;108.3611,46.3611;7.0000,49.2333;-62.7000,-32.6167;-16.1744,13.5053;-88.5833,13.3167;8.7333,41.9167;146.2667,-38.1667;-0.1953,35.5278;100.6333,4.2333;27.4833,-16.2833;-46.6361,-23.5475;30.9911,54.2833;-79.3333,-2.4000;19.7131,41.6356;15.0500,-7.6167;10.0000,53.5500;28.8575,47.0056;15.2066,59.2741;72.8667,32.9333;-64.8000,-23.8000;30.8667,-18.2667;-85.9833,14.0667;12.2431,44.1391;-77.2667,18.0833;5.1000,51.1833;-8.0000,31.6300;-47.9319,-19.7483;-56.3394,-34.1892;39.3833,-4.6500;23.1333,60.3833;100.4333,13.9167;-89.4169,16.3311;81.8167,7.4167;-78.7619,21.8400;-9.3333,38.6833;102.1020,14.9707;-6.0983,53.2028;34.0018,34.9821;5.9333,45.5667;18.3500,49.6833;50.1389,40.4922;43.2561,41.5739;-79.8072,8.8522;-171.5333,-13.9000;47.7083,40.2183;-67.4500,10.1833;33.7333,35.2583;173.1167,-41.3833;24.8825,57.7669;-78.2667,0.3000;32.4469,0.0644;-115.7688,49.4999;-86.5833,13.4833;-25.0167,17.1500;-63.2500,-32.4167;-2.7300,34.9300;29.9833,39.4242;26.8333,-29.2167;-73.1667,-38.7000;123.8917,10.3111;8.9000,9.9167;-2.7167,53.7667;-68.8500,-16.2333;14.4000,27.0500;44.5592,39.9589;33.2625,35.0625;44.3778,40.3339;105.8442,21.5928;-171.5667,-13.9167;-47.4008,-20.5386;-56.0333,-24.8000;43.3864,-11.3875;-17.0050,14.5128;9.6000,47.2331;8.0000,17.0167;-65.4167,-24.7833;103.9000,1.7333;27.4333,43.1833;-90.3667,15.4833;6.0625,50.8658;17.0777,59.6361;20.2251,67.8557;18.6694,44.5428;-6.2392,53.6819;-3.0074,16.7735;18.9167,50.2833;7.2000,51.1833;-77.1000,18.0833;-0.8833,52.2500;6.1514,49.5681;-79.9667,-3.4500;25.3167,54.6833;-82.4000,9.3000;28.8231,47.3831;23.8833,47.9333;122.2333,-17.9667;\
    71.4422,33.5869;103.4833,14.8833;49.2469,39.3586;16.2500,47.8000;-87.8936,13.6247;17.4667,4.3167;69.1628,54.8753;3.2649,50.8280;16.5425,46.0219;-25.0000,16.8833;47.8175,30.4956;98.6000,12.4333;18.0767,43.9425;-88.0333,15.5000;2.2833,48.9167;12.0528,44.2236;174.8833,-37.8000;15.6260,60.6036;43.9700,42.2300;-9.3333,38.7000;-23.5167,66.0500;-4.4203,36.7202;29.9833,-26.5333;70.6344,30.0561;19.7447,40.1017;41.1000,9.2167;140.3833,37.4000;7.4339,36.4661;33.6292,34.9167;-4.5333,42.0167;55.4500,-4.6167;5.8564,49.5464;44.4711,40.0575;46.3386,39.5078;16.2000,51.2000;72.3119,31.2742;-76.4633,-8.4592;112.2331,-7.5460;-78.4283,21.9469;-91.5667,14.8667;13.8217,47.9819;14.3083,45.3392;-82.6167,8.5167;37.8500,8.9833;4.4667,50.8833;32.0500,-18.2833;91.1425,23.2472;-7.2522,52.6542;20.0667,32.1167;9.7833,0.3833;2.5167,48.9500;-90.8875,14.7411;21.2500,-28.4500;6.7650,14.5106;70.7833,22.3000;18.7167,46.3500;8.7833,3.7500;36.9500,-0.4167;33.5867,-16.1564;-80.7497,43.1334;-77.7167,18.4167;-75.5622,6.3389;6.0233,49.6275;16.2517,43.5125;21.3278,42.0000;-84.1333,9.9333;2.4333,11.3000;-114.9819,36.0397;45.3567,40.1239;35.5686,33.2081;-64.7000,10.1333;-64.3260,-23.1322;-16.5813,13.4564;-85.0833,12.2667;36.9790,-3.5618;-54.1833,-32.3667;-60.2269,-33.3303;38.2569,-16.8403;-64.5167,-31.4000;10.8500,35.6833;24.8667,43.7500;5.9500,49.6700;10.6333,5.5000;38.4000,9.8000;-1.9333,5.3667;101.6911,2.9353;15.8969,45.0544;108.3167,22.8167;23.7000,37.9500;-6.1333,36.6833;115.7253,49.8807;116.9969,32.6264;113.2167,-8.1333;50.5831,26.2361;28.5833,43.8000;-86.1000,11.9000;71.6333,29.5333;-79.7933,9.3644;-59.0350,-34.0922;-68.6833,-16.0500;36.5544,40.3139;128.5008,35.8017;20.9458,41.9903;18.3097,44.7033;67.9000,27.8500;57.5508,-20.4853;-45.8869,-23.1794;127.5333,50.2667;28.7167,-16.5333;106.9272,-6.9167;21.0167,56.5333;79.8933,6.9897;26.8333,45.1500;123.3414,8.5894;-78.7833,-2.9000;79.8825,6.7733;23.0826,58.5592;26.4322,59.2958;39.8500,-3.6833;12.8500,47.4167;17.9897,45.1375;99.8333,19.9000;2.9386,11.1342;-75.6961,4.8133;\
    -70.6667,-33.4500;24.7535,59.4370;-77.8333,-10.6667;34.9175,32.4353;102.1500,2.2667;25.0833,56.6167;39.6333,11.1333;33.4958,34.8611;99.6167,2.9833;-10.0600,28.9800;-6.0400,35.4600;-88.8667,14.5667;-0.2500,6.0833;10.6167,2.1333;10.4662,61.1153;24.0833,61.4667;27.6000,-28.3167;136.9064,35.1815;27.6667,46.2333;8.5167,8.4833;-11.1000,28.4300;16.4217,45.8833;36.3667,33.5333;32.9292,34.6708;10.0436,12.4481;-77.5000,18.0333;-83.0481,22.7142;14.5239,59.3267;1.8266,41.7250;96.1561,16.8053;18.3575,49.9041;-64.2000,-21.5333;17.4822,42.7625;150.6000,-34.8833;-87.0333,14.2667;103.5167,1.5167;-47.9297,-15.7797;10.3050,36.8181;0.4708,10.3592;44.6519,41.1306;17.5928,43.3833;32.3136,0.2250;38.9094,47.2214;-89.8486,13.8703;129.3278,40.9642;-7.5606,12.8661;29.9586,46.7339;-82.3167,8.4000;-66.7667,-18.9000;25.4281,58.7900;2.3000,49.9000;36.5000,32.3333;-172.5500,-13.6833;46.7186,39.0300;35.8667,32.0167;51.6456,35.3253;28.8303,46.0550;40.5219,41.0208;88.9500,24.8000;34.7500,0.2833;174.4833,-39.7583;2.3667,51.0500;33.4167,49.0667;44.4361,40.0672;5.5333,50.6500;43.4361,-11.7544;17.2518,49.5955;-72.7833,19.2500;27.9289,47.7617;-16.5347,13.4839;-6.9803,52.9914;123.0778,10.9000;6.0667,33.1000;-16.2264,12.8103;0.9000,7.4667;21.9667,51.4167;34.9000,32.3167;27.6833,55.1333;18.2000,50.0833;98.4000,7.8833;25.5483,58.6283;7.6333,46.7833;-3.6104,14.3500;34.9739,32.1142;27.3667,44.5667;5.4167,8.8667;-86.4500,13.6500;-77.9667,0.4000;68.6733,37.3486;37.2667,10.7000;-76.8000,18.0500;19.0167,50.2667;-8.6167,41.1500;44.2819,41.1244;-77.5264,21.2350;11.4833,2.0833;19.9714,41.2156;162.9822,5.3631;44.9369,33.9786;26.1667,55.3500;-0.7500,5.2667;9.0167,36.0833;-87.9667,15.2333;18.1448,49.5995;49.4158,40.0842;40.0167,11.1833;33.1744,0.6422;90.9833,26.8667;-9.2914,38.6957;32.9167,34.6792;0.3000,7.0000;176.1667,-37.6861;14.4743,48.9745;-64.1069,-18.4897;44.2289,41.2017;44.5225,-12.3564;25.8333,-17.9333;-0.8514,10.7856;-69.2167,9.5667;30.9667,-25.4667;9.8503,55.8607;12.2223,55.5329;15.6000,5.9500;35.0703,32.8508;-8.7000,41.2000;\
    8.7833,-0.7167;-84.1333,9.9167;-80.5500,8.2667;-2.8900,34.4100;8.4778,13.4239;22.4500,60.5000;17.1667,11.7833;-8.9833,38.6500;35.7761,32.6983;-71.9364,18.8294;22.9331,54.0994;-118.2437,34.0522;-8.4667,39.4667;17.7903,44.1544;31.3600,0.5892;14.5144,46.0553;-84.0167,9.9333;-14.6500,13.4333;118.8933,-2.6786;19.4333,50.5000;29.8047,46.6514;23.6153,56.2403;145.0333,-38.2167;19.7178,41.4783;21.9333,47.0667;37.6500,0.0500;121.6172,13.9314;-9.2000,38.7667;-83.6981,22.4175;-72.2250,7.7669;1.9000,47.9167;32.9000,-13.8000;1.7167,6.6333;105.7833,10.0333;30.4167,19.5167;-57.3833,-33.8833;148.6167,-32.2500;44.6631,40.2511;28.2500,-26.2167;-2.3200,34.9200;-69.6667,-16.0833;7.8333,5.7833;51.3658,32.6331;48.1792,30.4392;102.1749,44.6702;43.4886,-11.9097;48.6911,31.3292;178.0042,-38.6533;-8.5333,41.1500;50.6206,26.1547;-59.4500,13.1000;10.3160,43.5426;22.8725,43.9900;-102.0667,19.4167;-2.9973,43.2956;-76.5125,-8.1842;-74.2017,11.2472;-85.1667,12.2167;3.0617,36.7128;2.1333,48.8000;120.3719,36.0986;38.0167,-5.4167;7.7000,11.0667;16.6445,49.3630;169.6500,5.9167;20.0978,39.9511;131.6044,33.2372;5.6333,6.3333;17.0079,58.7530;59.5289,22.5667;30.3797,10.6364;-40.3378,-20.3194;125.9778,38.9914;-74.9500,-14.8333;-15.4000,13.4167;-88.3000,18.5000;-72.6500,-38.9833;31.7775,-17.6469;10.8833,48.3667;46.3214,40.7917;24.7000,42.5000;30.7594,0.9381;35.2000,-0.1500;15.9667,5.6667;-171.7833,-14.0167;45.1342,40.2892;2.4500,41.5333;-89.0386,13.5436;-70.7167,-33.6000;52.0789,43.5942;8.7842,34.4250;-81.7833,8.4333;29.8500,62.7667;-87.8000,15.4000;66.4597,30.9169;5.0000,51.1667;-56.4333,-24.5333;44.2444,41.5442;50.2903,40.3636;105.4308,21.3019;6.1990,46.1881;34.1333,-5.9500;103.3167,2.0333;18.6839,45.2886;43.6747,-12.2508;106.8542,-6.4817;11.0333,50.9833;-85.8500,12.7500;34.2694,0.6328;66.6636,35.4897;20.4500,-11.6500;76.9167,52.0333;31.2000,-26.4667;-8.8500,53.5167;-84.1333,9.9833;18.2708,43.9508;4.8597,51.6450;78.5833,25.4333;-172.4333,-13.4500;13.6833,4.5833;46.6175,41.4783;11.6207,44.8268;14.3708,40.7839;\
    -57.2667,-25.9667;43.2764,-11.4208;-9.2100,30.4000;88.4667,26.0333;8.4500,55.4667;101.3650,6.8617;5.2917,52.1733;-120.9969,37.6391;1.4913,42.4637;105.6667,18.6667;19.9500,60.1000;10.1667,5.9333;2.6833,6.9667;-55.6833,-34.6833;44.0089,36.1900;22.7000,63.6667;124.3947,40.1292;16.3833,6.3167;46.9111,41.4725;-67.6000,-16.3000;9.7471,47.5031;23.7000,38.0333;-171.5667,-14.0000;3.5944,36.5622;38.2783,37.7592;-84.6333,9.6167;3.9333,6.8167;24.7159,42.8943;79.8764,6.9839;9.4898,55.2538;12.6499,42.5671;-65.1667,-18.1667;36.1333,-8.9333;34.9719,32.7617;128.8811,35.2342;3.9833,6.5833;104.8631,15.2331;114.3289,36.0994;21.6667,57.2167;28.5000,61.1000;177.3667,-39.0333;14.2333,-1.5833;7.7281,63.1103;43.7775,-12.3325;58.1442,48.8264;-64.4333,-31.6667;25.0833,54.6833;74.7333,19.0833;34.5833,31.4167;22.4128,41.9164;7.7500,48.5833;43.4631,56.2389;71.2000,30.0667;2.3597,13.1081;-67.8100,-9.9747;-98.9500,18.8000;-87.0667,15.6500;20.9061,41.7861;5.9289,49.5481;8.5635,47.2918;31.9000,-28.8833;48.1167,-20.0500;-3.1800,51.4800;21.0167,56.5167;-74.0058,5.0283;13.3000,4.9333;91.2000,27.2500;99.0500,2.9500;18.6000,53.0333;5.7486,52.7108;59.9650,57.9194;10.1530,55.9731;97.5675,16.0800;91.9147,22.5383;70.9644,30.4700;2.1590,41.3888;10.7667,35.6333;29.4769,4.9147;104.4250,43.5708;29.7044,54.8042;23.6167,54.8833;57.5244,-20.1261;44.5053,-12.2394;2.1019,34.1089;33.1792,35.0208;125.8042,39.8133;35.7833,32.5167;22.5167,56.2333;-15.3333,13.5500;18.4301,49.8453;-8.4167,11.9333;24.9667,55.8333;49.2180,36.1468;2.7500,36.2675;27.2500,42.7000;-58.1869,6.8075;14.3108,40.9260;31.0167,-29.8500;79.9000,6.9500;14.9058,44.9894;8.4853,47.3895;-60.1742,-35.4281;45.8853,35.9975;123.9933,12.9739;28.3167,-26.1833;32.9583,34.6042;-85.5833,10.2667;9.3047,48.7396;-81.2900,22.7272;113.9833,4.3833;4.8999,52.3731;14.1447,11.0425;22.6911,42.2839;-20.4000,63.8333;-84.3833,9.9833;-1.3150,34.8783;31.4833,-26.8833;69.1714,35.0136;69.5083,40.1686;15.3747,44.5461;-84.1833,9.9833;27.7058,38.5008;-89.3500,14.5667;-6.0600,33.8100;23.8000,38.0000;\
    57.5539,-20.1117;12.1975,56.0228;-97.3208,32.7254;-1.7000,4.9333;44.6703,39.8647;-16.4500,66.3000;43.3092,-11.8456;105.7772,20.9725;36.3500,35.0500;109.3333,-0.0333;-124.3195,49.3163;90.3500,23.2167;89.5500,22.8000;11.5667,0.7833;26.5322,57.8661;-51.1794,-29.1681;-8.2319,55.0503;35.4667,-0.2000;6.6328,46.5160;37.6833,-3.4000;-9.1333,39.4000;5.4667,50.6000;-9.1833,38.7833;-88.5000,13.3833;-78.9833,-2.8833;18.6997,45.6281;152.5167,-32.1667;6.1042,49.8475;48.6656,39.0342;-70.9667,-32.8500;6.9069,52.7792;26.5000,42.4833;82.6833,22.3500;34.9500,1.9500;97.0167,17.3000;95.5000,18.4833;47.9306,28.6392;24.8942,59.3542;69.7639,53.8478;-90.8244,13.9256;-92.9167,17.9833;15.3133,14.1194;-88.1833,15.3833;57.5725,-20.0650;-15.3167,13.5667;-8.6500,40.6333;121.1753,14.5864;14.8619,56.1706;175.5000,-37.2833;13.0333,47.8000;25.2997,53.8833;30.2000,-17.3667;34.5167,-15.6167;-70.7333,-32.7500;31.8167,18.4833;14.8037,40.6632;-72.1569,19.6967;15.2000,48.6167;144.2167,-5.8667;-10.1600,29.3800;91.9792,22.2947;138.6000,-34.9333;-4.9167,14.4667;2.8667,48.9500;35.6928,31.1589;24.4036,56.0594;34.9517,29.5611;92.2322,22.4714;13.6167,47.7167;29.8911,0.4186;20.8875,41.8658;5.7833,53.2000;-88.6333,13.9500;36.8006,35.8636;-88.4000,15.3500;-171.5833,-13.8500;8.9333,55.7500;-171.9167,-13.8167;10.3167,5.4500;-80.5856,22.9819;106.8167,10.9500;-88.5833,14.5833;-80.1667,-0.8500;-88.5333,13.5000;8.0182,58.1599;-84.2833,9.8333;24.0833,60.2500;135.1667,34.6833;-171.9167,-13.9500;135.5157,34.7614;50.5478,26.2186;-63.2492,10.6697;119.4386,-5.1464;-75.8508,20.1875;8.7500,47.5000;33.5750,34.8500;61.8347,30.4353;25.0500,57.2333;-79.4667,-1.4500;57.6875,-20.2889;161.9167,-10.4500;18.0500,46.9000;5.0893,60.4619;38.6000,7.2000;6.1000,53.1000;21.5667,53.0833;13.2422,46.0619;31.1333,-8.7667;28.5833,49.9000;16.1514,44.8825;11.0464,12.8681;18.2667,52.8000;-103.4000,20.7167;68.9533,34.5875;-63.5968,44.6520;-79.8830,44.7501;74.5000,15.8667;10.9000,35.6667;-86.5653,12.2675;-15.0667,13.6167;38.9167,5.7500;-55.9667,-34.3000;95.5500,20.0167;\
    5.0328,35.9461;-81.1833,8.1500;10.7511,36.4694;43.3775,-11.5014;8.4799,47.3824;8.5761,47.4037;19.9758,41.0317;-74.0776,40.7282;106.3500,10.3500;36.1606,36.1989;73.2667,6.1500;134.6342,7.4933;14.6928,45.1772;124.6472,8.4822;134.1294,6.9014;31.1475,-17.8900;15.6833,47.6000;27.5539,47.9572;15.6167,58.4167;11.9639,55.3235;20.6667,50.8333;115.7000,-32.5500;-84.1167,10.0000;57.7000,-20.4081;57.5789,-20.4628;70.3794,39.0242;-100.8167,20.5167;-77.0833,17.9000;30.2333,-13.2333;75.8500,30.9000;-99.1386,19.4342;-3.8044,43.4647;29.9167,-18.3500;-59.4344,-34.6544;16.1975,46.5208;24.2897,56.9411;-89.5333,14.9667;-23.2500,66.1500;27.9767,40.3522;20.9000,51.1167;24.7158,56.7442;11.0667,47.3000;15.8000,9.3167;12.1064,57.7395;19.6833,51.4000;145.4000,-36.3833;-0.3500,5.8000;37.7333,18.4333;122.8333,10.5378;45.1267,40.3589;19.1392,50.2423;41.9542,41.9069;-87.3742,13.2006;13.1833,3.9833;-88.4167,13.3833;49.0333,-17.3500;40.6728,-14.5428;-73.2094,19.6628;120.5931,16.4164;85.4167,27.6673;107.5183,-7.0331;-0.1833,5.8500;31.0333,-6.3667;-61.2350,13.1583;51.5333,25.2867;-73.4658,45.4501;33.0042,34.7500;-171.8000,-13.8000;27.5853,48.2244;44.5850,40.0006;26.7167,56.9000;38.5667,-3.3833;48.0167,-22.1333;45.3667,2.0667;-90.6569,14.6097;21.9031,45.6886;26.5892,43.2592;177.0000,-37.9833;-76.5225,3.4372;87.2833,26.6667;40.1500,35.3333;22.5025,41.1392;55.9939,30.4067;49.7717,40.4122;-89.6667,15.4000;150.5000,-23.3833;-71.9667,-36.4167;175.5833,-1.9167;66.8386,36.0261;-9.1685,38.8309;51.8667,47.0667;28.4000,-26.2500;-0.6342,35.6511;30.2461,53.5183;127.2142,37.6367;24.8422,59.4967;134.5578,7.3575;31.5167,-8.3000;32.4594,0.4044;100.4667,5.3667;-8.5333,41.1667;74.8667,32.7333;175.2167,-40.7167;30.4167,-15.6167;8.2500,50.0833;4.9000,8.0167;16.3167,48.0667;-81.2330,42.9834;38.7928,37.1511;-15.7333,13.3833;-14.3833,13.3000;35.0858,32.8275;-99.1833,18.8667;-21.0000,63.9333;33.1350,2.8006;135.0928,48.4808;24.2333,50.3833;47.7941,52.0247;47.6333,-21.3000;9.0036,33.4833;99.8539,20.1472;6.5667,53.2192;18.7000,4.9667;\
    -0.3667,43.3000;25.2667,62.7167;112.0167,-7.8167;36.5667,32.8667;25.1136,43.0258;80.2333,9.8167;43.2781,-11.8217;76.8333,17.3333;4.8667,50.4667;-74.7533,9.2414;32.2631,48.5042;-60.4739,-34.6417;15.0217,45.9042;20.0822,41.1125;96.5278,16.7619;-172.3333,-13.4333;-74.6692,5.4522;43.2500,3.8000;20.0167,42.0622;41.8500,36.3167;27.6833,62.9000;173.8833,-35.9333;80.5958,6.8972;-79.9061,-6.7011;9.6331,47.3331;104.7167,16.5333;36.1167,34.8167;-110.0016,53.2835;32.2167,14.4667;6.3258,49.6069;44.1850,33.3083;100.4167,6.2667;101.8667,3.8000;56.2436,26.1917;51.4219,36.6547;18.8533,44.8800;-2.2414,4.8683;-77.7667,18.1500;5.2500,13.8000;42.5992,42.3397;-80.0500,0.0833;26.4167,55.6000;40.3500,-11.3167;-17.4553,14.7089;114.6517,37.8792;36.5800,50.6100;-72.3992,18.5411;48.2167,-17.5833;-79.6533,22.3119;35.0333,31.0667;-66.3806,9.8622;4.6500,52.4600;-81.7006,12.5847;11.9424,58.3478;-61.1333,13.2000;10.9167,35.0333;-81.1944,22.8044;70.2000,52.9333;2.6167,9.3500;111.5167,-7.6167;20.9833,50.0167;-80.9333,8.0500;90.1833,24.3000;5.8000,51.1000;-123.1193,49.2497;38.0667,-0.9333;10.3290,44.8027;-8.5308,7.3022;35.6000,32.5500;-69.0833,-16.1500;109.2344,-7.4214;-14.6667,13.5333;57.6364,-20.2264;49.0500,-13.2000;5.9294,49.5128;-54.7000,-25.6500;44.4800,-12.2158;10.0500,56.4667;23.2500,-14.3833;6.1300,49.6803;10.0295,59.0538;-90.0647,14.8539;-78.7500,-8.4167;13.4000,52.5167;4.1667,50.4000;30.4208,0.2111;177.1500,-7.2500;-79.6833,8.9167;16.4428,43.1725;-74.1724,40.7357;1.6000,42.5667;-76.9533,20.2753;-68.3000,-54.8000;13.6619,45.5369;-15.4833,15.3333;33.3208,35.0750;-74.0333,18.2000;68.8333,24.6500;-90.0490,35.1495;151.9667,-27.5500;23.3056,57.0617;32.1000,-28.8000;-121.4944,38.5816;33.7167,-13.5167;-4.4100,34.0500;100.0667,6.6167;3.3958,6.4531;-60.0306,-34.9083;-14.9333,13.7833;-6.6814,53.6528;20.0283,41.2869;5.8806,49.5583;4.3889,51.9192;2.0833,41.4667;68.0167,36.2653;22.4075,40.7919;8.1333,53.5167;31.8667,13.0167;-49.0606,-22.3147;-84.4500,11.6833;44.0208,40.0456;31.1333,-25.9667;-16.7000,14.4167;\
    39.2667,8.5500;-62.0492,9.0686;86.0833,55.3333;-8.1000,55.0167;24.4019,40.9397;20.1397,41.4206;21.7167,47.9554;5.7069,51.2517;2.4333,6.3500;26.5325,54.0831;26.9500,56.5500;16.0000,45.8000;23.0471,-34.0363;44.2514,40.2336;75.7667,23.1833;-80.2664,43.1334;-56.6833,-24.5000;72.3683,33.7722;24.2000,-13.6000;14.5167,50.1333;106.6956,-6.2375;129.3522,36.1125;20.1933,41.3086;37.0833,34.8167;30.3364,53.9139;45.7222,36.7631;-8.5167,41.1500;23.1167,-13.5500;34.8397,32.0903;-88.3500,13.5500;-15.2667,13.4833;33.1197,0.9472;64.1925,31.1333;8.1333,33.9206;-80.0833,-1.8833;127.1247,36.4556;9.7903,56.6431;42.4500,1.6333;-1.7500,53.7833;44.2692,40.2706;95.9333,16.7167;20.1458,41.2825;12.0333,4.5167;5.0833,7.8167;125.6156,38.3992;29.2500,-26.9500;6.6500,46.7833;48.0947,29.1458;5.9839,49.5758;-172.2667,-13.4667;121.4667,-30.7500;7.1053,36.7394;135.5022,34.6937;23.8833,46.5500;29.5500,-11.3500;-83.5833,9.2667;13.7753,50.5485;36.7333,34.0167;104.1500,17.1667;-1.7833,52.4167;115.0882,-8.1120;121.8908,17.1481;105.2667,-5.4500;8.7209,47.3471;74.0833,32.5667;-2.9984,51.5877;-171.7833,-13.8167;39.2206,38.6753;-0.3353,53.7446;42.2167,3.3167;47.9781,29.2692;139.5522,35.6556;24.8000,60.4667;48.4500,-13.6833;3.1167,50.8000;24.1000,44.3500;10.6167,4.2333;27.8167,-20.4833;-78.0667,-7.8000;-89.5500,14.8000;66.4369,35.8497;102.2333,2.4667;-9.1000,38.6167;31.0481,-17.9939;22.7278,41.4406;10.1167,33.8833;12.5709,55.6777;-76.0564,1.8675;10.2889,36.6817;-74.7731,10.8589;173.0167,1.8167;40.6453,10.1664;19.6436,41.7836;32.6833,-18.8833;102.7333,16.0667;-79.6167,-2.6667;15.6167,47.2167;26.7000,13.6000;24.0156,35.5122;-3.6667,16.4167;104.7833,17.4000;50.8072,25.2142;18.0776,59.5344;28.3014,47.8933;7.4403,10.5231;74.4467,31.1156;110.4167,-6.9667;-59.8167,-22.3667;14.1910,40.9274;130.4622,42.3578;-80.2500,7.7500;57.7361,-20.2261;16.6394,43.7036;4.3556,52.0067;9.7936,37.1558;-84.0167,9.9667;19.0333,50.3000;-57.9167,6.6833;35.8969,32.4394;-18.1000,65.6667;32.1472,39.5842;-0.5667,44.8333;-101.2000,20.5667;17.9711,59.3613;\
    73.4458,30.8081;-84.3833,10.1000;36.5667,35.6000;10.4500,32.9333;73.0700,6.6221;34.3500,11.7667;29.8833,-27.3667;69.3778,40.5786;15.3975,46.2172;-73.3833,19.8000;129.4703,41.2006;4.4000,45.4333;-56.8500,-26.8667;-85.8333,12.9167;-59.5167,13.2167;-87.9667,14.7000;-71.4833,-35.5500;46.6333,-16.1000;11.2833,2.3833;-84.3167,9.5167;-56.2236,-34.7617;158.2711,6.9503;-13.9667,13.3667;24.1833,42.7000;1.2500,45.8500;69.1100,36.6800;-57.4167,-25.3000;-51.1836,-29.9178;-78.6333,-1.6667;13.8653,46.5656;-119.4857,49.8831;-21.9833,64.0833;-80.2500,-0.7833;22.3646,53.8288;127.0389,37.7486;8.4001,47.4017;12.7597,46.8311;6.0397,49.4892;70.4500,34.4200;41.9169,34.4686;29.0833,-9.7833;5.9815,50.8837;21.6158,63.0960;116.6783,23.3600;-46.4028,-24.0058;-9.4850,52.4464;33.5903,4.7717;-82.6793,27.7709;9.2667,34.4667;-57.2333,-25.8000;-55.7000,-25.3667;123.7558,13.1467;-87.1833,12.8500;5.2167,51.0500;17.9331,11.0597;-55.9667,-26.1000;-15.5569,12.7081;-75.6375,6.0900;-86.4500,11.8500;25.0000,-13.4667;57.3631,-20.2742;-84.0000,9.9000;29.2500,12.3500;55.9464,53.6331;26.9167,46.9167;-88.7833,14.7667;24.1906,56.4075;-9.3500,38.6833;28.1944,45.9075;-14.9333,13.5000;9.5329,46.8499;-67.4167,10.2167;35.3494,33.0150;3.0506,36.7631;37.5833,0.3500;35.1000,31.7167;-99.5167,27.5000;20.4167,47.1833;38.5000,9.0667;-73.8200,5.6189;160.1500,-9.1000;-7.8017,52.9511;-79.0092,-8.1711;-79.8711,22.6169;38.7167,7.9333;73.5099,4.1771;55.0986,51.7847;113.2167,-7.7500;21.0067,55.3081;0.2500,29.2500;145.8667,-6.2833;64.3033,34.1083;113.6486,34.7578;-55.1667,5.0167;4.4667,51.1667;18.9667,-33.7167;175.8667,-37.2000;-81.2047,23.0375;23.1833,55.6167;-60.9589,-34.5850;174.3000,-39.4292;-88.0333,14.1500;8.3000,47.0833;25.7283,43.1111;8.0833,5.8000;19.9771,47.9126;50.7981,30.3586;18.8500,50.3167;126.9850,35.0594;17.8339,44.9450;63.1206,52.9594;120.9367,14.4297;-60.5725,-33.8925;33.2958,34.9500;176.1333,-5.6667;-73.8547,7.0653;-72.0656,18.2372;10.9828,31.8683;25.9000,56.0167;-86.2167,11.8500;79.8731,7.1647;35.6000,30.7333;28.2725,53.0392;\
    -72.4333,-37.9500;95.4667,17.6333;23.6833,37.9833;37.2164,19.6158;104.9167,11.5500;31.5592,6.2072;114.0098,-7.7062;-7.3606,53.9908;48.0600,39.8697;-80.7500,-5.4000;6.2395,46.3832;-91.5000,14.5333;12.7500,42.0167;4.1333,51.1000;-1.3808,35.3036;22.2472,55.9814;-71.7833,19.4000;32.9667,34.9000;104.9000,12.7000;101.5328,3.0851;48.2372,38.9206;25.8667,-17.8500;24.8000,42.6333;-65.7667,-17.5667;27.5667,53.9000;13.7667,47.8000;71.5558,37.4917;-72.1167,19.3333;17.9819,59.2370;15.5807,49.6078;-76.1317,21.1097;18.8839,45.1467;14.4092,45.3431;-91.1500,14.7333;-38.6531,-3.7361;56.4872,49.1447;23.9747,55.2883;4.2167,50.5000;112.6281,-7.9778;45.4539,40.1608;44.6411,33.7500;9.4056,4.2972;-54.7167,-31.8000;23.8833,38.0000;27.6667,56.7833;66.0472,37.4128;11.2333,4.7500;26.9211,54.4975;-77.8500,18.0167;68.2000,37.4167;23.1814,57.0086;113.2792,46.6806;57.5208,-20.2175;36.9500,-0.5667;74.2167,16.7000;8.2833,47.3667;67.9692,41.2547;-10.0467,5.8808;12.5500,55.8333;43.0386,36.8086;10.6667,35.7167;38.7658,48.0169;12.5852,59.6553;69.3308,40.5217;-59.9570,46.1970;90.9500,24.2167;36.7500,35.1333;24.5667,46.5500;-80.5675,22.2775;40.1167,-3.2167;62.4936,52.5369;32.6333,-10.1333;-38.5431,-3.7172;37.9833,2.3333;42.9294,11.3103;32.4000,34.8250;-5.6500,40.9667;146.3333,-36.3667;-74.9183,10.7997;10.2423,55.0951;27.2333,-27.6500;25.9489,44.5614;22.4167,62.6167;-77.9833,-1.4667;-88.0833,13.7667;110.3417,20.0458;65.7000,31.6100;44.3056,40.0839;17.9319,44.4083;7.0500,51.3333;6.0694,49.7875;135.7538,35.0211;12.0000,51.5000;28.3000,-28.2333;-88.2478,13.2539;10.3361,36.7331;-76.4194,3.2336;-2.9167,53.2000;-8.6500,41.1167;-91.6833,14.5333;57.4989,-20.1619;-8.1736,51.9094;-78.1500,18.3000;-80.8500,-2.2333;26.5333,-33.3000;22.0189,41.4169;-16.5456,13.4828;-34.8553,-8.0089;98.6167,3.5500;38.7833,7.0667;-72.4000,19.7000;8.5478,47.4218;77.4000,23.2667;12.2000,-5.5500;19.7347,40.5989;112.6150,26.8881;33.9344,36.3778;174.9167,-41.2167;32.7308,39.9683;112.4536,34.6836;-69.1500,-16.0167;106.2078,50.2314;-89.8572,13.7869;\
    24.1667,54.6000;61.1978,34.6539;-77.2833,17.8333;-72.3889,18.9917;137.7333,34.7000;37.1833,-0.7000;42.3352,42.1537;10.8777,44.7824;11.3167,1.6333;120.9453,15.3075;50.9982,35.8288;44.3100,-12.1936;4.4500,51.1500;-91.0167,14.3333;175.6575,-40.9597;114.1833,4.5833;89.2500,24.0000;41.8361,11.7158;21.5333,56.2667;24.9839,52.5336;109.6667,-6.8833;126.8856,39.1694;32.6000,46.6333;95.1833,16.6000;80.9667,6.7667;66.3167,51.1000;19.4414,41.3231;126.7850,37.8594;10.1000,34.2833;27.8333,43.5667;17.9368,40.6277;9.2200,49.1403;14.9782,58.0372;100.0000,13.4000;114.6000,-28.7667;102.2167,5.9167;12.5334,57.9303;33.0917,35.1042;76.7872,30.7372;-72.6833,19.4500;127.7764,36.1750;36.7167,34.7333;114.0656,32.1228;10.6667,1.0333;88.0833,26.5333;-8.9333,39.7500;91.6419,48.0056;9.0333,56.5667;-91.7000,14.6000;-55.8833,-26.9833;21.8442,55.9114;102.1500,5.8000;27.1833,45.7000;22.5833,57.3667;-89.1897,13.9572;25.2500,57.3000;31.9500,-27.0333;23.8167,38.0667;-86.1194,12.0047;-67.5167,-17.5833;95.9797,21.8787;5.3244,60.3971;106.7833,21.0333;90.4000,24.7500;-3.0000,53.4167;5.6667,50.8167;13.1927,55.7028;37.9667,48.0333;131.2500,-0.8833;-72.2167,19.6167;126.6414,41.4608;38.4167,6.7500;-85.6500,10.5167;44.0122,40.1144;-83.6833,9.9000;45.2519,40.9744;8.5610,47.3548;45.2486,40.1328;-57.0833,-24.4500;31.1167,-23.9500;6.9114,46.4330;43.2647,-11.7161;4.8833,45.7667;-77.2333,17.9667;-100.5167,28.7000;73.0667,19.3000;-80.9330,44.5668;25.4167,55.2333;11.0697,11.7119;25.6372,58.3600;32.6256,41.1964;-55.9833,-31.7333;111.8167,2.3000;13.5036,59.3793;57.2072,50.2797;152.4667,-31.9000;-83.0333,12.1667;-63.6667,-17.4500;27.9167,43.2167;33.5917,34.9500;147.7333,-8.8833;5.9806,49.4958;19.7928,41.5092;32.4583,34.8167;-82.6667,8.5000;34.5167,-13.9833;120.5361,14.6761;-80.5164,43.4668;27.7036,48.3572;18.0000,3.8833;165.8333,-10.7167;44.4289,40.5108;43.2528,11.4611;-83.7500,12.0000;5.8694,50.9983;-77.0167,18.3167;100.5000,13.9167;49.2092,39.3214;-63.7500,-17.4000;71.3500,35.0300;28.6500,44.1833;68.3000,25.3667;101.8236,6.4244;\
    35.0667,-7.2500;-79.7697,-6.7178;-6.0494,52.9750;57.7706,-20.2992;24.5500,64.0833;17.3667,9.3667;126.0303,39.8672;48.8200,34.2942;14.1081,59.3098;-80.2022,22.1494;-123.1368,49.1700;35.5833,8.3000;43.5556,42.0225;27.5817,59.3997;173.9167,0.4167;12.3667,-6.1333;122.5644,10.6969;13.8558,46.6103;51.4244,35.6719;-15.8000,13.7500;44.2767,-12.1956;3.8336,36.5350;4.1879,50.4866;145.2833,-38.1000;-56.0372,-34.4975;57.7625,-20.2078;21.1000,46.6833;-73.6350,4.1533;7.7464,36.6839;50.6119,26.2572;-4.9000,13.3000;9.8442,10.3158;25.4833,54.8500;26.4833,46.4667;18.0878,44.7347;83.7667,27.9833;30.2333,-28.1667;26.6417,53.0653;20.2833,11.0333;-90.6764,14.6353;28.0833,-26.2000;4.4167,51.2167;77.7000,8.7333;-65.9667,-26.0833;73.4000,55.0000;-77.2333,17.8667;27.0497,59.3600;59.2200,35.2736;30.9000,-19.0167;74.2617,31.8025;72.8667,29.8000;103.1333,5.3333;33.1792,35.3375;-8.5500,41.2167;-15.8833,13.3167;33.2000,51.2333;22.6167,55.4167;110.2864,25.2819;29.2247,46.8789;37.2569,49.2128;99.5072,18.2983;5.9833,52.3475;-3.5275,50.7236;27.7000,-26.1667;69.7094,38.7200;20.9083,41.7972;36.2333,7.2833;126.6297,36.8944;-54.6464,-20.4428;-99.0964,19.6658;-1.7833,5.9667;48.4356,41.4264;-73.1500,-40.5667;-75.1972,7.9869;31.6167,-27.2167;47.9331,29.2950;120.9367,14.3294;33.0000,34.7000;14.2167,45.7769;2.3833,48.9167;-73.3872,-13.6556;33.2792,35.0875;35.7833,35.5167;-79.1167,9.2667;110.6556,47.3194;19.8522,41.2761;127.7378,34.7442;21.4167,12.2167;21.8892,45.3008;31.6667,-21.0500;-57.3500,-25.1500;23.5667,45.9667;66.8961,36.7586;1.0447,9.9578;17.9167,47.1000;100.2500,16.8333;22.7208,41.1875;-48.8456,-26.3044;11.3546,55.4028;-79.9342,-6.8367;-0.0333,39.9833;-79.6833,0.9333;139.4039,35.3261;19.7603,41.3817;127.0192,37.2842;35.1939,33.2733;34.8389,-19.8436;-55.7500,-31.2000;26.7252,58.3162;44.6789,32.3078;15.5911,45.9033;49.7333,-15.4333;-76.8672,17.9702;-60.3333,-33.2333;17.6252,59.1955;24.7000,62.2667;4.8333,51.1833;-0.7667,6.5500;21.2869,29.1081;22.2489,41.4050;0.9000,5.8000;-75.4478,8.9494;-56.7333,-25.5333;\
    4.7333,7.6167;8.3858,49.0047;-88.3667,13.7167;-172.7667,-13.4833;-15.3167,13.4000;-79.6167,9.1500;6.9986,4.7892;91.4100,23.0186;-171.3667,-13.9667;4.4333,50.4167;28.0333,43.7000;-172.7833,-13.4833;102.7444,17.8761;24.9667,45.8500;118.1833,39.6333;2.0811,49.0412;-14.3000,13.3167;27.0500,45.3833;15.1219,46.2564;48.4787,36.6736;71.6833,29.4000;151.1000,-33.7000;-14.3948,65.2653;22.4500,-33.9667;-2.1600,34.3100;-84.4000,10.1833;17.6628,44.8703;49.9481,40.4425;33.3875,34.8250;-34.8731,-7.9408;7.8472,36.7022;29.7339,52.6292;2.1833,7.7500;-79.9000,-1.3667;28.2225,36.4408;-84.3167,9.8500;-74.3678,4.3439;30.5333,-19.2833;126.5544,37.9708;23.7500,37.9667;15.1839,46.0681;23.7500,38.0167;1.0993,49.4431;6.0653,49.5683;-15.4799,15.3484;37.5178,36.3725;126.8450,36.6792;24.4333,-11.7333;-57.6667,-25.2667;12.5833,55.7500;-81.6954,41.4995;-96.8067,32.7831;116.1667,5.3333;-66.8333,10.5167;-75.6725,4.8347;26.1833,56.3667;10.0280,45.1362;6.1741,35.5560;23.8167,38.1000;6.1457,46.2022;-23.5000,65.8667;31.6167,-7.9667;35.8500,-0.1667;27.3167,56.5000;46.9500,-19.4000;10.7397,47.2450;-73.1158,46.0334;15.9127,50.5610;174.2500,-38.9250;-82.5667,8.8500;15.2730,37.0851;18.1181,44.1272;25.8000,46.3500;18.2000,50.3500;-14.0000,64.8000;-15.3333,13.4833;6.0128,49.5064;13.0657,52.3989;51.3667,51.2333;0.6167,41.6167;9.7526,55.5657;17.0011,43.7231;145.3833,-6.0833;-9.1015,38.4445;13.8237,58.7097;23.2667,45.3833;61.9528,33.2961;16.1000,47.9500;1.2644,10.6197;10.8000,35.7833;17.7038,50.0897;-75.6175,1.6175;5.7500,5.5167;99.7667,13.9000;6.1028,49.5186;19.2886,46.6206;27.6500,42.5500;21.9497,41.5161;14.2959,40.9066;-73.5167,18.5333;26.4333,-12.1000;114.9333,4.8833;25.1503,52.1897;-73.1917,19.8533;-106.6345,52.1168;102.2833,2.2167;23.3500,42.9667;-6.5400,53.5125;-72.3028,18.5447;-80.0756,22.8067;79.9942,7.0917;101.5000,2.8167;20.1467,41.4431;12.3607,55.8086;25.9667,43.8833;2.3833,49.0000;115.8833,28.6833;32.4583,34.7833;19.8778,40.7689;117.8917,-35.0169;-85.5833,11.5667;-75.0167,20.3667;-66.0167,-17.6167;13.6436,45.9614;\
    12.3633,55.7316;-3.8600,35.1500;11.9106,55.0080;46.7511,39.7564;45.0000,53.1958;129.6844,41.7136;14.4049,67.2804;175.1500,-40.7583;23.2167,42.8167;22.3644,55.7406;15.3000,44.0667;4.9500,51.3167;11.1211,46.0679;-80.4167,-1.5667;16.1629,50.4167;-79.4667,-1.5833;10.9333,35.3833;15.9308,45.0883;4.6889,52.3025;-63.2167,-17.5500;44.4817,-12.2297;27.3553,41.4067;-14.4000,13.3833;-64.3000,-19.3000;-6.2994,36.5336;-47.8103,-21.1775;25.4325,58.8086;125.3572,6.7497;108.4728,-6.7050;134.6389,7.6217;34.9167,32.3000;36.2042,16.1528;23.8500,49.2500;37.0500,35.0167;69.6403,40.5689;69.8306,37.1275;-63.4000,-18.0167;68.4458,32.8472;19.5569,41.1856;134.6331,7.5514;9.2203,4.0128;9.4333,5.3000;57.7256,-20.2975;124.3981,40.1006;39.9167,8.9000;9.3167,36.3333;8.5116,47.3738;15.0128,45.9625;123.9494,10.3103;23.7333,38.0333;35.1833,47.8167;-77.1500,18.1333;37.6167,-1.8000;100.9333,13.1667;19.2833,47.4000;-70.9353,-17.1956;22.5092,41.8828;37.6500,-0.3333;68.7917,33.1467;45.4539,41.1189;-43.1786,-22.5050;17.0861,44.4158;101.6067,3.1073;8.2203,33.9858;4.1333,51.1667;25.1500,55.9667;-69.1833,-12.7333;-3.9600,35.2300;17.9064,44.2014;-84.0500,9.9333;8.5211,47.3376;-64.2833,-36.6167;18.5417,49.8540;24.6675,58.4006;-80.7333,-0.9500;18.6139,45.5686;-54.5167,-24.0833;-6.4667,53.4000;-65.0500,-42.7667;35.2667,-16.9167;47.5167,-18.9167;10.1556,36.6769;113.7032,-8.1660;-171.6500,-14.0167;-72.4833,19.4833;9.1833,36.7333;23.7500,56.9608;73.0333,5.1167;-0.4655,39.4371;5.9750,51.5250;-77.1167,20.3433;34.7503,32.0231;101.5833,3.3167;28.0667,-14.6500;43.4819,41.8047;28.1833,-15.7667;-71.3000,-41.1500;-57.5667,-24.9667;35.6836,32.6550;20.9383,41.6072;116.8333,6.8833;21.1231,41.4672;-5.3198,35.8893;-85.6000,10.4000;35.7775,32.5667;47.1581,31.5750;-3.2000,55.9500;18.5833,49.9500;24.3944,56.7428;-35.9761,-8.2833;174.3239,-35.7317;-55.0667,-25.7833;19.3833,54.1667;140.7367,41.7758;44.5167,-22.2833;10.8000,52.4333;-84.6167,10.3667;29.6833,6.8000;-86.8600,12.5206;-80.8222,-5.5569;72.9789,31.7200;-82.4333,8.4333;90.2833,22.6333;\
    -79.8500,8.8394;-77.1500,-12.0667;142.1500,-34.2000;-80.4333,7.9667;-5.3667,13.9667;73.6833,32.0667;25.7214,59.5764;16.4889,13.6425;57.4453,-20.1894;-174.3500,-19.8000;8.4807,55.6211;175.9895,-1.3419;4.7597,52.9533;24.3000,-17.4667;35.0500,32.7650;14.9208,46.5469;28.7333,-9.3500;89.9625,48.9683;16.0833,48.5500;43.3875,-11.4281;24.4131,59.3036;32.4167,34.7667;172.6667,-43.3167;141.3000,-6.1167;-67.4833,10.0333;18.9500,50.4000;138.7333,-34.6000;-18.6500,66.0667;45.0622,9.4081;69.3883,53.2775;-75.7186,20.9625;44.2344,33.8467;9.5333,47.1000;36.0333,-7.6833;-16.4896,16.0179;15.1434,59.1277;-90.7928,14.6478;-86.5500,16.3000;-100.2000,25.7667;57.6303,-20.0903;14.0104,49.6899;14.1420,66.3137;106.6667,10.7500;-1.0000,51.4333;81.5000,6.8000;29.7167,11.0167;2.1833,48.9500;19.8661,41.3153;122.8122,9.9889;45.7283,40.1806;-9.3833,38.8000;-0.0667,5.6000;22.4000,8.0667;4.6119,50.6696;15.1875,44.0811;-74.2500,-12.9333;14.5750,46.1669;85.2833,27.6667;-46.8842,-23.1864;14.1524,56.0313;39.4500,-4.1833;112.9000,27.8500;-6.0833,13.6833;10.9981,35.4203;58.7989,36.2097;39.7333,11.0833;79.9186,45.4100;18.2000,47.3833;23.7667,37.8500;-80.4333,-1.2667;8.9339,44.4063;-56.4500,-25.4167;57.5703,-20.1039;13.7642,47.9022;-88.8000,16.1000;-87.9833,15.0333;21.5606,57.3894;-171.9333,-13.8167;14.3089,46.1672;-5.9247,43.5569;77.7500,20.9333;33.6000,-11.9000;100.9289,13.0470;9.9656,4.4544;7.0724,46.1028;-67.4267,8.9344;122.2247,40.6681;-77.3500,21.0500;-71.0500,-34.7000;-35.8811,-7.2306;70.6333,40.1167;-77.2333,18.1500;-72.4000,-37.0333;1.3278,9.7525;-43.3722,-22.8039;-90.4953,14.5689;54.8667,50.0667;46.5797,39.3439;57.5414,-20.4917;20.9853,39.1606;-171.4667,-14.0167;125.6842,7.3081;6.3642,36.1153;72.0458,34.1983;-85.1667,11.9667;2.6653,10.2289;18.1333,59.3667;57.6847,-20.1031;26.5667,42.1667;33.2150,37.1811;-14.2500,13.4333;18.0687,59.4439;26.8333,60.6833;-18.1833,65.9500;33.5833,34.8292;29.7033,54.4108;-76.7500,-7.1833;33.6917,35.0292;-1.0714,50.8091;9.8434,44.1105;-80.8431,35.2271;-7.9500,53.4333;46.4150,39.2011;\
    10.6333,36.7833;-80.4333,-0.9167;-63.4333,-17.7333;-111.8226,33.4223;16.0172,44.9353;-8.2194,53.3275;100.9833,13.3667;89.9500,22.3000;36.7333,34.1500;5.8931,49.5361;-99.5075,27.5064;-6.9667,53.1569;35.7167,0.0500;15.3833,8.6667;44.4986,-12.3083;-80.5500,8.3000;22.1833,60.4833;139.9689,35.8544;44.5978,31.9631;4.2000,50.6833;42.7297,43.0456;-3.8000,40.2833;4.4833,7.8333;16.7202,48.3392;-87.2167,14.1000;28.8619,47.1389;45.2897,40.1044;-84.4667,10.0833;31.1025,2.4758;11.5409,45.5573;37.7000,7.0667;129.0403,35.1028;62.1400,33.3000;10.6167,36.4000;3.1500,36.5667;-66.9833,10.3500;-90.4511,14.7867;-12.8667,15.2667;-93.2638,44.9800;52.3633,36.4708;-15.7500,13.5000;-1.7000,37.6667;-1.7833,53.6500;43.4372,-11.8511;51.0833,11.2833;15.9775,45.8131;19.3667,47.6000;-5.2600,35.1700;36.6000,35.2667;33.9583,35.0667;-57.5333,-25.3333;9.0047,4.2475;43.2658,-11.6458;-75.3303,2.9306;36.4333,-0.7167;3.9833,51.1000;66.4042,51.9556;73.1500,19.2500;38.3669,49.0158;72.9483,50.0544;25.3333,43.9833;-83.9833,9.9000;120.7731,15.1508;20.5500,46.8667;-48.2772,-18.9186;48.1519,40.7900;28.1167,8.5333;32.7553,0.3533;19.4000,47.4167;32.5678,4.4133;-86.1250,12.7306;27.3333,44.2000;74.3500,32.3333;-5.6000,34.8000;106.6822,20.8561;9.5468,47.2312;-66.8853,10.1622;34.4667,47.5000;92.9000,20.1500;43.9856,40.0678;35.7508,32.3339;4.6736,51.8100;57.5739,-20.0136;-73.1544,19.6297;-5.3700,35.5700;37.2833,-0.5000;125.7919,38.7986;-14.2000,13.2333;-101.1844,19.7008;-71.8397,19.6678;4.6303,43.6768;4.4931,52.1583;109.3667,-6.9000;106.4167,10.5333;24.3494,56.8611;29.4139,46.6442;-87.1667,12.4833;16.2833,48.0833;28.3206,37.9125;2.2474,41.4500;-48.3722,-1.3656;-78.5000,-0.2167;1.2584,36.1957;23.7000,38.0833;50.3566,35.0213;68.4167,26.2500;92.1000,26.9000;-63.0833,-15.7000;20.1389,40.0758;125.2414,39.8872;-118.2645,34.1606;-72.3167,10.3167;-73.3175,-37.2463;29.5167,64.1333;67.0500,24.8667;130.3003,42.2489;158.6500,53.0167;43.2822,-11.4050;30.0000,55.4603;-57.3167,-25.2667;69.2378,35.1183;57.5622,-20.1431;25.9167,47.8500;-42.8019,-5.0892;\
    -15.8333,13.5667;-88.0544,13.1969;70.5500,40.9333;6.1422,49.6686;26.1000,44.4333;-16.5469,12.4850;22.0681,40.6294;176.2452,-38.1387;-55.6333,-33.3500;101.5167,3.2833;-91.8000,14.9667;69.2633,42.5994;57.6608,-20.0064;72.8539,49.6311;176.7000,-38.1000;123.9233,41.8558;-81.5775,23.0411;-89.1703,13.7242;46.2756,41.8228;12.2886,58.2837;21.2667,40.5167;-1.6432,42.8169;20.2272,40.5042;44.4328,32.4794;-56.5167,-32.8167;-0.7667,51.6333;47.0489,40.7700;89.1167,23.1167;42.5244,42.0814;32.0667,-10.5500;23.3456,38.0000;45.3064,40.1447;26.5560,41.6772;72.9667,3.7833;63.3167,45.6167;-16.0492,13.5164;10.3833,35.2833;24.6919,41.5853;23.7128,56.6500;14.3247,10.5956;-86.1833,11.9333;15.0814,46.1461;104.2964,52.2978;-83.7078,22.6164;-16.0406,18.1194;51.2053,29.2622;-23.9000,64.9167;34.8162,31.3728;35.8236,33.1253;1.4078,18.4411;14.7000,55.1000;-73.1258,7.1297;21.6833,50.5833;29.6500,12.0500;27.7667,-15.8667;-22.4333,63.8333;44.3525,-12.1911;47.7406,40.6531;15.8328,50.2092;6.6500,51.4500;100.5000,5.6500;10.6873,53.8689;24.9500,55.0500;42.4153,41.6311;-76.2167,-13.7000;-0.4167,5.5333;121.1919,14.4656;68.0000,26.6667;26.3939,58.7467;35.7522,32.6744;33.7500,35.0167;-101.4167,26.9000;172.7333,-43.1667;170.9801,-45.0784;42.0536,42.7139;8.6189,44.9125;-46.3919,-23.9631;28.2833,-24.8833;26.5956,58.5542;71.9167,29.6167;20.9714,42.0106;-83.6833,10.2167;-121.9443,49.1747;89.8333,24.7333;-76.9500,17.9833;37.9333,0.2333;19.3667,52.2333;-86.4000,13.3500;20.3000,48.2167;-72.6000,-38.7333;125.4942,38.5108;31.8500,18.5500;75.8167,26.9167;56.8886,24.1722;45.0667,32.9083;-89.1186,13.7811;-53.9500,-32.9333;-68.1667,-15.2500;-9.6000,30.4000;21.3500,56.5833;-70.0500,-26.3667;16.8512,41.1177;-81.0189,22.7692;-57.5833,-30.2500;50.0378,40.5681;87.9333,26.9000;51.8169,55.6383;-100.2500,25.6833;72.3169,36.9489;-55.2333,5.7000;-87.6501,41.8500;20.8324,13.8292;11.5167,3.8667;-58.8333,-27.4667;100.3000,5.4667;36.9833,-1.4500;-0.0167,5.6167;-1.1500,46.1667;43.5900,41.9911;12.5346,55.6794;-172.7000,-13.4667;-55.7833,-27.0500;\
    20.5275,41.5250;-1.6167,6.7333;89.1000,22.7167;-73.0536,6.9894;29.2833,-20.5333;43.2167,42.0167;3.1667,50.7000;123.0617,0.5344;33.4333,-3.6667;5.8908,49.7656;-83.6667,12.3500;34.2694,0.2386;151.7765,-32.9271;106.5083,-6.2364;-65.3833,-10.8000;25.4667,65.0167;33.3803,46.7508;35.1064,32.8006;22.8167,57.5000;67.0233,34.3875;76.9500,52.3000;-89.7242,13.7189;151.2500,-23.8500;100.7333,4.8500;57.6592,-20.1308;0.1667,51.7833;-72.7167,-12.8667;-0.7333,5.2833;-84.1000,10.0167;-56.3564,-34.6050;-66.8167,10.4833;7.8797,33.8742;-82.4584,27.9475;117.2808,31.8639;-55.4667,5.8000;-64.7296,-21.5355;71.5553,34.2153;44.8319,35.5367;-69.2444,10.0228;79.4167,28.3500;-123.3693,48.4329;135.2000,34.1833;-72.9491,45.6168;106.5303,-6.1703;-68.8950,10.1603;12.2333,7.9667;31.9519,53.6094;21.5500,47.2167;19.9617,41.3967;27.7000,43.1833;29.2964,47.1536;8.7700,47.4765;100.9833,3.6667;-58.0756,-32.3214;62.1900,34.3400;81.5761,49.3314;-14.0167,13.5000;16.2333,48.0167;36.7833,-2.5500;22.5667,51.2500;9.2748,45.5825;37.1586,36.2028;-63.3500,-17.9667;-81.2167,8.3167;43.7036,-12.2611;-83.6000,10.1833;35.1833,-0.5833;13.2951,49.3955;19.6508,40.6428;35.7667,13.5167;48.8097,41.4708;14.1744,46.3444;5.9875,51.1942;103.2000,13.1000;27.7667,-15.8333;130.3667,47.4000;100.5333,13.6667;12.4833,41.9000;44.2892,40.0714;21.3167,46.1833;-76.6611,5.6947;-157.8583,21.3069;15.3156,46.0078;-89.3597,13.8761;40.9033,-2.2783;-64.2528,8.8925;-76.6028,21.1950;15.5000,51.9333;152.9500,-26.6333;14.1667,46.2333;24.8258,58.7789;-54.7000,-26.4000;41.0853,11.7361;25.5500,56.3500;-84.0667,9.8833;5.5718,50.6412;-171.9167,-13.8167;21.9000,4.6000;134.5122,7.3619;-16.6461,13.2675;41.3375,37.4261;29.6500,-0.8833;30.7967,0.9606;23.3167,55.9333;85.2500,52.5667;-71.0833,-34.4333;95.1333,22.1167;-55.8000,-27.2000;18.8667,50.4500;131.4333,31.9000;26.0101,57.7752;39.5689,52.6186;18.3484,49.6854;8.4667,47.1833;-76.2333,17.9167;9.7833,7.8500;99.8500,13.6833;46.1092,36.9611;-149.9003,61.2181;10.5167,5.1500;38.4000,40.9167;-58.5142,-33.0103;-90.0568,35.0868;\
    45.1114,40.3147;-55.2333,-34.8000;-49.2731,-25.4278;111.5167,2.1167;40.2844,33.0381;24.6833,45.1333;17.1118,49.4719;98.5667,16.7167;48.3014,38.2494;-76.6667,18.0333;12.2450,45.6667;48.4972,39.2089;25.8744,40.8475;172.9780,1.3291;34.9528,32.1558;-102.3000,21.8833;63.0506,26.0028;167.7167,-45.4167;13.2706,10.2689;-8.3986,51.8117;33.2042,0.4244;-8.0000,12.6500;-59.6500,13.1833;6.0422,49.5250;-4.9333,12.8333;13.7334,50.1037;4.3167,7.9000;37.0044,-14.8706;-68.0125,10.4731;44.2675,40.8372;86.9833,23.6833;-68.4667,-33.0667;44.0333,13.5667;25.9667,-27.2000;-2.4500,42.4667;9.8633,37.2336;8.0333,50.8667;96.1333,21.6000;36.7167,35.9333;23.9000,54.9000;44.4150,40.2481;171.2000,-42.4667;171.1333,-44.2667;35.2933,32.8622;6.1414,49.6528;2.0667,7.2000;44.1972,40.7875;-84.7333,10.1000;102.2833,2.2667;17.3839,45.8319;33.5417,35.0667;6.3503,49.8136;48.8786,38.4561;-70.2222,8.5944;32.2167,-17.4000;14.5376,50.6855;22.5667,62.7333;99.8000,2.9667;-89.2089,13.5753;-56.0167,-33.2500;0.5017,15.6594;27.5333,56.1000;43.7153,-12.3614;10.6821,56.1944;34.8044,31.9642;38.5481,35.8367;43.3833,-11.8706;26.3667,-27.8500;42.8356,42.6181;-2.9678,51.3458;13.4072,-12.5783;26.3558,59.3464;-84.1833,9.9167;48.1328,29.0961;43.1189,36.3350;32.9750,34.9375;-14.8833,13.6500;-6.6661,53.2489;5.9185,52.9593;38.0833,4.8833;-79.8167,-2.9167;126.9544,35.9439;-71.8991,45.4001;21.1333,46.7667;-66.9333,-20.3000;35.7833,-4.9000;-58.2956,-33.1325;32.0831,1.6858;15.5493,41.4609;177.4167,-17.8000;14.6094,46.2197;1.6833,6.9333;29.9883,0.2300;35.3125,32.7236;-8.5306,51.7075;173.0333,0.9000;-16.7089,13.4114;-85.8333,13.6167;40.8383,43.0806;8.5167,47.2000;-67.9167,-17.2333;-35.7353,-9.6658;10.2700,13.9869;30.5281,-2.8706;23.9500,54.6333;18.8375,45.1947;-91.3167,14.8500;143.9833,-6.2833;23.4667,55.2667;-16.2667,13.9833;79.8478,6.9319;24.3500,46.1667;46.3833,-25.0333;36.2500,34.6667;7.2167,51.5500;25.4333,61.0500;47.1753,31.8453;57.3650,-20.3931;43.3158,-11.8447;-88.0000,15.3167;-79.5058,-7.4328;29.3667,62.5333;41.2769,39.9086;20.9633,41.5144;\
    27.7547,59.3997;-74.0000,18.1833;1.8333,6.2833;11.3387,44.4938;-54.3333,-34.4833;4.3500,7.4667;57.6544,-20.2828;-100.3333,25.8167;36.2544,32.8578;24.6000,60.9000;23.5000,61.4667;-71.9667,19.0167;20.6167,52.8833;3.8833,43.6000;30.3833,-29.6167;46.4227,33.6374;38.5878,48.0383;-40.8394,-14.8661;80.2275,50.4111;5.5167,50.8667;18.7472,45.4981;-2.6167,53.5333;16.5947,31.2061;-56.5417,-34.3828;-8.6833,39.2333;39.6333,12.0167;108.1000,10.9333;-175.2000,-21.1333;-80.5167,8.3333;140.4667,37.7500;-1.5333,53.9833;79.9281,6.8461;4.6333,51.3500;65.5525,44.7728;-13.9500,13.3333;8.5213,47.3775;-86.0000,15.6833;31.4431,36.7867;36.6667,36.0167;22.8708,40.9919;39.4833,8.3167;8.5150,47.3620;17.0333,54.4500;-78.4350,-5.6111;-6.1414,52.7931;127.4436,39.1528;122.0494,41.1881;-72.2667,19.6833;-86.4833,13.3833;23.0500,47.2000;-55.5000,-24.0833;143.2000,-5.4667;-88.7500,15.8333;6.6625,52.3567;34.1667,13.2000;-78.6275,22.1094;8.1500,34.3833;48.9792,39.5950;175.5153,-41.4122;36.1650,36.5817;20.2000,47.1833;-55.0833,-26.7167;14.5176,50.2593;1.5000,6.5833;1.8521,50.9581;-172.0333,-13.8500;-78.5783,-9.0853;-90.9942,14.7669;-77.8000,18.3500;81.4333,21.2167;-12.5500,15.0333;117.6556,24.5133;-65.3051,-43.2490;50.5039,26.0686;16.2908,41.3118;106.0167,12.4833;40.5850,37.1933;108.3500,-7.3333;12.9833,31.4333;95.6500,17.0333;-91.1333,14.6333;-88.8333,13.7333;17.9448,59.4032;101.9000,3.5167;2.7508,35.8856;-86.4833,13.6333;25.7167,62.2833;50.0056,40.5561;116.0667,5.9833;24.9333,54.6333;19.9833,4.9667;3.2500,43.3500;12.3667,4.6833;-1.8224,52.5652;123.4242,10.9447;8.2711,50.0000;37.8867,15.5486;-173.9833,-18.6500;57.5203,-20.3147;23.2500,56.3592;-72.5500,19.6667;-55.9500,-34.7167;44.4167,40.5333;2.2000,48.9000;129.1686,35.4061;-35.0014,-8.1803;1.5151,42.5052;29.1833,65.9667;100.5167,13.7500;35.1000,0.2000;57.5878,-20.0283;172.8167,3.0833;88.9667,24.0333;-84.5833,14.0167;134.5836,7.3656;44.4158,-12.2781;1.5328,36.1842;-53.4553,-24.9558;30.4033,40.7806;32.8867,3.2783;34.4892,0.3364;-89.9000,16.9167;25.1864,58.0922;\
    101.7667,36.6167;-57.0833,-24.1333;-79.0300,-8.1160;77.3500,11.1000;27.7167,56.5500;22.2833,60.4500;54.3675,31.8972;-77.7000,17.9667;35.3722,32.6958;7.2441,47.1324;-1.2500,5.1000;7.7519,35.2731;-0.3774,39.4698;-175.2167,-21.1500;138.1833,36.6500;-2.6667,42.8500;4.9500,50.8000;18.5500,54.5000;-74.1000,18.2500;5.6194,51.6608;125.0894,7.0083;47.3700,10.6181;-63.9833,-19.8167;9.1577,56.1388;73.1389,29.6133;-70.7500,-33.7333;-171.8167,-13.8000;69.1339,36.0653;32.5342,15.5881;-2.1000,57.1333;68.6167,25.9333;33.4625,34.8250;89.4500,25.9000;5.8667,50.5833;-75.5639,6.1731;-14.0167,65.0667;82.9344,55.0411;-15.2833,13.4167;17.0078,43.8269;22.4500,13.4500;78.2083,44.8947;57.5333,22.3833;-84.2167,10.0167;3.0833,45.7833;-77.1167,17.9333;-55.9833,-34.4333;58.4819,35.2364;26.0000,55.1667;16.0356,46.2106;23.8453,58.6814;-80.4356,22.1461;57.5967,-20.4003;24.1000,43.2667;24.2372,56.8294;5.5589,52.0286;-83.9833,9.8333;-68.6833,-16.2000;26.2500,47.6333;40.4964,38.8847;4.7167,8.1500;125.7547,39.0194;6.8636,4.7358;25.1500,52.7500;-0.7558,52.0417;43.0975,40.6081;144.9500,-37.6000;35.4500,7.2000;-16.4339,13.3522;-84.1000,9.9000;27.8161,48.0350;69.3731,40.1539;16.2406,46.6053;0.4833,51.7333;97.6111,16.5314;-1.5534,47.2173;23.6371,37.9474;-55.1833,-25.9667;4.6368,52.3808;89.7167,24.4500;46.7228,30.9697;-57.3333,-25.3833;-58.7667,-28.1333;70.4667,40.2000;-90.2833,15.8167;41.2172,37.0700;59.0472,53.4186;149.1281,-35.2835;35.8667,-1.0833;-90.6153,14.4875;42.8675,39.2347;-74.4500,18.4000;10.9939,33.8083;-90.6442,14.7189;22.3000,60.3000;34.8778,32.0531;34.1639,39.1458;-56.0628,-34.6469;28.8703,47.0875;103.9167,12.5333;-1.4214,54.9878;19.0333,47.6500;-86.5708,12.6744;90.7667,24.4333;-80.6456,-4.8378;71.0667,31.6333;105.7667,10.3000;10.5500,36.5333;81.6667,7.2833;72.6833,31.1500;-0.7000,5.5333;-51.0233,-30.0811;37.3333,-3.3500;35.4853,38.7322;35.3333,-15.3833;100.3667,16.4333;-66.3500,-24.1833;14.1500,9.7667;-78.4411,-5.7561;35.0408,32.6936;18.3833,43.8500;-88.4500,13.3500;17.2500,46.7667;-53.4500,-33.6833;\
    7.1808,9.1758;26.7000,-30.7000;30.4000,-17.3833;99.9588,12.5706;35.6608,32.3864;-110.9333,31.3333;25.4000,42.6167;-56.3906,-34.4533;5.9103,49.5675;-83.3333,9.1667;16.5833,43.4833;-79.6167,-3.7167;31.3833,-8.8333;15.0486,46.3800;38.6667,-6.2500;17.2833,47.8667;85.9167,26.7000;44.3533,41.5228;18.7500,47.8000;36.7333,34.8333;-79.7333,-1.5500;-60.4500,-26.7833;12.0267,13.2178;-88.0167,13.6167;21.0525,42.0744;-3.8167,40.4333;27.9833,45.2667;31.1167,-16.8000;-7.2998,53.0344;26.6667,-26.8667;-82.6167,9.5000;10.7508,36.4447;-82.3886,22.9286;23.6833,38.0500;6.8259,47.0999;-56.2000,-26.8333;21.8286,42.1981;137.7667,-32.5000;90.0167,25.0167;72.4722,34.1200;15.1647,45.7925;-63.7000,-17.4833;33.4833,-13.0333;15.6500,46.5000;137.3833,34.7667;43.3772,-11.4786;-6.6892,53.2914;125.7483,40.2019;36.6500,-1.3667;12.4912,56.9055;19.2992,43.7847;24.1000,56.9500;89.2167,22.9167;-56.4833,-26.0500;26.6667,57.5333;65.1236,36.9525;48.5833,-17.0167;24.7500,57.8364;-72.1333,19.1667;70.9025,31.8328;23.8296,38.0338;173.0500,-41.4000;-3.7667,40.2333;117.1767,39.1422;-16.0369,13.4178;-7.5800,33.2600;12.7150,32.7631;35.5167,-14.9667;21.4167,50.0500;-72.8669,18.4314;14.5529,53.4294;6.0944,52.5125;-75.6469,6.1606;66.9114,50.2486;103.5000,10.6333;10.0667,5.4500;-83.5552,41.6639;11.8667,32.9500;106.7167,26.5833;121.1325,14.5692;-84.0500,9.9667;-8.3444,51.8739;33.4000,-3.5500;25.1306,35.3250;31.5833,-16.7833;-172.1333,-13.7000;105.1500,9.1769;44.2694,-12.1919;-2.1500,53.6167;91.2044,23.4578;6.9067,36.8792;127.3858,35.4100;12.0956,49.0150;80.2785,13.0878;43.2525,-11.7164;104.1833,10.6167;13.5939,45.2258;-70.9722,-31.7750;102.2655,-3.8004;29.9000,-10.2667;13.8333,-11.2000;16.5077,59.3666;-57.4333,-30.4333;104.3333,15.1167;-58.1333,-32.6833;-21.3667,63.8500;0.9833,5.9167;40.9858,56.9942;-82.9988,39.9612;42.2667,2.3500;9.7000,4.0503;-70.7500,-34.1667;33.6333,13.5500;15.9431,44.9669;-66.8833,10.4333;104.4500,0.9167;32.3833,34.8833;13.7800,45.6486;121.1450,14.7325;25.6000,42.0500;-0.5167,8.5500;139.3544,35.5531;\
    48.0572,29.2572;-4.7833,12.3667;128.5200,35.2497;-79.8403,21.8767;-87.6500,15.5833;20.0269,42.2647;48.2747,29.4425;27.1000,-26.7167;4.5347,51.8567;100.4000,5.4833;72.3525,32.2967;-84.0833,9.9333;31.2167,12.9000;9.6690,45.6980;80.3333,28.9167;20.0258,41.4783;-90.0158,13.8431;-0.3712,49.1811;77.6033,12.9762;-89.7367,13.7775;-72.2768,-50.3407;15.1661,46.6114;-70.7581,-28.5708;28.6753,41.0333;48.7516,33.8973;-70.9333,-33.6667;15.0833,47.0667;-5.6644,43.5411;-77.9333,-1.7000;6.3036,49.6508;44.8378,41.1478;-2.8146,36.7763;31.4500,-26.6167;8.5333,52.0333;31.3667,-26.4833;-75.3889,6.1553;8.0667,6.6333;17.6575,44.1711;43.3792,41.8375;-79.8833,8.7500;73.0591,33.6100;-84.2333,9.7500;-79.8722,-6.8781;64.5647,31.8189;18.2667,52.2167;16.0845,51.6662;7.3333,47.7500;85.5167,27.5833;-89.9000,14.2833;57.5892,-20.2439;22.9503,40.5825;26.4967,58.0583;18.6894,45.1819;-43.1036,-22.8833;26.3667,47.2000;43.6769,34.5967;17.2656,44.3422;26.4667,53.4500;-69.1208,9.3328;24.1394,41.1544;-72.2869,18.4506;139.5886,35.9697;53.3942,35.5753;22.2333,-10.7000;102.7931,17.4075;28.9333,-20.3000;6.2553,35.4864;104.9833,22.8333;51.4039,25.4169;-20.6449,64.0805;-8.5457,42.8805;139.0167,36.3333;8.1000,6.3333;35.7500,-6.1833;-76.8828,0.0847;3.2253,36.6194;120.9169,14.4825;44.0139,40.1361;107.9167,-6.8500;106.9833,13.5000;125.6097,39.5981;121.1417,41.1078;3.0953,36.6692;39.9833,9.9833;10.7333,3.2333;-8.1967,52.8619;15.2422,44.1197;23.3189,38.3250;-54.2000,-33.6667;-88.1833,13.4833;13.6400,45.0800;27.4167,45.8667;90.7167,22.3167;139.7206,35.8050;61.4978,31.0297;-55.1667,5.8333;118.1167,5.8333;-4.8300,33.8300;14.6633,50.1871;-89.4472,13.8403;100.2833,13.5333;3.6000,50.8500;104.7833,10.9833;6.5833,46.5333;18.1430,49.4585;120.9797,14.8386;20.1500,51.9667;96.4667,22.0333;33.5458,34.8542;44.5136,40.1811;47.9174,39.6482;13.5333,9.3833;-79.4667,0.3333;-172.4000,-13.4667;73.1000,6.8833;108.0000,14.3500;-89.2031,13.7086;-73.4000,-37.8000;43.3806,-11.6069;23.1500,-15.1000;-87.1167,15.6167;15.3667,-8.1833;14.0669,46.4306;\
    -9.4215,38.6979;-47.0608,-22.9056;27.1750,59.3517;17.0178,43.2969;6.1147,49.8647;44.4886,-12.3258;-56.0167,-25.4500;57.5786,-20.0431;7.5371,47.2079;43.2614,-11.6597;81.1333,28.5000;22.2375,41.8167;9.9175,37.2358;57.4008,-20.2117;4.6667,52.5167;175.0502,-41.1383;-0.7333,6.5833;30.9833,2.4167;3.2000,6.8167;6.1417,49.6831;17.4667,47.3333;110.7781,32.6475;35.1833,0.1167;175.9167,-37.5500;19.3167,-33.3667;-57.0833,-24.1000;22.7000,56.2000;33.3000,14.7333;-84.0833,9.8833;-76.6864,7.6769;-90.5333,19.8500;3.1931,36.7475;-85.8333,11.4333;22.5756,41.2031;12.9043,41.4661;5.7000,5.9167;86.7500,26.5333;12.3458,55.8750;35.7956,32.4919;126.7175,35.0283;-8.1558,52.4733;5.3497,31.9272;-54.1667,-34.6667;28.7000,-15.3500;-82.6333,8.7667;40.8667,9.0833;-55.8833,-34.5500;48.1211,29.1739;-89.2653,13.6486;-71.5169,10.5381;6.3383,49.7731;68.3611,40.8994;173.9528,-41.5160;34.9797,32.2847;34.8669,31.9253;115.4903,38.8511;73.1500,19.2167;43.7542,40.2983;6.6147,36.3650;80.4392,47.9714;17.1856,44.7758;14.1562,57.7815;-56.1708,-34.8581;-76.8000,18.0000;111.6522,40.8106;-8.6500,41.1833;110.6064,-7.7058;40.0689,1.7472;23.7667,55.7667;-80.1667,8.3833;42.9900,42.3442;10.3951,63.4305;100.1167,14.4667;-0.8833,47.0667;-76.9511,20.9617;36.8883,-17.8786;-8.4167,40.2000;-175.2000,-21.2000;-73.1333,-37.0167;44.9536,40.5550;9.7011,55.7704;-87.0333,14.1500;-88.2333,14.9167;-15.9806,12.8275;107.2831,-6.1483;101.2000,4.1667;20.6531,8.4092;64.3600,31.5800;22.4167,60.4167;-122.2526,49.0580;114.1694,22.3193;27.4667,52.7333;44.8211,41.2458;3.9531,36.5422;10.2558,43.8735;85.0333,27.4167;-16.9437,14.3735;134.5667,34.0667;1.1861,9.5511;3.1833,36.7167;6.2333,7.5500;-80.6853,-4.9039;-62.8667,-17.7833;74.2244,31.9744;-14.6500,16.5167;4.6694,52.5483;-8.6742,37.1020;48.9339,39.8078;43.2664,-11.6617;-80.1489,22.3772;-14.9167,13.6167;4.8333,52.4500;-102.5722,22.7756;-72.9667,-40.3167;30.7203,-3.5864;35.1550,32.9256;35.8425,32.2783;16.7819,45.4750;24.4500,61.0000;-88.5500,13.2833;-90.7344,14.5611;27.4333,-32.8667;11.1500,2.9000;\
    -21.9167,64.5333;16.3167,38.9833;-38.9667,-12.2667;9.9667,1.8833;35.9069,41.5678;35.4667,30.2500;-79.8500,8.8333;116.3972,39.9075;-56.9833,5.9500;-83.1500,8.6500;-67.8739,10.2297;33.9833,-2.8000;30.2664,0.6939;80.0833,6.9000;70.9333,30.9667;97.4000,25.3833;-86.0500,11.7500;29.2333,-25.8667;6.8493,51.2972;70.8344,36.8667;22.4297,37.0733;-77.6050,-11.1067;18.7050,59.7580;31.8122,-1.3317;15.2675,46.2361;-74.3167,18.5667;-85.1333,10.9667;21.4167,60.8000;13.8028,10.7403;12.8667,0.5667;-15.5833,13.4333;5.7595,58.9630;22.9342,55.6339;18.0833,51.7500;-73.6492,45.5168;-78.8333,-2.2000;38.1550,48.0353;-86.0000,13.1000;106.3833,10.2333;-79.1700,21.9419;36.6833,-3.3667;21.7833,61.4833;67.1122,36.7069;88.5667,26.3333;75.9167,17.6833;-9.0667,38.6667;-85.4500,10.1500;-0.0167,8.4667;95.1833,19.3167;12.3000,55.7333;-6.8300,34.0200;-87.0000,13.1167;29.0667,-26.5333;-72.9833,-36.7333;33.0819,68.9717;9.7500,1.8500;10.4667,2.0167;18.2820,49.8346;-34.8811,-8.0539;76.8578,43.3603;-13.9667,15.3000;104.7833,29.4000;15.8789,46.1608;27.5025,59.4236;72.8479,19.0144;28.2975,48.1558;1.5833,42.5333;-80.2667,8.6000;113.1167,23.0333;9.3167,55.2500;-57.5333,-25.1833;-5.5500,33.9000;-87.8833,15.2500;17.0312,59.3774;8.2440,56.0901;151.3500,-32.8333;41.8461,3.9261;2.3667,48.9333;34.8747,32.0603;69.8581,36.5722;6.0167,9.0833;71.5000,43.0000;46.9355,34.7956;-80.2333,-3.4833;17.4508,44.0572;137.1500,35.0833;-60.8914,-35.4542;-76.3500,-13.0667;-89.0714,17.1561;-76.5333,-10.4333;9.7500,47.4667;7.6683,13.9578;-63.8500,10.9500;25.6290,43.0812;24.7500,55.2500;-73.2659,45.3168;31.4500,-11.8333;81.0500,6.9833;-87.6000,15.3167;51.5378,25.1508;100.5000,5.1167;105.7667,19.8000;35.9333,31.8500;-6.4272,53.5108;-80.8508,22.3850;-84.6667,9.9833;18.3000,7.3000;135.1356,48.4431;51.0829,35.4846;-58.2283,-32.4833;31.3089,-2.6319;120.8747,32.0303;8.5030,47.3579;31.0500,-25.7833;39.8667,10.7167;99.9500,13.1000;32.1833,-3.6333;9.8833,5.4667;25.9586,59.2606;-7.8333,37.0333;-78.1464,21.2594;-7.5667,43.0000;43.2672,-11.5850;-88.1500,15.4000;\
    29.9133,46.6222;7.4146,46.9244;68.3331,38.5736;78.1792,26.2236;153.1333,-26.8000;-56.7136,-34.3375;57.6611,-20.1897;28.9497,41.0138;-90.8167,14.6686;49.9806,40.4422;-14.7667,13.5333;-78.9831,1.2492;49.9736,40.3967;76.9167,8.4833;107.1333,16.8167;107.7500,-7.1000;21.9527,41.8696;18.7000,12.1833;57.5294,23.2997;-0.3716,50.8115;40.8300,34.6897;-113.3187,53.5168;6.8958,52.2183;-73.0167,-36.9167;5.9192,49.6614;96.4333,18.9333;-2.1576,53.4098;68.4525,37.5603;105.5050,21.1378;5.1764,52.2233;-90.6064,14.6333;20.8333,32.5000;82.3114,50.1456;-72.0833,19.4167;111.4528,1.2472;9.3214,4.6839;19.9047,41.4081;-22.2000,65.4500;10.1011,36.8078;-56.0367,-34.7336;31.2000,-10.2000;-88.3167,14.9333;90.5011,23.6233;12.2011,44.4175;23.4167,45.4500;-37.0717,-10.9111;22.9189,55.9850;19.9333,47.7833;128.8200,35.9133;-43.2075,-22.9028;13.9367,45.2403;28.6839,7.2733;10.2500,56.2333;27.6667,63.0833;-9.2500,38.7167;-91.7333,15.6667;124.5317,40.1994;4.1833,7.6333;3.7780,50.6294;126.9125,36.2819;43.3772,36.2694;174.5500,-36.8500;21.8833,61.4333;17.4647,49.9884;116.3656,23.5297;7.6167,46.7500;44.3536,40.5961;24.3667,45.1000;-5.9000,35.0000;43.5975,15.6950;75.8333,22.7179;21.2272,45.7494;45.1167,2.1500;12.4736,55.6572;32.5122,0.7283;-61.2500,13.2833;30.4833,-22.9500;139.9000,35.7833;153.1586,-27.0797;-64.1833,-31.4000;67.3452,34.2654;44.7578,41.8244;24.3500,55.7333;35.1833,-15.7000;70.0308,38.1078;20.7167,49.6333;43.6667,-23.3500;-66.2667,-17.7167;-77.3500,18.3000;44.0833,2.7833;26.3167,56.9667;35.8497,34.4367;4.9750,51.8325;120.9483,14.7578;-56.2167,-27.2333;35.8833,34.8833;22.5919,42.0200;27.0667,50.1833;56.0375,54.7750;27.4667,5.6000;21.9311,55.7194;-78.6167,-0.9333;29.4000,-13.6167;-14.6500,13.4167;108.0500,12.6667;24.0333,61.2667;15.0872,37.5021;-6.3100,34.3000;-84.0000,9.9333;100.6500,14.0667;123.9222,10.3236;25.1167,60.4000;10.7000,4.9667;31.0333,-26.2333;-77.5167,18.2000;-0.8278,9.6247;33.3500,47.9167;15.5912,49.3961;12.3713,51.3396;8.3167,47.4667;28.1706,-25.8744;22.5333,56.7000;6.6618,46.5103;\
    19.4897,40.4667;127.0847,36.2039;74.8167,34.0833;28.4333,12.7000;-173.7500,-15.9500;6.8333,6.1500;-3.3667,47.7500;-88.1000,13.7000;-7.3500,53.5333;35.8097,32.2853;32.4000,13.8167;28.7842,46.5200;14.7667,47.9667;15.0365,58.5371;-11.4333,14.4667;51.2453,25.8297;-73.1731,7.0708;-62.8000,-17.6500;26.9000,-28.1167;128.2000,-3.7167;42.3739,11.1086;106.3167,20.9333;-171.8167,-13.8000;7.7383,36.8047;27.8375,38.9219;13.1667,-1.4333;-86.3500,13.0833;50.3217,40.4708;-79.7920,36.0726;-79.0000,-1.6000;15.1889,45.5711;58.8000,22.4500;102.7183,25.0389;19.5786,30.4167;8.6949,56.9552;36.9833,-6.8333;-6.1333,14.7333;21.2000,6.2500;-90.5269,14.6211;37.6167,48.8667;26.0333,61.2167;4.3667,50.4667;44.3964,-12.1628;-91.4086,15.0436;5.9317,49.6494;-2.1000,34.9500;44.6278,36.0839;57.5053,-20.5097;21.3822,42.1036;-1.9750,43.3128;19.9522,40.7058;9.1770,48.7823;-75.8197,9.2317;-95.9928,36.1540;6.8756,35.9764;-5.1500,31.0500;152.9500,-29.7000;84.8667,27.0000;6.0972,49.8103;19.2161,44.7569;-89.0297,13.4978;151.7500,-32.7500;16.1664,46.6625;-5.5300,31.5100;26.4064,40.1456;-73.6269,8.3125;26.7167,56.3000;44.5858,48.8047;16.2789,46.3294;104.6667,12.2500;-76.7167,18.0333;109.2333,13.7667;152.4333,-3.6667;22.7667,57.0333;-78.0333,18.1667;24.9658,59.2906;11.2667,1.1333;29.4667,-26.4500;-48.5492,-27.5967;15.8086,40.6443;140.7511,40.8211;175.5333,-37.1333;8.0500,47.3833;28.2553,46.4786;-91.4500,14.5833;91.8364,22.3331;94.9167,20.1500;14.8236,46.7661;43.7425,-12.2800;-8.7000,41.1833;34.8000,8.5333;-57.4167,-34.4333;21.1089,42.1067;5.9319,49.9689;8.3102,56.5486;-46.4614,-23.6678;106.6300,-6.1781;105.0833,10.0167;-88.3833,18.3833;57.4958,-20.2189;35.3472,-23.8597;11.9333,-0.1000;13.4000,9.3000;-6.5436,53.3386;5.0597,52.6425;101.3703,14.0503;34.2000,-4.3667;39.7139,47.2364;114.4786,38.0414;-83.3833,14.0333;10.2166,59.1313;32.4250,34.8083;149.2333,-35.3500;-23.5667,15.1333;151.6500,-30.5167;11.0000,35.6167;-85.6333,11.0667;-84.7000,11.6833;2.4000,15.9167;113.1500,27.8333;14.2307,37.0803;9.0522,48.5227;12.8622,56.2428;\
    -68.3047,9.9186;57.6292,-20.0597;26.6333,-32.7833;80.4173,8.3565;7.7833,47.2167;-76.6347,7.8856;14.1578,46.3611;24.8000,46.2167;43.4703,41.7339;68.2167,27.5500;-80.3500,8.5333;34.3333,-14.3667;-16.6558,13.4042;-88.3833,15.0833;-57.4000,-33.8833;8.7544,51.7191;-0.1310,38.5382;6.0500,49.6111;-72.9500,-38.7333;10.6000,36.6500;-88.4167,15.3000;-85.7000,11.5333;-72.6667,-37.5000;80.5767,7.1647;35.7667,31.2000;68.6608,37.5894;6.2483,49.6889;-5.3600,33.7000;-91.5167,14.8333;16.4441,49.9044;16.7161,10.4825;9.7331,47.4167;16.3453,-9.5447;11.6333,4.4500;49.1678,36.3187;43.7733,-12.2975;35.3333,32.6083;-85.4000,12.0833;-1.9167,52.4667;54.0828,17.0175;40.2106,37.9189;19.4333,-33.6500;12.5000,55.8833;-57.5500,-38.0000;2.4333,48.8667;46.8000,-20.5500;108.2208,16.0678;-78.9333,-2.5500;-18.2000,65.8500;-74.2683,4.7344;-84.0875,22.2025;77.3167,28.4333;-1.2560,51.7522;59.2144,32.8739;18.7833,50.3167;-56.0297,-34.8772;119.9081,32.4933;-55.2667,-25.4000;41.8692,42.5122;10.6584,59.4341;-78.5333,-1.1667;121.0022,6.0522;38.9000,-6.4333;38.8000,-10.7167;-7.3167,41.1000;-67.1500,-17.9833;-66.9333,10.3667;-70.8667,-34.4167;85.3167,27.7167;35.3689,33.5631;28.2769,47.5072;9.8167,47.1497;73.2000,34.3333;4.3250,52.0363;-80.3667,8.5333;4.6569,52.4833;-46.3483,-23.4861;19.5126,42.0683;24.5333,42.1333;-82.4000,8.4333;-87.6525,13.6208;130.9833,-12.4860;3.6000,50.7500;-0.6414,35.1939;128.7250,36.5656;25.0500,57.8667;26.8500,51.8833;-171.8167,-14.0000;68.0000,27.6000;27.7092,47.5736;103.6000,1.6667;100.0833,7.6167;-72.3961,18.2303;9.9833,3.7667;-2.1333,52.5833;2.4500,48.9333;26.4167,43.1000;49.2475,39.4200;39.5500,-6.4333;43.2903,-11.5514;-57.8833,-31.9500;-171.3833,-14.0000;-98.4936,29.4241;80.6517,7.8600;-9.4722,6.9956;65.9364,36.2156;-76.2994,20.9692;-98.3035,19.0641;-58.7500,-38.5500;15.2234,49.4313;173.8833,-35.3167;9.6831,47.3667;15.7766,50.0408;37.0667,10.7000;126.5094,35.2750;-76.4867,3.0131;174.7766,-41.2785;-75.6811,4.5339;33.4292,34.9500;24.4556,53.1519;28.4944,52.1281;8.9600,46.0101;6.1144,49.7417;\
    30.9308,3.0192;80.6350,7.2964;14.5767,45.9833;85.9667,23.7833;38.4686,48.0469;-63.8667,-18.1500;5.7767,15.8869;10.5833,36.6833;10.9333,9.9833;30.6897,36.9125;-77.4297,20.9881;103.7500,1.4667;14.7861,41.1295;27.5000,-26.4833;45.8017,41.9514;84.4333,27.6833;35.7333,-0.2500;10.2667,5.1500;-75.1331,9.7222;5.2475,14.4617;-72.1167,-36.6000;-78.7594,-3.8278;21.7833,49.6833;114.0683,22.5455;-8.9500,51.9000;108.4167,11.9333;16.5508,48.2028;42.5708,1.2533;-82.6875,22.9264;7.1833,51.2667;24.9642,58.1503;100.5500,14.3500;23.5167,54.2333;-9.5167,52.0500;144.3500,-38.1333;42.9522,14.7978;-82.3642,23.1319;-74.8008,4.3031;-15.9833,13.5500;-77.9956,20.7147;36.8833,6.3000;34.9167,-14.0833;15.6972,46.5058;9.4333,54.7833;-9.0667,38.8500;23.0272,57.0606;-66.9500,10.6000;2.8625,36.5231;-78.8986,35.9940;10.6667,6.2000;10.0667,6.3833;76.9500,43.2500;-119.5857,49.4998;-16.2833,13.5500;-58.1833,-26.1833;19.8764,41.2831;87.2833,26.8167;23.6500,37.9667;1.0422,35.0619;3.4167,50.8833;22.2036,40.5233;57.7061,-20.2639;37.6333,-1.7833;27.1333,62.3000;15.7956,49.9511;110.4050,-7.1397;30.5433,38.7567;-87.6167,14.3167;130.8333,33.8333;5.0806,52.0292;30.1167,49.7833;-0.1833,51.1167;-80.3667,8.5167;16.7500,53.1500;-24.5167,14.9000;46.9333,-21.8333;22.9000,45.8833;-91.5167,14.5833;13.7361,45.5786;69.6317,40.2800;11.3333,2.1500;125.7147,38.0406;-99.0167,21.9833;44.7717,48.7906;25.7167,56.6167;126.1611,37.9089;-46.3108,-23.5425;-77.8690,-6.2317;-68.8167,-32.8833;3.7500,8.8333;-65.7531,-19.5836;-79.8436,9.3606;35.8000,32.6833;158.1500,6.9167;119.9667,31.7833;-9.1833,38.6167;24.7333,-26.9500;-71.2333,-34.9833;24.5222,56.6064;-83.8667,9.8333;-60.1167,-22.6667;44.0650,9.5600;-8.8642,52.7039;-84.4777,37.9887;-1.0833,53.9667;155.5500,-6.2167;-90.6786,14.5758;101.0833,4.5833;106.3333,20.4500;-63.4333,-18.0500;45.1783,40.3822;3.4911,6.6028;-77.1650,-6.0586;2.3308,13.3086;126.6500,45.7500;45.4833,-25.3000;8.3833,51.9000;10.1141,44.0220;147.3333,-42.9167;-1.4167,54.5833;23.7333,37.8833;34.7667,0.6167;68.5775,43.5264;\
    46.9306,39.9931;6.1558,49.8678;6.0003,49.6572;36.1667,33.0833;102.9806,11.6175;-79.9496,43.2334;126.0961,39.1425;36.3000,33.5000;12.5269,32.0631;36.3500,8.4500;33.1000,34.7208;51.5383,30.1253;17.3167,50.4667;-89.4958,13.9089;5.4333,7.4667;49.6500,-14.6500;-80.6667,-1.0500;139.6917,35.6895;-65.8667,-17.3500;24.4500,54.8000;-72.1167,19.4667;178.4167,-18.1333;39.2833,8.4500;169.7500,-46.2339;39.3167,37.7500;-91.1461,15.4058;131.8735,43.1056;-68.6000,9.6667;-4.0226,39.8581;50.1667,-14.2667;-59.6167,13.1000;139.4500,35.4833;43.2628,-11.4658;22.9322,37.9378;-111.8412,33.3062;-6.5578,52.5008;-91.8333,18.6333;26.6667,-26.9833;-77.9333,9.1333;25.0333,60.4167;-85.2833,12.1833;-8.6444,42.4310;36.1167,33.0000;37.3833,11.6000;4.2917,51.4950;-73.1167,-36.7167;-7.8000,53.7333;100.8000,14.7167;124.8442,1.5017;-56.9667,-25.4000;-80.2833,7.8167;-75.9117,4.7464;-80.9497,43.3668;23.2242,57.1631;20.2219,41.4917;-8.4694,54.2697;34.5833,31.5167;-16.3056,13.2058;7.3833,6.8667;-54.9500,-34.9000;13.9486,9.9342;21.1833,4.3167;36.2333,33.3500;118.0931,24.4675;85.3333,23.3500;43.2889,-11.4211;43.8353,42.0350;-171.7000,-13.8333;-85.2167,12.9333;14.4241,50.0878;-86.1167,13.2167;-59.8500,-36.7833;3.8889,51.5042;21.5833,56.4333;10.0422,36.8342;127.1489,35.8219;7.1167,43.5833;123.7650,41.2886;33.0989,-25.0269;-86.8000,15.7833;128.5911,35.8703;-22.3667,63.9667;80.7829,6.9708;15.1103,46.3592;44.4814,40.3217;-72.4425,7.8178;10.9333,1.3000;25.8333,42.0333;-19.4333,65.9000;19.8189,41.3275;16.0667,3.5333;43.4722,-11.8758;9.9916,48.3984;25.6562,60.9833;138.1292,9.5144;3.1667,50.8000;43.8383,-12.3731;33.4833,51.8667;22.5611,41.3169;31.5706,-17.3156;-71.6217,-33.5933;33.4667,50.7500;99.9354,9.5342;14.5000,50.1167;26.5333,54.7167;36.2000,32.3500;-59.8167,-18.2833;-67.5419,10.1739;24.4500,54.8667;33.5868,34.8755;-79.8833,8.5833;-67.1333,-17.4167;139.3422,35.3231;20.6000,41.6100;-82.4992,22.8911;-72.6833,-38.2500;15.7833,4.2667;21.6667,48.4000;77.0000,20.7333;-86.0939,12.2006;9.0455,47.4615;35.3431,32.9458;13.5000,48.2167;\
    24.3578,57.8636;-76.3167,3.6881;-72.6833,19.6667;23.5333,38.0333;38.6167,-5.0500;125.7558,38.5072;24.6333,54.7667;31.3333,-26.4833;-20.7333,64.2167;-5.8300,30.3100;91.5667,27.0667;12.3000,55.5833;24.3581,57.7536;28.3667,-26.2333;19.9303,41.4400;35.7000,-7.7667;128.5886,34.8503;46.8503,40.5808;45.3875,39.7369;101.4131,14.7042;-70.4833,-25.4000;15.1711,50.7243;62.7925,31.5292;22.9000,45.7500;11.5744,48.1377;23.7167,37.9167;-0.7011,38.2622;5.2678,14.8903;-78.8496,43.9001;95.8667,20.8667;-90.7267,14.5183;89.4667,23.0167;4.8833,45.6833;-63.9333,-18.1167;3.3833,50.6000;-82.5053,22.7906;-0.9500,38.0833;25.1000,60.4667;57.5353,-20.1056;71.2667,29.1500;15.4519,43.9433;-71.2167,-32.7833;146.9167,-36.0833;3.3414,36.7422;-90.4994,14.7083;26.4000,47.9500;-101.6667,21.1167;-89.1392,17.0750;-85.5333,13.5333;18.7000,50.0500;28.0500,45.4500;127.6378,37.2939;-13.0000,15.2667;8.8491,45.6113;-80.0883,9.1933;-66.5333,10.4667;-7.1911,53.1622;11.5833,50.9333;39.6667,-4.0500;134.8500,34.7667;2.3000,7.2333;45.1861,40.2081;125.6128,7.0731;23.9167,47.0333;150.3000,-33.7000;8.3561,36.2247;8.5405,47.5220;25.9708,43.8564;-59.7500,-18.3333;91.7908,22.6883;90.5000,27.5167;-71.0598,42.3584;37.1333,-0.7833;-71.3667,-34.6333;74.5333,20.5500;15.0728,45.7800;31.3228,52.4094;39.7303,43.6000;24.5331,59.4303;30.0417,0.7414;85.8333,20.2333;20.9528,64.7507;13.2000,-1.5667;1.9833,7.1833;47.0581,34.3878;141.5000,40.5000;-98.9333,19.2667;-86.0175,12.0839;-80.0167,-1.5500;19.9286,41.3092;15.6814,46.4519;14.6578,49.4144;141.3000,-2.6833;-77.5500,18.2833;73.6833,24.5833;-171.3667,-13.9667;14.2148,50.7822;32.9833,-24.5333;-82.0281,22.8361;88.4333,26.7000;-71.7500,18.0500;11.0022,59.2324;-78.1662,43.9668;11.4333,48.7667;18.6975,45.0775;23.0319,57.2656;125.4736,39.4947;-60.0333,-22.3500;116.6947,39.5097;17.6853,45.3403;-83.6667,9.3333;42.4775,12.1494;-86.2000,11.9167;-7.7039,52.3550;76.6497,12.3072;32.8000,-5.0167;44.3247,15.1714;10.2500,6.1167;120.5986,15.4889;-3.7635,40.3272;10.8058,36.4672;-63.5000,-21.2500;15.8667,4.9333;\
    2.1167,13.5167;8.6833,50.1167;23.4833,12.9000;33.5078,48.3436;35.8575,32.5072;17.7617,43.6603;45.1750,40.1753;-84.1000,9.9000;-89.8833,16.9333;-171.7833,-14.0167;-55.6333,-27.0333;-79.5333,8.9667;30.2642,59.8944;-74.1325,45.2501;35.0078,29.5267;57.3661,-20.3603;100.2667,5.3333;175.1500,-37.4000;-88.9167,13.7833;-87.9500,15.8333;42.2242,42.1942;26.7833,-26.8500;47.1131,39.7953;1.2983,52.6278;120.9050,14.4456;-59.8000,3.3833;-74.2214,4.5872;7.3667,48.0833;36.0686,32.7381;26.9229,43.2706;18.3333,13.2167;103.0167,21.3833;7.5356,46.2919;68.9303,38.8411;35.3500,-0.7833;-84.0333,9.9500;20.5972,40.1514;-84.3167,10.1000;-82.3831,42.9668;15.6141,48.4092;45.8292,32.4975;8.5556,35.8914;-98.9000,19.2667;11.7158,44.3523;-84.7667,13.7333;15.2364,46.1519;-41.9494,-18.8511;19.9592,41.4217;23.7000,37.9333;108.6000,-6.8000;-2.1833,53.0000;13.8245,50.6404;28.7667,61.1667;35.8833,31.9000;-79.5714,-7.4006;-74.3667,4.8167;20.8839,41.8786;-71.8333,-36.1500;-66.3821,50.2001;4.9167,7.6500;57.0801,27.1467;26.8333,43.7167;-86.5500,13.8667;-2.2000,47.2833;-86.1333,11.8167;27.4000,-26.3667;9.0874,45.8100;83.4500,27.7000;-8.6167,41.2333;-21.0667,63.8333;16.8820,48.7590;44.6039,40.2536;34.6333,-14.8167;-73.3578,8.2364;-57.6167,-34.2833;-71.4494,-33.0450;14.6833,-13.7833;147.2667,-2.0167;20.9106,41.9108;-6.9508,37.2583;8.4353,49.4811;-43.4511,-22.7592;12.0761,57.4872;-79.7889,-6.6389;60.6919,27.2039;40.7358,-15.0342;88.4089,22.8714;129.2139,35.2442;28.4367,46.4275;7.6936,47.5207;106.0500,21.1833;31.5258,-3.2833;2.4667,27.2167;8.9000,47.5500;1.7833,6.8167;104.9667,13.8167;31.8958,-3.0203;38.9833,8.7500;22.0333,60.4500;-62.3989,8.0086;27.9667,-27.2833;-73.6658,45.5001;72.3667,30.4500;9.8667,5.8333;7.5333,47.5500;-68.1500,-16.5000;5.1000,7.1000;-74.2542,11.0094;38.6167,8.9167;-65.1000,-43.3000;-89.4012,43.0731;19.1333,47.7000;46.3167,2.7500;16.9414,45.5744;-85.1333,12.8000;7.1167,51.5167;47.2000,3.8500;88.3667,25.8667;153.0500,-26.6833;14.4123,50.0360;43.2906,11.9631;107.6206,-6.9128;137.9667,36.2333;100.8667,14.3333;\
    47.8575,4.6544;9.5094,47.2204;-40.3078,-20.1286;-69.0833,-16.2500;32.9914,35.1981;-84.1333,10.0167;98.2000,14.0833;23.0267,57.2486;16.2909,41.2306;-6.3300,33.9000;2.2667,9.2000;14.2333,48.2167;101.8000,2.8167;32.4083,34.8042;44.1111,41.9736;13.6362,50.5030;-86.2333,11.8500;-91.3167,14.5333;44.2869,-12.1917;23.2000,45.8333;126.6628,36.5994;17.7667,8.3500;19.0333,52.6500;-66.6656,45.9454;-71.0167,-33.6833;79.8978,6.9600;36.1333,34.8833;36.7167,35.2667;62.1561,45.8536;-71.7833,19.1667;21.1167,47.3167;-122.6762,45.5235;38.2833,-4.7833;37.3500,-1.3000;43.5325,41.7497;-94.5786,39.0997;23.2667,-16.1167;24.7575,59.1681;-57.3000,-25.6000;17.9711,49.4718;15.1792,44.6392;99.1667,10.5000;-2.2167,31.6167;-89.2889,13.5894;-16.0278,13.5228;2.6167,6.4833;107.0800,20.9511;35.8167,-8.8000;35.6500,-10.6833;-82.0108,23.0431;6.8833,51.4333;-0.5833,51.5000;120.9750,14.2306;45.4667,-19.5167;-79.4669,22.5206;2.1500,6.6500;-66.3833,-17.4500;-64.3500,-33.1333;-85.7594,38.2542;110.3833,21.2000;21.2167,5.0333;-2.3333,7.3333;19.4364,42.2136;16.2833,48.1000;151.8500,7.4458;-76.2447,3.4211;35.5000,-16.0333;21.4333,42.0000;-57.7833,-30.7833;-86.4000,13.4500;44.4258,-12.1322;-0.6667,5.8667;106.8294,-6.1744;-93.0933,44.9444;5.0417,52.1392;-12.1833,12.5500;38.6508,48.5633;108.7147,34.3456;0.7531,14.0106;25.7833,45.8667;88.3744,22.6942;32.6825,35.1803;-20.9500,65.4000;36.9667,-0.6833;70.8700,34.2400;100.3667,6.1167;9.9050,36.3669;-78.6833,-0.9500;-60.1548,-32.6184;6.2014,52.1383;9.6167,1.5833;51.6714,32.6597;-78.8156,1.7986;25.3219,53.0867;-72.3333,-39.6333;4.0333,50.9333;34.5667,0.5667;-63.2833,-20.4167;69.7819,37.9092;-15.9333,13.9167;69.4806,37.0842;50.8644,32.3256;14.1197,45.0950;48.6828,41.6281;43.3800,38.4942;-89.1731,13.8769;75.1667,15.3500;102.4167,2.8167;46.9289,40.3450;-77.5333,18.4667;10.1500,36.4000;8.8528,56.7933;18.4167,47.5667;-80.0333,0.6000;27.8458,37.8444;-74.7667,10.9172;-53.4167,-32.5667;-14.1167,13.2833;126.0025,38.8589;31.8833,49.2333;4.6167,50.7167;44.4242,41.9161;-57.4167,-25.3500;\
    -70.3333,-27.3667;-77.8167,0.6000;9.1333,36.3333;6.2644,36.4503;-88.6833,14.9500;15.2833,47.4167;174.0833,-35.2833;9.4020,56.4532;-135.0537,60.7161;-80.1937,25.7743;-82.8333,8.4500;77.9753,44.3583;82.3000,28.1333;-16.3167,28.4833;26.3147,58.3419;-3.0333,53.4000;69.7500,43.5667;-18.5286,65.9702;16.1500,9.3333;16.9900,46.0672;6.8500,51.4667;43.7975,-12.3375;138.6833,35.1667;173.8583,-39.4556;-84.4585,38.0498;-76.4167,17.8833;-91.0167,14.6833;16.7722,44.5350;33.1397,15.1603;-84.0333,9.6667;-77.2672,-11.5714;28.6422,53.2933;24.3667,49.0167;-14.9000,13.6333;175.5000,-36.7667;2.5167,48.8167;14.1033,46.4303;39.3772,14.8444;38.0167,-1.3667;32.1833,12.3500;-71.4388,-41.1673;14.6108,10.7361;23.4944,57.0244;-78.1453,8.4028;12.2508,57.1056;14.3167,46.3667;-80.4414,-3.5667;-77.6667,17.8833;23.6500,54.4000;33.4625,35.0000;-54.2000,-31.8667;24.4333,60.1167;10.7500,-0.1833;35.2603,32.8072;52.1167,44.1667;12.1821,55.4580;35.3417,32.7500;105.9228,49.4867;23.7878,56.6864;13.0203,32.1722;-55.1167,-33.4667;-72.4175,18.7358;-104.9000,21.5000;44.5783,39.9308;-56.4000,-25.7000;1.7336,42.5428;-72.0167,19.6333;71.3667,42.9000;141.6500,-8.5500;36.8667,-6.1500;66.6167,27.8000;46.3861,5.5350;-69.6203,9.9256;-7.5000,40.2833;145.7667,-7.9667;-87.3500,14.8000;1.1333,7.5333;120.6544,28.0192;-8.6167,41.1333;21.4167,50.2833;17.2250,45.5906;-172.3500,-13.7167;19.7833,47.0333;-15.6833,16.4667;13.0007,55.6059;15.3516,50.4372;-73.4492,46.0168;12.4000,55.6667;34.3000,-12.9167;15.8586,45.1342;25.7500,60.8000;22.6433,41.4375;34.9069,32.1750;173.1000,-41.2500;100.2167,13.7167;-91.7167,15.0833;15.3033,46.2933;102.2481,2.1969;142.3703,43.7678;8.3064,47.0505;3.1333,51.1500;-59.1050,-34.5703;152.3500,-24.8500;32.8644,39.9272;-84.9500,10.2833;89.3833,26.8500;49.6839,34.0856;-74.2239,-13.1583;-64.1592,8.8933;-82.5665,42.0834;50.0853,40.3744;28.2294,-25.7069;104.0833,49.0333;20.4219,42.0769;57.7456,-20.2158;72.3611,34.7761;79.8944,6.9317;17.1417,60.6745;90.6500,27.2333;-72.9072,11.5444;1.2500,41.1167;8.7667,50.1000;\
    34.6179,36.7953;21.5853,42.1564;123.6419,10.3792;19.0833,47.5000;27.2500,-25.6667;36.0833,33.4333;6.0472,49.8556;5.8786,49.6856;23.7167,46.3000;69.9622,38.8669;32.6667,-18.9667;-4.4833,48.4000;-85.8833,11.5000;20.6333,47.0000;-87.8411,13.5844;-13.8500,13.3833;-79.6163,44.1168;125.7800,38.2431;4.4500,51.2833;26.9397,58.8486;147.1667,-41.4500;-73.6500,-37.6167;22.7492,58.9978;88.6000,24.3667;16.1333,29.1667;35.0947,33.0058;15.0667,46.2833;6.7167,33.5667;106.3453,9.9347;141.4333,-31.9500;21.3500,47.9667;-0.2333,5.6500;127.5364,39.9183;-121.8950,37.3394;32.9333,-9.1167;-71.9167,19.5167;12.5108,38.0158;-90.4000,15.3833;-9.4167,38.7333;8.9167,50.1333;41.8500,-1.2167;114.6500,4.8000;-76.4000,-13.0833;122.9828,10.7731;2.8167,36.4667;61.0667,34.6667;-82.8000,8.4000;-9.1500,38.6833;6.2183,49.8019;-73.2500,18.4833;-80.4500,8.6000;-21.7667,65.1167;8.0500,34.7333;26.5000,43.8000;97.6256,16.4914;57.5131,-20.0628;-80.4500,-1.0500;19.7000,52.5500;30.8000,-27.7667;12.2765,41.7321;-8.9864,52.8436;15.9536,45.1636;121.1219,14.6969;-82.7167,8.4167;89.0833,24.1333;129.7228,33.1592;174.0333,-35.3833;-70.1333,-15.5000;-72.2042,19.7578;125.0467,40.3894;-79.6663,44.4001;120.2939,31.5772;20.0836,41.7797;-8.2992,51.8572;35.2406,-13.3128;24.5022,58.4239;142.3500,-3.1333;-5.9300,34.2600;-91.3667,14.2833;75.5792,31.3256;26.7500,-27.8333;-19.1333,65.7333;30.4053,54.5153;29.6433,46.8403;43.2619,-11.6981;20.0933,41.3025;46.9989,35.3172;9.1615,45.1845;145.7667,-16.9167;-1.1333,52.6333;140.7000,-2.5333;-3.6420,40.5475;-3.0500,53.8167;15.8500,46.4667;28.8833,61.8667;129.6000,44.5833;24.9667,42.3000;31.9639,-3.4919;18.4369,49.7798;13.6892,11.5183;-123.0693,49.3164;15.6613,38.1105;-63.6833,-22.0333;57.0842,30.2939;20.3517,40.2336;-77.9167,18.4667;19.9053,41.4933;96.2378,16.7722;30.9964,53.4453;13.1102,55.7919;-77.0054,18.1020;22.9000,61.3333;4.0289,13.6461;120.1689,30.2553;115.7667,-32.2500;-1.7167,8.0500;34.7667,31.8167;77.9167,45.0000;-73.3500,18.5167;26.2000,41.7667;-88.3500,13.3833;-172.4667,-13.7667;\
    172.1833,-43.3000;23.6000,54.9333;14.4000,46.0333;23.7167,37.9167;11.1079,42.7714;15.0014,46.1342;98.3500,7.9167;30.9642,-3.2828;72.7000,30.5333;-56.3167,5.8833;33.4803,0.4597;72.6500,30.7333;28.6833,-10.3833;-43.8514,-19.7697;33.6167,-9.2333;22.0819,38.2486;12.0225,55.9707;25.4706,58.5361;-103.3333,20.6667;11.0909,43.8843;22.7833,10.2833;-74.9228,10.6378;37.0667,0.0167;-0.4167,51.8833;-75.1638,39.9523;-58.4000,-16.3667;14.4213,48.0427;47.1706,41.1919;12.3638,55.6569;35.4025,33.1102;31.5500,-18.1833;10.5058,43.8436;43.6486,42.5231;29.9169,40.7669;34.3469,61.7849;22.7500,56.2500;13.4178,50.4605;11.3336,46.4927;-80.9900,46.4900;-71.2003,-30.5983;23.8333,38.0500;15.1500,47.0333;44.5947,40.0042;35.0000,31.7500;19.7861,41.7744;23.4885,41.8383;25.5000,64.9167;1.1500,49.0167;30.7326,46.4775;-65.3000,-24.1833;20.4000,-9.6500;73.0833,31.4167;174.0667,-35.3833;8.7667,55.3500;22.0833,53.1833;-71.6273,-33.0393;47.2833,-22.8167;-72.0167,19.1500;14.5167,48.1667;-100.9833,22.1500;34.3806,53.2875;5.2000,7.2500;143.8500,-37.5667;119.8597,-0.9017;57.4244,23.3908;80.3500,26.4667;15.9964,46.0411;133.3667,34.4833;-70.6036,9.3178;32.4333,-14.0667;45.2036,4.7358;43.3797,-11.5439;3.0667,50.6333;-91.7667,14.9667;-24.3333,15.0333;112.7508,-7.2492;8.6168,45.4406;28.3656,47.4997;100.6167,14.8000;10.8333,4.7667;95.2667,16.3833;28.1167,56.3833;35.8500,32.5556;-58.4088,-34.5761;1.2133,6.4261;37.1000,36.4667;-171.5833,-14.0167;62.5869,34.4792;-76.7936,17.9970;-75.5144,10.3997;-21.9500,64.0667;57.3706,-20.3256;-78.4167,-1.4000;175.9333,-37.4000;15.0192,46.5881;24.4000,41.7433;14.7172,36.9282;-5.5667,42.6000;-1.6140,54.9733;3.3500,7.1500;75.8333,25.1833;27.8170,-26.8136;4.8500,52.6667;22.3339,41.8833;127.2203,37.5897;14.3053,46.6247;-80.4167,7.9333;117.9000,4.2500;80.2117,6.0536;-1.6175,54.9450;31.0419,-5.1092;23.0333,40.5833;-88.0333,15.1167;11.8167,6.7500;121.1169,14.5800;14.2956,45.9661;-52.3425,-31.7719;11.7609,55.2299;22.7833,49.7833;-8.5000,33.2500;8.7500,47.1167;19.8511,41.4167;4.1975,12.4539;\
    38.8153,14.8872;-14.9167,13.5167;-25.6667,37.7333;-78.2258,21.5247;-57.5833,-30.2333;-63.1500,-14.4833;-78.5667,-0.5000;18.9333,47.3667;-39.3153,-7.2131;-22.9000,16.6000;-8.4958,51.8986;-15.9500,14.2667;40.2239,43.3283;-5.8333,43.3667;25.7333,62.6000;15.5866,56.1616;-9.2667,39.1000;-57.7667,-17.7833;45.1667,-21.6667;21.9833,6.5333;44.2833,-20.2833;136.9723,35.2476;13.3039,55.8393;-21.1000,65.2000;45.0164,10.4356;31.1667,39.6667;-88.3833,13.5167;36.7667,36.2167;10.5833,36.6500;14.8442,46.8406;34.8106,32.0192;100.4667,7.0167;-0.1395,50.8284;-76.5444,3.2639;135.0900,34.0903;106.1667,20.4167;57.6142,-19.9842;-15.8167,16.5000;18.1167,46.3833;15.1675,46.2536;33.0375,35.1417;102.1794,17.3075;-67.0333,10.6000;55.2000,45.3167;-117.0842,32.6401;44.4983,-12.1436;-86.2006,12.0233;-86.7844,36.1659;-116.6167,31.8667;-75.5131,6.3489;4.9000,44.9333;15.2375,44.8694;106.9528,47.7069;14.0000,32.6167;-80.2333,-1.8333;-0.0833,5.9667;22.5133,41.6258;83.0000,25.3333;17.3261,44.3667;-69.2333,-38.9333;6.9167,6.0167;32.6500,-13.6333;71.4753,30.1956;20.1561,41.3208;17.8983,43.9625;107.0667,10.3500;4.4931,52.0575;147.0667,-38.1000;-80.9000,-2.2333;125.4244,38.8561;-72.3500,-37.4667;-9.1833,38.8000;106.2472,-6.3547;14.3603,46.7681;128.6267,34.8825;-72.4874,-51.7236;28.4000,-13.1333;28.5833,-26.6500;20.0697,41.2817;33.4875,34.8083;88.2153,22.5083;2.1773,41.4416;-43.3994,-22.7642;42.8000,9.3500;31.7142,53.6989;33.7833,35.0583;33.6667,13.4167;-79.6333,-4.0667;27.5619,53.0272;85.5167,27.6333;-62.2833,-38.7167;121.1653,14.2117;-14.6667,13.4500;44.2994,-12.2056;27.4636,58.0981;40.1667,8.9833;57.4772,-20.1856;-63.5167,-20.0500;14.6833,47.2000;-85.7333,15.7333;34.3833,-8.7000;43.5906,41.2669;18.0000,53.1500;4.4333,50.9333;14.2408,45.5697;-110.3000,24.1667;35.8333,9.1667;15.8078,45.8564;-81.2719,-4.5772;80.0139,6.8408;30.0333,-20.3333;106.4579,48.8523;9.5215,47.1415;-78.1167,-2.3167;-68.7000,-16.5500;-72.8333,19.9333;49.6959,36.0696;49.9200,40.4617;71.5500,32.5833;-51.1306,-29.6783;26.8497,54.3167;-91.1167,14.9333;\
    22.4444,41.8542;14.9000,9.3667;44.3881,-12.1697;6.7833,6.1667;5.6667,53.0333;-80.3833,-2.6333;68.7000,36.1200;6.0710,46.2284;129.2131,42.2294;-7.1524,52.1624;69.6092,37.4672;106.5528,29.5628;26.5547,39.1100;168.3333,-46.1500;45.5333,2.9667;28.1667,43.4167;-79.5000,9.0333;10.7603,34.7406;-71.8972,19.6742;-77.9833,9.2000;34.4500,-0.5167;7.9864,13.7533;39.1167,8.6000;128.0847,35.1928;32.8625,34.7292;-6.4167,54.0000;31.3108,1.6200;-70.7333,-34.0667;15.5139,9.6272;3.1500,50.7167;-91.0500,14.0833;19.6550,41.3908;133.9167,34.6500;-56.2167,-26.9333;-114.0853,51.0501;-88.8667,13.5000;-54.1667,-31.8333;52.5714,28.8461;35.9667,0.4667;153.4333,-28.0000;81.6333,21.2333;28.1378,-25.9636;-87.8439,13.3369;18.1000,59.1167;-13.8667,13.3333;38.6617,48.5069;28.1972,45.4817;-55.7833,-27.1167;25.1833,56.6000;31.9000,-27.3167;6.9667,47.0000;73.9650,38.1689;-55.9000,-34.7000;-55.2744,-34.8682;-54.9167,-34.8000;12.4833,9.2000;47.4167,-19.3833;-88.3500,13.5000;32.6492,0.3475;52.5383,29.6150;-172.1333,-13.5833;-112.1860,33.5387;2.2667,48.8167;-73.7567,3.9875;6.4764,52.7225;-71.2375,8.5514;175.6667,-37.3667;23.1167,42.2667;4.8167,43.9500;27.7667,-26.1000;-86.3167,13.6833;14.0720,49.9638;94.8833,20.4667;5.6583,52.0333;-84.2167,12.1500;34.9333,32.2500;46.8250,40.5067;116.7917,33.9744;35.9042,33.8497;-79.1000,9.1667;24.1500,45.8000;-87.9714,13.8108;26.0167,42.4833;51.1750,24.9044;-1.3822,54.9046;-79.6086,9.2447;146.3500,-41.1667;-61.7333,12.1667;-66.3500,-33.3000;8.5313,47.3401;22.6869,60.4567;24.3500,44.1167;-10.9667,12.2833;-8.4300,32.6500;43.1192,36.4894;106.9919,-6.2367;16.4484,57.2646;-78.1522,-10.0681;3.3000,50.8500;20.3333,46.4167;-103.7167,19.2333;-79.4833,-7.2500;130.9500,33.9500;2.0833,41.3500;-89.6731,13.7447;33.4833,-19.1164;-9.5167,53.8000;35.3319,32.7944;11.1150,35.2372;-96.6989,33.0198;-108.2847,52.7834;20.6333,48.2500;66.6300,32.9200;23.9381,56.7853;-7.3200,53.1136;-66.0377,45.2594;-83.0333,10.0000;75.5667,21.0167;-90.7436,14.5819;24.5525,59.3203;8.6908,49.4077;22.5333,57.1500;135.8667,35.0000;\
    -71.6406,10.6317;15.8736,45.1742;17.2833,-12.1500;-89.3222,13.4883;-9.3977,38.7057;9.2792,47.3861;70.3333,40.6667;37.9833,8.5333;129.3450,35.9942;100.1008,17.6311;41.8661,9.5931;-8.3000,41.4500;-74.2167,4.7167;57.4072,-20.2869;22.6561,44.6319;11.7003,47.3350;139.4508,35.5403;-75.3978,9.3047;-65.2592,-19.0431;-79.9959,40.4406;40.7500,36.4833;71.4278,51.1811;35.9500,31.7000;-1.5667,54.5333;125.2103,39.6933;-89.7383,13.7164;5.9117,35.6286;9.6994,45.0468;37.1167,-1.7500;117.1500,-0.5000;-22.7333,65.0667;22.3794,37.5089;17.4333,46.8833;-8.7167,42.2333;13.5833,7.3167;73.8000,19.9833;-80.4830,43.4501;-15.2000,13.5833;8.2068,44.8979;18.0649,59.3326;-45.5553,-23.0264;23.0358,54.9547;8.5849,47.4515;-88.9333,13.7167;8.3059,47.4733;-7.7833,54.8000;42.4694,37.2497;128.2933,41.0903;24.5000,56.8333;3.2167,50.7333;14.8091,56.8777;28.3894,47.8639;-77.5167,18.3167;37.6156,55.7522;22.2361,41.9886;73.0994,49.7989;-86.1580,39.7684;3.3833,8.6667;15.6467,46.5547;159.9500,-9.4333;-122.4194,37.7749;33.5167,-3.5167;73.8554,18.5196;4.8333,7.1000;-171.6833,-14.0167;-85.9500,14.7667;18.6667,50.2833;111.7386,35.2089;6.8433,46.4612;14.3333,51.7667;-97.5164,35.4676;17.5250,45.0978;68.7800,37.2211;57.6383,-20.3642;-75.3439,10.2544;-57.2000,-32.3500;72.8333,21.1667;28.0833,-12.6500;19.8997,42.0444;176.0833,-38.6833;-10.8172,6.8708;20.1482,46.2530;12.2922,55.6517;47.9586,29.2775;38.7158,14.1297;-97.8333,22.2667;1.1570,52.0547;74.3436,31.5497;89.0500,25.6667;-14.7833,13.4333;-12.4985,16.6404;-6.2600,32.6000;55.1714,37.2550;27.7833,-25.6333;2.0833,6.3667;-79.9667,-3.2667;44.9658,41.7364;50.6542,26.2456;28.6103,47.1414;139.3239,35.6558;22.5500,4.3167;-57.5167,6.2500;6.1736,49.5842;-6.0631,53.1408;175.3000,-40.4667;14.1000,46.7167;103.9500,22.4833;174.7500,-37.2500;62.1100,32.3700;84.2719,49.7386;4.4653,51.5308;-7.4833,11.4167;-61.1833,12.8833;-70.7294,7.2472;39.6583,-0.4569;18.0156,45.1603;-66.6333,-18.4167;69.1667,37.1333;-75.9780,36.8529;44.4797,-12.2672;42.7103,41.6858;11.6667,52.1667;34.3000,-11.6000;\
    -8.5667,41.1000;12.7667,26.5833;126.5819,40.9561;143.7000,-5.4833;68.9486,33.9675;35.2956,32.7036;-5.7000,35.7000;-4.7400,32.6800;8.5833,53.5500;90.8592,23.2511;19.0333,49.8167;-172.3833,-13.7500;33.4208,35.0250;102.2500,2.2500;125.3619,37.9347;-15.4167,28.1000;13.7294,45.5469;4.2667,7.3833;10.4832,59.4171;-14.0000,65.2667;36.3833,48.8833;19.1667,50.3000;-104.8214,38.8339;-10.8047,6.3106;23.5667,61.4667;0.6333,6.9000;-15.5333,13.4333;49.3039,31.9364;-78.8784,42.8864;-1.8500,53.7167;89.5000,23.1667;-16.1383,13.3803;28.7778,46.9431;86.1833,22.8000;24.8667,44.8500;3.8500,9.0833;15.2406,10.3428;-171.9667,-13.8333;125.4889,9.7839;21.5833,48.3167;147.1925,-9.4647;-78.1169,21.8525;99.5000,16.4667;-73.2472,-3.7481;11.4500,47.2833;35.7667,-4.0167;14.4333,46.1333;35.1000,32.7500;40.7833,8.9000;28.4028,45.6842;16.3092,38.9659;39.2833,-10.9500;20.9758,41.8389;35.0753,32.8364;-71.2167,-33.7000;32.6667,13.1667;-171.8000,-13.8167;74.2000,32.6333;9.8074,54.9093;101.2500,3.3500;-89.2186,13.9736;18.4297,42.8719;23.8667,54.8167;-77.7167,18.1667;-56.7000,-24.4333;23.8000,37.9667;3.2667,51.3500;9.6500,47.4331;15.5836,46.5536;50.1500,53.2000;-12.9500,15.2000;24.8783,59.4581;-88.7333,15.0500;135.6168,34.8483;-172.1167,-13.6833;47.2519,56.1322;22.6392,32.7650;8.8388,47.2298;-101.0000,25.4167;68.3333,51.8000;42.3781,34.1397;-15.8667,13.3000;45.3292,40.1378;-91.6167,14.8667;21.0000,42.0328;-8.3960,43.3713;-6.6708,53.4022;24.2022,56.5986;-61.4667,12.4833;-88.3500,13.3500;4.7009,50.8796;19.0756,47.6694;11.9833,2.9333;34.8142,32.0806;21.5822,41.9961;-71.9781,-13.5183;-98.8833,19.3167;-74.4000,18.3333;129.7758,41.7956;-57.6500,-25.3500;-76.3500,17.9500;47.0261,39.4000;33.3539,16.7097;43.3858,-11.6431;-73.3500,-37.4667;-98.7333,20.1170;-15.4333,13.3667;25.3167,60.6333;44.7547,40.5017;37.1150,36.7161;12.0667,47.4833;-9.2500,38.7167;8.9167,3.7000;39.1979,-6.1639;168.9436,-46.1028;-87.8736,13.7406;-6.4956,53.3658;23.7522,58.9539;4.1167,51.0333;-79.4425,21.9297;22.9000,40.6569;28.9333,-18.2167;27.4667,55.3667;\
    -171.3667,-13.9833;-68.5364,-31.5375;85.8000,26.6333;35.9919,32.6414;12.4667,9.2833;69.3356,38.0964;30.2167,50.5667;10.2108,56.1567;11.6333,52.1500;-5.4539,36.1275;13.4500,4.2333;20.8019,41.1172;-72.5347,18.2342;26.8326,55.1168;27.3050,48.1681;20.9625,41.8242;-79.4833,-2.2167;102.2000,2.2167;10.4167,5.4667;100.7333,14.0167;28.6633,46.3658;31.3000,-29.3333;1.1053,9.7681;46.7892,39.9111;70.7000,34.1200;19.2000,47.6167;-73.0203,5.8269;-60.9000,-16.4333;10.4667,1.5833;121.1497,14.5256;2.5667,48.8500;19.8030,48.0987;-1.3331,34.6425;10.8000,35.7000;1.5341,42.5073;-171.7333,-13.8333;-61.1167,13.2667;36.0833,33.7167;71.5733,34.0078;-122.3321,47.6062;-89.5667,-0.9167;14.0333,48.1667;-47.4581,-23.5017;35.5792,31.9522;11.4000,47.2667;92.0514,22.0756;36.3667,0.0333;124.6075,11.0064;44.5506,39.9539;35.8164,37.4506;57.4783,-20.2981;22.0500,50.5667;-79.5167,-1.9167;25.0772,59.3644;-66.8333,-20.4667;18.5694,43.8161;17.9872,44.6117;19.3833,51.3667;22.7333,37.6333;8.6687,47.2268;-1.2167,6.6167;9.1916,48.8973;8.7581,36.9544;32.4333,34.8375;16.4339,46.3844;-86.5667,15.5000;32.0400,54.7817;43.3978,-11.8736;11.0683,49.4478;-9.2308,38.7538;-6.1375,53.4231;27.0500,53.9667;19.1833,6.9833;133.8833,-23.7000;-8.8706,51.6231;-86.3167,11.9833;28.9728,46.3311;126.7114,35.9786;8.5122,47.4181;17.3931,49.2979;10.1333,3.8000;45.2075,42.0311;14.8544,46.5214;33.9833,18.0167;102.6667,2.0500;100.5968,13.5993;127.5078,37.2917;6.7417,52.6625;24.7667,-28.7333;-89.3667,15.1333;30.7000,-17.8833;0.2050,10.8639;-89.2797,13.6769;73.2333,-0.5833;47.4333,-21.8833;26.2833,54.3167;144.9667,-6.0167;-1.0833,6.8000;34.8511,31.9867;16.9333,8.2667;121.7269,17.6131;-90.7850,14.3050;-70.9467,-30.6919;115.2417,5.2767;-88.7667,17.2500;-85.9500,11.9333;-8.5500,41.3333;35.2167,-15.0667;117.1333,-20.7833;68.7167,25.4667;99.3167,9.1333;34.9500,-14.9833;24.0833,41.9833;86.0333,44.3000;-78.1667,-2.4667;102.8167,2.5000;20.6761,41.1781;9.4386,4.6439;113.2500,23.1167;174.8500,-41.1333;-1.4977,53.6833;43.3739,-11.5178;-9.3333,38.7667;-82.5464,22.9828;\
    -1.5000,52.9333;-57.6000,6.4000;16.8094,45.1767;9.5681,4.5092;28.2167,55.5667;87.3333,26.9833;33.4342,16.6914;-64.1667,10.4667;42.6872,37.1442;11.2500,43.7667;5.8528,51.8425;47.9783,29.3697;53.5425,36.6944;6.1636,49.5650;-92.0500,14.9000;44.8108,41.4636;120.8492,14.9017;4.7083,52.0167;105.9139,20.5411;-87.4500,15.7833;73.7000,30.4500;-55.9333,-26.0333;22.9250,40.5003;21.4547,41.4253;101.0031,14.5881;-84.0528,9.9461;-92.0000,14.9333;25.1831,59.3664;72.9008,32.2658;109.1167,-6.9333;50.0039,40.5328;17.9962,49.3387;90.7181,23.9208;119.8861,29.4603;-87.8386,13.7836;-65.1333,-16.9833;67.6981,36.6953;-77.0500,-12.0500;33.5708,34.8500;33.1833,-12.3000;22.8578,41.7031;6.9167,46.4500;-79.9000,-2.1667;48.5333,-20.5833;9.6167,36.3500;1.3333,47.5833;-7.4500,55.1333;-61.2167,13.1333;8.2833,47.0333;-88.8167,15.4833;-71.4461,10.4019;6.0850,46.2170;36.5833,35.3667;35.4833,30.3167;16.0500,48.3333;175.2000,-38.1833;89.8833,27.5333;-57.3167,-30.7333;6.1639,52.2550;29.0167,-24.1833;11.0022,59.2324;7.3959,47.1921;-84.7833,9.9833;41.6947,41.6753;79.1000,21.1500;-78.9567,-4.0692;9.3667,36.0833;-79.2217,-7.7914;120.2828,14.8292;-6.9100,30.9200;22.3333,47.9500;22.6833,-14.9667;23.5414,58.9431;-56.0983,-34.7422;73.3333,5.4333;-73.8833,18.2000;-77.4667,18.0667;6.7333,0.3333;73.4333,31.3333;25.6000,55.5000;8.7724,58.4618;16.3252,48.3052;-55.7167,-34.7333;10.5839,57.7209;46.3606,40.6828;7.3333,51.4333;73.2667,29.9833;107.4728,-6.8378;-76.2000,4.0867;16.6080,49.1952;120.7658,14.9164;139.8044,35.8203;34.6500,31.8167;73.8000,18.6167;114.8794,40.8100;-4.1000,48.0000;-42.5367,-19.4683;-74.8886,4.1528;8.5500,47.3667;22.4469,41.9228;-6.5700,34.2600;50.1492,37.1953;109.3000,13.0833;147.3667,-35.1167;-65.4533,-19.5506;-66.2333,-20.9333;19.6358,40.5306;-0.3960,51.6553;7.0000,51.0333;2.6000,8.8833;9.9361,49.7878;39.1833,6.9833;-104.6178,50.4501;73.0300,26.2867;103.4500,4.5167;-86.0833,14.7167;-79.2392,-7.9589;-8.7500,41.3500;46.5833,-21.1500;48.5450,39.4583;44.7025,39.8303;-57.1500,-25.3833;-72.7111,-16.6228;\
    -57.8738,-21.0415;-90.8075,14.4803;100.2000,6.4333;18.0269,45.1903\
";

#[test]
fn example() {
    let mut tr = RTree::new();

    // insert some points
    tr.insert(Rect::new_point([-112.0078, 33.4373]), "PHX");
    tr.insert(Rect::new_point([-118.4071, 33.9425]), "LAX");
    tr.insert(Rect::new_point([-73.7822, 40.6441]), "JFK");

    // Search a rectangle
    for item in tr.search(Rect::new([-112.1, 33.4], [-112.0, 33.5])) {
        println!("{}", item.data);
    }

    // OUTPUT:
    // PHX
}
