#![no_std]

use bsp::hal;
use pyportal as bsp;

#[cfg(feature = "display")]
mod display;
#[cfg(feature = "neopixel")]
mod neopixel;

use hal::clock::GenericClockController;
use hal::delay::Delay;
use hal::pac::{CorePeripherals, Peripherals, MCLK};

pub use bsp::entry;
pub use hal::prelude;

pub(crate) struct SharedPeripherals {
    pub clocks: GenericClockController,
    pub mclk: MCLK,
}

pub struct RsPortal {
    pub delay: Delay,

    pub(crate) peripherals: SharedPeripherals,

    #[cfg(feature = "neopixel")]
    neopixel: Option<neopixel::NeopixelBuilder>,
    #[cfg(feature = "display")]
    display: Option<display::DisplayBuilder>,
}

#[derive(Debug)]
pub enum Error {}

impl RsPortal {
    pub fn new() -> Result<Self, Error> {
        let mut peripherals = Peripherals::take().unwrap();
        let core_peripherals = CorePeripherals::take().unwrap();
        let pins = bsp::Pins::new(peripherals.PORT).split();
        let mut clocks = GenericClockController::with_internal_32kosc(
            peripherals.GCLK,
            &mut peripherals.MCLK,
            &mut peripherals.OSC32KCTRL,
            &mut peripherals.OSCCTRL,
            &mut peripherals.NVMCTRL,
        );
        let delay = Delay::new(core_peripherals.SYST, &mut clocks);

        let mut portal = RsPortal {
            delay,
            peripherals: SharedPeripherals {
                clocks,
                mclk: peripherals.MCLK,
            },

            #[cfg(feature = "neopixel")]
            neopixel: None,
            #[cfg(feature = "display")]
            display: None,
        };

        #[cfg(feature = "neopixel")]
        {
            portal.neopixel = Some(neopixel::NeopixelBuilder {
                pin: pins.neopixel.into(),
                tc3: peripherals.TC3,
            });
        }

        #[cfg(feature = "display")]
        {
            portal.display = Some(display::DisplayBuilder { pins: pins.display })
        }

        Ok(portal)
    }

    #[cfg(feature = "neopixel")]
    pub fn neopixel(&mut self) -> Option<neopixel::Neopixel> {
        self.neopixel
            .take()
            .map(|builder| builder.create(&mut self.peripherals))
    }

    #[cfg(feature = "display")]
    pub fn display(&mut self) -> Option<display::Display> {
        self.display
            .take()
            .map(|builder| builder.create(&mut self.delay))
    }
}
