//! Error codes.
//!
//! # References
//! - <https://kafka.apache.org/protocol#protocol_error_codes>

use super::primitives::Int16;

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
#[allow(clippy::enum_variant_names)]
pub enum Error {
    UnknownServerError,
    OffsetOutOfRange,
    CorruptMessage,
    UnknownTopicOrPartition,
    InvalidFetchSize,
    LeaderNotAvailable,
    NotLeaderOrFollower,
    RequestTimedOut,
    BrokerNotAvailable,
    ReplicaNotAvailable,
    MessageTooLarge,
    StaleControllerEpoch,
    OffsetMetadataTooLarge,
    NetworkException,
    CoordinatorLoadInProgress,
    CoordinatorNotAvailable,
    NotCoordinator,
    InvalidTopicException,
    RecordListTooLarge,
    NotEnoughReplicas,
    NotEnoughReplicasAfterAppend,
    InvalidRequiredAcks,
    IllegalGeneration,
    InconsistentGroupProtocol,
    InvalidGroupId,
    UnknownMemberId,
    InvalidSessionTimeout,
    RebalanceInProgress,
    InvalidCommitOffsetSize,
    TopicAuthorizationFailed,
    GroupAuthorizationFailed,
    ClusterAuthorizationFailed,
    InvalidTimestamp,
    UnsupportedSaslMechanism,
    IllegalSaslState,
    UnsupportedVersion,
    TopicAlreadyExists,
    InvalidPartitions,
    InvalidReplicationFactor,
    InvalidReplicaAssignment,
    InvalidConfig,
    NotController,
    InvalidRequest,
    UnsupportedForMessageFormat,
    PolicyViolation,
    OutOfOrderSequenceNumber,
    DuplicateSequenceNumber,
    InvalidProducerEpoch,
    InvalidTxnState,
    InvalidProducerIdMapping,
    InvalidTransactionTimeout,
    ConcurrentTransactions,
    TransactionCoordinatorFenced,
    TransactionalIdAuthorizationFailed,
    SecurityDisabled,
    OperationNotAttempted,
    KafkaStorageError,
    LogDirNotFound,
    SaslAuthenticationFailed,
    UnknownProducerId,
    ReassignmentInProgress,
    DelegationTokenAuthDisabled,
    DelegationTokenNotFound,
    DelegationTokenOwnerMismatch,
    DelegationTokenRequestNotAllowed,
    DelegationTokenAuthorizationFailed,
    DelegationTokenExpired,
    InvalidPrincipalType,
    NonEmptyGroup,
    GroupIdNotFound,
    FetchSessionIdNotFound,
    InvalidFetchSessionEpoch,
    ListenerNotFound,
    TopicDeletionDisabled,
    FencedLeaderEpoch,
    UnknownLeaderEpoch,
    UnsupportedCompressionType,
    StaleBrokerEpoch,
    OffsetNotAvailable,
    MemberIdRequired,
    PreferredLeaderNotAvailable,
    GroupMaxSizeReached,
    FencedInstanceId,
    EligibleLeadersNotAvailable,
    ElectionNotNeeded,
    NoReassignmentInProgress,
    GroupSubscribedToTopic,
    InvalidRecord,
    UnstableOffsetCommit,
    ThrottlingQuotaExceeded,
    ProducerFenced,
    ResourceNotFound,
    DuplicateResource,
    UnacceptableCredential,
    InconsistentVoterSet,
    InvalidUpdateVersion,
    FeatureUpdateFailed,
    PrincipalDeserializationFailure,
    SnapshotNotFound,
    PositionOutOfRange,
    UnknownTopicId,
    DuplicateBrokerRegistration,
    BrokerIdNotRegistered,
    InconsistentTopicId,
    InconsistentClusterId,
    TransactionalIdNotFound,
    Unknown(i16),
}

impl Error {
    /// Coversion from [`i16`].
    ///
    /// This cannot be a `From` trait due to <https://github.com/rust-lang/rfcs/issues/1402>.
    pub fn new(code: i16) -> Option<Self> {
        match code {
            -1 => Some(Self::UnknownServerError),
            0 => None,
            1 => Some(Self::OffsetOutOfRange),
            2 => Some(Self::CorruptMessage),
            3 => Some(Self::UnknownTopicOrPartition),
            4 => Some(Self::InvalidFetchSize),
            5 => Some(Self::LeaderNotAvailable),
            6 => Some(Self::NotLeaderOrFollower),
            7 => Some(Self::RequestTimedOut),
            8 => Some(Self::BrokerNotAvailable),
            9 => Some(Self::ReplicaNotAvailable),
            10 => Some(Self::MessageTooLarge),
            11 => Some(Self::StaleControllerEpoch),
            12 => Some(Self::OffsetMetadataTooLarge),
            13 => Some(Self::NetworkException),
            14 => Some(Self::CoordinatorLoadInProgress),
            15 => Some(Self::CoordinatorNotAvailable),
            16 => Some(Self::NotCoordinator),
            17 => Some(Self::InvalidTopicException),
            18 => Some(Self::RecordListTooLarge),
            19 => Some(Self::NotEnoughReplicas),
            20 => Some(Self::NotEnoughReplicasAfterAppend),
            21 => Some(Self::InvalidRequiredAcks),
            22 => Some(Self::IllegalGeneration),
            23 => Some(Self::InconsistentGroupProtocol),
            24 => Some(Self::InvalidGroupId),
            25 => Some(Self::UnknownMemberId),
            26 => Some(Self::InvalidSessionTimeout),
            27 => Some(Self::RebalanceInProgress),
            28 => Some(Self::InvalidCommitOffsetSize),
            29 => Some(Self::TopicAuthorizationFailed),
            30 => Some(Self::GroupAuthorizationFailed),
            31 => Some(Self::ClusterAuthorizationFailed),
            32 => Some(Self::InvalidTimestamp),
            33 => Some(Self::UnsupportedSaslMechanism),
            34 => Some(Self::IllegalSaslState),
            35 => Some(Self::UnsupportedVersion),
            36 => Some(Self::TopicAlreadyExists),
            37 => Some(Self::InvalidPartitions),
            38 => Some(Self::InvalidReplicationFactor),
            39 => Some(Self::InvalidReplicaAssignment),
            40 => Some(Self::InvalidConfig),
            41 => Some(Self::NotController),
            42 => Some(Self::InvalidRequest),
            43 => Some(Self::UnsupportedForMessageFormat),
            44 => Some(Self::PolicyViolation),
            45 => Some(Self::OutOfOrderSequenceNumber),
            46 => Some(Self::DuplicateSequenceNumber),
            47 => Some(Self::InvalidProducerEpoch),
            48 => Some(Self::InvalidTxnState),
            49 => Some(Self::InvalidProducerIdMapping),
            50 => Some(Self::InvalidTransactionTimeout),
            51 => Some(Self::ConcurrentTransactions),
            52 => Some(Self::TransactionCoordinatorFenced),
            53 => Some(Self::TransactionalIdAuthorizationFailed),
            54 => Some(Self::SecurityDisabled),
            55 => Some(Self::OperationNotAttempted),
            56 => Some(Self::KafkaStorageError),
            57 => Some(Self::LogDirNotFound),
            58 => Some(Self::SaslAuthenticationFailed),
            59 => Some(Self::UnknownProducerId),
            60 => Some(Self::ReassignmentInProgress),
            61 => Some(Self::DelegationTokenAuthDisabled),
            62 => Some(Self::DelegationTokenNotFound),
            63 => Some(Self::DelegationTokenOwnerMismatch),
            64 => Some(Self::DelegationTokenRequestNotAllowed),
            65 => Some(Self::DelegationTokenAuthorizationFailed),
            66 => Some(Self::DelegationTokenExpired),
            67 => Some(Self::InvalidPrincipalType),
            68 => Some(Self::NonEmptyGroup),
            69 => Some(Self::GroupIdNotFound),
            70 => Some(Self::FetchSessionIdNotFound),
            71 => Some(Self::InvalidFetchSessionEpoch),
            72 => Some(Self::ListenerNotFound),
            73 => Some(Self::TopicDeletionDisabled),
            74 => Some(Self::FencedLeaderEpoch),
            75 => Some(Self::UnknownLeaderEpoch),
            76 => Some(Self::UnsupportedCompressionType),
            77 => Some(Self::StaleBrokerEpoch),
            78 => Some(Self::OffsetNotAvailable),
            79 => Some(Self::MemberIdRequired),
            80 => Some(Self::PreferredLeaderNotAvailable),
            81 => Some(Self::GroupMaxSizeReached),
            82 => Some(Self::FencedInstanceId),
            83 => Some(Self::EligibleLeadersNotAvailable),
            84 => Some(Self::ElectionNotNeeded),
            85 => Some(Self::NoReassignmentInProgress),
            86 => Some(Self::GroupSubscribedToTopic),
            87 => Some(Self::InvalidRecord),
            88 => Some(Self::UnstableOffsetCommit),
            89 => Some(Self::ThrottlingQuotaExceeded),
            90 => Some(Self::ProducerFenced),
            91 => Some(Self::ResourceNotFound),
            92 => Some(Self::DuplicateResource),
            93 => Some(Self::UnacceptableCredential),
            94 => Some(Self::InconsistentVoterSet),
            95 => Some(Self::InvalidUpdateVersion),
            96 => Some(Self::FeatureUpdateFailed),
            97 => Some(Self::PrincipalDeserializationFailure),
            98 => Some(Self::SnapshotNotFound),
            99 => Some(Self::PositionOutOfRange),
            100 => Some(Self::UnknownTopicId),
            101 => Some(Self::DuplicateBrokerRegistration),
            102 => Some(Self::BrokerIdNotRegistered),
            103 => Some(Self::InconsistentTopicId),
            104 => Some(Self::InconsistentClusterId),
            105 => Some(Self::TransactionalIdNotFound),
            _ => Some(Self::Unknown(code)),
        }
    }
}

impl From<Option<Error>> for Int16 {
    fn from(error: Option<Error>) -> Self {
        let error = match error {
            Some(error) => error,
            None => return Self(0),
        };

        match error {
            Error::UnknownServerError => Self(-1),
            Error::OffsetOutOfRange => Self(1),
            Error::CorruptMessage => Self(2),
            Error::UnknownTopicOrPartition => Self(3),
            Error::InvalidFetchSize => Self(4),
            Error::LeaderNotAvailable => Self(5),
            Error::NotLeaderOrFollower => Self(6),
            Error::RequestTimedOut => Self(7),
            Error::BrokerNotAvailable => Self(8),
            Error::ReplicaNotAvailable => Self(9),
            Error::MessageTooLarge => Self(10),
            Error::StaleControllerEpoch => Self(11),
            Error::OffsetMetadataTooLarge => Self(12),
            Error::NetworkException => Self(13),
            Error::CoordinatorLoadInProgress => Self(14),
            Error::CoordinatorNotAvailable => Self(15),
            Error::NotCoordinator => Self(16),
            Error::InvalidTopicException => Self(17),
            Error::RecordListTooLarge => Self(18),
            Error::NotEnoughReplicas => Self(19),
            Error::NotEnoughReplicasAfterAppend => Self(20),
            Error::InvalidRequiredAcks => Self(21),
            Error::IllegalGeneration => Self(22),
            Error::InconsistentGroupProtocol => Self(23),
            Error::InvalidGroupId => Self(24),
            Error::UnknownMemberId => Self(25),
            Error::InvalidSessionTimeout => Self(26),
            Error::RebalanceInProgress => Self(27),
            Error::InvalidCommitOffsetSize => Self(28),
            Error::TopicAuthorizationFailed => Self(29),
            Error::GroupAuthorizationFailed => Self(30),
            Error::ClusterAuthorizationFailed => Self(31),
            Error::InvalidTimestamp => Self(32),
            Error::UnsupportedSaslMechanism => Self(33),
            Error::IllegalSaslState => Self(34),
            Error::UnsupportedVersion => Self(35),
            Error::TopicAlreadyExists => Self(36),
            Error::InvalidPartitions => Self(37),
            Error::InvalidReplicationFactor => Self(38),
            Error::InvalidReplicaAssignment => Self(39),
            Error::InvalidConfig => Self(40),
            Error::NotController => Self(41),
            Error::InvalidRequest => Self(42),
            Error::UnsupportedForMessageFormat => Self(43),
            Error::PolicyViolation => Self(44),
            Error::OutOfOrderSequenceNumber => Self(45),
            Error::DuplicateSequenceNumber => Self(46),
            Error::InvalidProducerEpoch => Self(47),
            Error::InvalidTxnState => Self(48),
            Error::InvalidProducerIdMapping => Self(49),
            Error::InvalidTransactionTimeout => Self(50),
            Error::ConcurrentTransactions => Self(51),
            Error::TransactionCoordinatorFenced => Self(52),
            Error::TransactionalIdAuthorizationFailed => Self(53),
            Error::SecurityDisabled => Self(54),
            Error::OperationNotAttempted => Self(55),
            Error::KafkaStorageError => Self(56),
            Error::LogDirNotFound => Self(57),
            Error::SaslAuthenticationFailed => Self(58),
            Error::UnknownProducerId => Self(59),
            Error::ReassignmentInProgress => Self(60),
            Error::DelegationTokenAuthDisabled => Self(61),
            Error::DelegationTokenNotFound => Self(62),
            Error::DelegationTokenOwnerMismatch => Self(63),
            Error::DelegationTokenRequestNotAllowed => Self(64),
            Error::DelegationTokenAuthorizationFailed => Self(65),
            Error::DelegationTokenExpired => Self(66),
            Error::InvalidPrincipalType => Self(67),
            Error::NonEmptyGroup => Self(68),
            Error::GroupIdNotFound => Self(69),
            Error::FetchSessionIdNotFound => Self(70),
            Error::InvalidFetchSessionEpoch => Self(71),
            Error::ListenerNotFound => Self(72),
            Error::TopicDeletionDisabled => Self(73),
            Error::FencedLeaderEpoch => Self(74),
            Error::UnknownLeaderEpoch => Self(75),
            Error::UnsupportedCompressionType => Self(76),
            Error::StaleBrokerEpoch => Self(77),
            Error::OffsetNotAvailable => Self(78),
            Error::MemberIdRequired => Self(79),
            Error::PreferredLeaderNotAvailable => Self(80),
            Error::GroupMaxSizeReached => Self(81),
            Error::FencedInstanceId => Self(82),
            Error::EligibleLeadersNotAvailable => Self(83),
            Error::ElectionNotNeeded => Self(84),
            Error::NoReassignmentInProgress => Self(85),
            Error::GroupSubscribedToTopic => Self(86),
            Error::InvalidRecord => Self(87),
            Error::UnstableOffsetCommit => Self(88),
            Error::ThrottlingQuotaExceeded => Self(89),
            Error::ProducerFenced => Self(90),
            Error::ResourceNotFound => Self(91),
            Error::DuplicateResource => Self(92),
            Error::UnacceptableCredential => Self(93),
            Error::InconsistentVoterSet => Self(94),
            Error::InvalidUpdateVersion => Self(95),
            Error::FeatureUpdateFailed => Self(96),
            Error::PrincipalDeserializationFailure => Self(97),
            Error::SnapshotNotFound => Self(98),
            Error::PositionOutOfRange => Self(99),
            Error::UnknownTopicId => Self(100),
            Error::DuplicateBrokerRegistration => Self(101),
            Error::BrokerIdNotRegistered => Self(102),
            Error::InconsistentTopicId => Self(103),
            Error::InconsistentClusterId => Self(104),
            Error::TransactionalIdNotFound => Self(105),
            Error::Unknown(code) => Self(code),
        }
    }
}

impl std::fmt::Display for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{:?}", self)
    }
}

impl std::error::Error for Error {}
