//
// SPDX-License-Identifier: GPL-3.0-or-later
//
// Copyright (C) 2021 Shun Sakai
//

use std::path::Path;

use crate::value::{Checksum, HashAlgorithm};

impl Checksum {
    /// Compute message digest for the specified hash algorithm.
    pub fn digest<P: AsRef<Path>, T: AsRef<[u8]>>(algorithm: HashAlgorithm, input: (P, T)) -> Self {
        let digest = match algorithm {
            HashAlgorithm::Blake2b => {
                use blake2::{Blake2b, Digest};

                Blake2b::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Blake2s => {
                use blake2::{Blake2s, Digest};

                Blake2s::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Blake3 => blake3::hash(input.1.as_ref()).as_bytes().to_vec(),
            HashAlgorithm::Gost => {
                use gost94::{Digest, Gost94Test};

                Gost94Test::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::GostCryptoPro => {
                use gost94::{Digest, Gost94CryptoPro};

                Gost94CryptoPro::digest(input.1.as_ref())
                    .as_slice()
                    .to_vec()
            }
            HashAlgorithm::Groestl224 => {
                use groestl::{Digest, Groestl224};

                Groestl224::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Groestl256 => {
                use groestl::{Digest, Groestl256};

                Groestl256::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Groestl384 => {
                use groestl::{Digest, Groestl384};

                Groestl384::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Groestl512 => {
                use groestl::{Digest, Groestl512};

                Groestl512::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Keccak224 => {
                use sha3::{Digest, Keccak224};

                Keccak224::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Keccak256 => {
                use sha3::{Digest, Keccak256};

                Keccak256::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Keccak384 => {
                use sha3::{Digest, Keccak384};

                Keccak384::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Keccak512 => {
                use sha3::{Digest, Keccak512};

                Keccak512::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Md2 => {
                use md2::{Digest, Md2};

                Md2::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Md4 => {
                use md4::{Digest, Md4};

                Md4::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Md5 => {
                use md5::{Digest, Md5};

                Md5::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Ripemd160 => {
                use ripemd160::{Digest, Ripemd160};

                Ripemd160::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Ripemd256 => {
                use ripemd256::{Digest, Ripemd256};

                Ripemd256::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Ripemd320 => {
                use ripemd320::{Digest, Ripemd320};

                Ripemd320::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha1 => {
                use sha1::{Digest, Sha1};

                Sha1::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha224 => {
                use sha2::{Digest, Sha224};

                Sha224::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha256 => {
                use sha2::{Digest, Sha256};

                Sha256::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha384 => {
                use sha2::{Digest, Sha384};

                Sha384::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha512 => {
                use sha2::{Digest, Sha512};

                Sha512::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha3_224 => {
                use sha3::{Digest, Sha3_224};

                Sha3_224::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha3_256 => {
                use sha3::{Digest, Sha3_256};

                Sha3_256::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha3_384 => {
                use sha3::{Digest, Sha3_384};

                Sha3_384::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sha3_512 => {
                use sha3::{Digest, Sha3_512};

                Sha3_512::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Shabal192 => {
                use shabal::{Digest, Shabal192};

                Shabal192::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Shabal224 => {
                use shabal::{Digest, Shabal224};

                Shabal224::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Shabal256 => {
                use shabal::{Digest, Shabal256};

                Shabal256::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Shabal384 => {
                use shabal::{Digest, Shabal384};

                Shabal384::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Shabal512 => {
                use shabal::{Digest, Shabal512};

                Shabal512::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Sm3 => {
                use sm3::{Digest, Sm3};

                Sm3::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Streebog256 => {
                use streebog::{Digest, Streebog256};

                Streebog256::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Streebog512 => {
                use streebog::{Digest, Streebog512};

                Streebog512::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Tiger => {
                use tiger::{digest::Digest, Tiger};

                Tiger::digest(input.1.as_ref()).as_slice().to_vec()
            }
            HashAlgorithm::Whirlpool => {
                use whirlpool::{Digest, Whirlpool};

                Whirlpool::digest(input.1.as_ref()).as_slice().to_vec()
            }
        };

        Checksum {
            algorithm: Some(algorithm),
            file: input.0.as_ref().to_path_buf(),
            digest,
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn verify_blake2() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Blake2b, ("-", b"Hello, world!")).digest,
            [
                0xa2, 0x76, 0x4d, 0x13, 0x3a, 0x16, 0x81, 0x6b, 0x58, 0x47, 0xa7, 0x37, 0xa7, 0x86,
                0xf2, 0xec, 0xe4, 0xc1, 0x48, 0x9, 0x5c, 0x5f, 0xaa, 0x73, 0xe2, 0x4b, 0x4c, 0xc5,
                0xd6, 0x66, 0xc3, 0xe4, 0x5e, 0xc2, 0x71, 0x50, 0x4e, 0x14, 0xdc, 0x61, 0x27, 0xdd,
                0xfc, 0xe4, 0xe1, 0x44, 0xfb, 0x23, 0xb9, 0x1a, 0x6f, 0x7b, 0x4, 0xb5, 0x3d, 0x69,
                0x55, 0x2, 0x29, 0x7, 0x22, 0x95, 0x3b, 0xf
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Blake2s, ("-", b"Hello, world!")).digest,
            [
                0x30, 0xd8, 0x77, 0x7f, 0xe, 0x17, 0x85, 0x82, 0xec, 0x8c, 0xd2, 0xfc, 0xdc, 0x18,
                0xaf, 0x57, 0xc8, 0x28, 0xee, 0x2f, 0x89, 0xe9, 0x78, 0xdf, 0x52, 0xc8, 0xe7, 0xaf,
                0x7, 0x8b, 0xd5, 0xcf
            ]
        );
    }

    #[test]
    fn verify_blake3() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Blake3, ("-", b"Hello, world!")).digest,
            [
                0xed, 0xe5, 0xc0, 0xb1, 0xf, 0x2e, 0xc4, 0x97, 0x9c, 0x69, 0xb5, 0x2f, 0x61, 0xe4,
                0x2f, 0xf5, 0xb4, 0x13, 0x51, 0x9c, 0xe0, 0x9b, 0xe0, 0xf1, 0x4d, 0x9, 0x8d, 0xcf,
                0xe5, 0xf6, 0xf9, 0x8d
            ]
        );
    }

    #[test]
    fn verify_gost() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Gost, ("-", b"Hello, world!")).digest,
            [
                0x71, 0x1e, 0x0, 0xe0, 0x34, 0xa9, 0x25, 0x47, 0x65, 0xf6, 0x27, 0xb, 0xd0, 0x2b,
                0x6b, 0xad, 0xf9, 0xdf, 0xe3, 0x80, 0xa1, 0x65, 0x93, 0xef, 0xf6, 0xe1, 0xef, 0x1e,
                0xec, 0x7c, 0xa0, 0x23
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::GostCryptoPro, ("-", b"Hello, world!")).digest,
            [
                0xc0, 0x3, 0xab, 0xf7, 0xee, 0x48, 0xc4, 0x2f, 0xe2, 0x3c, 0xad, 0x86, 0xd5, 0x6d,
                0x2c, 0x98, 0x24, 0x61, 0xf9, 0x4d, 0x46, 0xb1, 0x9, 0xa9, 0xf6, 0xb2, 0xe9, 0x60,
                0xf5, 0x83, 0xcf, 0x52
            ]
        );
    }

    #[test]
    fn verify_groestl() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Groestl224, ("-", b"Hello, world!")).digest,
            [
                0xc6, 0xf1, 0x65, 0x83, 0xeb, 0xfb, 0x25, 0x44, 0x96, 0x9f, 0x67, 0x3d, 0x1f, 0xb4,
                0x3d, 0x73, 0xa3, 0xa5, 0x1c, 0xd6, 0x92, 0x7c, 0xdc, 0x1b, 0x7f, 0xf5, 0xe2, 0xa
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Groestl256, ("-", b"Hello, world!")).digest,
            [
                0x63, 0xe4, 0xab, 0x20, 0x44, 0xe3, 0x8c, 0x1f, 0xb1, 0x72, 0x53, 0x13, 0xf2, 0x22,
                0x9e, 0x3, 0x89, 0x26, 0xaf, 0x83, 0x9c, 0x86, 0xea, 0xf9, 0x65, 0x53, 0x2, 0x7d,
                0x2c, 0x85, 0x1e, 0x18
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Groestl384, ("-", b"Hello, world!")).digest,
            [
                0xfc, 0x49, 0xed, 0xd6, 0xb6, 0x1c, 0x56, 0x30, 0xc6, 0x11, 0x1e, 0x51, 0xb7, 0xb7,
                0x21, 0xff, 0x18, 0x45, 0x4e, 0x45, 0x1f, 0x82, 0x94, 0x98, 0xcc, 0xd, 0x76, 0x1,
                0x8c, 0x11, 0xf9, 0xf1, 0x38, 0x36, 0x54, 0x5f, 0x5d, 0x61, 0xac, 0x32, 0x9, 0xa2,
                0xa9, 0xfb, 0x2b, 0x5c, 0xdc, 0xfd
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Groestl512, ("-", b"Hello, world!")).digest,
            [
                0xb6, 0x6, 0x58, 0xe7, 0x23, 0xa8, 0xeb, 0x17, 0x43, 0x82, 0x3a, 0x80, 0x2, 0x17,
                0x54, 0x86, 0xbc, 0x24, 0x22, 0x3b, 0xa3, 0xdc, 0x6d, 0x8c, 0xb4, 0x35, 0xa9, 0x48,
                0xf6, 0xd2, 0xb9, 0x74, 0x4a, 0xc9, 0xe3, 0x7, 0xe1, 0xd3, 0x80, 0x21, 0xea, 0x18,
                0xc4, 0xd5, 0x36, 0xd2, 0x8f, 0xc2, 0x34, 0x91, 0xd7, 0x77, 0x1a, 0x5a, 0x5b, 0xd,
                0x2, 0xff, 0xad, 0x9a, 0x7, 0x3d, 0xcc, 0x28
            ]
        );
    }

    #[test]
    fn verify_keccak() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Keccak224, ("-", b"Hello, world!")).digest,
            [
                0xf8, 0x9e, 0x15, 0x34, 0x7f, 0xc7, 0x11, 0xf2, 0x5f, 0xc6, 0x29, 0xf4, 0xba, 0x60,
                0xe3, 0x32, 0x66, 0x43, 0xdc, 0x1d, 0xaf, 0x5a, 0xe9, 0xc0, 0x4e, 0x86, 0x96, 0x1d
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Keccak256, ("-", b"Hello, world!")).digest,
            [
                0xb6, 0xe1, 0x6d, 0x27, 0xac, 0x5a, 0xb4, 0x27, 0xa7, 0xf6, 0x89, 0x0, 0xac, 0x55,
                0x59, 0xce, 0x27, 0x2d, 0xc6, 0xc3, 0x7c, 0x82, 0xb3, 0xe0, 0x52, 0x24, 0x6c, 0x82,
                0x24, 0x4c, 0x50, 0xe4
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Keccak384, ("-", b"Hello, world!")).digest,
            [
                0x93, 0x9e, 0x56, 0xd1, 0xf6, 0x78, 0xb0, 0xb2, 0x1f, 0x5c, 0x17, 0x6a, 0xc1, 0xa5,
                0xfe, 0xd3, 0x47, 0xa3, 0x5c, 0x68, 0x8c, 0xf6, 0x4b, 0xd9, 0x97, 0xbc, 0x57, 0x11,
                0x3b, 0x6b, 0xa6, 0x24, 0x51, 0x49, 0x15, 0x76, 0x65, 0xb7, 0xdd, 0x23, 0x35, 0x82,
                0x28, 0xdc, 0xda, 0x58, 0x3, 0xde
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Keccak512, ("-", b"Hello, world!")).digest,
            [
                0x10, 0x1f, 0x35, 0x3a, 0x47, 0x27, 0xcc, 0x94, 0xef, 0x81, 0x61, 0x3b, 0xb3, 0x8a,
                0x80, 0x7e, 0xbc, 0x88, 0x8e, 0x20, 0x61, 0xba, 0xa4, 0xf8, 0x45, 0xc8, 0x4c, 0xd3,
                0xc3, 0x17, 0xf3, 0x43, 0xf, 0xda, 0x3d, 0xbe, 0xb4, 0x40, 0x10, 0x84, 0x4b, 0x35,
                0xbc, 0xcc, 0x8e, 0x19, 0x0, 0x61, 0xd0, 0x5b, 0x4d, 0x0, 0x2c, 0x70, 0x96, 0x15,
                0x27, 0x5a, 0x44, 0xe1, 0x8e, 0x49, 0x4f, 0xc
            ]
        );
    }

    #[test]
    fn verify_md2() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Md2, ("-", b"Hello, world!")).digest,
            [
                0x8c, 0xca, 0xe, 0x96, 0x5e, 0xdd, 0xe, 0x22, 0x3b, 0x74, 0x4f, 0x9c, 0xed, 0xf8,
                0xe1, 0x41
            ]
        );
    }

    #[test]
    fn verify_md4() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Md4, ("-", b"Hello, world!")).digest,
            [
                0xa, 0xbe, 0x9e, 0xe1, 0xf3, 0x76, 0xca, 0xa1, 0xbc, 0xec, 0xad, 0x90, 0x42, 0xf1,
                0x6e, 0x73
            ]
        );
    }

    #[test]
    fn verify_md5() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Md5, ("-", b"Hello, world!")).digest,
            [
                0x6c, 0xd3, 0x55, 0x6d, 0xeb, 0xd, 0xa5, 0x4b, 0xca, 0x6, 0xb, 0x4c, 0x39, 0x47,
                0x98, 0x39
            ]
        );
    }

    #[test]
    fn verify_ripemd() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Ripemd160, ("-", b"Hello, world!")).digest,
            [
                0x58, 0x26, 0x2d, 0x1f, 0xbd, 0xbe, 0x45, 0x30, 0xd8, 0x86, 0x5d, 0x35, 0x18, 0xc6,
                0xd6, 0xe4, 0x10, 0x2, 0x61, 0xf
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Ripemd256, ("-", b"Hello, world!")).digest,
            [
                0x41, 0x21, 0xb1, 0xd1, 0xe6, 0x8b, 0xe2, 0xc6, 0x27, 0x19, 0xef, 0xbd, 0xc4, 0x32,
                0x19, 0x57, 0x7, 0x4a, 0x9f, 0xd3, 0xf5, 0x97, 0xcd, 0xa5, 0xc9, 0x2, 0x35, 0xa6,
                0xa8, 0x50, 0x61, 0xe5
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Ripemd320, ("-", b"Hello, world!")).digest,
            [
                0x38, 0xe0, 0x63, 0x6b, 0x7e, 0xfa, 0x3c, 0x6c, 0x3c, 0xce, 0x53, 0xa3, 0x34, 0xf4,
                0xff, 0x12, 0xcf, 0xee, 0x2a, 0x97, 0x4, 0xcd, 0xf9, 0xc2, 0xe7, 0xd0, 0xfe, 0x3,
                0x99, 0xcf, 0x6e, 0xe6, 0x6a, 0x71, 0xba, 0xbb, 0x49, 0xf5, 0x87, 0xd
            ]
        );
    }

    #[test]
    fn verify_sha1() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha1, ("-", b"Hello, world!")).digest,
            [
                0x94, 0x3a, 0x70, 0x2d, 0x6, 0xf3, 0x45, 0x99, 0xae, 0xe1, 0xf8, 0xda, 0x8e, 0xf9,
                0xf7, 0x29, 0x60, 0x31, 0xd6, 0x99
            ]
        );
    }

    #[test]
    fn verify_sha2() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha224, ("-", b"Hello, world!")).digest,
            [
                0x85, 0x52, 0xd8, 0xb7, 0xa7, 0xdc, 0x54, 0x76, 0xcb, 0x9e, 0x25, 0xde, 0xe6, 0x9a,
                0x80, 0x91, 0x29, 0x7, 0x64, 0xb7, 0xf2, 0xa6, 0x4f, 0xe6, 0xe7, 0x8e, 0x95, 0x68
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha256, ("-", b"Hello, world!")).digest,
            [
                0x31, 0x5f, 0x5b, 0xdb, 0x76, 0xd0, 0x78, 0xc4, 0x3b, 0x8a, 0xc0, 0x6, 0x4e, 0x4a,
                0x1, 0x64, 0x61, 0x2b, 0x1f, 0xce, 0x77, 0xc8, 0x69, 0x34, 0x5b, 0xfc, 0x94, 0xc7,
                0x58, 0x94, 0xed, 0xd3
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha384, ("-", b"Hello, world!")).digest,
            [
                0x55, 0xbc, 0x55, 0x6b, 0xd, 0x2f, 0xe0, 0xfc, 0xe5, 0x82, 0xba, 0x5f, 0xe0, 0x7b,
                0xaa, 0xff, 0xf0, 0x35, 0x65, 0x36, 0x38, 0xc7, 0xac, 0xd, 0x54, 0x94, 0xc2, 0xa6,
                0x4c, 0xb, 0xea, 0x1c, 0xc5, 0x73, 0x31, 0xc7, 0xc1, 0x2a, 0x45, 0xcd, 0xbc, 0xa7,
                0xf4, 0xc3, 0x4a, 0x8, 0x9e, 0xeb
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha512, ("-", b"Hello, world!")).digest,
            [
                0xc1, 0x52, 0x7c, 0xd8, 0x93, 0xc1, 0x24, 0x77, 0x3d, 0x81, 0x19, 0x11, 0x97, 0xc,
                0x8f, 0xe6, 0xe8, 0x57, 0xd6, 0xdf, 0x5d, 0xc9, 0x22, 0x6b, 0xd8, 0xa1, 0x60, 0x61,
                0x4c, 0xc, 0xd9, 0x63, 0xa4, 0xdd, 0xea, 0x2b, 0x94, 0xbb, 0x7d, 0x36, 0x2, 0x1e,
                0xf9, 0xd8, 0x65, 0xd5, 0xce, 0xa2, 0x94, 0xa8, 0x2d, 0xd4, 0x9a, 0xb, 0xb2, 0x69,
                0xf5, 0x1f, 0x6e, 0x7a, 0x57, 0xf7, 0x94, 0x21
            ]
        );
    }

    #[test]
    fn verify_sha3() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha3_224, ("-", b"Hello, world!")).digest,
            [
                0x6a, 0x33, 0xe2, 0x2f, 0x20, 0xf1, 0x66, 0x42, 0x69, 0x7e, 0x8b, 0xd5, 0x49, 0xff,
                0x7b, 0x75, 0x92, 0x52, 0xad, 0x56, 0xc0, 0x5a, 0x1b, 0xa, 0xcc, 0x31, 0xdc, 0x69
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha3_256, ("-", b"Hello, world!")).digest,
            [
                0xf3, 0x45, 0xa2, 0x19, 0xda, 0x0, 0x5e, 0xbe, 0x9c, 0x1a, 0x1e, 0xaa, 0xd9, 0x7b,
                0xbf, 0x38, 0xa1, 0xc, 0x84, 0x73, 0xe4, 0x1d, 0xa, 0xf7, 0xfb, 0x61, 0x7c, 0xaa,
                0xc, 0x6a, 0xa7, 0x22
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha3_384, ("-", b"Hello, world!")).digest,
            [
                0x6b, 0xa9, 0xea, 0x26, 0x89, 0x65, 0x91, 0x6f, 0x59, 0x37, 0x22, 0x8d, 0xde, 0x67,
                0x8c, 0x20, 0x2f, 0x9f, 0xe7, 0x56, 0xa8, 0x7d, 0x8b, 0x1b, 0x73, 0x62, 0x86, 0x95,
                0x83, 0xa4, 0x59, 0x1, 0xfd, 0x1a, 0x27, 0x28, 0x9d, 0x72, 0xfc, 0xe, 0x3f, 0xf4,
                0x8b, 0x1b, 0x78, 0x82, 0x7d, 0x3a
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sha3_512, ("-", b"Hello, world!")).digest,
            [
                0x8e, 0x47, 0xf1, 0x18, 0x5f, 0xfd, 0x1, 0x4d, 0x23, 0x8f, 0xab, 0xd0, 0x2a, 0x1a,
                0x32, 0xde, 0xfe, 0x69, 0x8c, 0xbf, 0x38, 0xc0, 0x37, 0xa9, 0xe, 0x3c, 0xa, 0xa,
                0x32, 0x37, 0xf, 0xb5, 0x2c, 0xbd, 0x64, 0x12, 0x50, 0x50, 0x85, 0x2, 0x29, 0x5f,
                0xca, 0xbc, 0xbf, 0x67, 0x6c, 0x9, 0x47, 0xb, 0x27, 0x44, 0x38, 0x68, 0xc8, 0xe5,
                0xf7, 0xe, 0x26, 0xdc, 0x33, 0x72, 0x88, 0xaf
            ]
        );
    }

    #[test]
    fn verify_shabal() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Shabal192, ("-", b"Hello, world!")).digest,
            [
                0x55, 0x30, 0xac, 0xe9, 0xc4, 0xf7, 0x25, 0x42, 0xda, 0x20, 0xb, 0x10, 0x9f, 0x2f,
                0x31, 0xac, 0xdf, 0xd0, 0xf5, 0xcb, 0x59, 0x99, 0x17, 0xa6
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Shabal224, ("-", b"Hello, world!")).digest,
            [
                0x64, 0xe0, 0x6c, 0xbe, 0x6, 0xf9, 0x82, 0x27, 0x31, 0xbd, 0xd2, 0xa2, 0xcc, 0x8e,
                0x1, 0x63, 0x72, 0x2, 0xe2, 0xec, 0xd3, 0xef, 0x6b, 0x33, 0x60, 0xb8, 0x73, 0xb5
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Shabal256, ("-", b"Hello, world!")).digest,
            [
                0xe5, 0x8d, 0x8d, 0x76, 0x4a, 0xd4, 0xdb, 0x5e, 0x71, 0x6d, 0xf1, 0x84, 0x2, 0x83,
                0x68, 0x1a, 0x40, 0x10, 0xa7, 0x7d, 0xfe, 0x59, 0xb4, 0x94, 0xfc, 0x7a, 0xc9, 0xfc,
                0x8c, 0x64, 0xaf, 0x76
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Shabal384, ("-", b"Hello, world!")).digest,
            [
                0xbf, 0xae, 0x21, 0xd6, 0xdc, 0xb2, 0x52, 0x24, 0x9f, 0x8d, 0xf3, 0x85, 0xdf, 0xa4,
                0x38, 0x2b, 0xb3, 0x47, 0x48, 0xc8, 0x18, 0x54, 0xa4, 0x2c, 0x8e, 0xd9, 0x47, 0xf5,
                0x7e, 0xe9, 0x10, 0x21, 0x69, 0xf4, 0x43, 0x56, 0xa, 0x72, 0xe5, 0x53, 0xfb, 0x65,
                0xd3, 0xf1, 0x4f, 0xd5, 0x4d, 0x35
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Shabal512, ("-", b"Hello, world!")).digest,
            [
                0x70, 0x48, 0xf0, 0xa5, 0x89, 0x33, 0x9d, 0x2d, 0x26, 0x89, 0x7, 0x1, 0xed, 0x3b,
                0x2d, 0x1e, 0xd7, 0xc8, 0xdd, 0x1a, 0xc3, 0x7f, 0xec, 0x51, 0x7c, 0x7a, 0x8c, 0x39,
                0xd5, 0xd5, 0x15, 0x48, 0xe9, 0x6e, 0xa8, 0xdf, 0xac, 0xeb, 0x5b, 0x99, 0xf9, 0xd1,
                0xdb, 0x3b, 0x18, 0xa7, 0x65, 0x2e, 0x4, 0x12, 0x34, 0x8e, 0xbf, 0xd6, 0x1d, 0x32,
                0xd7, 0x55, 0xd6, 0x9, 0x8b, 0xff, 0x8c, 0xb3
            ]
        );
    }

    #[test]
    fn verify_sm3() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Sm3, ("-", b"Hello, world!")).digest,
            [
                0xe3, 0xbc, 0xa1, 0x1, 0xb4, 0x96, 0x88, 0xc, 0x36, 0x53, 0xda, 0xd8, 0x58, 0x61,
                0xd0, 0xe7, 0x84, 0xb0, 0xa, 0x8c, 0x18, 0xf7, 0x57, 0x44, 0x72, 0xd1, 0x56, 0x6,
                0xe, 0x90, 0x96, 0xbf
            ]
        );
    }

    #[test]
    fn verify_streebog() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Streebog256, ("-", b"Hello, world!")).digest,
            [
                0xcc, 0xb6, 0xfa, 0xe3, 0x55, 0x3c, 0x10, 0x17, 0x15, 0xda, 0x53, 0x53, 0x28, 0xde,
                0x71, 0x8f, 0x6f, 0x6e, 0x41, 0x2d, 0xb8, 0x61, 0x1a, 0x38, 0x2, 0x5c, 0x51, 0xa,
                0xc8, 0xf8, 0x5a, 0xeb
            ]
        );
        assert_eq!(
            Checksum::digest(HashAlgorithm::Streebog512, ("-", b"Hello, world!")).digest,
            [
                0xa8, 0x33, 0x52, 0xd3, 0x5d, 0xc8, 0xf0, 0x7c, 0xa8, 0x4, 0x8e, 0x67, 0x52, 0x41,
                0x5e, 0x5e, 0x99, 0x15, 0x27, 0xe2, 0x94, 0x15, 0xad, 0xe0, 0xea, 0xad, 0x6e, 0x48,
                0xd6, 0x7b, 0xf3, 0x7b, 0x60, 0xdf, 0xd7, 0xbb, 0x44, 0x75, 0xcb, 0xcb, 0xe2, 0x97,
                0xed, 0x1, 0x61, 0x28, 0x39, 0x1c, 0x31, 0x2d, 0xfe, 0x3a, 0x0, 0xe0, 0xa9, 0xbd,
                0xe, 0x49, 0x73, 0x89, 0xc8, 0x88, 0xee, 0xdc
            ]
        );
    }

    #[test]
    fn verify_tiger() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Tiger, ("-", b"Hello, world!")).digest,
            [
                0xb5, 0xe5, 0xdd, 0x73, 0xa5, 0x89, 0x42, 0x36, 0x93, 0x70, 0x84, 0x13, 0x1b, 0xb8,
                0x45, 0x18, 0x9c, 0xdc, 0x54, 0x77, 0x57, 0x9b, 0x9f, 0x36
            ]
        );
    }

    #[test]
    fn verify_whirlpool() {
        assert_eq!(
            Checksum::digest(HashAlgorithm::Whirlpool, ("-", b"Hello, world!")).digest,
            [
                0xa1, 0xa8, 0x70, 0x3b, 0xe5, 0x31, 0x2b, 0x13, 0x9b, 0x42, 0xeb, 0x33, 0x1a, 0xa8,
                0x0, 0xcc, 0xac, 0xa0, 0xc3, 0x4d, 0x58, 0xc6, 0x98, 0x8e, 0x44, 0xf4, 0x54, 0x89,
                0xcf, 0xb1, 0x6b, 0xeb, 0x4b, 0x6b, 0xf0, 0xce, 0x20, 0xbe, 0x1d, 0xb2, 0x2a, 0x10,
                0xb0, 0xe4, 0xbb, 0x68, 0x4, 0x80, 0xa3, 0xd2, 0x42, 0x9e, 0x6c, 0x48, 0x30, 0x85,
                0x45, 0x3c, 0x9, 0x8b, 0x65, 0x85, 0x24, 0x95
            ]
        );
    }
}
