//!
//! Rust Firebird Client
//!
//! Example of insert
//!
//! You need create a database with this table:
//! create table test (col_a int generated by default as identity, col_b float, col_c varchar(10));
//!

#![allow(unused_variables, unused_mut)]

use rsfbclient::{prelude::*, FbError};

const SQL_INSERT: &str = "insert into test (col_b, col_c) values (?, ?)";
const SQL_INSERT_NAMED: &str = "insert into test (col_b, col_c) values (:colb, :colc)";
const SQL_INSERT_PURE: &str = "insert into test (col_b, col_c) values (32, 'Leite')";

#[derive(Clone, IntoParams)]
struct ParamTest {
    colb: f32,
    colc: String,
}

fn main() -> Result<(), FbError> {
    #[cfg(feature = "linking")]
    let mut conn = rsfbclient::builder_native()
        .with_dyn_link()
        .with_remote()
        .host("localhost")
        .db_name("examples.fdb")
        .user("SYSDBA")
        .pass("masterkey")
        .connect()?;

    #[cfg(feature = "dynamic_loading")]
    let mut conn = rsfbclient::builder_native()
        .with_dyn_load("./fbclient.lib")
        .with_remote()
        .host("localhost")
        .db_name("examples.fdb")
        .user("SYSDBA")
        .pass("masterkey")
        .connect()?;

    #[cfg(feature = "pure_rust")]
    let mut conn = rsfbclient::builder_pure_rust()
        .host("localhost")
        .db_name("examples.fdb")
        .user("SYSDBA")
        .pass("masterkey")
        .connect()?;

    let p1 = ParamTest {
        colb: 150.0,
        colc: "Café".to_string(),
    };
    let p2 = ParamTest {
        colb: 132.0,
        colc: "Arroz".to_string(),
    };

    conn.with_transaction(|tr| {
        // First alternative (Recommended) (Prepares if needed and executes automatically)
        tr.execute(SQL_INSERT, (94, "Banana"))?; // with position params
        tr.execute(SQL_INSERT_NAMED, p1.clone())?; // with named params

        // Second alternative
        tr.execute_immediate(SQL_INSERT_PURE)?;

        // Third alternative, with position params
        {
            let mut stmt = tr.prepare(SQL_INSERT, false)?;

            stmt.execute((-39, "test"))?;
            stmt.execute((12, "test 2"))?;
        }
        // Fourth alternative, with named params
        {
            let mut stmt = tr.prepare(SQL_INSERT_NAMED, false)?;

            stmt.execute(p1.clone())?;
            stmt.execute(p2.clone())?;
        }

        Ok(())
    })?;

    // Explicit close is optional
    conn.close()?;

    Ok(())
}
