struct CyclicCheckVisitor {
	path_visitor: PathVisitor,
}

impl CyclicCheckVisitor {
	fn new() -> Self {
		Self {
			path_visitor: PathVisitor::new(),
		}
	}

	fn pre(&mut self, node: usize) {
		if self.path_visitor.has_node(node) {
			panic!("Cyclic dependency");
		}
		self.path_visitor.pre(node);
	}

	fn post(&mut self) {
		self.path_visitor.post();
	}
}

struct PathVisitor {
	path: Vec<usize>,
}

impl PathVisitor {
	fn new() -> Self {
		Self { path: Vec::new() }
	}

	fn has_node(&self, node: usize) -> bool {
		self.path.contains(&node)
	}

	fn pre(&mut self, node: usize) {
		self.path.push(node);
	}

	fn post(&mut self) {
		self.path.pop();
	}
}

struct VisitedNodesVisitor {
	visited: Vec<usize>,
}

impl VisitedNodesVisitor {
	fn new() -> Self {
		Self {
			visited: Vec::new(),
		}
	}

	fn should_enter(&self, node: usize) -> bool {
		!self.visited.contains(&node)
	}

	fn post(&mut self, node: usize) {
		self.visited.push(node);
	}
}

pub(crate) struct ResolutionVisitor {
	cyclic_visitor: CyclicCheckVisitor,
	visited_nodes_visitor: VisitedNodesVisitor,
	resolved: Vec<usize>,
}

impl ResolutionVisitor {
	pub(crate) fn new() -> Self {
		Self {
			cyclic_visitor: CyclicCheckVisitor::new(),
			visited_nodes_visitor: VisitedNodesVisitor::new(),
			resolved: Vec::new(),
		}
	}

	pub(crate) fn should_enter(&self, node: usize) -> bool {
		self.visited_nodes_visitor.should_enter(node)
	}

	pub(crate) fn pre(&mut self, node: usize) {
		self.cyclic_visitor.pre(node);
	}

	pub(crate) fn post(&mut self, node: usize) {
		self.cyclic_visitor.post();
		self.visited_nodes_visitor.post(node);
		self.resolved.push(node);
	}

	pub(crate) fn get_resolved(mut self) -> Vec<usize> {
		self.resolved.reverse();
		self.resolved
	}
}
