use std::marker::PhantomData;

use crate::connect_structs::Connect;

pub(crate) enum Nodes {
	Observed(Box<dyn Node>),
	ObservedAny(Box<dyn Node>),
	Computed(Box<dyn ComputedNode>),
	ComputedAny(Box<dyn ComputedNode>),
	Effected(Box<dyn ComputedNode>),
	EffectedAny(Box<dyn ComputedNode>),
}

pub(crate) trait ComputedNode: Node {
	fn get_dependencies(&self) -> Vec<usize>;
	fn set_dependencies(&mut self, dependencies: Vec<usize>);
	fn recalculate(&mut self, c: &Connect) -> bool;
	fn run_effect(&self);
}

pub(crate) trait Node {
	fn get_derived_clone(&self) -> Vec<usize>;
	fn get_mut_derived(&mut self) -> &mut Vec<usize>;
}

impl Node for Nodes {
	fn get_derived_clone(&self) -> Vec<usize> {
		match &self {
			Nodes::Observed(observed) | Nodes::ObservedAny(observed) => {
				observed.get_derived_clone()
			}
			Nodes::Computed(computed)
			| Nodes::ComputedAny(computed)
			| Nodes::Effected(computed)
			| Nodes::EffectedAny(computed) => computed.get_derived_clone(),
		}
	}

	fn get_mut_derived(&mut self) -> &mut Vec<usize> {
		match self {
			Nodes::Observed(observed) | Nodes::ObservedAny(observed) => {
				observed.get_mut_derived()
			}
			Nodes::Computed(computed)
			| Nodes::ComputedAny(computed)
			| Nodes::Effected(computed)
			| Nodes::EffectedAny(computed) => computed.get_mut_derived(),
		}
	}
}

pub trait NodeValueAccess {
	type Value;
	type CompFun;

	fn get(&self) -> &Self::Value;
	fn set(&mut self, value: Self::Value) -> Self::Value;
	fn equal(&self, value: &Self::Value) -> bool;
}

pub struct NodeValue<T> {
	pub(crate) id: usize,
	pub(crate) connect: *const Connect,
	pub(crate) value: PhantomData<T>,
}

impl<T> NodeValue<T> {
	pub(crate) fn new(id: usize, connect: &Connect) -> Self {
		Self {
			id,
			connect,
			value: PhantomData,
		}
	}
}

impl<T> Copy for NodeValue<T> {}

impl<T> Clone for NodeValue<T> {
	fn clone(&self) -> Self {
		NodeValue {
			id: self.id,
			connect: self.connect,
			value: PhantomData,
		}
	}
}
