use std::marker::PhantomData;

use crate::{
	comparable_value::{ComparableValue, NonComparableValue, Value},
	connect_structs::Connect,
	node_traits::{ComputedNode, Node, NodeValueAccess},
};

pub struct Observed<T, V: Value<T>> {
	value: V,
	derived: Vec<usize>,
	value_type: PhantomData<T>,
}

impl<T: PartialEq> Observed<T, ComparableValue<T>> {
	pub(crate) fn new(value: T) -> Self {
		Self {
			value: ComparableValue(value),
			derived: Vec::new(),
			value_type: PhantomData,
		}
	}
}

impl<T> Observed<T, NonComparableValue<T>> {
	pub(crate) fn new(value: T) -> Self {
		Self {
			value: NonComparableValue(value),
			derived: Vec::new(),
			value_type: PhantomData,
		}
	}
}

pub struct Computed<T, V: Value<T>, F: Fn(&Connect) -> T> {
	comp_fun: F,
	value: V,
	derived: Vec<usize>,
	dependencies: Vec<usize>,
}

impl<T: PartialEq, F: Fn(&Connect) -> T> Computed<T, ComparableValue<T>, F> {
	pub(crate) fn new(comp_fun: F, value: T, dependencies: Vec<usize>) -> Self {
		Self {
			value: ComparableValue(value),
			comp_fun,
			derived: Vec::new(),
			dependencies,
		}
	}
}

impl<T, F: Fn(&Connect) -> T> Computed<T, NonComparableValue<T>, F> {
	pub(crate) fn new(comp_fun: F, value: T, dependencies: Vec<usize>) -> Self {
		Self {
			value: NonComparableValue(value),
			comp_fun,
			derived: Vec::new(),
			dependencies,
		}
	}
}

pub(crate) struct Effected<T, V: Value<T>, F: Fn(&Connect) -> T, E: Fn(&T)> {
	comp_fun: F,
	value: V,
	effect: E,
	derived: Vec<usize>,
	dependencies: Vec<usize>,
}

impl<T: PartialEq, F: Fn(&Connect) -> T, E: Fn(&T)>
	Effected<T, ComparableValue<T>, F, E>
{
	pub(crate) fn new(
		comp_fun: F,
		value: T,
		effect: E,
		dependencies: Vec<usize>,
	) -> Self {
		Self {
			value: ComparableValue(value),
			comp_fun,
			effect,
			derived: Vec::new(),
			dependencies,
		}
	}
}

impl<T, F: Fn(&Connect) -> T, E: Fn(&T)> Effected<T, NonComparableValue<T>, F, E> {
	pub(crate) fn new(
		comp_fun: F,
		value: T,
		effect: E,
		dependencies: Vec<usize>,
	) -> Self {
		Self {
			value: NonComparableValue(value),
			comp_fun,
			effect,
			derived: Vec::new(),
			dependencies,
		}
	}
}

impl<T, V: Value<T>> Node for Observed<T, V> {
	fn get_derived_clone(&self) -> Vec<usize> {
		self.derived.clone()
	}

	fn get_mut_derived(&mut self) -> &mut Vec<usize> {
		&mut self.derived
	}
}

impl<T, V: Value<T>, F: Fn(&Connect) -> T> Node for Computed<T, V, F> {
	fn get_derived_clone(&self) -> Vec<usize> {
		self.derived.clone()
	}

	fn get_mut_derived(&mut self) -> &mut Vec<usize> {
		&mut self.derived
	}
}

impl<T, V: Value<T>, F: Fn(&Connect) -> T> ComputedNode for Computed<T, V, F> {
	fn get_dependencies(&self) -> Vec<usize> {
		self.dependencies.clone()
	}

	fn recalculate(&mut self, c: &Connect) -> bool {
		let value = self.value.set_own((self.comp_fun)(c));
		!self.value.equal(&value)
	}

	fn set_dependencies(&mut self, dependencies: Vec<usize>) {
		self.dependencies = dependencies;
	}

	fn run_effect(&self) {}
}

impl<T, V: Value<T>, F: Fn(&Connect) -> T, E: Fn(&T)> Node for Effected<T, V, F, E> {
	fn get_derived_clone(&self) -> Vec<usize> {
		self.derived.clone()
	}

	fn get_mut_derived(&mut self) -> &mut Vec<usize> {
		&mut self.derived
	}
}

impl<T, V: Value<T>, F: Fn(&Connect) -> T, E: Fn(&T)> ComputedNode
	for Effected<T, V, F, E>
{
	fn get_dependencies(&self) -> Vec<usize> {
		self.dependencies.clone()
	}

	fn recalculate(&mut self, c: &Connect) -> bool {
		let value = self.value.set_own((self.comp_fun)(c));
		!self.value.equal(&value)
	}

	fn set_dependencies(&mut self, dependencies: Vec<usize>) {
		self.dependencies = dependencies;
	}

	fn run_effect(&self) {
		(self.effect)(self.value.get_ref())
	}
}

impl<T, V: Value<T>> NodeValueAccess for Observed<T, V> {
	type Value = T;
	type CompFun = ();

	fn get(&self) -> &T {
		self.value.get_ref()
	}

	fn set(&mut self, value: T) -> T {
		self.value.set_own(value)
	}

	fn equal(&self, value: &T) -> bool {
		self.value.equal(value)
	}
}

impl<T, V: Value<T>, F: Fn(&Connect) -> T> NodeValueAccess for Computed<T, V, F> {
	type Value = T;
	type CompFun = F;

	fn get(&self) -> &T {
		self.value.get_ref()
	}

	fn set(&mut self, value: T) -> T {
		self.value.set_own(value)
	}

	fn equal(&self, _value: &T) -> bool {
		false
	}
}
