use crate::{
	connect_dependencies::ConnectDependencies, connect_structs::Connect,
	node_traits::Node, resolution::ResolutionVisitor,
};

pub(crate) trait ConnectUpdate {
	fn get_update_list(&self, modified_node_ids: &mut Vec<usize>) -> Vec<usize>;
	fn visit_node(&self, resolution_visitor: &mut ResolutionVisitor, node_id: usize);
	fn recalculate_update_list(
		&self,
		modified_node_ids: &mut Vec<usize>,
		update_id_list: Vec<usize>,
	);
}

impl ConnectUpdate for Connect {
	fn get_update_list(&self, modified_node_ids: &mut Vec<usize>) -> Vec<usize> {
		let mut resolution_visitor = ResolutionVisitor::new();
		for node_id in modified_node_ids {
			let node = &self.nodes.borrow()[*node_id];
			for derived_id in node.borrow().get_derived_clone() {
				self.visit_node(&mut resolution_visitor, derived_id);
			}
		}
		resolution_visitor.get_resolved()
	}

	fn visit_node(&self, resolution_visitor: &mut ResolutionVisitor, node_id: usize) {
		if !resolution_visitor.should_enter(node_id) {
			return;
		}
		resolution_visitor.pre(node_id);
		let nodes = self.nodes.borrow();
		let node = nodes[node_id].borrow();
		for derived_id in node.get_derived_clone() {
			self.visit_node(resolution_visitor, derived_id);
		}
		resolution_visitor.post(node_id)
	}

	fn recalculate_update_list(
		&self,
		modified_node_ids: &mut Vec<usize>,
		update_id_list: Vec<usize>,
	) {
		for computed_id in update_id_list {
			if self.is_any_dependency_changed(computed_id, modified_node_ids) {
				self.remove_computed_from_derived_lists(computed_id);
				self.recalculate_value_and_set_dependencies(
					computed_id,
					modified_node_ids,
				);
				self.add_computed_to_derived_lists(computed_id);
			}
		}
	}
}
