use crate::{
	connect_structs::{Connect, CurrentUpdate},
	node_traits::Node,
};

pub(crate) trait NodeUpdate {
	fn start_shared_update(&self);
	fn stop_shared_update(&self) -> Vec<usize>;
	fn get_computed_value<T, F: FnOnce(&Connect) -> T>(
		&self,
		comp_fun: F,
	) -> (T, Vec<usize>);
	fn add_to_current_update_dependencies(&self, id: usize);
	fn add_to_derived(&self, computed_id: usize, dependency_id: usize);
	fn remove_from_derived(&self, computed_id: usize, dependency_id: usize);
}

impl NodeUpdate for Connect {
	fn start_shared_update(&self) {
		*self.current_update.borrow_mut() = Some(CurrentUpdate::new());
	}

	fn stop_shared_update(&self) -> Vec<usize> {
		if let Some(current_update) =
			std::mem::replace(&mut *self.current_update.borrow_mut(), None)
		{
			current_update.dependencies
		} else {
			Vec::new()
		}
	}

	fn get_computed_value<T, F: FnOnce(&Connect) -> T>(
		&self,
		comp_fun: F,
	) -> (T, Vec<usize>) {
		self.start_shared_update();
		let value = comp_fun(self);
		(value, self.stop_shared_update())
	}

	fn add_to_current_update_dependencies(&self, id: usize) {
		if let Some(current_update) = &mut *self.current_update.borrow_mut() {
			if !current_update
				.dependencies
				.iter()
				.any(|dependency| dependency == &id)
			{
				current_update.dependencies.push(id);
			}
		}
	}

	fn add_to_derived(&self, computed_id: usize, dependency_id: usize) {
		let dependency = &self.nodes.borrow()[dependency_id];
		let mut dependency = dependency.borrow_mut();
		let derived = dependency.get_mut_derived();
		if !derived.iter().any(|derived_id| derived_id == &computed_id) {
			derived.push(computed_id);
		}
	}

	fn remove_from_derived(&self, computed_id: usize, dependency_id: usize) {
		let dependency = &self.nodes.borrow()[dependency_id];
		let mut dependency = dependency.borrow_mut();
		let derived = dependency.get_mut_derived();
		if let Some(index) = derived
			.iter()
			.position(|derived_id| derived_id == &computed_id)
		{
			derived.remove(index);
		}
	}
}
