use crate::{
	connect_node_update::NodeUpdate,
	connect_structs::Connect,
	connect_update::ConnectUpdate,
	node_traits::{NodeValue, NodeValueAccess, Nodes},
};

pub(crate) trait ConnectNodeUpdateInitiator {
	fn before_get_node(&self, id: usize);
	fn after_set_node(&self, id: usize);
	fn after_set_nodes(&self, modified_ids: Vec<usize>);
}

impl ConnectNodeUpdateInitiator for Connect {
	fn before_get_node(&self, id: usize) {
		self.add_to_current_update_dependencies(id);
	}

	fn after_set_node(&self, id: usize) {
		match &mut *self.modified_ids.borrow_mut() {
			None => {
				self.after_set_nodes(vec![id]);
			}
			Some(modified_ids) => {
				modified_ids.push(id);
			}
		}
	}

	fn after_set_nodes(&self, mut modified_ids: Vec<usize>) {
		let update_list = self.get_update_list(&mut modified_ids);
		self.recalculate_update_list(&mut modified_ids, update_list);
	}
}

trait ConnectNodeAccess {
	fn get_concrete_from_node<'a, N: ?Sized, R>(&self, node: &N) -> &'a R;
	fn get_mut_concrete_from_node<'a, N: ?Sized, R>(&self, node: &mut N) -> &'a mut R;
}

impl ConnectNodeAccess for Connect {
	fn get_concrete_from_node<'a, N: ?Sized, R>(&self, node: &N) -> &'a R {
		let node_dyn_ptr: *const N = node;
		let node_ptr = node_dyn_ptr.cast::<R>();
		unsafe {
			if let Some(node_ref) = node_ptr.as_ref() {
				node_ref
			} else {
				unreachable!()
			}
		}
	}

	fn get_mut_concrete_from_node<'a, N: ?Sized, R>(&self, node: &mut N) -> &'a mut R {
		let node_dyn_ptr: *mut N = node;
		let node_ptr = node_dyn_ptr.cast::<R>();
		unsafe {
			if let Some(node_mut) = node_ptr.as_mut() {
				node_mut
			} else {
				unreachable!()
			}
		}
	}
}

pub trait ConnectNodeInterface {
	fn get<'a, T: NodeValueAccess + 'a>(&self, node: NodeValue<T>) -> &'a T::Value;
	fn set<T: NodeValueAccess>(&self, node: NodeValue<T>, value: T::Value) -> T::Value;
}

impl ConnectNodeInterface for Connect {
	fn get<'a, T: NodeValueAccess + 'a>(&self, node: NodeValue<T>) -> &'a T::Value {
		if !std::ptr::eq(self, node.connect) {
			panic!("This node does not belong in current Connect graph");
		}
		self.before_get_node(node.id);
		match &*self.nodes.borrow()[node.id].borrow() {
			Nodes::Observed(node) | Nodes::ObservedAny(node) => {
				self.get_concrete_from_node::<_, T>(node.as_ref()).get()
			}
			Nodes::Computed(computed_node) | Nodes::ComputedAny(computed_node) => self
				.get_concrete_from_node::<_, T>(computed_node.as_ref())
				.get(),
			_ => unreachable!(),
		}
	}

	fn set<T: NodeValueAccess>(&self, node: NodeValue<T>, value: T::Value) -> T::Value {
		if !std::ptr::eq(self, node.connect) {
			panic!("This node does not belong in current Connect graph");
		}
		let (is_changed, value) = match &mut *self.nodes.borrow()[node.id].borrow_mut() {
			Nodes::Observed(node) | Nodes::ObservedAny(node) => {
				let observed_mut = self.get_mut_concrete_from_node::<_, T>(node.as_mut());
				(!observed_mut.equal(&value), observed_mut.set(value))
			}
			Nodes::Computed(computed_node) | Nodes::ComputedAny(computed_node) => {
				let computed_mut =
					self.get_mut_concrete_from_node::<_, T>(computed_node.as_mut());
				(!computed_mut.equal(&value), computed_mut.set(value))
			}
			_ => unreachable!(),
		};
		if is_changed {
			self.after_set_node(node.id);
		}
		value
	}
}
