use std::cell::RefCell;

use crate::{
	comparable_value::{ComparableValue, NonComparableValue},
	connect_dependencies::ConnectDependencies,
	connect_node_update::NodeUpdate,
	connect_structs::Connect,
	node_structs::{Computed, Effected, Observed},
	node_traits::{ComputedNode, Node, NodeValue, Nodes},
};

pub trait ConnectInterface {
	fn observed<T: PartialEq + 'static>(
		&self,
		value: T,
	) -> NodeValue<Observed<T, ComparableValue<T>>>;
	fn observed_any<T: 'static>(
		&self,
		value: T,
	) -> NodeValue<Observed<T, NonComparableValue<T>>>;
	fn computed<T: PartialEq + 'static, F: Fn(&Connect) -> T + 'static>(
		&self,
		comp_fun: F,
	) -> NodeValue<Computed<T, ComparableValue<T>, F>>;
	fn computed_any<T: 'static, F: Fn(&Connect) -> T + 'static>(
		&self,
		comp_fun: F,
	) -> NodeValue<Computed<T, NonComparableValue<T>, F>>;
	fn effected<
		T: PartialEq + 'static,
		F: Fn(&Connect) -> T + 'static,
		E: Fn(&T) + 'static,
	>(
		&self,
		comp_fun: F,
		effect: E,
	);
	fn effected_any<T: 'static, F: Fn(&Connect) -> T + 'static, E: Fn(&T) + 'static>(
		&self,
		comp_fun: F,
		effect: E,
	);
}

impl ConnectInterface for Connect {
	fn observed<T: PartialEq + 'static>(
		&self,
		value: T,
	) -> NodeValue<Observed<T, ComparableValue<T>>> {
		let id = self.nodes.borrow().len();
		let node_value = NodeValue::new(id, self);
		let observed = Box::new(Observed::<T, ComparableValue<T>>::new(value));
		let node = RefCell::new(Nodes::Observed(observed as Box<dyn Node>));
		self.nodes.borrow_mut().push(node);
		node_value
	}

	fn observed_any<T: 'static>(
		&self,
		value: T,
	) -> NodeValue<Observed<T, NonComparableValue<T>>> {
		let id = self.nodes.borrow().len();
		let node_value = NodeValue::new(id, self);
		let observed = Box::new(Observed::<T, NonComparableValue<T>>::new(value));
		let node = RefCell::new(Nodes::ObservedAny(observed as Box<dyn Node>));
		self.nodes.borrow_mut().push(node);
		node_value
	}

	fn computed<T: PartialEq + 'static, F: Fn(&Connect) -> T + 'static>(
		&self,
		comp_fun: F,
	) -> NodeValue<Computed<T, ComparableValue<T>, F>> {
		let id = self.nodes.borrow().len();
		let node_value = NodeValue::new(id, self);
		let (value, dependencies) = self.get_computed_value(&comp_fun);
		let computed = Box::new(Computed::<T, ComparableValue<T>, F>::new(
			comp_fun,
			value,
			dependencies,
		));
		let node = RefCell::new(Nodes::Computed(computed as Box<dyn ComputedNode>));
		self.nodes.borrow_mut().push(node);
		self.add_computed_to_derived_lists(id);
		node_value
	}

	fn computed_any<T: 'static, F: Fn(&Connect) -> T + 'static>(
		&self,
		comp_fun: F,
	) -> NodeValue<Computed<T, NonComparableValue<T>, F>> {
		let id = self.nodes.borrow().len();
		let node_value = NodeValue::new(id, self);
		let (value, dependencies) = self.get_computed_value(&comp_fun);
		let computed = Box::new(Computed::<T, NonComparableValue<T>, F>::new(
			comp_fun,
			value,
			dependencies,
		));
		let node = RefCell::new(Nodes::ComputedAny(computed as Box<dyn ComputedNode>));
		self.nodes.borrow_mut().push(node);
		self.add_computed_to_derived_lists(id);
		node_value
	}

	fn effected<
		T: PartialEq + 'static,
		F: Fn(&Connect) -> T + 'static,
		E: Fn(&T) + 'static,
	>(
		&self,
		comp_fun: F,
		effect: E,
	) {
		let id = self.nodes.borrow().len();
		let (value, dependencies) = self.get_computed_value(&comp_fun);
		let effected = Box::new(Effected::<T, ComparableValue<T>, F, E>::new(
			comp_fun,
			value,
			effect,
			dependencies,
		));
		effected.run_effect();
		let node = RefCell::new(Nodes::Effected(effected as Box<dyn ComputedNode>));
		self.nodes.borrow_mut().push(node);
		self.add_computed_to_derived_lists(id);
	}

	fn effected_any<T: 'static, F: Fn(&Connect) -> T + 'static, E: Fn(&T) + 'static>(
		&self,
		comp_fun: F,
		effect: E,
	) {
		let id = self.nodes.borrow().len();
		let (value, dependencies) = self.get_computed_value(&comp_fun);
		let effected = Box::new(Effected::<T, NonComparableValue<T>, F, E>::new(
			comp_fun,
			value,
			effect,
			dependencies,
		));
		effected.run_effect();
		let node = RefCell::new(Nodes::EffectedAny(effected as Box<dyn ComputedNode>));
		self.nodes.borrow_mut().push(node);
		self.add_computed_to_derived_lists(id);
	}
}
