use crate::{
	connect_node_update::NodeUpdate, connect_structs::Connect, node_traits::Nodes,
};

pub(crate) trait ConnectDependencies {
	fn remove_computed_from_derived_lists(&self, computed_id: usize);
	fn add_computed_to_derived_lists(&self, computed_id: usize);
	fn recalculate_value_and_set_dependencies(
		&self,
		computed_id: usize,
		modified_node_ids: &mut Vec<usize>,
	);
	fn is_any_dependency_changed(
		&self,
		computed_id: usize,
		modified_node_ids: &[usize],
	) -> bool;
}

impl ConnectDependencies for Connect {
	fn remove_computed_from_derived_lists(&self, computed_id: usize) {
		if let Nodes::Computed(computed)
		| Nodes::ComputedAny(computed)
		| Nodes::Effected(computed)
		| Nodes::EffectedAny(computed) = &*self.nodes.borrow()[computed_id].borrow()
		{
			let dependency_ids = computed.get_dependencies();
			for dependency_id in dependency_ids {
				self.remove_from_derived(computed_id, dependency_id);
			}
		}
	}

	fn add_computed_to_derived_lists(&self, computed_id: usize) {
		if let Nodes::Computed(computed)
		| Nodes::ComputedAny(computed)
		| Nodes::Effected(computed)
		| Nodes::EffectedAny(computed) = &*self.nodes.borrow()[computed_id].borrow()
		{
			let dependency_ids = computed.get_dependencies();
			for dependency_id in dependency_ids {
				self.add_to_derived(computed_id, dependency_id);
			}
		}
	}

	fn recalculate_value_and_set_dependencies(
		&self,
		computed_id: usize,
		modified_node_ids: &mut Vec<usize>,
	) {
		if let Nodes::Computed(computed)
		| Nodes::ComputedAny(computed)
		| Nodes::Effected(computed)
		| Nodes::EffectedAny(computed) = &mut *self.nodes.borrow()[computed_id].borrow_mut()
		{
			let (is_changed, dependencies) =
				self.get_computed_value(|c| computed.recalculate(c));
			computed.run_effect();
			computed.set_dependencies(dependencies);
			if is_changed {
				modified_node_ids.push(computed_id);
			}
		}
	}

	fn is_any_dependency_changed(
		&self,
		computed_id: usize,
		modified_node_ids: &[usize],
	) -> bool {
		if let Nodes::Computed(computed)
		| Nodes::ComputedAny(computed)
		| Nodes::Effected(computed)
		| Nodes::EffectedAny(computed) = &*self.nodes.borrow()[computed_id].borrow()
		{
			for dependency_id in computed.get_dependencies() {
				if modified_node_ids.contains(&dependency_id) {
					return true;
				}
			}
		}
		false
	}
}
