// Copyright (c) 2015, 2016, 2017, 2018, 2020, 2021 Frank Fischer <frank-fischer@shadow-soft.de>
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see  <http://www.gnu.org/licenses/>
//

//! Maximum Network Flow algorithms.

use crate::num::traits::{Num, NumAssign};
use crate::traits::{GraphType, IndexDigraph, IndexGraph};
use crate::vec::EdgeVec;

/// Trait for max flow algorithms.
pub trait MaxFlow<'a> {
    /// Type of the underlying graph.
    type Graph: IndexGraph<'a>;

    /// Type of flows.
    type Flow: Num + Ord;

    /// Create a new maxflow algorithm instance for a graph.
    fn new(g: &'a Self::Graph) -> Self;

    /// Return the underlying graph.
    fn as_graph(&self) -> &'a Self::Graph;

    /// Return the value of the latest computed maximum flow.
    fn value(&self) -> Self::Flow;

    /// The flow of an Edge.
    fn flow(&self, a: <Self::Graph as GraphType<'a>>::Edge) -> Self::Flow;

    /// The flow as vector.
    fn flow_vec(&self) -> EdgeVec<'a, &'a Self::Graph, Self::Flow> {
        EdgeVec::new_with(self.as_graph(), |e| self.flow(e))
    }

    /// Solve the maxflow problem.
    ///
    /// The method solves the max flow problem from the source nodes
    /// `src` to the sink node `snk` with the given `upper` bounds on
    /// the edges.
    fn solve<Us>(
        &mut self,
        src: <Self::Graph as GraphType<'a>>::Node,
        snk: <Self::Graph as GraphType<'a>>::Node,
        upper: Us,
    ) where
        Us: Fn(<Self::Graph as GraphType<'a>>::Edge) -> Self::Flow;

    /// Return the mincut associated with the current flow.
    fn mincut(&self) -> Vec<<Self::Graph as GraphType<'a>>::Node>;
}

/// Solve the maxflow problem using the specified algorithm.
///
/// The method solves the max flow problem from the source nodes
/// `src` to the sink node `snk` with the given `upper` bounds on
/// the edges.
///
/// The function returns the flow value, the flow on each edge and the
/// minimal cut containing the source.
pub fn solve<'a, G, F, Us, M>(
    g: &'a G,
    src: G::Node,
    snk: G::Node,
    upper: Us,
) -> (F, EdgeVec<'a, &'a G, F>, Vec<G::Node>)
where
    G: IndexGraph<'a>,
    F: Num + Ord + Copy,
    Us: Fn(G::Edge) -> F,
    M: 'a + MaxFlow<'a, Graph = G, Flow = F>,
{
    let mut maxflow = M::new(g);
    maxflow.solve(src, snk, upper);
    (maxflow.value(), maxflow.flow_vec(), maxflow.mincut())
}

/// Solve the maxflow problem using the algorithm of Edmonds-Karp.
pub fn edmondskarp<'a, G, F, Us>(
    g: &'a G,
    src: G::Node,
    snk: G::Node,
    upper: Us,
) -> (F, EdgeVec<&'a G, F>, Vec<G::Node>)
where
    G: IndexDigraph<'a>,
    F: 'a + NumAssign + Ord + Copy,
    Us: Fn(G::Edge) -> F,
{
    solve::<G, F, Us, EdmondsKarp<G, F>>(g, src, snk, upper)
}

/// Solve the maxflow problem using Dinic' algorithm.
pub fn dinic<'a, G, F, Us>(g: &'a G, src: G::Node, snk: G::Node, upper: Us) -> (F, EdgeVec<'a, &'a G, F>, Vec<G::Node>)
where
    G: IndexDigraph<'a>,
    F: 'a + NumAssign + Ord + Copy,
    Us: Fn(G::Edge) -> F,
{
    solve::<G, F, Us, Dinic<G, F>>(g, src, snk, upper)
}

/// Solve the maxflow problem using the push-relabel algorithm.
pub fn pushrelabel<'a, G, F, Us>(
    g: &'a G,
    src: G::Node,
    snk: G::Node,
    upper: Us,
) -> (F, EdgeVec<'a, &'a G, F>, Vec<G::Node>)
where
    G: IndexDigraph<'a>,
    F: 'a + NumAssign + Ord + Copy,
    Us: Fn(G::Edge) -> F,
{
    solve::<G, F, Us, PushRelabel<G, F>>(g, src, snk, upper)
}

pub mod edmondskarp;
pub use self::edmondskarp::EdmondsKarp;

pub mod dinic;
pub use self::dinic::Dinic;

pub mod pushrelabel;
pub use self::pushrelabel::PushRelabel;
