/*
 * Copyright (c) 2017-2021 Frank Fischer <frank-fischer@shadow-soft.de>
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see  <http://www.gnu.org/licenses/>
 */

#![deprecated(since = "0.17.0", note = "use rs-graph-derive crate instead")]

//! Extend a graph with attributes.
//!
//! Sometimes one needs additional attributes associated with the nodes or edges of a graph.
//! The simplest way is to store them in a `NodeVec` or `EdgeVec`. If
//! these attributes are tightly related to the graph, it is convenient to store the graph
//! together with the attributes in a single struct, e.g.
//!
//! ```
//! use rs_graph::LinkedListGraph;
//!
//! struct MyGraph {
//!     graph: LinkedListGraph,
//!     balances: Vec<f64>,
//!     bounds: Vec<f64>,
//! }
//! ```
//!
//! The problem is that `MyGraph` itself does not implement the graph
//! traits `Graph`, `Digraph` and so on, and can therefore not used
//! directly for methods that need values of this type.
//!
//! The struct `Attributed` in this module can be used to automate the
//! tedious task of implementing the traits.
//!
//! For a different approach, see the `rs-graph-derive` crate.

use crate::builder::{Buildable, Builder};
use crate::traits::{Directed, GraphIterator, GraphSize, GraphType, IndexGraph, Undirected};

use std::marker::PhantomData;
use std::ops::{Deref, DerefMut};

/// An attributed graph.
///
/// This trait adds three attributes to a graph:
///
/// 1. attributes for the graph itself via `attr` and `attr_mut`.
/// 2. attributes for the nodes via `node` and `node_mut`.
/// 3. attributes for the edges via `node` and `node_mut`.
///
/// Furthermore, if you need to mutate some of the attributes, the
/// `split` method, splits the graph into two references: one for pure
/// read-only graph access and one for mutable access to the
/// attributes. This allows to mutate the attributes while ensuring
/// that the graph structure remains unchanged (and thus no nodes or
/// edges may be added.)
pub trait AttributedGraph<'a> {
    type Graph: GraphType<'a>;

    type Attributes: Attributes<
        Node = <Self::Graph as GraphType<'a>>::Node,
        Edge = <Self::Graph as GraphType<'a>>::Edge,
    >;

    /// Return a read-only graph reference and a mutable attributes reference.
    fn split(&'a mut self) -> (&Self::Graph, Self::Attributes);

    /// Return the graph attributes.
    fn attr(&'a self) -> &'a <Self::Attributes as Attributes>::GraphAttr;

    /// Return the graph attributes.
    fn attr_mut(&'a mut self) -> &'a mut <Self::Attributes as Attributes>::GraphAttr;

    /// Return the attributes of a node.
    fn node(&'a self, u: <Self::Graph as GraphType<'a>>::Node) -> &'a <Self::Attributes as Attributes>::NodeAttr;

    /// Return the attributes of a node.
    fn node_mut(
        &'a mut self,
        u: <Self::Graph as GraphType<'a>>::Node,
    ) -> &'a mut <Self::Attributes as Attributes>::NodeAttr;

    /// Return the attributes of an edge.
    fn edge(&'a self, e: <Self::Graph as GraphType<'a>>::Edge) -> &'a <Self::Attributes as Attributes>::EdgeAttr;

    /// Return the attributes of an edge.
    fn edge_mut(
        &'a mut self,
        e: <Self::Graph as GraphType<'a>>::Edge,
    ) -> &'a mut <Self::Attributes as Attributes>::EdgeAttr;
}

/// This trait provides (mutable) access to the attributes of an attributed graph.
///
/// A structure implementing this trait is returned as the second
/// value by the `split` method of an attributed graph.
pub trait Attributes {
    /// Type of nodes of the associated graph.
    type Node: Copy + Eq;
    /// Type of edges of the associated graph.
    type Edge: Copy + Eq;
    /// Type of graph attributes.
    type GraphAttr;
    /// Type of node attributes.
    type NodeAttr;
    /// Type of edge attributes.
    type EdgeAttr;

    /// Return the graph attributes.
    fn attr(&self) -> &Self::GraphAttr;

    /// Return the graph attributes.
    fn attr_mut(&mut self) -> &mut Self::GraphAttr;

    /// Return the attributes of a node.
    fn node(&self, u: Self::Node) -> &Self::NodeAttr;

    /// Return the attributes of a node.
    fn node_mut(&mut self, u: Self::Node) -> &mut Self::NodeAttr;

    /// Return the attributes of an edge.
    fn edge(&self, e: Self::Edge) -> &Self::EdgeAttr;

    /// Return the attributes of an edge.
    fn edge_mut(&mut self, e: Self::Edge) -> &mut Self::EdgeAttr;
}

/// Wrapper to attach attributes with a graph.
///
/// This is a wrapper struct that adds additional attributes to an
/// arbitrary graph. The type parameters are
///
/// - 'Gx` the type of graph attributes.
/// - 'Nx` the type of node attributes.
/// - 'Ex` the type of edge attributes.
/// - 'Ax` the type of biedge attributes.
///
/// The attributed graph implements `Graph`, `Digraph` and `Network` if the
/// the wrapped graph does.
///
/// # Example
///
/// ```
/// use rs_graph::{LinkedListGraph, classes, traits::*};
/// use rs_graph::attributed::{Attributes, Attributed, AttributedGraph};
///
/// #[derive(Default)]
/// struct NodeAttr {
///     balance: f64,
/// }
///
/// #[derive(Default)]
/// struct EdgeAttr {
///     flow: f64,
/// }
///
/// type MyGraph = Attributed<LinkedListGraph, (), NodeAttr, EdgeAttr>;
/// let mut g: MyGraph = classes::complete_bipartite(3,4);
///
/// {
///     let (g, mut attr) = g.split();
///     for u in g.nodes() {
///         attr.node_mut(u).balance = 42.0;
///     }
///     for e in g.edges() {
///         attr.edge_mut(e).flow = 1.5;
///     }
/// }
///
/// assert!(g.nodes().all(|u| g.node(u).balance == 42.0));
/// assert!(g.edges().all(|e| g.edge(e).flow == 1.5));
/// ```
pub struct Attributed<G, Gx = (), Nx = (), Ex = ()> {
    graph: G,
    attrs: AttributedAttrs<Gx, Nx, Ex>,
}

/// The default implementation of attributes.
///
/// The node, edge and biedge attributes are stored in vectors.
#[derive(Default)]
pub struct AttributedAttrs<Gx, Nx, Ex> {
    attr: Gx,
    nattrs: Vec<Nx>,
    eattrs: Vec<Ex>,
}

impl<Gx, Nx, Ex> AttributedAttrs<Gx, Nx, Ex>
where
    Gx: Default,
    Nx: Default,
    Ex: Default,
{
    fn with_size(n: usize, m: usize) -> Self {
        AttributedAttrs {
            attr: Gx::default(),
            nattrs: (0..n).map(|_| Default::default()).collect(),
            eattrs: (0..m).map(|_| Default::default()).collect(),
        }
    }
}

/// Accessor for graph attributes.
pub struct GraphAttrs<'a, G, Gx, Nx, Ex>
where
    G: 'a + GraphType<'a>,
    Gx: 'a,
    Nx: 'a,
    Ex: 'a,
{
    graph: &'a G,
    attrs: &'a mut AttributedAttrs<Gx, Nx, Ex>,
}

impl<'a, G, Gx, Nx, Ex> Attributes for GraphAttrs<'a, G, Gx, Nx, Ex>
where
    G: IndexGraph<'a>,
{
    type Node = G::Node;
    type Edge = G::Edge;
    type GraphAttr = Gx;
    type NodeAttr = Nx;
    type EdgeAttr = Ex;

    fn attr(&self) -> &Self::GraphAttr {
        &self.attrs.attr
    }

    fn attr_mut(&mut self) -> &mut Self::GraphAttr {
        &mut self.attrs.attr
    }

    fn node(&self, u: Self::Node) -> &Self::NodeAttr {
        &self.attrs.nattrs[self.graph.node_id(u)]
    }

    fn node_mut(&mut self, u: Self::Node) -> &mut Self::NodeAttr {
        &mut self.attrs.nattrs[self.graph.node_id(u)]
    }

    fn edge(&self, e: Self::Edge) -> &Self::EdgeAttr {
        &self.attrs.eattrs[self.graph.edge_id(e)]
    }

    fn edge_mut(&mut self, e: Self::Edge) -> &mut Self::EdgeAttr {
        &mut self.attrs.eattrs[self.graph.edge_id(e)]
    }
}

impl<'a, G, Gx, Nx, Ex> Deref for GraphAttrs<'a, G, Gx, Nx, Ex>
where
    G: IndexGraph<'a>,
{
    type Target = Gx;
    fn deref(&self) -> &Gx {
        &self.attrs.attr
    }
}

impl<'a, G, Gx, Nx, Ex> DerefMut for GraphAttrs<'a, G, Gx, Nx, Ex>
where
    G: IndexGraph<'a>,
{
    fn deref_mut(&mut self) -> &mut Gx {
        &mut self.attrs.attr
    }
}

/// Default builder for an attributes graph.
///
/// This builder requires the attributes to be default constructable.
/// All attributes will be initialized with these default values.
pub struct AttributedBuilder<B, Gx, Nx, Ex>(B, PhantomData<(Gx, Nx, Ex)>);

impl<'b, B, G, Gx, Nx, Ex> Builder for AttributedBuilder<B, Gx, Nx, Ex>
where
    B: Builder<Graph = G>,
    G: GraphSize<'b>,
    Gx: Default,
    Nx: Default,
    Ex: Default,
{
    type Graph = Attributed<G, Gx, Nx, Ex>;
    type Node = B::Node;
    type Edge = B::Edge;

    fn with_capacities(nnodes: usize, nedges: usize) -> Self {
        AttributedBuilder(B::with_capacities(nnodes, nedges), PhantomData)
    }

    fn reserve(&mut self, nnodes: usize, nedges: usize) {
        self.0.reserve(nnodes, nedges)
    }

    fn num_nodes(&self) -> usize {
        self.0.num_nodes()
    }

    fn num_edges(&self) -> usize {
        self.0.num_edges()
    }

    fn add_node(&mut self) -> Self::Node {
        self.0.add_node()
    }

    fn add_nodes(&mut self, n: usize) -> Vec<Self::Node> {
        self.0.add_nodes(n)
    }

    fn add_edge(&mut self, u: Self::Node, v: Self::Node) -> Self::Edge {
        self.0.add_edge(u, v)
    }

    fn node2id(&self, u: Self::Node) -> usize {
        self.0.node2id(u)
    }

    fn edge2id(&self, e: Self::Edge) -> usize {
        self.0.edge2id(e)
    }

    fn into_graph(self) -> Attributed<G, Gx, Nx, Ex> {
        let graph = self.0.into_graph();
        let attrs = AttributedAttrs::with_size(graph.num_nodes(), graph.num_edges());
        Attributed { graph, attrs }
    }
}

#[derive(Clone)]
pub struct AttributedWrapIt<I>(I);

impl<G, Gx, Nx, Ex, I> GraphIterator<Attributed<G, Gx, Nx, Ex>> for AttributedWrapIt<I>
where
    I: GraphIterator<G>,
{
    type Item = I::Item;

    fn next(&mut self, g: &Attributed<G, Gx, Nx, Ex>) -> Option<I::Item> {
        self.0.next(&g.graph)
    }

    fn size_hint(&self, g: &Attributed<G, Gx, Nx, Ex>) -> (usize, Option<usize>) {
        self.0.size_hint(&g.graph)
    }

    fn count(self, g: &Attributed<G, Gx, Nx, Ex>) -> usize {
        self.0.count(&g.graph)
    }
}

impl<'a, G, Gx, Nx, Ex> GraphType<'a> for Attributed<G, Gx, Nx, Ex>
where
    G: GraphType<'a>,
    Gx: Default,
    Nx: Default,
    Ex: Default,
{
    type Node = G::Node;
    type Edge = G::Edge;
}

impl<'a, G, Gx, Nx, Ex> GraphSize<'a> for Attributed<G, Gx, Nx, Ex>
where
    G: GraphSize<'a>,
    Gx: 'a + Default,
    Nx: 'a + Default,
    Ex: 'a + Default,
{
    type NodeIt = AttributedWrapIt<G::NodeIt>;

    type EdgeIt = AttributedWrapIt<G::EdgeIt>;

    fn num_nodes(&self) -> usize {
        self.graph.num_nodes()
    }

    fn num_edges(&self) -> usize {
        self.graph.num_edges()
    }

    fn nodes_iter(&'a self) -> Self::NodeIt {
        AttributedWrapIt(self.graph.nodes_iter())
    }

    fn edges_iter(&'a self) -> Self::EdgeIt {
        AttributedWrapIt(self.graph.edges_iter())
    }
}

impl<'a, G, Gx, Nx, Ex> Undirected<'a> for Attributed<G, Gx, Nx, Ex>
where
    G: Undirected<'a>,
    Gx: 'a + Default,
    Nx: 'a + Default,
    Ex: 'a + Default,
{
    type NeighIt = AttributedWrapIt<G::NeighIt>;

    fn enodes(&'a self, e: Self::Edge) -> (Self::Node, Self::Node) {
        self.graph.enodes(e)
    }

    fn neigh_iter(&'a self, u: Self::Node) -> Self::NeighIt {
        AttributedWrapIt(self.graph.neigh_iter(u))
    }
}

impl<'a, G, Gx, Nx, Ex> Directed<'a> for Attributed<G, Gx, Nx, Ex>
where
    G: Directed<'a>,
    Gx: 'a + Default,
    Nx: 'a + Default,
    Ex: 'a + Default,
{
    type OutIt = AttributedWrapIt<G::OutIt>;
    type InIt = AttributedWrapIt<G::InIt>;
    type IncidentIt = AttributedWrapIt<G::IncidentIt>;
    type DirectedEdge = G::DirectedEdge;

    fn src(&'a self, e: Self::Edge) -> Self::Node {
        self.graph.src(e)
    }

    fn snk(&'a self, e: Self::Edge) -> Self::Node {
        self.graph.snk(e)
    }

    fn out_iter(&'a self, u: Self::Node) -> Self::OutIt {
        AttributedWrapIt(self.graph.out_iter(u))
    }

    fn in_iter(&'a self, u: Self::Node) -> Self::InIt {
        AttributedWrapIt(self.graph.in_iter(u))
    }

    fn incident_iter(&'a self, u: Self::Node) -> Self::IncidentIt {
        AttributedWrapIt(self.graph.incident_iter(u))
    }
}

impl<'a, G, Gx, Nx, Ex> IndexGraph<'a> for Attributed<G, Gx, Nx, Ex>
where
    G: IndexGraph<'a>,
    Gx: 'a + Default,
    Nx: 'a + Default,
    Ex: 'a + Default,
{
    fn node_id(&self, u: Self::Node) -> usize {
        self.graph.node_id(u)
    }

    fn id2node(&'a self, id: usize) -> Self::Node {
        self.graph.id2node(id)
    }

    fn edge_id(&self, e: Self::Edge) -> usize {
        self.graph.edge_id(e)
    }

    fn id2edge(&'a self, id: usize) -> Self::Edge {
        self.graph.id2edge(id)
    }
}

impl<'a, G, Gx, Nx, Ex> AttributedGraph<'a> for Attributed<G, Gx, Nx, Ex>
where
    G: 'a + IndexGraph<'a>,
    Gx: 'a + Default,
    Nx: 'a + Default,
    Ex: 'a + Default,
{
    type Graph = G;
    type Attributes = GraphAttrs<'a, G, Gx, Nx, Ex>;

    fn split(&'a mut self) -> (&G, Self::Attributes) {
        (
            &self.graph,
            GraphAttrs {
                graph: &self.graph,
                attrs: &mut self.attrs,
            },
        )
    }

    fn attr(&self) -> &Gx {
        &self.attrs.attr
    }

    fn attr_mut(&mut self) -> &mut Gx {
        &mut self.attrs.attr
    }

    fn node(&self, u: G::Node) -> &Nx {
        &self.attrs.nattrs[self.graph.node_id(u)]
    }

    fn node_mut(&mut self, u: G::Node) -> &mut Nx {
        &mut self.attrs.nattrs[self.graph.node_id(u)]
    }

    fn edge(&self, e: G::Edge) -> &Ex {
        &self.attrs.eattrs[self.graph.edge_id(e)]
    }

    fn edge_mut(&mut self, e: G::Edge) -> &mut Ex {
        &mut self.attrs.eattrs[self.graph.edge_id(e)]
    }
}

impl<'a, G, Gx, Nx, Ex> Buildable for Attributed<G, Gx, Nx, Ex>
where
    G: GraphSize<'a> + Buildable,
    Gx: 'a + Default,
    Nx: 'a + Default,
    Ex: 'a + Default,
{
    type Builder = AttributedBuilder<G::Builder, Gx, Nx, Ex>;
}
