use serde::{de::DeserializeOwned, Serialize};

use crate::{
    throttle::{Throttle, ThrottleMechanism},
    RestRequest,
};

/// A configuration to simplify REST-API-requests.
///
/// # Examples
///
/// Simple [RestConfig].
///
/// ```rust
/// # use std::error::Error;
/// # use rrw::RestConfig;
/// #
/// # fn main() -> Result<(), Box<dyn Error>> {
/// let config = RestConfig::new("https://example.com/rest");
/// #
/// #     Ok(())
/// # }
/// ```
///
/// More Complicated [RestConfig] using [RestConfigBuilder].
///
/// ```rust
/// # use std::error::Error;
/// # use rrw::RestConfigBuilder;
/// # use rrw::throttle::StupidThrottle;
/// #
/// # fn main() -> Result<(), Box<dyn Error>> {
/// let config = RestConfigBuilder::new("https://example.com/rest")
///     .client(reqwest::Client::new())
///     .throttle(StupidThrottle::new(10.0))
///     .build();
/// #
/// #     Ok(())
/// # }
/// ```

pub struct RestConfig {
    endpoint: String,
    client: reqwest::Client,
    throttle: Option<Throttle>,
}

impl RestConfig {
    /// Construct a new, simple [RestConfig] using the endpoint to use.
    ///
    /// The client of the requests will be set to [reqwest::Client::new()], no throttling will be
    /// applied.
    ///
    /// For more sophisticated construction, look at [RestConfigBuilder].
    pub fn new<S: AsRef<str>>(endpoint: S) -> Self {
        log::trace!(
            "Create a new RestConfig with endpoint {}.",
            endpoint.as_ref()
        );
        Self {
            endpoint: endpoint.as_ref().to_string(),
            client: reqwest::Client::new(),
            throttle: None,
        }
    }

    /// Execute a [RestRequest] using the configuration.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use std::error::Error;
    /// # use rrw::RestConfig;
    /// # use rrw::throttle::StupidThrottle;
    /// # use rrw::RestRequest;
    /// # use serde::Deserialize;
    /// # use httpmock::prelude::*;
    /// #
    /// #[derive(Deserialize)]
    /// struct Json {
    ///     key: String
    /// }
    ///
    /// # #[tokio::main]
    /// # async fn main() -> Result<(), Box<dyn Error>> {
    /// # let server = MockServer::start();
    /// #
    /// # let mock = server.mock(|when, then| {
    /// #     when.method(GET).path("/test");
    /// #     then.status(200)
    /// #         .header("content-type", "application/json; charset=UTF-8")
    /// #         .body(r#"{"key": "value"}"#);
    /// # });
    /// #
    /// # let config = RestConfig::new(server.base_url());
    /// let result: Json = config
    ///     .execute(&RestRequest::<(), ()>::get("/test"))
    ///     .await?;
    /// #
    /// # mock.assert();
    /// #
    /// # Ok(())
    /// # }
    /// ```
    pub async fn execute<T: DeserializeOwned, Q: Serialize, B: Serialize>(
        &self,
        req: &RestRequest<Q, B>,
    ) -> Result<T, reqwest::Error> {
        // Throttle request.
        if let Some(throttle) = &self.throttle {
            throttle.throttle().await;
        }

        // Calculating full request path.
        let full_path = self.endpoint.clone() + &req.path;
        log::debug!("Executing a REST-Request to {:?}", full_path);

        // Building request.
        let request = self.client.request(req.method.clone(), full_path);
        let request = if let Some(query) = &req.query {
            log::trace!("Request got a query.");
            request.query(query)
        } else {
            request
        };
        let request = if let Some(body) = &req.body {
            log::trace!("Request got a body.");
            request.json(body)
        } else {
            request
        };

        // Sending request.
        log::trace!("Sending REST-Request.");
        let response = request.send().await?;

        // Converting request to json.
        log::trace!("Converting REST-Request to wanted result type.");
        Ok(response.json::<T>().await?)
    }
}

/// A builder for a [RestConfig].
pub struct RestConfigBuilder {
    config: RestConfig,
}

impl RestConfigBuilder {
    /// Construct a [RestConfigBuilder] with the given endpoint.
    ///
    /// The client of the requests will be set to [reqwest::Client::new()], no throttling will be
    /// applied.
    pub fn new<S: AsRef<str>>(endpoint: S) -> Self {
        Self {
            config: RestConfig::new(endpoint),
        }
    }

    /// Customize the [reqwest::Client] to use to make the requests.
    pub fn client(mut self, client: reqwest::Client) -> Self {
        self.config.client = client;
        self
    }

    /// Apply a [ThrottleMechanism] to throttle requests.
    pub fn throttle<T: 'static + ThrottleMechanism>(mut self, throttle: T) -> Self {
        self.config.throttle = Some(Throttle::new(throttle));
        self
    }

    /// Build the [RestConfig] from the [RestConfigBuilder].
    pub fn build(self) -> RestConfig {
        self.config
    }
}

#[cfg(test)]
mod test {
    use super::*;

    use httpmock::prelude::*;
    use serde::Deserialize;

    #[derive(Deserialize)]
    struct Empty {}

    #[derive(Debug, PartialEq, Eq, Serialize, Deserialize)]
    struct Json {
        key: String,
    }

    #[tokio::test]
    async fn config_simple_get() -> Result<(), Box<dyn std::error::Error>> {
        let server = MockServer::start();

        let mock = server.mock(|when, then| {
            when.method(GET).path("/test");
            then.status(200)
                .header("content-type", "application/json; charset=UTF-8")
                .body("{}");
        });

        let config = RestConfig::new(server.base_url());
        let _: Empty = config.execute(&RestRequest::<(), ()>::get("/test")).await?;

        mock.assert();

        Ok(())
    }

    #[tokio::test]
    async fn config_query_get() -> Result<(), Box<dyn std::error::Error>> {
        let server = MockServer::start();

        let mock = server.mock(|when, then| {
            when.method(GET).path("/test").query_param("key", "value");
            then.status(200)
                .header("content-type", "application/json; charset=UTF-8")
                .body("{}");
        });

        let config = RestConfig::new(server.base_url());
        let _: Empty = config
            .execute(&RestRequest::<&Json, ()>::get("/test").query(&Json {
                key: "value".to_string(),
            }))
            .await?;

        mock.assert();

        Ok(())
    }

    #[tokio::test]
    async fn config_simple_post() -> Result<(), Box<dyn std::error::Error>> {
        let server = MockServer::start();

        let mock = server.mock(|when, then| {
            when.method(POST).path("/test").body(r#"{"key":"value"}"#);
            then.status(200)
                .header("content-type", "application/json; charset=UTF-8")
                .body("{}");
        });

        let config = RestConfig::new(server.base_url());
        let _: Empty = config
            .execute(&RestRequest::<(), &Json>::post("/test").body(&Json {
                key: "value".to_string(),
            }))
            .await?;

        mock.assert();

        Ok(())
    }

    #[tokio::test]
    async fn config_simple_return() -> Result<(), Box<dyn std::error::Error>> {
        let server = MockServer::start();

        let mock = server.mock(|when, then| {
            when.method(GET).path("/test");
            then.status(200)
                .header("content-type", "application/json; charset=UTF-8")
                .body(r#"{"key":"value"}"#);
        });

        let config = RestConfig::new(server.base_url());
        let query: Json = config.execute(&RestRequest::<(), ()>::get("/test")).await?;

        mock.assert();
        assert_eq!(
            query,
            Json {
                key: "value".to_string()
            }
        );

        Ok(())
    }
}
