/*
 * Copyright (c) 2022 Contributors to the Rrise project
 */

use std::any::Any;
use std::ffi::c_void;

#[derive(Debug, Copy, Clone)]
pub struct Cookie {
    cookie: *mut c_void,
}

impl Default for Cookie {
    fn default() -> Self {
        Cookie {
            cookie: std::ptr::null_mut(),
        }
    }
}

// impl Drop for Cookie {
//     fn drop(&mut self) {
//         self.free();
//     }
// }

impl Cookie {
    pub fn new<T: Any>(value: Box<T>) -> Self {
        let mut c = Self::default();
        c.set_cookie(value);
        c
    }

    #[allow(dead_code)]
    pub(crate) fn from_addr<T>(address: *mut T) -> Self {
        let mut c = Self::default();
        c.cookie = address as *mut c_void;
        c
    }

    pub fn set_cookie<T: Any>(&mut self, value: Box<T>) {
        self.free();

        let raw = Box::into_raw(Box::new(value as Box<dyn Any>));
        self.cookie = raw as *mut c_void;
    }

    pub fn free(&mut self) {
        unsafe {
            let raw = self.cookie as *mut Box<dyn Any>;
            if !raw.is_null() {
                Box::from_raw(raw); // box takes ownership of the pointer, then frees it when it goes out of scope
            }
            self.cookie = std::ptr::null_mut();
        }
    }

    #[allow(dead_code)]
    pub(crate) fn get_addr(&self) -> *mut c_void {
        let c = Self::default();
        c.cookie
    }

    pub fn get<T: Any>(&self) -> Option<&T> {
        unsafe {
            let raw = self.cookie as *const Box<dyn Any>;
            if !raw.is_null() {
                let b: &Box<dyn Any> = &*raw;
                b.downcast_ref()
            } else {
                None
            }
        }
    }

    pub fn get_mut<T: Any>(&mut self) -> Option<&mut T> {
        unsafe {
            let raw = self.cookie as *mut Box<dyn Any>;
            if !raw.is_null() {
                let b: &mut Box<dyn Any> = &mut *raw;
                b.downcast_mut()
            } else {
                None
            }
        }
    }
}
