use rqrr::{PreparedImage, DeQRError};
use image;
use std::io::{Write, Error, ErrorKind};

struct BrokenWriter {}

impl Write for BrokenWriter {
    fn write(&mut self, buf: &[u8]) -> Result<usize, Error> {
        println!("Writer got this data: {:?} (decoded: {:?})", buf, String::from_utf8_lossy(buf));
        Err(Error::new(ErrorKind::PermissionDenied, "testing IoError"))
    }
    fn flush(&mut self) -> Result<(), Error> {
        println!("Writer was flushed");
        Err(Error::new(ErrorKind::PermissionDenied, "testing IoError"))
    }
}

#[test]
fn test_io_error() {
    let img = image::open("tests/data/errors/io_error.png").unwrap().to_luma8();

    let mut search_img = PreparedImage::prepare(img);
    let grids = search_img.detect_grids();

    assert_eq!(grids.len(), 1);

    let writer = BrokenWriter {};

    let result = grids[0].decode_to(writer);
    assert!(result.is_err());
    let err = result.unwrap_err();
    assert_eq!(err, DeQRError::IoError);

}

#[test]
fn test_invalid_version() {
    let img = image::open("tests/data/errors/invalid_version.gif").unwrap().to_luma8();

    let mut search_img = PreparedImage::prepare(img);
    let grids = search_img.detect_grids();
    assert_eq!(grids.len(), 1);

    let decoded = grids[0].decode();
    assert!(decoded.is_err());

    let err = decoded.unwrap_err();
    assert_eq!(err, DeQRError::InvalidVersion);
}

#[test]
fn test_format_ecc() {
    let img = image::open("tests/data/errors/format_ecc.png").unwrap().to_luma8();

    let mut search_img = PreparedImage::prepare(img);
    let grids = search_img.detect_grids();
    assert_eq!(grids.len(), 1);

    let decoded = grids[0].decode();
    println!("{:?}", decoded);
    assert!(decoded.is_err());

    let err = decoded.unwrap_err();
    assert_eq!(err, DeQRError::FormatEcc);
}

#[test]
fn test_data_ecc() {
    let img = image::open("tests/data/errors/data_ecc.png").unwrap().to_luma8();

    let mut search_img = PreparedImage::prepare(img);
    let grids = search_img.detect_grids();
    assert_eq!(grids.len(), 1);

    let decoded = grids[0].decode();
    println!("{:?}", decoded);
    assert!(decoded.is_err());

    let err = decoded.unwrap_err();
    assert_eq!(err, DeQRError::DataEcc);
}

// As of commit 956686877c964731559463dc645aa14e44e691b3, the following elements
// of the DeQRError enum are not used anywhere:
// - InvalidGridSize
// - EncodingError

// Also, these errors require grids to be manipulated at a deep level, which requires involved knowledge of the QR code structure:
// - DataUnderflow
// - DataOverflow
// - UnknownDataType
