use serde::{Deserialize, Serialize};

pub use account::*;
pub use balance::*;
pub use common::*;
pub use institutions::*;
pub use item::*;
pub use sandbox::*;
pub use token::*;
pub use transactions::*;

mod common {
    use super::*;

    #[derive(thiserror::Error, Debug, Serialize, Deserialize)]
    #[error("request failed with code {error_code:?}: {display_message:?}")]
    pub struct ErrorResponse {
        pub display_message: Option<String>,
        pub documentation: Option<String>,
        pub error_code: Option<String>,
        pub error_message: Option<String>,
        pub error_type: Option<String>,
        pub request_id: Option<String>,
        pub suggested_action: Option<String>,
    }
}

mod institutions {
    use super::*;

    #[derive(Debug, Serialize)]
    pub struct InstitutionsSearchRequest<'a, T: AsRef<str>> {
        pub query: T,
        pub products: Option<&'a [T]>,
        pub country_codes: &'a [T],
    }

    #[derive(Debug, Serialize, Deserialize)]
    pub struct InstitutionSearchResponse {
        pub institutions: Vec<Institution>,
    }

    #[derive(Debug, Serialize)]
    pub struct InstitutionGetRequest<'a, T: AsRef<str>> {
        pub institution_id: T,
        pub country_codes: &'a [T],
    }

    #[derive(Debug, Serialize, Deserialize)]
    pub struct InstitutionGetResponse {
        pub institution: Institution,
    }

    #[derive(Debug, Serialize)]
    pub struct InstitutionsGetRequest<'a, T: AsRef<str>> {
        pub count: usize,
        pub offset: usize,
        pub country_codes: &'a [T],
    }

    #[derive(Debug, Serialize, Deserialize)]
    pub struct InstitutionsGetResponse {
        pub institutions: Vec<Institution>,
    }

    #[derive(Debug, Serialize, Deserialize)]
    pub struct Institution {
        pub institution_id: String,
        pub name: String,
        pub products: Vec<String>,
        pub country_codes: Vec<String>,
        pub url: Option<String>,
        pub primary_color: Option<String>,
        pub logo: Option<String>,
        pub routing_numbers: Option<Vec<String>>,
        pub oauth: bool,
    }
}

mod sandbox {
    use super::*;

    #[derive(Debug, Serialize)]
    pub struct CreatePublicTokenRequest<'a, T: AsRef<str>> {
        pub institution_id: T,
        pub initial_products: &'a [T],
    }

    #[derive(Debug, Serialize, Deserialize)]
    pub struct CreatePublicTokenResponse {
        pub public_token: String,
    }

    #[derive(Debug, Serialize)]
    pub struct ResetLoginRequest<T: AsRef<str>> {
        pub access_token: T,
    }

    #[derive(Debug, Serialize, Deserialize)]
    pub struct ResetLoginResponse {
        pub reset_login: bool,
    }
}

mod token {
    use super::*;

    #[derive(Debug, Serialize)]
    pub struct ExchangePublicTokenRequest<T: AsRef<str>> {
        pub public_token: T,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct ExchangePublicTokenResponse {
        pub access_token: String,
        pub item_id: String,
        pub request_id: String,
    }

    #[derive(Debug, Serialize)]
    pub struct CreateLinkTokenRequest<'a, T: AsRef<str>> {
        pub client_name: T,
        pub language: T,
        pub country_codes: &'a [T],
        pub user: LinkUser<T>,
        pub products: &'a [T],
        pub webhook: Option<T>,
        pub access_token: Option<T>,
        pub link_customization_name: Option<T>,
        pub redirect_uri: Option<T>,
        pub android_package_name: Option<T>,
        // TODO(allancalix): Implement missing options.
        // pub account_filters: Option<&T>,
        // pub eu_config: Option<&T>,
        // payment_initiation: Option<_>,
        // deposit_switch: Option<_>,
        // income_verification: Option<_>,
        // auth: Option<_>,
        pub institution_id: Option<T>,
    }

    #[derive(Debug, Serialize, Default)]
    pub struct LinkUser<T: AsRef<str>> {
        pub client_user_id: T,
        pub legal_name: Option<T>,
        pub phone_number: Option<T>,
        pub phone_number_verified_time: Option<T>,
        pub email_address: Option<T>,
        pub email_address_verified_time: Option<T>,
        pub ssn: Option<T>,
        pub date_of_birth: Option<T>,
    }

    impl<T: AsRef<str> + Default> LinkUser<T> {
        pub fn new(user_id: T) -> Self {
            Self {
                client_user_id: user_id,
                ..Self::default()
            }
        }
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct CreateLinkTokenResponse {
        pub link_token: String,
        pub expiration: String,
        pub request_id: String,
    }
}

mod item {
    use super::*;

    #[derive(Debug, Serialize)]
    pub struct GetItemRequest<T: AsRef<str>> {
        pub access_token: T,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct GetItemResponse {
        pub item: Item,
        pub status: Option<Status>,
        pub request_id: String,
    }

    #[derive(Debug, Serialize)]
    pub struct RemoveItemRequest<T: AsRef<str>> {
        pub access_token: T,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct RemoveItemResponse {
        pub request_id: String,
    }

    #[derive(Debug, Serialize)]
    pub struct UpdateItemWebhookRequest<T: AsRef<str>> {
        pub access_token: T,
        /// The new url to associate with the item.
        pub webhook: T,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct UpdateItemWebhookResponse {
        pub item: Item,
        pub request_id: String,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct Item {
        pub item_id: String,
        pub institution_id: Option<String>,
        pub webhook: Option<String>,
        pub error: Option<ErrorResponse>,
        pub available_products: Vec<String>,
        pub billed_products: Vec<String>,
        // An RFC 3339 timestamp after which the consent provided by the end user will expire.
        pub consent_expiration_time: Option<String>,
        pub update_type: String,
        pub status: Option<Status>,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct Status {
        pub investments: Option<StatusMessage>,
        pub transactions: Option<StatusMessage>,
        pub last_webhook: Option<WebhookStatus>,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct StatusMessage {
        pub last_successful_update: Option<String>,
        pub last_failed_update: Option<String>,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct WebhookStatus {
        pub sent_at: Option<String>,
        pub code_sent: Option<String>,
    }
}

mod account {
    use super::*;

    #[derive(Debug, Serialize)]
    pub struct GetAccountsRequest<T: AsRef<str>> {
        pub access_token: T,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct GetAccountsResponse {
        pub accounts: Vec<Account>,
        pub item: Item,
        pub request_id: String,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct Account {
        pub account_id: String,
        pub balances: Balance,
        pub mask: Option<String>,
        pub name: String,
        pub official_name: Option<String>,
        /// One of investment | credit | depository | loan | brokerage | other.
        pub r#type: String,
        pub subtype: Option<String>,
        // This field is listed on the documentation for this type as non-nullable
        // but doesn't appear to be returned in payloads. TODO(allancalix): Find
        // out what the status is of this value.
        // https://plaid.com/docs/api/accounts/#accounts-get-response-verification-status_accounts
        // pub verification_status: String,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct Balance {
        pub available: Option<f64>,
        pub current: Option<f64>,
        pub iso_current_code: Option<String>,
        pub limit: Option<f64>,
        pub unofficial_currency_code: Option<String>,
    }
}

mod balance {
    use super::*;

    #[derive(Debug, Serialize)]
    pub struct AccountBalancesGetRequest<T: AsRef<str>> {
        pub access_token: T,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct AccountBalancesGetResponse {
        pub accounts: Vec<Account>,
        pub item: Item,
        pub request_id: String,
    }
}

mod identity {
    use super::*;

    #[derive(Debug, Serialize)]
    pub struct GetIdentityRequest<T: AsRef<str>> {
        pub access_token: T,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct GetIdentityResponse {
        pub accounts: Vec<Account>,
        pub item: Item,
        pub request_id: String,
    }
}

mod transactions {
    use super::*;

    #[derive(Debug, Serialize, Copy, Clone)]
    pub struct GetTransactionsRequest<T: AsRef<str>> {
        pub access_token: T,
        /// A string date with the format YYYY-MM-DD. Start date is inclusive.
        pub start_date: T,
        /// A string date with the format YYYY-MM-DD. End date is inclusive.
        pub end_date: T,
        pub options: Option<GetTransactionsOptions<T>>,
    }

    #[derive(Debug, Serialize, Copy, Clone)]
    pub struct GetTransactionsOptions<T: AsRef<str>> {
        pub account_ids: Option<T>,
        pub count: Option<usize>,
        pub offset: Option<usize>,
        pub include_original_description: Option<T>,
    }

    #[derive(Debug, Deserialize, Serialize)]
    pub struct GetTransactionsResponse {
        pub accounts: Vec<Account>,
        pub transactions: Vec<Transaction>,
        pub total_transactions: usize,
        pub item: Item,
        pub request_id: String,
    }

    #[derive(Debug, Deserialize, Serialize, Clone)]
    pub struct Transaction {
        /// DEPRECATED: do not depend on this type, it will be deleted in the future.
        pub transaction_type: String,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub pending_transaction_id: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub category_id: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub category: Option<Vec<String>>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub location: Option<TransactionLocation>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub payment_meta: Option<PaymentMetadata>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub account_owner: Option<String>,
        pub name: String,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub original_description: Option<String>,
        pub account_id: String,
        pub amount: f64,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub iso_currency_code: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub unofficial_currency_code: Option<String>,
        pub date: String,
        pub pending: bool,
        pub transaction_id: String,
        pub payment_channel: String,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub merchant_name: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub authorized_date: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub authorized_datetime: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub datetime: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub check_number: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub transaction_code: Option<String>,
    }

    #[derive(Debug, Deserialize, Serialize, Clone)]
    pub struct TransactionLocation {
        #[serde(skip_serializing_if = "Option::is_none")]
        pub address: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub city: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub region: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub postal_code: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub country: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub lat: Option<f64>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub lon: Option<f64>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub store_number: Option<String>,
    }

    #[derive(Debug, Deserialize, Serialize, Clone)]
    pub struct PaymentMetadata {
        #[serde(skip_serializing_if = "Option::is_none")]
        pub reference_number: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub ppd_id: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub payee: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub by_order_of: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub payer: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub payment_method: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub payment_processor: Option<String>,
        #[serde(skip_serializing_if = "Option::is_none")]
        pub reason: Option<String>,
    }
}
