use http_client::h1::H1Client;
use http_client::{Body, Error as HttpError, Request, Response};
use serde::{Deserialize, Serialize};
use thiserror::Error;

use crate::HttpClient;
use crate::model::*;

const SANDBOX_DOMAIN: &str = "https://sandbox.plaid.com";
const DEVELOPMENT_DOMAIN: &str = "https://development.plaid.com";
const PRODUCTION_DOMAIN: &str = "https://production.plaid.com";

#[derive(Error, Debug)]
pub enum ClientError {
    #[error("http request failed: {0}")]
    Http(HttpError),
    #[error(transparent)]
    Parse(#[from] serde_json::Error),
    #[error(transparent)]
    App(#[from] ErrorResponse),
}

#[derive(Debug, Default)]
pub struct Credentials {
    pub client_id: String,
    pub secret: String,
}

impl From<HttpError> for ClientError {
    fn from(error: HttpError) -> Self {
        Self::Http(error)
    }
}

#[derive(Debug, Serialize, Deserialize, Clone, PartialEq)]
pub enum Environment {
    Custom(String),
    Sandbox,
    Development,
    Production,
}

impl std::default::Default for Environment {
    fn default() -> Self {
        Environment::Sandbox
    }
}

impl std::string::ToString for Environment {
    fn to_string(&self) -> String {
        match self {
            Environment::Sandbox => SANDBOX_DOMAIN.into(),
            Environment::Development => DEVELOPMENT_DOMAIN.into(),
            Environment::Production => PRODUCTION_DOMAIN.into(),
            Environment::Custom(s) => s.into(),
        }
    }
}

pub struct Plaid<T: HttpClient> {
    http: T,
    credentials: Credentials,
    env: Environment,
}

pub struct Builder {
    http: Option<Box<dyn HttpClient>>,
    credentials: Option<Credentials>,
    env: Option<Environment>,
}

impl Default for Builder {
    fn default() -> Self {
        Self::new()
    }
}

impl Builder {
    pub fn new() -> Self {
        Self {
            http: None,
            credentials: None,
            env: None,
        }
    }

    pub fn with_http_client(mut self, client: impl HttpClient) -> Self {
        self.http = Some(Box::new(client));
        self
    }

    pub fn with_credentials(mut self, creds: Credentials) -> Self {
        self.credentials = Some(creds);
        self
    }

    pub fn with_env(mut self, env: Environment) -> Self {
        self.env = Some(env);
        self
    }

    pub fn build(self) -> Plaid<Box<dyn HttpClient>> {
        let http = self.http.unwrap_or_else(|| Box::new(H1Client::new()));
        Plaid {
            http,
            credentials: self.credentials.unwrap_or_default(),
            env: self.env.unwrap_or_default(),
        }
    }
}

impl<T: HttpClient> Plaid<T> {
    pub fn new(client: T, credentials: Credentials) -> Self {
        Self {
            http: client,
            credentials,
            env: Environment::Sandbox,
        }
    }

    async fn send(
        &self,
        path: &str,
        payload: &impl http_types::convert::Serialize,
    ) -> Result<Response, ClientError> {
        let mut req = Request::post(format!("{}{}", self.env.to_string(), path).as_str());
        req.insert_header("Content-Type", "application/json");
        req.insert_header("PLAID-CLIENT-ID", &self.credentials.client_id);
        req.insert_header("PLAID-SECRET", &self.credentials.secret);
        req.set_body(Body::from_json(&payload)?);
        let mut res = self.http.send(req).await?;

        match res.status() {
            http_client::http_types::StatusCode::Ok => Ok(res),
            _ => Err(ClientError::from(res.body_json::<ErrorResponse>().await?)),
        }
    }

    pub async fn search_institutions<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        req: &InstitutionsSearchRequest<'_, P>,
    ) -> Result<Vec<Institution>, ClientError> {
        let payload: InstitutionsGetResponse = self
            .send("/institutions/search", &req)
            .await?
            .body_json()
            .await?;
        Ok(payload.institutions)
    }

    pub async fn get_institution_by_id<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        req: &InstitutionGetRequest<'_, P>,
    ) -> Result<Institution, ClientError> {
        let payload: InstitutionGetResponse = self
            .send("/institutions/get_by_id", &req)
            .await?
            .body_json()
            .await?;
        Ok(payload.institution)
    }

    pub async fn get_institutions<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        req: &InstitutionsGetRequest<'_, P>,
    ) -> Result<Vec<Institution>, ClientError> {
        let payload: InstitutionsGetResponse = self
            .send("/institutions/get", &req)
            .await?
            .body_json()
            .await?;
        Ok(payload.institutions)
    }

    pub async fn create_public_token<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        req: CreatePublicTokenRequest<'_, P>,
    ) -> Result<String, ClientError> {
        let payload: CreatePublicTokenResponse = self
            .send("/sandbox/public_token/create", &req)
            .await?
            .body_json()
            .await?;

        Ok(payload.public_token)
    }

    pub async fn reset_login<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        access_token: P,
    ) -> Result<(), ClientError> {
        let payload: ResetLoginResponse = self
            .send(
                "/sandbox/item/reset_login",
                &ResetLoginRequest { access_token },
            )
            .await?
            .body_json()
            .await?;

        match payload.reset_login {
            true => Ok(()),
            false => Err(ClientError::App(ErrorResponse {
                error_message: Some("failed to reset login".into()),
                display_message: None,
                documentation: None,
                error_code: None,
                error_type: None,
                request_id: None,
                suggested_action: None,
            })),
        }
    }

    pub async fn exchange_public_token<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        public_token: P,
    ) -> Result<ExchangePublicTokenResponse, ClientError> {
        Ok(self
            .send(
                "/item/public_token/exchange",
                &ExchangePublicTokenRequest { public_token },
            )
            .await?
            .body_json()
            .await?)
    }

    pub async fn create_link_token<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        req: &CreateLinkTokenRequest<'_, P>,
    ) -> Result<CreateLinkTokenResponse, ClientError> {
        Ok(self
            .send("/link/token/create", req)
            .await?
            .body_json()
            .await?)
    }

    pub async fn accounts<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        access_token: P,
    ) -> Result<Vec<Account>, ClientError> {
        let res: GetAccountsResponse = self
            .send("/accounts/get", &GetAccountsRequest { access_token })
            .await?
            .body_json()
            .await?;

        Ok(res.accounts)
    }

    pub async fn item<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        access_token: P,
    ) -> Result<Item, ClientError> {
        let res: GetItemResponse = self
            .send("/item/get", &GetItemRequest { access_token })
            .await?
            .body_json()
            .await?;

        Ok(res.item)
    }

    pub async fn item_del<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        access_token: P,
    ) -> Result<(), ClientError> {
        self.send("/item/remove", &RemoveItemRequest { access_token })
            .await?
            .body_json::<RemoveItemResponse>()
            .await?;

        Ok(())
    }

    pub async fn item_webhook_update<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        access_token: P,
        webhook: P,
    ) -> Result<Item, ClientError> {
        let res: UpdateItemWebhookResponse = self
            .send(
                "/item/webhook/update",
                &UpdateItemWebhookRequest {
                    access_token,
                    webhook,
                },
            )
            .await?
            .body_json()
            .await?;

        Ok(res.item)
    }

    pub async fn balances<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        access_token: P,
    ) -> Result<Vec<Account>, ClientError> {
        let res: AccountBalancesGetResponse = self
            .send(
                "/accounts/balance/get",
                &AccountBalancesGetRequest { access_token },
            )
            .await?
            .body_json()
            .await?;

        Ok(res.accounts)
    }

    pub async fn transactions<P: AsRef<str> + http_types::convert::Serialize>(
        &self,
        req: &GetTransactionsRequest<P>,
    ) -> Result<GetTransactionsResponse, ClientError> {
        Ok(self
            .send("/transactions/get", &req)
            .await?
            .body_json::<GetTransactionsResponse>()
            .await?)
    }

    #[cfg(feature = "streams")]
    pub fn transactions_iter<'a, P: AsRef<str> + http_types::convert::Serialize + Clone + 'a>(
        &'a self,
        req: GetTransactionsRequest<P>,
    ) -> impl futures_core::stream::Stream<Item = Result<Vec<Transaction>, ClientError>> + 'a {
        async_stream::try_stream! {
            let mut yielded = 0;
            let mut total_xacts = None;
            let mut request = req.clone();
            let count = req.options.as_ref().unwrap().count.unwrap_or(100);
            let mut offset = req.options.as_ref().unwrap().offset.unwrap_or(0);

            while total_xacts.is_none() || total_xacts.unwrap() > yielded {
                if let Some(ref mut opts) = &mut request.options {
                    opts.count = Some(count);
                    opts.offset = Some(offset);
                } else {
                    request.options = Some(GetTransactionsOptions{
                        count: Some(count),
                        offset: Some(offset),
                        account_ids: None,
                        include_original_description: None,
                    });
                }

                let res = self.transactions(&request).await?;
                if total_xacts.is_none() {
                    total_xacts = Some(res.total_transactions - offset);
                }
                yielded += res.transactions.len();
                offset += yielded;

                yield res.transactions;
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use futures_util::pin_mut;
    use futures_util::StreamExt;

    const INSTITUTION_ID: &str = "ins_129571";

    fn credentials() -> Credentials {
        Credentials {
            client_id: std::env::var("PLAID_CLIENT_ID")
                .expect("Variable PLAID_CLIENT_ID must be defined."),
            secret: std::env::var("PLAID_SECRET").expect("Variable PLAID_SECRET must be defined."),
        }
    }

    #[tokio::test]
    async fn can_get_multiple_institutions() {
        let client = Builder::new().with_credentials(credentials()).build();
        let res = client
            .get_institutions(&InstitutionsGetRequest {
                count: 10,
                offset: 0,
                country_codes: &["US"],
            })
            .await
            .unwrap();

        insta::assert_json_snapshot!(res);
    }

    #[tokio::test]
    async fn can_fetch_single_institution() {
        let client = Builder::new().with_credentials(credentials()).build();
        let res = client
            .get_institution_by_id(&InstitutionGetRequest {
                institution_id: INSTITUTION_ID,
                country_codes: &[],
            })
            .await
            .unwrap();

        insta::assert_json_snapshot!(res);
    }

    #[tokio::test]
    async fn can_search_institutions() {
        let client = Builder::new().with_credentials(credentials()).build();
        let res = client
            .search_institutions(&InstitutionsSearchRequest {
                query: "Banque Populaire",
                products: None,
                country_codes: &[],
            })
            .await
            .unwrap();

        insta::assert_json_snapshot!(res);
    }

    #[tokio::test]
    async fn can_create_sandbox_pub_token() {
        let client = Builder::new().with_credentials(credentials()).build();
        let public_token = client
            .create_public_token(CreatePublicTokenRequest {
                institution_id: INSTITUTION_ID,
                initial_products: &["assets", "auth", "balance"],
            })
            .await
            .unwrap();

        let res = client.exchange_public_token(public_token).await.unwrap();
        assert!(!res.access_token.is_empty());
        // Should succeed.
        client.reset_login(res.access_token).await.unwrap();
    }

    #[tokio::test]
    async fn can_fetch_accounts_with_token() {
        let client = Builder::new().with_credentials(credentials()).build();
        let public_token = client
            .create_public_token(CreatePublicTokenRequest {
                institution_id: INSTITUTION_ID,
                initial_products: &["assets", "auth", "balance"],
            })
            .await
            .unwrap();

        let res = client.exchange_public_token(public_token).await.unwrap();
        assert!(!res.access_token.is_empty());
        let accounts = client.accounts(res.access_token).await.unwrap();

        insta::assert_json_snapshot!(accounts, {
            "[].account_id" => "[account_id]"
        });
    }

    #[tokio::test]
    async fn can_modify_items() {
        let client = Builder::new().with_credentials(credentials()).build();
        let public_token = client
            .create_public_token(CreatePublicTokenRequest {
                institution_id: INSTITUTION_ID,
                initial_products: &["assets", "auth", "balance"],
            })
            .await
            .unwrap();

        let res = client.exchange_public_token(public_token).await.unwrap();
        assert!(!res.access_token.is_empty());
        let item = client.item(&res.access_token).await.unwrap();

        insta::assert_json_snapshot!(item, {
            ".item_id" => "[item_id]"
        });

        // Should succeed.
        client.item_del(res.access_token).await.unwrap();
    }

    #[tokio::test]
    async fn can_create_link_token() {
        let client = Builder::new().with_credentials(credentials()).build();
        let res = client
            .create_link_token(&CreateLinkTokenRequest {
                client_name: "test_client",
                user: LinkUser::new("test-user"),
                language: "en",
                country_codes: &["US"],
                products: &["transactions"],
                webhook: None,
                access_token: None,
                link_customization_name: None,
                redirect_uri: None,
                android_package_name: None,
                institution_id: None,
            })
            .await
            .unwrap();

        assert!(!res.link_token.is_empty());
    }

    #[tokio::test]
    async fn can_read_transactions() {
        let client = Builder::new().with_credentials(credentials()).build();
        let public_token = client
            .create_public_token(CreatePublicTokenRequest {
                institution_id: INSTITUTION_ID,
                initial_products: &["assets", "auth", "balance", "transactions"],
            })
            .await
            .unwrap();

        let res = client.exchange_public_token(public_token).await.unwrap();
        assert!(!res.access_token.is_empty());
        // TODO(allancalix): Transaction isn't available immediately after the
        // token is created, we probably want to find a better way to find out if
        // the product is ready.
        tokio::time::sleep(std::time::Duration::from_millis(500)).await;
        let res = client
            .transactions(&GetTransactionsRequest {
                access_token: res.access_token.as_str(),
                start_date: "1999-01-01",
                end_date: "2021-09-03",
                options: None,
            })
            .await
            .unwrap();

        insta::assert_json_snapshot!(res.transactions, {
            "[].transaction_id" => "[transaction_id]",
            "[].account_id" => "[account_id]",
        });
    }

    #[tokio::test]
    async fn can_drain_transaction_stream() {
        let client = Builder::new().with_credentials(credentials()).build();
        let public_token = client
            .create_public_token(CreatePublicTokenRequest {
                institution_id: INSTITUTION_ID,
                initial_products: &["assets", "auth", "balance", "transactions"],
            })
            .await
            .unwrap();

        let res = client.exchange_public_token(public_token).await.unwrap();
        assert!(!res.access_token.is_empty());
        // TODO(allancalix): Transaction isn't available immediately after the
        // token is created, we probably want to find a better way to find out if
        // the product is ready.
        tokio::time::sleep(std::time::Duration::from_millis(500)).await;

        let req = GetTransactionsRequest {
            access_token: res.access_token.as_str(),
            start_date: "2019-09-06",
            end_date: "2021-09-06",
            options: Some(GetTransactionsOptions {
                count: Some(10),
                offset: Some(5),
                account_ids: None,
                include_original_description: None,
            }),
        };
        let iter = client.transactions_iter(req);
        pin_mut!(iter);

        let xacts = iter
            .fold(vec![], |mut acc, x| async move {
                acc.append(&mut x.unwrap());
                acc
            })
            .await;
        assert_eq!(xacts.len(), 11);
    }
}
