//! `rp1` is a [diesel] based CRUD for [rocket]. `rp1` is
//! intented as a quick starting point. It allows you to quickly scaffold a
//! basic application and should implement all basic functionality you may
//! expect from a basic Create-Read-Update-Delete cycle.
//!
//! Writing an application using `rp1` starts by defining your database
//! schema using [diesel]. Based on this schema and a model struct (which
//! in diesel are normally intended for querying) `rp1` will generate
//! some routes and handlers that you can directly plug into your rocket
//! application.
//!
//! To get started, read the details of the [crud] macro to see some basic
//! usage and an explanation of all parameters.

mod error;
mod filter;
mod sort;

pub mod helper;

pub mod access_control;

use ::rocket::serde::json::Json;

pub use access_control::*;
pub use error::*;
pub use filter::*;
pub use sort::*;

pub use rp1_macros::crud;

/// This trait is implemented on the main struct and indicates the type that
/// is the diesel table struct.
pub trait CrudStruct {
    type TableType;
}

/// This trait is implemented on the main struct and indicates the type that
/// is used for insert operations.
///
/// Note that if the `create` action is disabled this type will be implemented
/// with a unit (`()`) insert type.
pub trait CrudInsertable {
    type InsertType;
}

/// This trait is implemented on the main struct and indicates the type that
/// is used for update operations.
///
/// Note that if the `update` action is disabled this type will be implemented
/// with a unit (`()`) update type.
pub trait CrudUpdatable {
    type UpdateType;
}

/// This trait is implemented on the main struct and indicates the type that
/// is used for filtering on properties.
pub trait CrudFilterSpec {
    type FilterSpecType;
}

/// Return type for all handler functions generated by the [crud] macro.
pub type CrudResult<T, E = crate::CrudError> = Result<T, E>;

/// Return type for all handler functions that return json generated by the
/// [crud] macro.
pub type CrudJsonResult<T, E = crate::CrudError> = Result<Json<T>, E>;
