extern crate alloc;

use alloc::boxed::Box;
use alloc::collections::BTreeMap;
use chrono::{DateTime, NaiveDateTime, Utc};
use core::cell::RefCell;
use stm32f1xx_hal_bxcan::pac::{BKP, PWR, RTC};
use stm32f1xx_hal_bxcan::prelude::*;
use stm32f1xx_hal_bxcan::rcc::{APB1, BKP as RCC_BKP};
use stm32f1xx_hal_bxcan::rtc::Rtc;
use stm32f1xx_hal_bxcan::stm32::SYST;

use ross_config::config::Config;
use ross_config::event_processor::EventProcessor;
use ross_config::state_manager::StateManager;
use ross_config::Value;
use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error, Logger};
use ross_protocol::convert_packet::ConvertPacket;
use ross_protocol::event::internal::SystemTickEvent;

use crate::helper::type_helper::CanProtocol;
use crate::module::*;

pub struct ConfigModuleConfig<'a, 'b> {
    pub protocol: Rc<RefCell<CanProtocol<'a>>>,
    pub logger: &'a RefCell<Logger>,
    pub device_info: &'a DeviceInfo,
    pub syst: SYST,
    pub rtc: RTC,
    pub rcc_bkp: RCC_BKP,
    pub bkp: BKP,
    pub apb1: &'b mut APB1,
    pub pwr: &'b mut PWR,
    pub config: Rc<RefCell<Config>>,
}

pub struct ConfigModule<'a> {
    protocol: Rc<RefCell<CanProtocol<'a>>>,
    logger: &'a RefCell<Logger>,
    device_info: &'a DeviceInfo,
    syst: SYST,
    state_manager: Rc<RefCell<StateManager>>,
    rtc: Rtc,
}

impl<'a> ConfigModule<'a> {
    fn set_config(module: Rc<RefCell<Self>>, config: Rc<RefCell<Config>>) {
        while let Some(event_processor) = config.borrow_mut().event_processors.pop() {
            Self::add_event_processor(Rc::clone(&module), event_processor);
        }

        Self::set_initial_state(module, &mut config.borrow_mut().initial_state);
    }

    fn add_event_processor(module: Rc<RefCell<Self>>, mut event_processor: EventProcessor) {
        let protocol = Rc::clone(&module.borrow_mut().protocol);
        let logger = module.borrow_mut().logger;
        let device_info = module.borrow_mut().device_info;
        let state_manager = Rc::clone(&module.borrow_mut().state_manager);

        protocol
            .borrow_mut()
            .add_packet_handler(
                Box::new(move |packet, protocol| {
                    match event_processor
                        .matcher
                        .do_match(&packet, &mut state_manager.borrow_mut())
                    {
                        Ok(matches) if !matches => return,
                        Ok(_) => {}
                        Err(err) => {
                            log_error!(logger, "Matcher failed with error ({:?}).", err);
                            return;
                        }
                    }

                    for creator in event_processor.creators.iter_mut() {
                        let produced_packet = match creator.create(
                            &packet,
                            &mut state_manager.borrow_mut(),
                            device_info.device_address,
                        ) {
                            Ok(Some(packet)) => packet,
                            Ok(None) => continue,
                            Err(err) => {
                                log_error!(logger, "Creator failed with error ({:?}).", err);
                                return;
                            }
                        };

                        if let Err(err) = protocol.send_packet(&produced_packet) {
                            log_error!(logger, "Failed to send packet ({:?}).", err);
                        } else {
                            log_debug!(logger, "Sent packet ({:?}).", produced_packet);
                        }
                    }
                }),
                true,
            )
            .unwrap();
    }

    fn set_initial_state(module: Rc<RefCell<Self>>, initial_state: &mut BTreeMap<u32, Value>) {
        while let Some((id, value)) = initial_state.pop_first() {
            module
                .borrow_mut()
                .state_manager
                .borrow_mut()
                .set_value(id, value);
        }
    }
}

impl<'a> Module<'a> for ConfigModule<'a> {
    fn new<'b>(config: ModuleConfig<'a, 'b>) -> Rc<RefCell<Self>> {
        let config = match config {
            ModuleConfig::ConfigModule(config) => config,
            _ => {
                panic!("Wrong config provided for button module.");
            }
        };

        let protocol = config.protocol;
        let logger = config.logger;
        let device_info = config.device_info;
        let mut syst = config.syst;

        // Actually gets ticks per 1ms for some reason
        syst.set_reload(SYST::get_ticks_per_10ms());
        syst.clear_current();
        syst.enable_counter();

        let state_manager = Rc::new(RefCell::new(StateManager::new()));

        let mut rtc = Rtc::rtc(
            config.rtc,
            &mut config
                .rcc_bkp
                .constrain(config.bkp, config.apb1, config.pwr),
        );
        rtc.select_frequency(1.hz());

        log_debug!(logger, "Config module initialized.");

        let module = Rc::new(RefCell::new(Self {
            protocol,
            logger,
            device_info,
            syst,
            state_manager,
            rtc,
        }));

        ConfigModule::set_config(Rc::clone(&module), config.config);

        module
    }

    fn tick(module: Rc<RefCell<Self>>, current_time: &mut u32) {
        let borrowed_module: &mut Self = &mut module.borrow_mut();
        let syst = &mut borrowed_module.syst;
        let rtc = &borrowed_module.rtc;
        let state_manager = &borrowed_module.state_manager;
        let device_info = borrowed_module.device_info;
        let logger = borrowed_module.logger;
        let protocol = &mut borrowed_module.protocol;

        if syst.has_wrapped() {
            *current_time += 1;

            state_manager.borrow_mut().set_date_time(DateTime::from_utc(
                NaiveDateTime::from_timestamp(rtc.current_time().into(), 0),
                Utc,
            ));

            let system_tick_event = SystemTickEvent {
                receiver_address: device_info.device_address,
            };

            if let Err(err) = protocol
                .borrow_mut()
                .send_packet(&system_tick_event.to_packet())
            {
                log_error!(
                    logger,
                    "Failed to send `system_tick_event` with error ({:?}).",
                    err
                );
            }
        }
    }
}
