#![no_std]
#![no_main]
#![feature(alloc_error_handler)]

extern crate alloc;

use panic_itm as _;

use alloc::rc::Rc;
use alloc_cortex_m::CortexMHeap;
use alloc::boxed::Box;
use core::alloc::Layout;
use core::cell::RefCell;
use cortex_m_rt::entry;
use stm32f1xx_hal_bxcan::pac::{CorePeripherals, Peripherals};
use stm32f1xx_hal_bxcan::prelude::*;

use ross_firmware::config::*;
use ross_firmware::helper::can_helper::setup_can_protocol;
use ross_firmware::helper::eeprom_helper::setup_eeprom;
use ross_firmware::helper::heap_helper::allocate_heap;
use ross_firmware::module::relay_module::*;
use ross_firmware::module::hello_module::*;
use ross_firmware::module::config_module::*;
use ross_firmware::module::*;
use ross_logger::{log_info, log_warning, LogLevel, Logger};

#[global_allocator]
static ALLOCATOR: CortexMHeap = CortexMHeap::empty();

#[entry]
fn main() -> ! {
    let mut dp = Peripherals::take().unwrap();
    let cp = CorePeripherals::take().unwrap();

    let mut flash = dp.FLASH.constrain();
    let mut rcc = dp.RCC.constrain();

    let clocks = rcc
        .cfgr
        .use_hse(8.mhz())
        .sysclk(72.mhz())
        .hclk(72.mhz())
        .pclk1(36.mhz())
        .pclk2(72.mhz())
        .freeze(&mut flash.acr);

    let config = Config {
        heap_config: Some(HeapConfig { size: 4096 }),
        eeprom_config: Some(EepromConfig {
            bitrate: 400_000,
            device_info_address: 0,
        }),
        can_config: Some(CanConfig {
            bitrate: 50_000,
            tseg1: 13,
            tseg2: 2,
            sjw: 1,
        }),
        usart_config: None,
    };

    allocate_heap(&config.unwrap_heap_config(), &ALLOCATOR);

    let logger = RefCell::new(Logger::new(LogLevel::Debug, cp.ITM));

    log_info!(
        logger,
        "Firmware v{} initialized.",
        env!("CARGO_PKG_VERSION")
    );

    let mut gpioa = dp.GPIOA.split(&mut rcc.apb2);
    let mut gpiob = dp.GPIOB.split(&mut rcc.apb2);
    let mut afio = dp.AFIO.constrain(&mut rcc.apb2);

    let mut eeprom = setup_eeprom(
        config.unwrap_eeprom_config(),
        clocks,
        gpiob.pb6.into_alternate_open_drain(&mut gpiob.crl),
        gpiob.pb7.into_alternate_open_drain(&mut gpiob.crl),
        dp.I2C1,
        &mut rcc.apb1,
        &mut afio.mapr,
    );

    let device_info = eeprom.read_device_info().unwrap();

    log_info!(
        logger,
        "Loaded device information from EEPROM ({:?}).",
        device_info
    );

    let protocol = Rc::new(RefCell::new(setup_can_protocol(
        config.unwrap_can_config(),
        &device_info,
        clocks,
        gpioa.pa11.into_floating_input(&mut gpioa.crh),
        gpioa.pa12.into_alternate_push_pull(&mut gpioa.crh),
        dp.CAN1,
        dp.USB,
        &mut rcc.apb1,
        &mut afio.mapr,
    )));

    let config = Rc::new(RefCell::new(eeprom.read_config().unwrap()));

    // HELLO MODULE
    let hello_module_config = HelloModuleConfig {
        protocol: Rc::clone(&protocol),
        logger: &logger,
        device_info: &device_info,
    };
    let hello_module = HelloModule::new(ModuleConfig::HelloModule(hello_module_config));

    // RELAY MODULE
    let relay_module_config = RelayModuleConfig {
        protocol: Rc::clone(&protocol),
        logger: &logger,
        config: Rc::clone(&config),
    };

    let relay_module = RelayModule::new(ModuleConfig::RelayModule(relay_module_config));

    RelayModule::add_relay(
        Rc::clone(&relay_module),
        0,
        Box::new(gpioa.pa2.into_push_pull_output(&mut gpioa.crl)),
    );
    RelayModule::add_relay(
        Rc::clone(&relay_module),
        1,
        Box::new(gpioa.pa3.into_push_pull_output(&mut gpioa.crl)),
    );
    RelayModule::add_relay(
        Rc::clone(&relay_module),
        2,
        Box::new(gpioa.pa4.into_push_pull_output(&mut gpioa.crl)),
    );
    RelayModule::add_relay(
        Rc::clone(&relay_module),
        3,
        Box::new(gpioa.pa5.into_push_pull_output(&mut gpioa.crl)),
    );

    // CONFIG MODULE
    let config_module_config = ConfigModuleConfig {
        protocol: Rc::clone(&protocol),
        logger: &logger,
        device_info: &device_info,
        syst: cp.SYST,
        rtc: dp.RTC,
        rcc_bkp: rcc.bkp,
        bkp: dp.BKP,
        apb1: &mut rcc.apb1,
        pwr: &mut dp.PWR,
        config: Rc::clone(&config),
    };

    let config_module = ConfigModule::new(ModuleConfig::ConfigModule(config_module_config));

    let mut current_time = 0;

    loop {
        if let Err(err) = protocol.borrow_mut().tick() {
            log_warning!(logger, "Unexpected error occurred ({:?}).", err);
        }

        HelloModule::tick(Rc::clone(&hello_module), &mut current_time);
        RelayModule::tick(Rc::clone(&relay_module), &mut current_time);
        ConfigModule::tick(Rc::clone(&config_module), &mut current_time);
    }
}

#[alloc_error_handler]
fn oom(_: Layout) -> ! {
    loop {}
}
