#![no_std]
#![no_main]
#![feature(alloc_error_handler)]

extern crate alloc;

use panic_itm as _;

use alloc::rc::Rc;
use alloc_cortex_m::CortexMHeap;
use core::alloc::Layout;
use core::cell::RefCell;
use cortex_m_rt::entry;
use stm32f1xx_hal_bxcan::pac::{CorePeripherals, Peripherals};
use stm32f1xx_hal_bxcan::prelude::*;
use stm32f1xx_hal_bxcan::serial::{Config as SerialConfig, Serial};

use ross_firmware::config::*;
use ross_firmware::helper::can_helper::setup_can_protocol;
use ross_firmware::helper::eeprom_helper::setup_eeprom;
use ross_firmware::helper::heap_helper::allocate_heap;
use ross_firmware::module::programmer_module::*;
use ross_firmware::module::*;
use ross_logger::{log_info, log_warning, LogLevel, Logger};
use ross_protocol::interface::usart::Usart;
use ross_protocol::protocol::Protocol;

#[global_allocator]
static ALLOCATOR: CortexMHeap = CortexMHeap::empty();

#[entry]
fn main() -> ! {
    let dp = Peripherals::take().unwrap();
    let cp = CorePeripherals::take().unwrap();

    let mut flash = dp.FLASH.constrain();
    let mut rcc = dp.RCC.constrain();

    let clocks = rcc
        .cfgr
        .use_hse(8.mhz())
        .sysclk(72.mhz())
        .hclk(72.mhz())
        .pclk1(36.mhz())
        .pclk2(72.mhz())
        .freeze(&mut flash.acr);

    let config = Config {
        heap_config: Some(HeapConfig { size: 4096 }),
        eeprom_config: Some(EepromConfig {
            bitrate: 400_000,
            device_info_address: 0,
        }),
        can_config: Some(CanConfig {
            bitrate: 50_000,
            tseg1: 13,
            tseg2: 2,
            sjw: 1,
        }),
        usart_config: Some(UsartConfig { baudrate: 115200 }),
    };

    allocate_heap(&config.unwrap_heap_config(), &ALLOCATOR);

    let logger = RefCell::new(Logger::new(LogLevel::Debug, cp.ITM));

    log_info!(
        logger,
        "Firmware v{} initialized.",
        env!("CARGO_PKG_VERSION")
    );

    let mut gpioa = dp.GPIOA.split(&mut rcc.apb2);
    let mut gpiob = dp.GPIOB.split(&mut rcc.apb2);
    let mut afio = dp.AFIO.constrain(&mut rcc.apb2);

    let mut eeprom = setup_eeprom(
        config.unwrap_eeprom_config(),
        clocks,
        gpiob.pb6.into_alternate_open_drain(&mut gpiob.crl),
        gpiob.pb7.into_alternate_open_drain(&mut gpiob.crl),
        dp.I2C1,
        &mut rcc.apb1,
        &mut afio.mapr,
    );

    let device_info = eeprom.read_device_info().unwrap();

    log_info!(
        logger,
        "Loaded device information from EEPROM ({:?}).",
        device_info
    );

    let can_protocol = Rc::new(RefCell::new(setup_can_protocol(
        config.unwrap_can_config(),
        &device_info,
        clocks,
        gpioa.pa11.into_floating_input(&mut gpioa.crh),
        gpioa.pa12.into_alternate_push_pull(&mut gpioa.crh),
        dp.CAN1,
        dp.USB,
        &mut rcc.apb1,
        &mut afio.mapr,
    )));

    let usart_protocol = {
        let pin_tx = gpioa.pa9.into_alternate_push_pull(&mut gpioa.crh);
        let pin_rx = gpioa.pa10;

        let serial = Serial::usart1(
            dp.USART1,
            (pin_tx, pin_rx),
            &mut afio.mapr,
            SerialConfig::default().baudrate(config.unwrap_usart_config().baudrate.bps()),
            clocks,
            &mut rcc.apb2,
        );

        let usart = Usart::new(serial);
        Rc::new(RefCell::new(Protocol::new(
            device_info.device_address,
            usart,
        )))
    };

    let programmer_module_config = ProgrammerModuleConfig {
        can_protocol: Rc::clone(&can_protocol),
        usart_protocol: Rc::clone(&usart_protocol),
        logger: &logger,
        device_info: &device_info,
    };

    let programmer_module =
        ProgrammerModule::new(ModuleConfig::ProgrammerModule(programmer_module_config));

    let mut current_time = 0;

    loop {
        if let Err(err) = can_protocol.borrow_mut().tick() {
            log_warning!(logger, "Unexpected error occurred ({:?}).", err);
        }

        if let Err(err) = usart_protocol.borrow_mut().tick() {
            log_warning!(logger, "Unexpected error occurred ({:?}).", err);
        }

        ProgrammerModule::tick(Rc::clone(&programmer_module), &mut current_time);
    }
}

#[alloc_error_handler]
fn oom(_: Layout) -> ! {
    loop {}
}
