extern crate alloc;

use alloc::boxed::Box;
use alloc::rc::Rc;
use alloc::vec;

use core::cell::RefCell;
use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error, log_warning, Logger};
use ross_protocol::convert_packet::*;
use ross_protocol::event::configurator::ConfiguratorHelloEvent;
use ross_protocol::event::programmer::ProgrammerHelloEvent;
use ross_protocol::packet::Packet;
use ross_protocol::protocol::BROADCAST_ADDRESS;

use crate::helper::type_helper::{CanProtocol, UsartProtocol};
use crate::module::*;

pub struct ProgrammerModuleConfig<'a> {
    pub can_protocol: Rc<RefCell<CanProtocol<'a>>>,
    pub usart_protocol: Rc<RefCell<UsartProtocol<'a>>>,
    pub logger: &'a RefCell<Logger>,
    pub device_info: &'a DeviceInfo,
}

pub struct ProgrammerModule {}

impl<'a> Module<'a> for ProgrammerModule {
    fn new(config: ModuleConfig) -> Rc<RefCell<Self>> {
        let config = match config {
            ModuleConfig::ProgrammerModule(config) => config,
            _ => {
                panic!("Wrong config provided for programmer module.");
            }
        };

        let can_protocol = config.can_protocol;
        let usart_protocol = config.usart_protocol;
        let logger = config.logger;
        let device_info = config.device_info;

        let usart_protocol_clone = Rc::clone(&usart_protocol);
        can_protocol
            .borrow_mut()
            .add_packet_handler(
                Box::new(move |packet, _protocol| {
                    log_debug!(logger, "Received can packet ({:?}).", packet);

                    let mut packet_data = vec![];

                    for byte in &packet.data {
                        packet_data.push(*byte);
                    }

                    let usart_packet = Packet {
                        is_error: packet.is_error,
                        device_address: BROADCAST_ADDRESS,
                        data: packet_data,
                    };

                    if let Err(err) = usart_protocol_clone.borrow_mut().send_packet(&usart_packet) {
                        log_warning!(
                            logger,
                            "Failed to send usart packet with error ({:?}).",
                            err
                        );
                    } else {
                        log_debug!(logger, "Sent usart packet ({:?}).", usart_packet);
                    };
                }),
                false,
            )
            .unwrap();

        usart_protocol
            .borrow_mut()
            .add_packet_handler(
                Box::new(move |packet, protocol| {
                    log_debug!(logger, "Received usart packet ({:?}).", packet);

                    if let Ok(_) = ConfiguratorHelloEvent::try_from_packet(packet) {
                        let programmer_hello_event = ProgrammerHelloEvent {
                            programmer_address: device_info.device_address,
                        };

                        let event_packet = programmer_hello_event.to_packet();

                        if let Err(err) = protocol.send_packet(&event_packet) {
                            log_error!(logger, "Failed to send usart packet ({:?}).", err);
                        } else {
                            log_debug!(logger, "Sent usart packet ({:?}).", event_packet);
                        }
                    } else {
                        if let Err(err) = can_protocol.borrow_mut().send_packet(packet) {
                            log_warning!(
                                logger,
                                "Failed to send can packet with error ({:?}).",
                                err
                            );
                        } else {
                            log_debug!(logger, "Sent can packet ({:?}).", packet);
                        };
                    }
                }),
                true,
            )
            .unwrap();

        log_debug!(logger, "Programmer module initialized.");

        Rc::new(RefCell::new(Self {}))
    }

    fn tick(&mut self) {}
}
