extern crate alloc;

use alloc::boxed::Box;
use alloc::collections::BTreeMap;
use core::cell::RefCell;
use core::convert::Infallible;
use debouncr::{debounce_16, Debouncer, Edge, Repeat16};
use embedded_hal::digital::v2::InputPin;

use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error, Logger};
use ross_protocol::convert_packet::ConvertPacket;
use ross_protocol::event::button::{ButtonPressedEvent, ButtonReleasedEvent};

use crate::helper::type_helper::CanProtocol;
use crate::module::*;

pub struct ButtonModuleConfig<'a> {
    pub protocol: Rc<RefCell<CanProtocol<'a>>>,
    pub logger: &'a RefCell<Logger>,
    pub device_info: &'a DeviceInfo,
}

pub struct ButtonModule<'a> {
    buttons: BTreeMap<
        u8,
        (
            Box<dyn InputPin<Error = Infallible>>,
            Debouncer<u16, Repeat16>,
        ),
    >,
    protocol: Rc<RefCell<CanProtocol<'a>>>,
    logger: &'a RefCell<Logger>,
    device_info: &'a DeviceInfo,
}

impl<'a> ButtonModule<'a> {
    pub fn add_button(
        module: Rc<RefCell<Self>>,
        index: u8,
        input: Box<dyn InputPin<Error = Infallible>>,
    ) {
        let logger = module.borrow_mut().logger;
        let buttons = &mut module.borrow_mut().buttons;

        let debouncer = debounce_16(input.is_high().unwrap());

        log_debug!(
            logger,
            "Button ({}) has been added to the button module.",
            index
        );

        buttons.insert(index, (input, debouncer));
    }
}

impl<'a> Module<'a> for ButtonModule<'a> {
    fn new(config: ModuleConfig<'a>) -> Rc<RefCell<Self>> {
        let config = match config {
            ModuleConfig::ButtonModule(config) => config,
            _ => {
                panic!("Wrong config provided for button module.");
            }
        };

        let protocol = config.protocol;
        let logger = config.logger;
        let device_info = config.device_info;

        log_debug!(logger, "Button module initialized.");

        Rc::new(RefCell::new(Self {
            buttons: BTreeMap::new(),
            protocol,
            logger,
            device_info,
        }))
    }

    fn tick(&mut self) {
        for button in self.buttons.iter_mut() {
            let transition = button.1 .1.update(button.1 .0.is_high().unwrap());
            match transition {
                Some(Edge::Rising) => {
                    log_debug!(self.logger, "Button ({}) has been pressed.", button.0);

                    let button_pressed_event = ButtonPressedEvent {
                        receiver_address: self.device_info.device_address,
                        button_address: self.device_info.device_address,
                        index: *button.0,
                    };

                    if let Err(err) = self
                        .protocol
                        .borrow_mut()
                        .send_packet(&button_pressed_event.to_packet())
                    {
                        log_error!(
                            self.logger,
                            "Failed to send `button_pressed_event` ({:?}).",
                            err
                        );
                    } else {
                        log_debug!(
                            self.logger,
                            "Sent `button_pressed_event` ({:?}).",
                            button_pressed_event
                        );
                    }
                }
                Some(Edge::Falling) => {
                    log_debug!(self.logger, "Button ({}) has been released.", button.0);

                    let button_released_event = ButtonReleasedEvent {
                        receiver_address: self.device_info.device_address,
                        button_address: self.device_info.device_address,
                        index: *button.0,
                    };

                    if let Err(err) = self
                        .protocol
                        .borrow_mut()
                        .send_packet(&button_released_event.to_packet())
                    {
                        log_error!(
                            self.logger,
                            "Failed to send `button_released_event` ({:?}).",
                            err
                        );
                    } else {
                        log_debug!(
                            self.logger,
                            "Sent `button_released_event` ({:?}).",
                            button_released_event
                        );
                    }
                }
                None => {}
            };
        }
    }
}
