use bxcan::filter::Mask32;
use bxcan::Can as BxCan;
use nb::block;
use stm32f1xx_hal_bxcan::afio;
use stm32f1xx_hal_bxcan::can::Can as HalCan;
use stm32f1xx_hal_bxcan::pac::{CAN1 as PacCan, USB};
use stm32f1xx_hal_bxcan::rcc::{Clocks, APB1};

use ross_eeprom::DeviceInfo;
use ross_protocol::interface::can::Can;
use ross_protocol::protocol::Protocol;

use crate::config::CanConfig;
use crate::helper::type_helper::*;

pub fn setup_can_protocol<'a>(
    can_config: &CanConfig,
    device_info: &DeviceInfo,
    clocks: Clocks,
    rx: CanRxPin,
    tx: CanTxPin,
    can1: PacCan,
    usb: USB,
    apb1: &mut APB1,
    mapr: &mut afio::MAPR,
) -> CanProtocol<'a> {
    let mut can1 = {
        let can = HalCan::new(can1, apb1, usb);
        can.assign_pins((tx, rx), mapr);
        BxCan::new(can)
    };

    can1.configure(|c| {
        unsafe {
            // Enable non-automatic retransmission mode
            (*PacCan::ptr()).mcr.modify(|_r, w| w.nart().set_bit());
        }

        c.set_bit_timing(calc_can_btr(can_config, clocks.pclk1().0));
        c.set_loopback(false);
        c.set_silent(false);
    });

    let mut filters = can1.modify_filters();
    filters.enable_bank(0, Mask32::accept_all());
    drop(filters);

    block!(can1.enable()).unwrap();

    let can = Can::new(can1);
    Protocol::new(device_info.device_address, can)
}

fn calc_can_btr(config: &CanConfig, clock_rate: u32) -> u32 {
    let brp = clock_rate / config.bitrate / (config.tseg1 + config.tseg2);

    (brp - 1) | ((config.tseg1 - 1) << 16) | ((config.tseg2 - 1) << 20) | ((config.sjw - 1) << 24)
}
