extern crate alloc;

use alloc::boxed::Box;
use alloc::rc::Rc;
use core::cell::RefCell;

use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error, Logger};
use ross_protocol::convert_packet::*;
use ross_protocol::event::bootloader_event::BootloaderHelloEvent;
use ross_protocol::event::programmer_event::ProgrammerHelloEvent;

use crate::helper::type_helper::CanProtocol;
use crate::module::*;

pub struct HelloModuleConfig<'a> {
    pub protocol: Rc<RefCell<CanProtocol<'a>>>,
    pub logger: &'a RefCell<Logger>,
    pub device_info: &'a DeviceInfo,
}

pub struct HelloModule {}

impl<'a> Module<'a> for HelloModule {
    fn new(config: ModuleConfig) -> Rc<RefCell<Self>> {
        let config = match config {
            ModuleConfig::HelloModule(config) => config,
            _ => {
                panic!("Wrong config provided for hello module.");
            }
        };

        let protocol = config.protocol;
        let logger = config.logger;
        let device_info = config.device_info;

        protocol
            .borrow_mut()
            .add_packet_handler(
                Box::new(move |packet, protocol| {
                    if let Ok(programmer_hello_event) =
                        ProgrammerHelloEvent::try_from_packet(&packet)
                    {
                        log_debug!(
                            logger,
                            "Received `programmer_hello_event` ({:?}).",
                            programmer_hello_event
                        );
                        let bootloader_hello_event = BootloaderHelloEvent {
                            bootloader_address: device_info.device_address,
                            programmer_address: programmer_hello_event.programmer_address,
                        };

                        if let Err(err) = protocol.send_packet(&bootloader_hello_event.to_packet())
                        {
                            log_error!(
                                logger,
                                "Failed to send `bootloader_hello_event` ({:?}).",
                                err
                            );
                        } else {
                            log_debug!(
                                logger,
                                "Sent `bootloader_hello_event` ({:?}).",
                                bootloader_hello_event
                            );
                        }
                    }
                }),
                false,
            )
            .unwrap();

        log_debug!(logger, "Hello module initialized.");

        Rc::new(RefCell::new(Self {}))
    }

    fn tick(&mut self) {}
}
