extern crate alloc;

use alloc::boxed::Box;
use alloc::collections::BTreeMap;
use core::cell::RefCell;
use stm32f1xx_hal_bxcan::stm32::SYST;

use ross_config::config::Config;
use ross_config::event_processor::EventProcessor;
use ross_config::state::StateManager;
use ross_config::Value;
use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error, Logger};
use ross_protocol::convert_packet::ConvertPacket;
use ross_protocol::event::internal_event::SystemTickEvent;

use crate::helper::type_helper::CanProtocol;
use crate::module::*;

pub struct ConfigModuleConfig<'a> {
    pub protocol: Rc<RefCell<CanProtocol<'a>>>,
    pub logger: &'a RefCell<Logger>,
    pub device_info: &'a DeviceInfo,
    pub syst: SYST,
}

pub struct ConfigModule<'a> {
    protocol: Rc<RefCell<CanProtocol<'a>>>,
    logger: &'a RefCell<Logger>,
    device_info: &'a DeviceInfo,
    syst: SYST,
    state_manager: Rc<RefCell<StateManager>>,
}

impl<'a> ConfigModule<'a> {
    pub fn set_config(module: Rc<RefCell<Self>>, mut config: Config) {
        while let Some(event_processor) = config.event_processors.pop() {
            Self::add_event_processor(Rc::clone(&module), event_processor);
        }

        Self::set_initial_state(module, config.initial_state);
    }

    fn add_event_processor(module: Rc<RefCell<Self>>, mut event_processor: EventProcessor) {
        let protocol = Rc::clone(&module.borrow_mut().protocol);
        let logger = module.borrow_mut().logger;
        let device_info = module.borrow_mut().device_info;
        let state_manager = Rc::clone(&module.borrow_mut().state_manager);

        protocol
            .borrow_mut()
            .add_packet_handler(
                Box::new(move |packet, protocol| {
                    for matcher in event_processor.matchers.iter_mut() {
                        if !matcher.do_match(&packet, &mut state_manager.borrow_mut()) {
                            return;
                        }
                    }

                    let value = event_processor.extractor.extract(packet);
                    let packet = match event_processor.producer.produce(
                        &value,
                        &mut state_manager.borrow_mut(),
                        device_info.device_address,
                    ) {
                        Some(packet) => packet,
                        None => return,
                    };

                    if let Err(err) = protocol.send_packet(&packet) {
                        log_error!(logger, "Failed to send packet ({:?}).", err);
                    } else {
                        log_debug!(logger, "Sent packet ({:?}).", packet);
                    }
                }),
                true,
            )
            .unwrap();
    }

    fn set_initial_state(module: Rc<RefCell<Self>>, mut initial_state: BTreeMap<u32, Value>) {
        while let Some((id, value)) = initial_state.pop_first() {
            module
                .borrow_mut()
                .state_manager
                .borrow_mut()
                .set_value(id, value);
        }
    }
}

impl<'a> Module<'a> for ConfigModule<'a> {
    fn new(config: ModuleConfig<'a>) -> Rc<RefCell<Self>> {
        let config = match config {
            ModuleConfig::ConfigModule(config) => config,
            _ => {
                panic!("Wrong config provided for button module.");
            }
        };

        let protocol = config.protocol;
        let logger = config.logger;
        let device_info = config.device_info;
        let mut syst = config.syst;

        // Actually gets ticks per 1ms for some reason
        syst.set_reload(SYST::get_ticks_per_10ms());
        syst.clear_current();
        syst.enable_counter();

        let state_manager = Rc::new(RefCell::new(StateManager::new()));

        log_debug!(logger, "Config module initialized.");

        Rc::new(RefCell::new(Self {
            protocol,
            logger,
            device_info,
            syst,
            state_manager,
        }))
    }

    fn tick(&mut self) {
        if self.syst.has_wrapped() {
            let system_tick_event = SystemTickEvent {
                receiver_address: self.device_info.device_address,
            };

            if let Err(err) = self
                .protocol
                .borrow_mut()
                .send_packet(&system_tick_event.to_packet())
            {
                log_error!(
                    self.logger,
                    "Failed to send `system_tick_event` with error ({:?}).",
                    err
                );
            }
        }
    }
}
