#![no_std]
#![no_main]
#![feature(alloc_error_handler)]

extern crate alloc;

use panic_itm as _;

use alloc::rc::Rc;
use alloc_cortex_m::CortexMHeap;
use core::alloc::Layout;
use core::cell::RefCell;
use cortex_m_rt::entry;
use stm32f1xx_hal_bxcan::pac::{CorePeripherals, Peripherals};
use stm32f1xx_hal_bxcan::prelude::*;

use ross_firmware::config::*;
use ross_firmware::helper::can_helper::setup_can_protocol;
use ross_firmware::helper::eeprom_helper::setup_eeprom;
use ross_firmware::helper::heap_helper::allocate_heap;
use ross_firmware::module::bcm_module::*;
use ross_firmware::module::hello_module::*;
use ross_firmware::module::*;
use ross_logger::{log_info, log_warning, LogLevel, Logger};

#[global_allocator]
static ALLOCATOR: CortexMHeap = CortexMHeap::empty();

#[entry]
fn main() -> ! {
    let dp = Peripherals::take().unwrap();
    let cp = CorePeripherals::take().unwrap();

    let mut flash = dp.FLASH.constrain();
    let mut rcc = dp.RCC.constrain();

    let clocks = rcc
        .cfgr
        .use_hse(8.mhz())
        .sysclk(72.mhz())
        .hclk(72.mhz())
        .pclk1(36.mhz())
        .pclk2(72.mhz())
        .freeze(&mut flash.acr);

    let config = Config {
        heap_config: Some(HeapConfig { size: 4096 }),
        eeprom_config: Some(EepromConfig {
            bitrate: 400_000,
            device_info_address: 0,
        }),
        can_config: Some(CanConfig {
            bitrate: 50_000,
            tseg1: 13,
            tseg2: 2,
            sjw: 1,
        }),
        usart_config: None,
    };

    allocate_heap(&config.unwrap_heap_config(), &ALLOCATOR);

    let logger = RefCell::new(Logger::new(LogLevel::Info, cp.ITM));

    log_info!(
        logger,
        "Firmware v{} initialized.",
        env!("CARGO_PKG_VERSION")
    );

    let mut gpioa = dp.GPIOA.split(&mut rcc.apb2);
    let mut gpiob = dp.GPIOB.split(&mut rcc.apb2);
    let mut afio = dp.AFIO.constrain(&mut rcc.apb2);

    let mut eeprom = setup_eeprom(
        config.unwrap_eeprom_config(),
        clocks,
        gpiob.pb6.into_alternate_open_drain(&mut gpiob.crl),
        gpiob.pb7.into_alternate_open_drain(&mut gpiob.crl),
        dp.I2C1,
        &mut rcc.apb1,
        &mut afio.mapr,
    );

    let device_info = eeprom.read_device_info().unwrap();

    log_info!(
        logger,
        "Loaded device information from EEPROM ({:?}).",
        device_info
    );

    let protocol = Rc::new(RefCell::new(setup_can_protocol(
        config.unwrap_can_config(),
        &device_info,
        clocks,
        gpioa.pa11.into_floating_input(&mut gpioa.crh),
        gpioa.pa12.into_alternate_push_pull(&mut gpioa.crh),
        dp.CAN1,
        dp.USB,
        &mut rcc.apb1,
        &mut afio.mapr,
    )));

    // HELLO MODULE
    let hello_module_config = HelloModuleConfig {
        protocol: Rc::clone(&protocol),
        logger: &logger,
        device_info: &device_info,
    };
    let hello_module = HelloModule::new(ModuleConfig::HelloModule(hello_module_config));

    // BCM MODULE
    let bcm_module_config = BcmModuleConfig {
        protocol: Rc::clone(&protocol),
        logger: &logger,
        device_info: &device_info,
        clock_pin: gpiob.pb13.into_alternate_push_pull(&mut gpiob.crh),
        latch_pin: gpiob.pb14.into_push_pull_output(&mut gpiob.crh),
        data_pin: gpiob.pb15.into_alternate_push_pull(&mut gpiob.crh),
        spi2: dp.SPI2,
        clocks,
        apb1: &mut rcc.apb1,
        tim2: dp.TIM2,
    };

    let bcm_module = BcmModule::new(ModuleConfig::BcmModule(bcm_module_config));

    loop {
        if let Err(err) = protocol.borrow_mut().tick() {
            log_warning!(logger, "Unexpected error occurred ({:?}).", err);
        }

        hello_module.borrow_mut().tick();
        bcm_module.borrow_mut().tick();
    }
}

#[alloc_error_handler]
fn oom(_: Layout) -> ! {
    loop {}
}
