extern crate alloc;

use alloc::boxed::Box;
use alloc::format;
use alloc::vec;

use core::cell::UnsafeCell;
use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error, log_warning};
use ross_protocol::convert_packet::*;
use ross_protocol::event::configurator_event::*;
use ross_protocol::event::programmer_event::ProgrammerHelloEvent;
use ross_protocol::interface::Interface;
use ross_protocol::packet::Packet;
use ross_protocol::protocol::BROADCAST_ADDRESS;

use crate::helper::cell_helper::get_from_cell;
use crate::helper::type_helper::{UnsafeCanProtocol, UnsafeLogger, UnsafeUsartProtocol, UnsafeProgrammerModule};

pub struct ProgrammerModule {}

impl ProgrammerModule {
    pub fn new() -> UnsafeProgrammerModule {
        UnsafeCell::new(ProgrammerModule {})
    }

    pub fn init<'a>(
        _module: &'a UnsafeProgrammerModule,
        can_protocol: &'a UnsafeCanProtocol<'a>,
        usart_protocol: &'a UnsafeUsartProtocol<'a>,
        logger: &'a UnsafeLogger,
        device_info: &'a DeviceInfo,
    ) {
        get_from_cell(can_protocol)
            .add_packet_handler(
                Box::new(move |packet, _can| {
                    log_debug!(logger, "Received can packet ({:?}).", packet);

                    let mut packet_data = vec![];

                    for byte in &packet.data {
                        packet_data.push(*byte);
                    }

                    let usart_packet = Packet {
                        is_error: packet.is_error,
                        device_address: BROADCAST_ADDRESS,
                        data: packet_data,
                    };

                    if let Err(err) = get_from_cell(usart_protocol).send_packet(&usart_packet) {
                        log_warning!(
                            logger,
                            "Failed to send usart packet with error ({:?}).",
                            err
                        );
                    } else {
                        log_debug!(logger, "Sent usart packet ({:?}).", usart_packet);
                    };
                }),
                false,
            )
            .unwrap();

        get_from_cell(usart_protocol)
            .add_packet_handler(
                Box::new(move |packet, usart| {
                    log_debug!(logger, "Received usart packet ({:?}).", packet);

                    if let Ok(_) = ConfiguratorHelloEvent::try_from_packet(packet) {
                        let programmer_hello_event = ProgrammerHelloEvent {
                            programmer_address: device_info.device_address,
                            firmware_version: device_info.firmware_version,
                        };

                        let event_packet = programmer_hello_event.to_packet();

                        if let Err(err) = usart.try_send_packet(&event_packet) {
                            log_error!(logger, "Failed to send usart packet ({:?}).", err);
                        } else {
                            log_debug!(logger, "Sent usart packet ({:?}).", event_packet);
                        }
                    } else {
                        if let Err(err) = get_from_cell(can_protocol).send_packet(packet) {
                            log_warning!(
                                logger,
                                "Failed to send can packet with error ({:?}).",
                                err
                            );
                        } else {
                            log_debug!(logger, "Sent can packet ({:?}).", packet);
                        };
                    }
                }),
                true,
            )
            .unwrap();
            
        log_debug!(logger, "Programmer module initialized.");
    }

    pub fn tick(&mut self) {}
}
