extern crate alloc;

use alloc::boxed::Box;
use alloc::format;

use core::cell::UnsafeCell;
use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error};
use ross_protocol::convert_packet::*;
use ross_protocol::event::programmer_event::ProgrammerHelloEvent;
use ross_protocol::event::bootloader_event::BootloaderHelloEvent;
use ross_protocol::interface::Interface;

use crate::helper::cell_helper::get_from_cell;
use crate::helper::type_helper::{UnsafeCanProtocol, UnsafeLogger, UnsafeHelloModule};

pub struct HelloModule {}

impl HelloModule {
    pub fn new() -> UnsafeHelloModule {
        UnsafeCell::new(HelloModule {})
    }

    pub fn init<'a>(_module: &'a UnsafeHelloModule, protocol: &'a UnsafeCanProtocol<'a>, logger: &'a UnsafeLogger, device_info: &'a DeviceInfo) {
        get_from_cell(protocol).add_packet_handler(
            Box::new(move |packet, can| {
                if let Ok(programmer_hello_event) = ProgrammerHelloEvent::try_from_packet(&packet) {
                    log_debug!(logger, "Received `programmer_hello_event` ({:?}).", programmer_hello_event);
                    let bootloader_hello_event = BootloaderHelloEvent {
                        bootloader_address: device_info.device_address,
                        programmer_address: programmer_hello_event.programmer_address,
                        firmware_version: device_info.firmware_version,
                    };
                
                    if let Err(err) = can.try_send_packet(&bootloader_hello_event.to_packet()) {
                        log_error!(
                            logger,
                            "Failed to send `bootloader_hello_event` ({:?}).",
                            err
                        );
                    } else {
                        log_debug!(logger, "Sent `bootloader_hello_event` ({:?}).", bootloader_hello_event);
                    }
                }
            }),
            false,
        ).unwrap();
            
        log_debug!(logger, "Hello module initialized.");
    }

    pub fn tick(&mut self) {}
}
